/*
 *  Vload.c
 *  
 *  This is a bootloader for the V kernel on the Sun workstation.
 *  Its use is as follows:
 *	Boot the workstation.
 *	Use the n command to load and start the loader.
 *	  (or the d command if you don't have a network monitor)
 *	The loader prompts for the name of the file containing the
 *	  initial team.
 *	It then loads the kernel and the initial team, and jumps to
 *	  the kernel.  Both the team's and the kernel's b.out headers
 *	  are placed in known locations so the kernel can look at them.
 *      To load the kernel from other than its standard location, follow
 *	  the team name on the command line with a comma.  You will
 *	  then be prompted on the next line for the kernel filename.
 *	To cause the loader to break to the monitor before starting
 *	  the kernel, follow the kernel name on the command line with
 *	  a comma.
 *
 *  This same source also compiles to various "hardwired" versions.  If
 *    the symbol FIRST_TEAM is defined, its value is taken as the filename
 *    of a program to load as the first team, without prompting for the
 *    program name or kernel name.  If GRAPHIC_TEAM is defined, it is
 *    used instead of FIRST_TEAM if a framebuffer is present.  If 
 *    SMALL_GRAPHIC_TEAM is defined, it is used instead of GRAPHIC_TEAM
 *    if there is 1/4 megabyte or less of memory.  
 *
 *  If XWORLD is defined, the default kernel is XV_KERNEL rather than 
 *    V_KERNEL, and the default directory for teams is XVBOOT_DIR rather
 *    than VBOOT_DIR.
 *
 *  If FIRST_TEAM is NOT defined, we look at the configuration register
 *    to see if this copy of Vload was autobooted.  If the bootfile number
 *    is neither 0 nor 0xF, assume we were.  Look at the user-specified
 *    bits (1 and 0) to determine whether to load AUTOBOOT_TEAM0, 1, 2,
 *    or 3.  Teams 2 and 3 get the XV_KERNEL instead of the V_KERNEL.
 *    This doesn't work on SMI Suns, which have no config register, so in
 *    that case we don't try it.
 *
 * AUTHOR
 *      Tim Mann
 *
 * HACKED UPON BY
 *	Myles Cagney, Bill Nowicki, probably others
 *
 * HISTORY
 *	5.0: (TPM) Added AUTOBOOT_TEAM stuff.  Flushed old history.
 */

#include <b.out.h>
#include <pcmap.h>
#include <framebuf.h>
#include <confreg.h>

/* Default names */
#define V_KERNEL "Vkernel"
#define XV_KERNEL "xVkernel"
#define VBOOT_DIR  "/usr/sun/Vboot/"
#define XVBOOT_DIR "/usr/sun/xVboot/"

#define AUTOBOOT_TEAM0 "/usr/sun/Vboot/vgtsexec"
#define AUTOBOOT_TEAM1 "/usr/sun/Vboot/serverexec"
#define AUTOBOOT_TEAM2 "/usr/sun/xVboot/vgtsexec"
#define AUTOBOOT_TEAM3 "/usr/sun/xVboot/serverexec"

#ifdef XWORLD
#define DEFAULT_KERNEL XV_KERNEL
#define DEFAULT_DIR XVBOOT_DIR
#else
#define DEFAULT_KERNEL V_KERNEL
#define DEFAULT_DIR VBOOT_DIR
#endif XWORLD

/* Constants */
#define KERNSTK KERNLOC
#define KERNBHDR (struct bhdr *) (TEAMLOC- 2 * sizeof(struct bhdr))
    /* Kernel's b.out header */
#define KERNLOC 0x1000	/* kernel loads here */
#define TEAMBHDR (struct bhdr *) (TEAMLOC- sizeof(struct bhdr))
    /* Team's b.out header */
#define TEAMLOC 0x10000	/* initial team loads here */

#define TRUE 1
#define FALSE 0
#define NUL '\0'
#define disable asm("	orw	#/0700,sr")	/* Disable interrupts */
#define jump(addr) asm("	jmp	addr")	/* Jump to address */
#define setstack(expr) sp = (unsigned) (expr)

extern unsigned a0,a1,sp;	/* Hack to allow getting at registers */
extern char _end;
char *MemTop;			/* Top of memory */
static int (*mmloc)(); 		/* This points to the moved movemem() */
static int kernMaxSize;		/* Space available to load kernel */

main()
  {

    static char kfilename[101];
    static char tfilename[101];
    char *teamFile, *kernFile;
    struct bhdr *bheader;
    register unsigned tentry,kentry,montrap = 0;
				 /* Must be the only register variables */
    int movekern(), dummy(), i , j,	moversize;
    short c, version;
    struct ConfReg *config;
    char *moverloc;
    char *emt_getmemsize();

    MemTop = emt_getmemsize(); /* highest segment(s) in machine */
    moversize = (char *)dummy - (char *)movekern;
    kernMaxSize = ((char *)TEAMLOC-2*sizeof(struct bhdr)-moversize-&_end);

#ifdef FIRST_TEAM
    teamFile = FIRST_TEAM;
    kernFile = DEFAULT_KERNEL;

#ifdef GRAPHICS_TEAM
    if (ProbeAddress(GXDefaultBase) != 0)  /* if frame buffer exists */
      {
        teamFile = GRAPHICS_TEAM;

#ifdef SMALL_GRAPHICS_TEAM
        if ((int) emt_getmemsize() <= 256*1024 && !ProbeAddress(MBMEMLOWADR))
	    teamFile = SMALL_GRAPHICS_TEAM;
#endif SMALL_GRAPHICS_TEAM

      }
#endif GRAPHICS_TEAM

#else FIRST_TEAM
    /* Check if we need to autoboot */
    version = emt_version();
    c = emt_getconfig();
    config = (struct ConfReg *) &c;
    if (version != 101 && version != 0x200	/* if not an SMI */
	&& config->Bootfile != BOOT_TO_MON	/* and not booting monitor */
	&& config->Bootfile != BOOT_SELF_TEST)
      {
	/* Autoboot */
	switch (config->UserAssigned)
	  {
	  case 0:
	    teamFile = AUTOBOOT_TEAM0;
	    kernFile = V_KERNEL;
	    break;

	  case 1:
	    teamFile = AUTOBOOT_TEAM1;
	    kernFile = V_KERNEL;
	    break;

	  case 2:
	    teamFile = AUTOBOOT_TEAM2;
	    kernFile = XV_KERNEL;
	    break;

	  case 3:
	    teamFile = AUTOBOOT_TEAM3;
	    kernFile = XV_KERNEL;
	    break;
	  }
      }
    else
      {
	/* Sign on and prompt for program name */
#ifdef XWORLD
	printf("\n\nxV");
#else
	printf("\n\nV");
#endif XWORLD
	printf(" Kernel Loader - Version 5.0 - August 19, 1983\n\n");
	printf("Program name: ");
	scanf("%100s",tfilename);
	teamFile = tfilename;
	/* Check for comma at end of filename */
	if (tfilename[strlen(tfilename)-1] == ',')
	  {
	    /* If found, read another filename */
	    tfilename[strlen(tfilename)-1] = NUL;	/* delete comma */
	    printf("Kernel name: ");
	    scanf("%100s",kfilename);
	    kernFile = kfilename;
	    if (kfilename[strlen(kfilename)-1] == ',')
	      {
		kfilename[strlen(kfilename)-1] = NUL;
		montrap = 1;
	      }
	  }
	else
	  {
	    kernFile = DEFAULT_KERNEL;
	  }
      }
#endif FIRST_TEAM

    printf("Team filename = %s\nKernel filename = %s\n",teamFile,kernFile);

    /* Read in initial team */
    bheader = TEAMBHDR;
    loadfile(teamFile,bheader,TEAMLOC,MemTop-TEAMLOC,TRUE,TRUE,DEFAULT_DIR);
    tentry = bheader->entry;
    if (tentry < TEAMLOC)
      {
	printf("?Program starts too low in memory.\n");
	exit();
      }
	
    /* Read kernel just above Vload itself */
    bheader = KERNBHDR;
    loadfile(kernFile,bheader,&_end,TEAMLOC-KERNLOC,FALSE,FALSE,"");
    kentry = bheader->entry;
	
    /* Get ready to move kernel */
#ifdef DEBUG
    printf("Team entry point is %x;\
	    kernel entry point is %x.\n",tentry,kentry);
#endif

  /* This assumes that the movemem routine will compile into
   *  position-independent code.  This seems reasonable.
   * It also assumes the register declarations for tentry and kentry here
   *  will put them in the same register as the ones in movekern do.
   */

#ifdef DEBUG
    printf("About to move mover.\n");
#endif DEBUG
	
    moverloc = (char *)TEAMLOC-2*sizeof(struct bhdr)-moversize;
    movemem(movekern,moverloc,moversize);
    /* Find where movemem() has moved to */
    mmloc = (int (*)()) (moverloc + ((char *)movemem - (char *)movekern));
	
#ifdef DEBUG
    printf("Mover moved--jumping to it.\n");
#endif

    a0 = (unsigned)moverloc;
    jump(a0@);  /* Jump to moved code */

  }


movekern()	/* Not called--jumped to */
  {
    register unsigned tentry,kentry,montrap;
		
    disable /* interrupts */;

#ifdef DEBUG
    printf("About to move kernel, movemem is at %x now.\n",mmloc);
#endif

    /* Move the kernel to where it belongs */
    (*mmloc)(&_end,KERNLOC,kernMaxSize);
    setstack(KERNSTK);
    a0 = tentry;
    a1 = kentry;
    asm("	movl	a0,sp@-");
#ifndef FIRST_TEAM	
    if (montrap)
      {
	/* Take a monitor breakpoint trap */
	asm("	trap	#14");
      }
    else ;	/* Needed because of following asm */
#endif FIRST_TEAM
    /* Jump into it */
    asm("	jsr	a1@");

  }


movemem(from,to,bytes)
    char *from, *to;
    unsigned bytes;

  /* A very inefficient routine that moves an arbitrary number of
   *  bytes from one place to another.  It moves one byte at a
   *  time.
   */
  
  {

#ifdef DEBUG
    printf("In movemem().  From = %x, to = %x, bytes = %x.\n",
	from,to,bytes);
#endif

    for(;bytes>0;bytes--)
	*(to++) = *(from++);

  }

dummy() { }	/* Dummy function to help us find the end of movemem() */






