/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/


/*
 * FILE: window.c
 *
 * Based on a package for the Yale layout editor by Thomas R. Davis
 *
 * Major changes by Bill Nowicki August 1982
 *
 * This file contains routines pertinent to the managing of views
 * and graphical vgts.  Included are the various initialization
 * routines for the viewing package, for converting between world and screen
 * coordinates, routines for creating/deleting graphical vgts
 * and views, and routines for changing the magnification (zoom)
 * of a view.
 */


# include "Vgts.h"
# include "sdf.h"

/*
 * Slightly device dependent:
 * Mask to make grids come out right
 * on SUN workstations
 */
# define GRIDMASK 0xfff0
# define BannerLength 128	/* max size of banner */

short ScreenLimitX, ScreenLimitY;
				/* Frame buffer coordinates of limits */

extern short Debug;		/* print extra debugging information */

VGT VGTtable[MAX_VGTS];		/* table for virtual graphics terminals */
View ViewTable[MAX_Views];	/* state table for views */

short TtyVGT;

/*******************************************************************/


/*
 * InitViewPackage:
 * Initializes the window package.
 */

InitViewPackage()
  {
    register short i;

    InitDraw();

    for (i = 0; i < MAX_Views; i++)
        ViewTable[i].usage = UNUSED;
    ViewList = NULL;

    for (i = 0; i < MAX_VGTS; i++)
      {
	VGTtable[i].type = UNUSED;
	VGTtable[i].sdfPtr = NIL;
      }
    InitSDF();
    DrawBackground(0, 0, 1024, 1024); 	/* Provide a "grey" background. */
    Cursor();
  }


/*******************************************************************/



/*
 * FindTopView:
 * Finds the vgt which is currently on top, and
 * which contains (or nearly contains) the point (x, y).  -1 is
 * returned if an appropriate window is not found.
 *
 * The edgeFlag is true if we are looking for an edge instead of
 * the inside of a view.
 */

short FindTopView(x, y, edgeFlag)
  short x, y;
  BOOLEAN edgeFlag;
{
    short border;
    register View_PTR w = ViewList;
    
    border = edgeFlag ? -4 : 1;

    for (; w; w = w->next)
    	if (w->usage != UNUSED &&
	    x < w->ScreenXmax - border &&
	    x > w->ScreenXmin + border &&
	    y < w->ScreenYmax - border &&
	    y > w->ScreenYmin + border)
		return(w->ViewID);

    /* if we haven't returned yet, try looking near windows which are
     * on the top.
     */


    for ( w = ViewList; w; w = w->next)
	if (w->usage != UNUSED &&
	    x < w->ScreenXmax + MaxPixelMiss - border &&
	    x > w->ScreenXmin - MaxPixelMiss + border &&
	    y < w->ScreenYmax + MaxPixelMiss - border &&
	    y > w->ScreenYmin - MaxPixelMiss + border)
		return(w->ViewID);

    return(-1);			/* couldn't find it */
}


short FindTopVGT(x, y)
  short x, y;
 {
 	/*
	 * This is the client's callable routine --
	 * return the VGT index for the indicated coordinates
	 */
    register short i = FindTopView(x,y,FALSE);
    if (i==-1) return(-1);
    return(ViewTable[i].vgt);
 }



extern short MouseX, MouseY, MouseButtons;

short FindMouseClick( x, y, xp, yp )
    short x, y;
    short *xp, *yp;
  {
 	/*
	 * Converts the given screen coordinates to world
	 * coordinates, and returns them in the last two parameters.
	 *  Returns the window number of the hit.
	 * -1 means outside of a window.
	 */
    short window, round;
    register View_PTR w;

    window = FindTopView(x, y, FALSE);
    if (window<0)
	return(window);

    w = ViewTable + window;

    if (w->zoom > 0)	round = ( 1 << (w->zoom - 1) );
     else		round = 0;
    if (xp) 	
      {
        if (w->zoom>=0) *xp = (x - w->Xconst + round) >> w->zoom;
	else 		*xp = (x - w->Xconst) << - w->zoom;
      }

    if (yp) 	
      {
        if (w->zoom>=0) *yp = (w->Yconst - y + round) >> w->zoom;
	else 		*yp = (w->Yconst - y) << - w->zoom;
      }
    return(window);
}


short GetMouseClick( x, y, buttons)
  short *x, *y, *buttons;
  {
    register View_PTR w;
    short round, window, b;
	/*
	 * The client calls this to return information about a mouse event.
	 * Any non-null parameters return the appropriate values.
	 * NOTE: returned x and y are World coordinates!
	 */

    b = GetMouseInput();
    if (buttons)
      {
        *buttons = b;
      }

    window = FindMouseClick(MouseX, MouseY, x, y);

    if (window<0) 
	return(window);

    w = ViewTable + window;
    return(w->vgt);
  }


/*******************************************************************/


/*
 * CreateVGT:
 * Creates entry for another vgt.
 */

int CreateVGT(sdf, type, topSymbol, name)
    short sdf;
    int type;			/* Type of vgt */
    short topSymbol;		/* Top-level symbol */
    char *name;			/* human-readable name */
{
    DISPLAY_RECORD_PTR Find();
    register VGT *p = VGTtable;
    int i;

    if (sdf<0 || sdf>=NumberSDF) return(-1);
    for (i = 0; (i != MAX_VGTS) && 
    		(p->type != UNUSED); i++, p++);
    if (i == MAX_VGTS)
	return(-1);
    p->type = type;
    p->sdfPtr = topSymbol ? Find(sdf, topSymbol) : NULL;
    
    if (name)
      {
        strncpy( p->name, name, VGTnameLength );
        p->name[VGTnameLength] = 0;
      }
    else
      {
        strcpy(p->name," [Unknown]");
      }
    p->banner = "";
    if (Debug)
      {
        printf( "created vgt %s (%d) of type %d, topSymbol=%d\n", 
			p->name, i, type, topSymbol );
      }
    return(i);
  }


SetBanner(vgt,string)
   short vgt;
   char *string;
  {
    register VGT *p = VGTtable + vgt;
    register View *w;

    if ( p->banner && strlen(p->banner) ) free(p->banner);
    p->banner = (char *)malloc(strlen(string)+1);
    if (p->banner==NULL)
        p->banner = "";
     else
        strcpy( p->banner, string );

    for ( w = ViewList; w; w = w->next)
      if (w->usage != UNUSED && w->vgt == vgt)
        {
          SetupBanner(w);
	  RedrawBanner(w);
	}
  }




/*******************************************************************/


/*
 * DeleteVGT:
 * Deletes a vgt entry.
 * This routine should eventually free the storage associated with
 * a vgt, etc.
 */

DeleteVGT(vgtId,andViews)
    int vgtId;
    int andViews;		/* true means remove views too */
  {
    register View_PTR wp = ViewTable;
    register i;

    if (vgtId<0 || vgtId>=MAX_VGTS) return(-1);
    VGTtable[vgtId].type = UNUSED;
    VGTtable[vgtId].sdfPtr = NULL;
    if (!andViews) return(vgtId);
    if ( VGTtable[vgtId].banner && 
         strlen(VGTtable[vgtId].banner) ) free(VGTtable[vgtId].banner);

    for (i = 0; i < MAX_Views; i++, wp++)
      if (wp->usage != UNUSED && wp->vgt==vgtId)
        DeleteView(i);

    return(vgtId);
  }


DisplayItem(sdf, globalName, vgt )
  short sdf;
  short globalName;
  {
  	/*
	 * Set the currentd top-level item to be displayed
	 */
    register DISPLAY_RECORD_PTR dispRec;
    DISPLAY_RECORD_PTR Find();
    
    dispRec = Find(sdf, globalName);
    if (vgt<0 || vgt>=MAX_VGTS) return(-1);
    VGTtable[vgt].sdfPtr = dispRec;
    if (Debug)
      {
        printf( "Display Item %d in VGT %d\n", globalName, vgt );
      }
    RedrawVGT(vgt, -MaxCoord, MaxCoord, -MaxCoord, MaxCoord);
  }


/*******************************************************************/

/*
 * CreateView:
 * Creates a new view of a VGT.
 */

int CreateView(vgt, sXmin, sYmin, sXmax, sYmax, 
		WXmin, WYmin, zoom, showGrid)
  int vgt;
  short sXmin, sYmin, WXmin, WYmin, sXmax, sYmax, zoom;
  BOOLEAN showGrid;
{
    register View_PTR w = ViewTable;
    int i;

    if (vgt < 0 || vgt >= MAX_VGTS || 
    		   sXmax - sXmin < 2 || sYmax - sYmin < 2)
	return (-1);

    for (i = 0; (i != MAX_Views) && (w -> usage != UNUSED); w++, i++);
    if (i == MAX_Views) 
      {
	if (Debug)
	    printf ("No more free View descriptors!");
	return (-1);
      }

/*
 * At this point, i is a number of a view,
 * and w points to its descriptor. 
 */

    w -> vgt = vgt;
    w -> WorldXmin = WXmin;
    w -> WorldYmin = WYmin;
    w -> usage = VGTtable[vgt].type;
    w -> showGrid = showGrid;
    w -> ViewID = i;
    w -> banner = (char *) malloc(BannerLength);
    if (w->banner)
       SetupBanner(w);
    else w -> banner = VGTtable[vgt].name;
    w -> expansionDepth = MaxCoord;
    if ((w -> usage & ZOOMABLE) == 0) 
      {
	zoom = 0;
	showGrid = FALSE;
      }
    w -> next = ViewList;
    ViewList = w;
    Recalculate (w, zoom, sXmin, sXmax, sYmin, sYmax);
    RedrawEntireView (i);
    if (Debug)
	printf ("created view %d\n", i);

    return (i);
}


SetupBanner(v)
 register View *v;
  {    
   register char *s;

   if (v->banner==NULL || v->banner==VGTtable[v->vgt].name) return;
    
    if (v->vgt==TtyVGT)
      sprintf( v->banner, "%s    %s", 
                VGTtable[v->vgt].name,  
		VGTtable[v->vgt].banner );
    else if (v->usage & ZOOMABLE)
       sprintf( v->banner, "%.32s     Vgt %d    View %d    Zoom %d   %.50s", 
                VGTtable[v->vgt].name,
	 	v->vgt, v->ViewID,  v->zoom,
		VGTtable[v->vgt].banner );
    else
       sprintf( v->banner, "%.20s     %s Vgt %d      %.70s", 
                VGTtable[v->vgt].name,  
	 	v->usage & TTY ? "" : "Graphics", v->vgt,
		VGTtable[v->vgt].banner );

    if (Debug && strlen(v->banner) >= BannerLength)
      printf("strlen(v->banner)=%d\n",strlen(v->banner) );
    if (IsInput(v->vgt))
        for ( s = v->banner; *s;)
              *s++ = *s | 0200;
    }


/*******************************************************************/

/*
 * Obscures:
 * Returns TRUE if the proposed window (xMin, yMin, xMax, yMax)
 * obscures window w.
 */

static BOOLEAN Obscures(xMin, yMin, xMax, yMax, w)
short xMin, yMin, xMax, yMax;
  register View *w;
{
    if (w->ScreenXmin < xMax + 1 &&
        w->ScreenXmax > xMin - 1 &&
	w->ScreenYmin < yMax + 1 &&
	w->ScreenYmax > yMin - 1)
          {
	    return(TRUE);
          }
    return(FALSE);
}


/*
 * DeleteView:
 * Deletes a window. Internal use only.
 */

DeleteView(windowId)
short windowId;
{
    short xMin, yMin, xMax, yMax;
    register View_PTR w = ViewTable+windowId;  
    
    if (windowId<0 || windowId>=MAX_Views) return(-1);
    if (w->usage==UNUSED) return;

    xMin = w->ScreenXmin;
    xMax = w->ScreenXmax;
    yMin = w->ScreenYmin;
    yMax = w->ScreenYmax;
    RemoveView(w);
    SmashViews();
    w->usage = UNUSED;
    DisplayBackground(w);
    RedrawScreen(xMin,xMax,yMin,yMax,w->next);
    ReleaseSubViews(w);
 }


RedrawScreen(xMin,xMax,yMin,yMax,w)
 register View *w;
  {
    	/*
	 * Redraw all windows that we obscured
	 */
    if (w==NULL) w = ViewList;
    for (;w;w = w->next)
      if (Obscures(xMin,yMin,xMax,yMax,w))
	  RedrawEntireView(w->ViewID);

}


static RemoveView(w)
 View *w;
  {
    register View *v;
    if (w==ViewList)
      ViewList = w->next;
    for (v=ViewList;v;v=v->next)
      if (v->next==w)
        {
		/*
		 * remove the view from the view list
		 */
	  v->next = w->next;
	}

  }

static ReleaseSubViews(w)
 View *w;
  {
  	/*
	 * Return all the subviews for a given window
	 * to the free storage pool.
	 */
    register struct SubView *r1, *r2;
    
    for (r1=w->rect;r1;r1=r2)
      {
        r2=r1->next;
	free(r1);
      }
    w->rect = NULL;
  }



MakeTopXY(x,y)
 short x,y;
  {
  	/*
	 * Given screen coodinates we bring the selected view
	 * to the top of the list
	 */
    short view;
    
    if ( (view = FindTopView(x,y,FALSE))<0) return;
    if (ViewTable+view == ViewList) return;
    MakeTop(ViewTable+view);
    SmashViews();
    RedrawEntireView(view);
  }

MakeTop(w)
  View *w;
  {
    if (w==NULL) return;
    RemoveView(w);
    w->next = ViewList;
    ViewList = w;
  }


MakePadTop(vgt)
  short vgt;
  {
  	/*
	 * make the indicated pad on the top of the screen.
	 * Go through and find the first view of the pad.
	 * If it is already on top, return.
	 * Else, bring it to the top and redraw.
	 */
    short v;
    View *p = ViewTable;

    for (v=0;v<MAX_Views;v++,p++)
      if ( (p->vgt == vgt) && (p->usage != UNUSED) )
        {
	  if (p==ViewList) return;
	  MakeTop(p);
          SmashViews();
          RedrawEntireView(v);
	  return;
	}
  }

/*******************************************************************/


/*
 * ZoomIn:
 * Zooms the current window in by a factor of two and re-draws it.  This
 * routine puts the point (WXcenter, WYcenter) at the new center of the
 * window.
 */

ZoomIn(wXcenter, wYcenter, windowId)
  short wXcenter, wYcenter;
  short windowId;
{
    int xmin, xmax, ymin, ymax;
    BOOLEAN zoomFlag;
    register View_PTR w = ViewTable+windowId;

    if (windowId<0 || windowId>=MAX_Views) return(-1);
    if (w->zoom == 5)
        return;

    if ( (w->usage & ZOOMABLE) == 0 )
        return;

    xmin = wXcenter - (w->WorldXmax - w->WorldXmin)/4;
    xmax = wXcenter + (w->WorldXmax - w->WorldXmin)/4;
    ymin = wYcenter - (w->WorldYmax - w->WorldYmin)/4;
    ymax = wYcenter + (w->WorldYmax - w->WorldYmin)/4;

/* The following lines of code make sure that the window's world coordinates
 * are in some sense "reasonable".  The window views at least one pixel and
 * is not larger than allowable world coordinates.
 */

    zoomFlag = FALSE;
    if (xmin >= xmax)
      {
        xmax = xmin + 1;
        zoomFlag = TRUE;
      }
    if (ymin >= ymax)
      {
        ymax = ymin + 1;
        zoomFlag = TRUE;
      }
    if (!zoomFlag) w->zoom += 1;
    w->WorldXmin = xmin;
    w->WorldYmin = ymin;
    Recalculate(w, w->zoom, 
    		w->ScreenXmin, w->ScreenXmax, w->ScreenYmin, w->ScreenYmax);
    SetupBanner(w);
    RedrawEntireView(windowId);
}


/*******************************************************************/


/*
 * ZoomOut:
 * Zooms the current window out by a factor of two and re-draws it.  It puts
 * (wXcenter, wYcenter) at the middle of the new window in the same way that
 * ZoomIn does, but this is mostly for symmetry.  It would probably be fine
 * to zoom out only from the center of the window.  As in the ZoomIn routine,
 * "reasonableness" checks are done to make sure that the window stays a
 * reasonable size and that the zoom factor does not get out of hand.
 */

ZoomOut(wXcenter, wYcenter, windowId)
  short wXcenter, wYcenter;
  short windowId;
{
    register View_PTR w = ViewTable+windowId;
    int xmin, xmax, ymin, ymax;

    if (windowId<0 || windowId>=MAX_Views) return(-1);
    if ( (w->usage & ZOOMABLE) == 0 )
        return;

    xmin = wXcenter - (w->WorldXmax - w->WorldXmin);
    xmax = wXcenter + (w->WorldXmax - w->WorldXmin);
    ymin = wYcenter - (w->WorldYmax - w->WorldYmin);
    ymax = wYcenter + (w->WorldYmax - w->WorldYmin);

    xmax = min(xmax, MaxCoord);
    xmin = max(xmin, -MaxCoord);
    ymax = min(ymax, MaxCoord);
    ymin = max(ymin, -MaxCoord);

    if (xmax == MaxCoord || xmin == -MaxCoord || 
        ymax == MaxCoord || ymin == -MaxCoord)
        return(ZoomTo(wXcenter, wYcenter, windowId));

    w->WorldXmin = xmin;
    w->WorldYmin = ymin;
    Recalculate(w, w->zoom-1, 
    		w->ScreenXmin, w->ScreenXmax, w->ScreenYmin, w->ScreenYmax);
    SetupBanner(w);
    RedrawEntireView(windowId);
}


/*******************************************************************/


/*
 * ZoomTo:
 * Zooms the point (wXcenter, wYcenter) to the center of the current window,
 * but does not change the zoom factor of the screen.  This routine might just
 * as well be named "pan_to".  As before, "reasonableness" checks are done on
 * the final world coordinates of the window produced.
 */

ZoomTo(wXcenter, wYcenter, windowId)
  short wXcenter, wYcenter;
  short windowId;
{
    register View_PTR w = ViewTable+windowId;
    int xmin, xmax, ymin, ymax;

    xmin = wXcenter - (w->WorldXmax - w->WorldXmin)/2;
    xmax = wXcenter + (w->WorldXmax - w->WorldXmin)/2;
    ymin = wYcenter - (w->WorldYmax - w->WorldYmin)/2;
    ymax = wYcenter + (w->WorldYmax - w->WorldYmin)/2;

    if (xmax > MaxCoord)
      {
        xmax = MaxCoord;
        xmin = MaxCoord - (w->WorldXmax - w->WorldXmin);
      }
    if (xmin < -MaxCoord)
      {
        xmin = -MaxCoord;
        xmax = xmin + (w->WorldXmax - w->WorldXmin);
      }
    if (ymax > MaxCoord)
      {
        ymax = MaxCoord;
        ymin = MaxCoord - (w->WorldYmax - w->WorldYmin);
      }
    if (ymin < -MaxCoord)
      {
        ymin = -MaxCoord;
        ymax = ymin + (w->WorldYmax - w->WorldYmin);
      }
    w->WorldXmin = xmin;
    w->WorldYmin = ymin;
    Recalculate(w, w->zoom, 
    		w->ScreenXmin, w->ScreenXmax, w->ScreenYmin, w->ScreenYmax);
    RedrawEntireView(windowId);
}



Recalculate(w, zoom, sXmin, sXmax, sYmin, sYmax)
     register View_PTR w;
     short zoom, sXmin, sXmax, sYmin, sYmax;
  {
  	/*
	 * recalculate the world limits and constants for this window.
	 * Also checks to make sure zoomable windows lie on 16 bit boundaries,
	 * so the grid lines up.
	 */
    int modulus;		/* world mins mod modulus must be zero */

    if (w->usage & ZOOMABLE)
      {
        sXmax -= (sXmin & ~GRIDMASK);
        sYmin -= (sYmax & ~GRIDMASK);
        sXmin &= GRIDMASK;
	sYmax &= GRIDMASK;
      }
    w -> ScreenXmin = sXmin;
    w -> ScreenYmin = min(sYmin,sYmax-1);
    w -> ScreenXmax = max(sXmax,sXmin+1);
    w -> ScreenYmax = sYmax;
    w->zoom = zoom;
    
    if (w->usage & ZOOMABLE && zoom<4)
      {
      		/*
		 * here we fudge world coordinates to get the grid to line up
		 */
          modulus = 1 << (4-zoom);
	  w->WorldXmin -= w->WorldXmin % modulus;
	  w->WorldYmin -= w->WorldYmin % modulus;
      }
    
    if (zoom >= 0)
      {
        w->WorldXmax = w->WorldXmin + ((sXmax - sXmin)>>zoom);
        w->WorldYmax = w->WorldYmin + ((sYmax - sYmin)>>zoom);
        w->Xconst = w->ScreenXmin - (w->WorldXmin<<zoom);
        w->Yconst = w->ScreenYmax + (w->WorldYmin<<zoom);
      }
    else
      {
        w->WorldXmax = w->WorldXmin + ((sXmax - sXmin)<<(-zoom));
        w->WorldYmax = w->WorldYmin + ((sYmax - sYmin)<<(-zoom));
        w->Xconst = w->ScreenXmin - (w->WorldXmin>>(-zoom));
        w->Yconst = w->ScreenYmax + (w->WorldYmin>>(-zoom));
      }
    
    w->rect = NULL;
    SmashViews();
}


struct SubView *PushSubView(xmin,xmax,ymin,ymax,edges,w,bannerFlag)
  short xmin, xmax, ymin,ymax;	/* screen coordinates of subview */
  short edges;			/* set of edges that are borders */
  View *w;
  short bannerFlag;		/* TRUE if banner, false if object */
  {
  	/*
	 * Push a new rectangle onto the list, and
	 * return the new list.
  	*/
     register struct SubView *r, *list;

    r = (struct SubView *)malloc(sizeof(struct SubView));
    if (r==NULL) return(w->rect);
    list = w->rect;
    w->rect = r;
    r->ScreenXmin = xmin;
    r->ScreenXmax = xmax;
    r->ScreenYmin = ymin;
    r->ScreenYmax = ymax;
    r->EdgeSet = edges;
    r->next = list;
    r->bannerFlag = bannerFlag;
   /**  if (Debug)
      {
        register struct SubView *p;
	printf("Pushing a subviewport, window %d\n", w->ViewID );
	for (p=w->rect;p;p=p->next)
	  printf("  xmin=%d, xmax=%d, ymin=%d, ymax=%d\n",
	    p->ScreenXmin, p->ScreenXmax, p->ScreenYmin, p->ScreenYmax );
      }  **/
    return(r);
  }

SmashViews()
  {
      register View *w, *top;
      register struct SubView *r;
      struct SubView *temp;
      short rxmin, rxmax, rymin, rymax, edges;	/* save values for current subview */
    /*
     * Here we slice up each viewport into rectangular
     * non-overlapping regions.
     */
     
   for (w=ViewList;w;w=w->next)
    {
       ReleaseSubViews(w);
       if (w->ScreenXmin<ScreenLimitX-1 && w->ScreenXmax>1 &&
           w->ScreenYmin<ScreenLimitY-1 && w->ScreenYmax>1 )
        {
		/*
	 	 * first push the initial banner sub-view
		 */
	  r = PushSubView(max(w->ScreenXmin+2, 2), 
	  			min(w->ScreenXmax-2,ScreenLimitX-3),
          	       max(w->ScreenYmin+2, 2), 
		       		min(w->ScreenYmin+BannerHeight+1,
				       ScreenLimitY-3),
		     AllEdges, w, TRUE);
		/*
		 * then push the initial object sub-view
		 */
	  r = PushSubView(max(w->ScreenXmin+2, 2),
	  			min(w->ScreenXmax-2,ScreenLimitX-3),
          	       max(w->ScreenYmin+BannerHeight+4, 2), 
		       		min(w->ScreenYmax-2,ScreenLimitY-3),
		     AllEdges, w, FALSE);
	  for (top=ViewList;top && top!=w;top=top->next)
	   for (r=w->rect;r;)
	    {
	      /*
	       * At this stage, top points to a view that is on top
	       * of the SubView r.  If it overlaps, we must reduce
	       * r and try again.
	       */
	      if (r->ScreenXmin >= top->ScreenXmax ||
	          r->ScreenXmax <= top->ScreenXmin ||
		  r->ScreenYmin >= top->ScreenYmax ||
		  r->ScreenYmax <= top->ScreenYmin )
		   {
		     r=r->next;
		     continue;
		   }
	      rxmin = r->ScreenXmin;
	      rxmax = r->ScreenXmax;
	      rymin = r->ScreenYmin;
	      rymax = r->ScreenYmax;
	      edges = r->EdgeSet;

	      if (rymin < top->ScreenYmin)
	        {
		  /*
		   * r peeks out on the top.
		   * We try to slice up in the Y direction first
		   * so that scrolling looks OK.
		   */
		  r->ScreenYmin = top->ScreenYmin;
		  r->EdgeSet &= ~TopEdge;
		  r = PushSubView(rxmin,rxmax,rymin,top->ScreenYmin-1,
		  	edges&~BottomEdge,w,r->bannerFlag);
		  continue;
		}
	      if (rymax > top->ScreenYmax)
	        {
		  /*
		   * r peeks out on the bottom
		   */
		  r->ScreenYmax = top->ScreenYmax;
		  r->EdgeSet &= ~BottomEdge;
		  r = PushSubView(rxmin,rxmax,top->ScreenYmax+1,rymax,
		  	edges&~TopEdge,w,r->bannerFlag);
		  continue;
		}
	      if (rxmin < top->ScreenXmin)
	        {
		  /*
		   * r peeks out on the left
		   */
		  r->ScreenXmin = top->ScreenXmin;
		  r->EdgeSet &= ~LeftEdge;		  
		  r = PushSubView(rxmin,top->ScreenXmin-1,rymin,rymax,
		  	edges&~RightEdge,w,r->bannerFlag);
		  continue;
		}
	      if (rxmax > top->ScreenXmax)
	        {
		  /*
		   * r peeks out on the right
		   */
		  r->ScreenXmax = top->ScreenXmax;
		  r->EdgeSet &= ~RightEdge;
		  r = PushSubView(top->ScreenXmax+1,rxmax,rymin,rymax,
		  	edges&~LeftEdge,w,r->bannerFlag);
		  continue;
		}
		/*
		 * If we get here, the bottom SubView
		 * is completely obscured, so we remove it from the list.
		 */
	 	if (w->rect==r) 
		  {
		    w->rect=r->next;
		  }
		else
	         for (temp=w->rect;temp;temp=temp->next)
		  if (temp->next==r)
		    {
			temp->next = r->next;
			free(r);
			break;
		    }
		 r = w->rect;
	    }
        }
    }

  }

