/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* mux.c - Virtual Graphics Terminal Server Master Multiplexor
 *
 *
 * Bill Nowicki February 1983
 *
 * April 1983 (WIN)
 *	- Integrated mouse changes by Kenneth Brooks
 * 
 * Marvin Theimer, 7/83
 *	Added SET_BREAK_PROCESS case to the message handling.
 *	Changed SetBanner into a msg-sent request.
 *
 * Craig Dunwoody July 1983
 * 	Added initialization of lineEditBuf and call to InitEditor in
 *	InitVgts, added calls to CreateBuffer and DestroyBuffer in
 *	HandleCreate, HandleRelease, HandleModify, and SetPadMode.
 *
 * The process structure of the vgts consists of 4 processes: 
 *	the vgts process,
 * 	a mouse device helper process,
 *	a keyboard helper process.
 *	a timer helper process
 * The helper processes block on input from resp. the mouse
 * and the keyboard kernel calls and then send the input to the
 * vgts process.  The vgts process receives messages from both the
 * helper processes and from applications processes who desire 
 * virtual graphics terminal services.
 *
 */

# include <Venviron.h>
# include <Vioprotocol.h>
# include "Vgts.h"
# include "sdf.h"
# include "pad.h"
# include "client.h"

short Debug;		/* extra debugging information */
short Unique = 37;	/* my favorite random number */

extern short MouseX, MouseY, MouseButtons;

struct Event *FreeEvents;
ProcessId VgtsPid;	/* Well-known process id for the vgts process. */

/*
 * Number of clock ticks before declaring screen to be idle
 */
# define TicksPerMinute 240
static IdleLimit = 10*TicksPerMinute;
static IdleTime = 0;

# define DateAndTimeLength 24	/* size of VGTS banner */

/*
 * Process stack sizes.
 */
# define HelperSize 500

struct Buffer *CreateBuffer();

/*******************************************************************/


/*
 * InitHelperProcesses:
 * Starts up keyboard and mouse helper processes for Vgts.
 */

InitHelperProcesses()
  {
    Message msg;
    ProcessId pid;
    int MouseProcess(), KeyboardProcess(), TimerProcess();

    pid = Create( 5, KeyboardProcess, HelperSize );
    Reply(msg, pid);
    pid = Create( 6, MouseProcess, HelperSize);
    Reply(msg, pid);
    pid = Create( 100, TimerProcess, HelperSize);
    Reply(msg, pid);
  }


/*******************************************************************/

short InputPad;		/* the current input pad */
short TtyVGT;		/* VGT for view manager interaction */
short TtySDF;		/* SDF for Tty VGT */


InitVgts(xmin,ymin,xmax,ymax)
  {
/*
 * InitVgts:
 * Initializes the Vgts multiplexor.
 * Called with the coordinates of the first VGTS view.
 * This process must eventually be the one who calls GetMessage()
 */
    int i;
    register struct Client *client = ClientTable; 
    register struct Event *e = EventTable;
    
    VgtsPid = GetPid(VGT_SERVER,LOCAL_PID);
    if (VgtsPid && ValidPid(VgtsPid))
      {
        printf("You can only run one copy of the VGTS\r\n");
	exit();
      }

    VgtsPid = GetPid(ACTIVE_PROCESS,LOCAL_PID);
    SetPid(VGT_SERVER,VgtsPid,LOCAL_PID);
    InitHelperProcesses();

    for (i = 0; i < MaxClients; i++, client++)
      {
	client->pid = 0;
	client->owner = 0;
	client->termPid = 0;
	client->master = -1;
	client->exec = NULL;
	client->requestFlag = FALSE;
	client->eventQ = NULL;
	client->eventTail = NULL;
	client->lineEditBuf = NULL;
	client->mode = CR_Input+LF_Output+Echo+LineBuffer;
      }
 
    for (i = 0; i < MaxEvents; i++, e++)
      {
      		/*
		 * put all of the event descriptors onto the
		 * free event list
		 */
        e->next = FreeEvents;
	FreeEvents = e;
      }

    InitViewPackage();
    InitTtyVGT(xmin,ymin,xmax,ymax);
    ClientTable[TtyVGT].requestFlag = TRUE;
    ClientTable[TtyVGT].owner = VgtsPid;
  }


/*******************************************************************/


GetMessage(mouseFlag, keyboardFlag, msg)
  int mouseFlag;	/* true if we return on mouse input */
  int keyboardFlag;	/* true if we retun on keyboard input */
  register IoRequest *msg;	/* message that we return */
   {
	/*
	 * handle a request by dispatching to the correct handler
	 */
    ProcessId pid;
    int index;
    char c;

    while (1)
      {
	pid = Receive(msg);
	if (msg->requestcode != (SystemCode) Timer)
	  {
	    if (IdleTime > IdleLimit) ScreenEnable();
	    IdleTime = 0;
	  }
	index = msg->fileid;
	if (msg->requestcode != CREATE_INSTANCE)
	  {
	  	/*
		 * Do all the file id Validity checking here.
		 * after this point, we can index the Client table
		 * with impunity, since we know that the index is in range
		 */
	    if (index >= MaxClients) 
	      {
		msg->requestcode = INVALID_FILE_ID;
		Reply( msg, pid);
	  	return;
	      }
	  }
	switch (msg->requestcode)
	  {
	    case QUERY_INSTANCE:
		HandleQuery(msg,pid);
	        Reply(msg, pid);
	        break;

	    case CREATE_INSTANCE:
		HandleCreate(msg,pid);
	        Reply(msg, pid);
	        break;

	    case QUERY_FILE:
		HandleQueryFile(msg,pid);
	        Reply(msg, pid);
	        break;

	    case MODIFY_FILE:
		HandleModify(msg);
	        Reply(msg, pid);
	        break;

	    case SET_BREAK_PROCESS:
		HandleSetBreak(msg);
		Reply(msg,pid);
		break;

	    case SET_INSTANCE_OWNER:
		HandleSetOwner(msg);
		Reply(msg,pid);
		break;

	    case SetBannerRequest:
		HandleBanner(msg, pid);
		Reply(msg,pid);
		break;

	    case RELEASE_INSTANCE:
		HandleRelease(msg);
	        Reply(msg, pid);
	        break;

	    case WRITE_INSTANCE:
	        HandleWrite(msg, pid);
	        Reply(msg, pid);
	        break;

	    case WRITESHORT_INSTANCE:
	        HandleShortWrite(msg);
		Reply(msg,pid);
	        break;

	    case READ_INSTANCE:
	        if (index == DirectoryInstance)
		  HandleDirectoryRead(msg,pid);
		else HandleRead(ClientTable+index,pid);
		break;

	    case EventRequest:
		HandleEventRequest(ClientTable+index, msg, pid);
		break;

	    case LineEditRequest:	 
		HandleShoveInput(ClientTable+index, msg, pid);
		break;

	    case RedrawRequest:
		if (ClientTable[index].mode & NoCursor)  PadCursorOff(index);
		  else PadCursorOn(index);
		if (!mouseFlag) PadRedraw(index);
		Reply( msg, pid);
		break;

	    case SwitchInput:
		Reply( msg, pid );
		SelectForInput(index);
		break;

	    case Keyboard:
		Reply(msg, pid);
		if (keyboardFlag) return(pid);
		c = (char) msg->shortbuffer[0];
		UserCook(ClientTable+InputPad,c);
		break;

	    case GraphicsHelperMsg:
		Reply(msg, pid);
		if (mouseFlag) return(pid);
	        HandleGraphics(msg);
		break;
		
	    case Timer:
		HandleTimer(mouseFlag);
		Reply(msg, pid);
		break;

	    default:
		if (Debug) printf("Bad request: 0x%x, pid 0x%x\n", 
			msg->requestcode, pid);
		msg->requestcode = ILLEGAL_REQUEST;
		Reply(msg, pid);
		break;
	  }
      }
  }


/*******************************************************************/


static HandleTimer(mouseFlag)
  {
  	/*
	 * handle clean-up operations for the pad driven
	 * by the timer, and flash the cursorin the input pad
	 */
    register struct Client *client = ClientTable;
    register int index;
    int seconds, clicks;

    for (index=0; index<MaxClients; index++, client++)
      {
	static blinkCount = 0;

        if (client->owner && ValidPid(client->owner)==0)
	  {
	  	/*
		 * we just encountered a pad whose owner has died. RIP.
		 */
  	    if (index==InputPad) MakeNotCurrent(index);
            if (Debug) printf("Owner died in vgt %d\n", index );
	    DeleteVGT(client->vgt,TRUE);
	    PadDelete(client->vgt);
	    FlushEvents(client);
	    client->owner = 0;
	    if (client->exec)
	      {
	        if (Debug) printf("Exec died in vgt %d\n", index );
		free(client->exec);
		client->exec = NULL;
	      }
	  }
	if (index==InputPad && !mouseFlag && 
           client->requestFlag &&
	 !(client->mode & NoCursor) &&
		  blinkCount++ == 2)
          {
	    PadCursorBlink(index);
	    blinkCount = 0;
          }
	  else if (client->mode & NoCursor)
		PadCursorOff(index);
	  else PadCursorOn(index);
	   if (!mouseFlag)
 	        PadRedraw(index);
     }
     if (IdleTime++ > IdleLimit)
	  ScreenDisable();
     seconds = GetTime( &clicks);
     if ( (seconds % 10) == 0)
       {
         char dateAndTime[32];

	 strcpy( dateAndTime, ctime( &seconds ));
	 dateAndTime[DateAndTimeLength] = 0;
         SetBanner( TtyVGT, dateAndTime);
       }
  }


static HandleWrite(msg, pid)
  IoRequest *msg;
  ProcessId pid;
  {
  	/*
	 * Do a MoveFrom to get clients' written data
	 * if in a different team.
	 */
    unsigned user, total;
    static char buf[BlockSize];
    char *from;
    register count;
    register struct Client *client;

    user = msg->fileid;

    client = ClientTable + user;
    from = msg->bufferptr;
    total = msg->bytecount;

    while (total>0)
      {
        count = total;
	if (count>BlockSize) count=BlockSize;
	total -= count;
	MoveFrom( pid, buf, from, count);
	from += count;
        InterpPutString(client->interp, buf, count, client->mode);
      }
    client->block += msg->bytecount;
    msg->requestcode = OK;
  }


static HandleShortWrite(msg)
  IoRequest *msg;
  {
    register struct Client *client;
    unsigned int user;

    user = msg->fileid;
    client = ClientTable+user;
    InterpPutString(client->interp, msg->shortbuffer, 
    			msg->bytecount, client->mode);
    client->block += msg->bytecount;
    msg->requestcode = OK;
  }



static HandleCreate(msg,pid)
  register CreateInstanceReply *msg;
  ProcessId pid;
  {
  	/*
	 * Return apropriate stuff to create an instance.
	 * msg becomes the reply message.
	 * This creates a new pad, unless the Directory bit is set,
	 * in which case it just returns the Directory instance.
	 */
    register CreateInstanceRequest *req = (CreateInstanceRequest *)msg;
    int readMode = (req->filemode & FBASIC_MASK) == FREAD;
    short lines, columns, width, height, xCorner, yCorner;
    register struct Client *client;
    int interp;
    char name[128];
    
    if (req->filemode & FDIRECTORY)
      {
        msg->fileid = DirectoryInstance;
	msg->fileserver = VgtsPid;
        msg->blocksize = VgtDescriptorSize();
	msg->filetype = READABLE+FIXED_LENGTH;
	msg->filelastblock = DirectoryInstance;
	msg->filelastbytes = VgtDescriptorSize();
	msg->filenextblock = 0;
	msg->replycode = OK;
	if (Debug) printf("Replied OK to Directory create\n");
	return;
      }

    lines = req->unspecified[0];
    columns = req->unspecified[1];
    if (lines<2 || lines>96) lines = PadHeight;
    if (columns<10 || columns>127) columns = PadWidth;
    width = ViewWidth(columns);
    height = ViewHeight(lines);
    req->filename += req->filenameindex;
    req->filenamelen -= req->filenameindex;
    if (req->filenamelen>0)
      { 
        if (req->filenamelen>VGTnameLength) req->filenamelen = VGTnameLength;
        MoveFrom( pid, name, req->filename, req->filenamelen);
	name[req->filenamelen] = 0;
      }
     else strcpy(name, "Mystery Pad");
    msg->replycode = OK;
    msg->fileid = PadInit(TtySDF,Unique++, name, lines, columns);
    if (msg->fileid>= MaxClients || (interp = CreateInterp(msg->fileid)) == 0) 
      {
        msg->replycode = NO_SERVER_RESOURCES;
        return;
      }

    TtyPutString("Creating another Pad, upper left corner at B: ");
    SetPadCursor();
    WaitForDownButton();
    xCorner = MouseX;
    yCorner = MouseY;
    FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    while (MouseButtons!=0)
      {
        ReadMouseInput();

	FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
	if (MouseButtons==7)
          {
            SetMainCursor();
	    TtyPutString(" Aborted\r\n");
            msg->replycode = RETRY;
	    return;
      	 }
        xCorner = MouseX;
        yCorner = MouseY;
	CursorUpdate();
	FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    }
    FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    TtyPutString("\r\n");
    SetMainCursor();

    CreateView( msg->fileid, xCorner, yCorner, 
        	xCorner+width, yCorner+height, 0, 0, 0, 0);

    client = ClientTable+msg->fileid;
    client->interp = interp;
    client->vgt = msg->fileid;
    client->pid = 0;
    client->termPid = 0;
    client->owner = pid;
    client->master = -1;
    client->exec = NULL;
    client->requestFlag = 0;
    client->block = -1;
    client->lineEditBuf = CreateBuffer();
    client->mode = CR_Input+LF_Output+Echo+LineBuffer;
       
    msg->fileserver = VgtsPid;
    if (readMode)
      {
        msg->blocksize = IO_MSG_BUFFER;
	msg->filetype = READABLE+WRITEABLE+STREAM+VARIABLE_BLOCK+INTERACTIVE;
	msg->filelastblock = 0;
	msg->filelastbytes = 0;
	msg->filenextblock = 0;
      }
    else 
      {
        msg->blocksize = BlockSize;
        msg->filetype = READABLE+WRITEABLE+STREAM+INTERACTIVE;
	msg->filelastblock = 0;
	msg->filelastbytes = 0;
	msg->filenextblock = 0;
      }
  }


static HandleRelease(msg)
  register CreateInstanceReply *msg;
  {
  	/*
	 * Return apropriate stuff to release an instance.
	 * msg becomes the reply message.
	 *
	 * If we are deleting the current input pad, we need
	 * to find another one.
	 */
    
    if (Debug) printf("Release of vgt %d\n", msg->fileid );
    msg->replycode = OK;
    DeleteVGT(msg->fileid, 1);
    PadDelete(msg->fileid);

    FlushEvents(ClientTable + msg->fileid);
    if (ClientTable[msg->fileid].interp)
      {
        free(ClientTable[msg->fileid].interp);
        ClientTable[msg->fileid].interp = 0;
      }
    MakeNotCurrent(msg->fileid);
  }


MakeNotCurrent(pad)
    short pad;
  {
	/*
	 * checks if we are referring to the current input pad.
	 * If so, select some other likely candidate.
	 * Returns the new input pad.
	 */
    register struct Client *client;
    register int i;

    if (InputPad==pad)
      {
        for (i=1, client=ClientTable+1; 
		i<MaxClients; i++, client++)
	  if (client->vgt>0 && client->requestFlag && client->pid)
	      {
	        if (client->owner && ValidPid(client->owner))
		  {
		    SelectForInput(i);
	            return;
		  }
		DeleteVGT(client->vgt,TRUE);
		PadDelete(client->vgt);
		FlushEvents(client);
		client->pid = 0;
		client->owner = 0;
		client->master = -1;
	      }
        for (i=1, client=ClientTable+1; 
		i<MaxClients; i++, client++)
	  if (client->vgt>0 && client->pid)
	    {
	        SelectForInput(i);
	        return(i);
	    }
      }
  return(i);
  }



static HandleQuery(msg,pid)
  register CreateInstanceReply *msg;
  {
  	/*
	 * Return apropriate stuff to a query instance.
	 * msg becomes the reply message.
	 */
    QueryInstanceRequest *req = (QueryInstanceRequest *)msg;
    int readMode = (req->filemode & 0x3F)
    			== FREAD;
    register struct Client *client = ClientTable + msg->fileid;
    
    if (req->filemode & 0xFF00) readMode = 0;

    if (Debug) 
        printf("Query on id %d, mode=0x%x, readMode=%d\n", msg->fileid,
		 ((QueryInstanceRequest *)msg) ->filemode,
		 readMode);
    msg->replycode = OK;
    msg->fileserver = VgtsPid;
    
    if (msg->fileid == DirectoryInstance)
      {        
	msg->fileserver = VgtsPid;
        msg->blocksize = IO_MSG_BUFFER;
	msg->filetype = READABLE+WRITEABLE;
	msg->filelastblock = MaxClients-1;
	msg->filelastbytes = 0;
	msg->filenextblock = 0;
	msg->replycode = OK;
	return;
      }
    if (client->owner==0)
	client->owner = pid;      
    if (client->vgt <= 0)
	client->vgt = msg->fileid;
    if (client->interp==0)
        client->interp = CreateInterp(msg->fileid);

    if (readMode)
      {
        msg->blocksize = IO_MSG_BUFFER;
	msg->filetype = READABLE+WRITEABLE+STREAM+VARIABLE_BLOCK+INTERACTIVE;
	msg->filelastblock = client->block;
	msg->filelastbytes = 0;
	msg->filenextblock = 0;
      }
    else 
      {
        msg->blocksize = BlockSize;
        msg->filetype = READABLE+WRITEABLE+STREAM+INTERACTIVE;
	msg->filelastblock = client->block;
	msg->filelastbytes = 0;
	msg->filenextblock = 0;
      }
  }

static HandleQueryFile(msg)
  register struct ModifyMsg *msg;
  {
  	/*
	 * Return apropriate stuff to a query file.
	 * Right now this is just the degree of cooking.
	 * msg becomes the reply message.
	 */
    int pad = msg->fileid;
    
    if (pad<MaxClients)
      {
        msg->mode = ClientTable[pad].mode;
	msg->requestcode = OK;
      }
  }


static HandleModify(msg)
  register struct ModifyMsg *msg;
  {
   	/*
	 * Set stuff for a Modify file request.
	 * Right now this is just the degree of cooking.
	 * msg becomes the reply message.
	 */

    int pad = msg->fileid;
    register struct Client *client = ClientTable + msg->fileid;
    
    if (pad<MaxClients)
      {
        client->mode = msg->mode;

	if ( (client->mode & LineBuffer) && !(client->lineEditBuf) )
	  {
	    client->lineEditBuf = CreateBuffer(pad);
	    if (client->lineEditBuf == NULL)
	      {
		msg->mode &= ~LineBuffer;
		client->mode = msg->mode;
		msg->requestcode = NO_SERVER_RESOURCES;
		return;
	      }
          }
	else if ( !(client->mode & LineBuffer) && (client->lineEditBuf) )
	  {
	    DestroyBuffer(client->lineEditBuf);
	    client->lineEditBuf = NULL;
	  }

	msg->requestcode = OK;
      }
  }


static HandleSetBreak(msg)
    SetBreakRequest *msg;
  {
  	/*
	 * The break process is destroyed when the Kill Program
	 * command is selected, or the user types the break character.
	 */
    if (msg->fileid >= MaxClients)
      {
	msg->requestcode = INVALID_FILE_ID;
	return;
      }
    ClientTable[msg->fileid].termPid = msg->breakprocess;
    msg->requestcode = OK;
  }

static HandleSetOwner(msg)
    register IoRequest *msg;
  {
  	/*
	 * Change the "owner" of an instance
	 */
    if (msg->fileid >= MaxClients)
      {
	msg->requestcode = INVALID_FILE_ID;
	return;
      }
    ClientTable[msg->fileid].owner = msg->instanceowner;
    msg->requestcode = OK;
  }


static HandleBanner(msg, pid)
    IoRequest *msg;
    ProcessId pid;
  {
    char string[BlockSize];
    SystemCode status;
    int len = BlockSize-1;

    if (msg->bytecount < BlockSize-1)
        len = msg->bytecount;
    status = MoveFrom(pid, string, msg->bufferptr, len);
    string[len] = '\0';
    SetBanner(msg->fileid, string);
    msg->requestcode = OK;
  }


SetPadMode(pad,mode)
  {
  	/*
	 * Procedural hook for the interpeter
	 */

    if (pad<MaxClients)
      {
        ClientTable[pad].mode = mode;
	if (Debug) printf("Pad %d set to mode %d\n", pad, mode );
      }
  }


SelectForInput(i)
 short i;
  {
  	/*
	 * select the given pad for input.
	 * Zero means do not change.
	 * Bad values give an error message.
	 */

     if (i<1)
       {
         TtyBlinkError("Commands are control up-arrow followed by:");
	 TtyBlinkError("  1, 2, 3, etc to select");
	 return;
       }
    if (i>=MaxClients || ClientTable[i].owner==0) return;
    if (ClientTable[i].interp==0)
      {
      	  /*
	   * if this is a graphics VGT instead of a TTY,
	   * then just bring it to the top
	   */
	MakePadTop(i);
	return;
      }
    PadCursorOn(InputPad);
    PadRedrawLine(InputPad);
    if (InputPad !=i ) 
      {
        InputPad = i;
        FixBanners(i);
      }
    MakePadTop(i);
  }


FixBanners(new)
   short new;
  {
	/*
	 * Set the banner to be Highlighted or Normal depending
	 * on the change in input.
	 */
    register View *w;
    register char *s;
    short master;

    for ( w = ViewList; w; w = w->next)
      if ( *w->banner & 0200)
        {
		/*
		 * Banner is now ON.  Check if it is about
		 * to go off, and redraw if so.
		 */
	 if (IsInput(w->vgt)) continue;
	 for ( s = w->banner; *s;)
              *s++ = (*s & 0177);
	 RedrawBanner(w);
	}
      else
	{
		/*
		 * Banner is now OFF.  Check if it is about
		 * to go on, and redraw if so.
		 */
	 if ( !IsInput(w->vgt)) continue;
	 for ( s = w->banner; *s;)
              *s++ = *s | 0200;
	 RedrawBanner(w);
	}
  }



IsInput(vgt)
  {
  	/*
	 * return true if this VGT is accepting input
	 */
   if (vgt==InputPad) return 1;
   vgt = ClientTable[vgt].master;
   return (vgt==InputPad);
  }
