/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*
 * mouse.c - Mouse helper process 
 *
 * Bill Nowicki November 1982
 */

# define MouseGain 2	/* factor to multiply the mouse motion */

# include <Venviron.h>
# include <Vgts.h>
# include <Vmouse.h>
# include <Vioprotocol.h>
# include "client.h"

extern Process_id VgtsPid; 	/* master multiplexor */

static File *MouseFAD;	/* mouse globals */
short MouseX, MouseY, MouseButtons;
extern short Debug;

/*
 * MouseProcess:
 * It blocks on input from the kernel mouse device, updates the
 * cursor image on the screen, and sends a mouse click message
 * to the vgts process if a mouse button has been clicked.
 */

MouseProcess()
  {
    struct EventMsg msg;
    static MouseBlock buffer;
    int size;
    short oldButtons;

    InitMouse();
    /*
     * Initially put the mouse in a reasonable place
     */
    SetMouseTo(100,100);
    oldButtons = 7;

    while (1)
      {
	MouseX = (short) buffer.xcoordinate*MouseGain;
	MouseY = (short) buffer.ycoordinate*MouseGain;
        MouseButtons = buffer.buttons;
# ifdef OLDWAY
	CursorUpdate();
		/* This routine will update the cursor image according
		   to the values of MouseX and MouseY.  It will draw
		   a cursor image only if other drawing routines have
		   not signalled otherwise. */

	if (oldButtons != buffer.buttons)
			/* A mouse button has changed. */
	  {
# endif OLDWAY
	    msg.requestcode = (SystemCode)GraphicsHelperMsg;
	    msg.fileid = 0;
	    msg.x = MouseX;
	    msg.y = MouseY;
	    msg.buttons = buffer.buttons; 
	    Send( &msg, VgtsPid);
# ifdef OLDWAY
	  }
# endif OLDWAY

	oldButtons = buffer.buttons;
		/* Save the state of the mouse buttons so that it
		   can be compared to the new one for a change. */

        size = Read(MouseFAD, &buffer, 10);
		/* Block on input from the kernel mouse device. */
        if (size <= 0)
	  {
	    if (MouseFAD->lastexception==END_OF_FILE)
	        while (1) Receive(&msg);  /* Hang if mouse has been released */
	    printf("Error %d on Mouse\n", ErrorString(MouseFAD->lastexception));
	    Quit();
	  }
      }
  }


/*
 * InitMouse:
 * Initializes the mouse device.
 */

static InitMouse()
  {
    CreateInstanceRequest req;
    SystemCode error;
    File *_Open();

    req.requestcode = CREATE_INSTANCE;
    req.filenamelen = 5;
    req.filename = "Mouse";
    req.type = MOUSE;
    MouseFAD = _Open(&req, FCREATE|FRELEASE_ON_CLOSE, 
    	GetPid(DEVICE_SERVER,LOCAL_PID), &error);

    /*
     * Hang ourselves if we don't have a mouse.
     */
    if (error==NOT_FOUND)
      {
	MouseX = MouseY = 800;
    	while (1) Receive(&req);
      }

    if (error || (MouseFAD == NULL))
      {
	printf("Error %s in Mouse Open\n", ErrorString(error));
	exit();
      }
  }


Quit()
  {
  	/*
	 * Shut off the mouse and exit, by sending a message to
	 * the creator of the root process.
	 */
    exit();
  }


/*
 * ReadMouseInput:
 *
 */

ReadMouseInput()
  {
    struct EventMsg msg;

    GetMessage( 1, 0, &msg );
  }

/*
 * GetMouseInput: waits until some button is pressed, then returns as
 * soon as all buttons are released.  This is in contrast with
 * ReadMouseInput above, which returns on an button change.
 * Returns with the button set.
 */

GetMouseInput()
  {
    short buttons = 0;

    WaitForDownButton();
    while (MouseButtons)
      {
	buttons |= MouseButtons;
        CursorUpdate();
	ReadMouseInput();
      }
    return(buttons);
  }

WaitForDownButton()
  {
	/*
	 * Wait until a button goes down, tracking the mouse all the time
	 */
    while (MouseButtons==0)
      {
        CursorUpdate();
	ReadMouseInput();
      }
  }

/*******************************************************************/


SetMouseTo(x,y)
    short x,y;
  {
  	/*
	 * defines the mouse to be at a given location.
	 * This is because the mouse is a relative device,
	 * but we do not want it going too far off the screen.
	 */
    MouseBlock buffer;
    short size;

    buffer.xcoordinate = x/MouseGain;
    buffer.ycoordinate = y/MouseGain;
    buffer.buttons = MouseButtons;
    size = Write(MouseFAD, &buffer, 10);
    if (size != 10)
      {
	printf("Error %s on Mouse Write\n", ErrorString(MouseFAD->lastexception));
	Quit();
      }
  }
