/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*
 * File: graphdisp.c
 *
 * Based on a package for the Yale layout editor by Thomas R. Davis
 *
 * This file contains the routines used to draw graphical
 * displays.  DrawSegment() is the primary drawing "engine" and is
 * the routine called to interprete vgts (i.e. the
 * structured display file).  Is is only visible internally, however.
 * There are three externally visible drawing routines.  These resp.
 * redraw everything inside a bounding box, inside a window, and
 * inside the entire sdf.
 */

# include "Vgts.h"
# include "sdf.h"


/* For a given set of viewport and world coordinates, we need to have
 * instantly available the coefficients for the transformations from
 * world to screen coordinates.  In the first implementation, all "zoom"
 * factors will be restricted to powers of two.
 *
 *    zoom
 *   2    * (WorldXmax - WorldXmin) = ScreenXmax - ScreenXmin
 *
 *    zoom
 *   2    * (WorldYmax - WorldYmin) = ScreenYmax - ScreenYmin
 *
 */

/* Given a point (XWorld, YWorld) in world coordinates, the following
 * transformations can be used to convert them to screen coordinates:
 *
 *                           zoom
 *   XScreen = ScreenXmin + 2    *(XWorld - WorldXmin)
 *
 *                           zoom
 *   YScreen = ScreenYmax - 2    *(YWorld - WorldYmin)
 *
 * The apparent asymmetry in the two equations above is due to the fact that
 * the screen coordinates are left-handed and the world coordinates are
 * right-handed.
 *
 * These equations can be simplified to:
 *
 *                       zoom
 *   XScreen = Xconst + 2    * XWorld
 *
 *                       zoom
 *   YScreen = Yconst - 2    * YWorld
 *
 * where:
 *
 *                          zoom
 *   Xconst = ScreenXmin - 2    * WorldXmin
 *
 *                          zoom
 *   Yconst = ScreenYmax + 2    * WorldYmin
 */

extern short Debug;

/*******************************************************************/


/*
 * ClearWorld:
 * Clears the region of the screen corresponding to the
 * input coordinates (which are in world coordinates).
 */

ClearWorld(xmin, xmax, ymin, ymax, wp)
  int xmin, xmax, ymin, ymax;
  register View_PTR wp;
{
    register struct SubView *r;
    int sxmin, sxmax, symin, symax, h, w;
    int rxmin, rxmax, rymin, rymax;

    sxmin = XcvtToScreen(xmin, wp);
    sxmax = XcvtToScreen(xmax, wp);
    symax = YcvtToScreen(ymin, wp);
    symin = YcvtToScreen(ymax, wp);

    for (r=wp->rect;r;r=r->next)
      {
        if (r->bannerFlag) continue;

    	rxmin = max(sxmin, r->ScreenXmin);
    	rymin = max(symin, r->ScreenYmin);
    	rxmax = min(sxmax, r->ScreenXmax);
    	rymax = min(symax, r->ScreenYmax);

    	h = rymax - rymin + 1;
    	w = rxmax - rxmin + 1;
    	if ((h > 0) && (w > 0))
        	ClearRectangle(rxmin, rymin, h, w, wp->showGrid);
    /** if (Debug) printf( "Clearing xmin=%d, ymin=%d, h=%d, w=%d\n",
    	rxmin, rymin, h, w); **/
      }
}
  

/*******************************************************************/


/*
 * drawRefPoint:
 * This is the routine which is used to draw SILT reference
 * points in an arbitrary window.  It assumes that there is a value in
 * typedata which is the world coordinate (either x or y for vertical
 * and horizontal reference points, respectively), and that the xmin,
 * ..., ymax values completely surround all parts of the icon.  This is
 * not easy to do, since the reference point is half in screen
 * coordinates, and it is not clear how long and high the text might be.
 * The delete reference point routine will have to be very careful or very
 * wasteful.  The leftChild pointer is to the textual name for the
 * reference point.
 */

static drawRefPoint(sdf,wp,r)
  register DISPLAY_RECORD_PTR sdf;
  register View_PTR wp;
  register struct SubView *r;
  {
    int screenX, screenY, screenMax, screenMin, barMin, barMax;
    
/**    if (Debug) printf("drawRefPt xmin=%d, xmax=%d, ymin=%d, ymax=%d\n",
          sdf->xmin, sdf->xmax, sdf->ymin, sdf->ymax); **/

    if (sdf->type == SDF_HORIZONTAL_REF || sdf->type == SDF_SEL_HORIZ_REF)
      {
        screenY = YcvtToScreen((sdf->ymax+sdf->ymin)/2, wp);
        screenMin = r->ScreenXmin;
        screenMax = r->ScreenXmax;
	if (r->EdgeSet&LeftEdge)
	    screenMin += 16;
	if (r->EdgeSet&RightEdge)
	    screenMax -= 16;

        if ((r->ScreenYmin < screenY) && (screenY < r->ScreenYmax))
    	    FillRegion (screenMin, screenY,
    	      (sdf->type == SDF_HORIZONTAL_REF) ? 1 : 2, 
	      screenMax - screenMin + 1);
	if (r->EdgeSet&LeftEdge)
	  {
	    if (sdf->leftChild)
              DisplayText(screenMin+3, screenY-2, sdf->leftChild, r, TRUE);
            barMin = max(screenY - 4, r->ScreenYmin);
    	    barMax = min(screenY + 5, r->ScreenYmax);
            if (barMax - barMin > 0)
                FillRegion (screenMin, barMin, barMax - barMin, 1);
	  }
      }
    else	/* it's a vertical reference point */
      {
        screenX = XcvtToScreen((sdf->xmax + sdf->xmin)/2, wp);
        screenMin = r->ScreenYmin;
        screenMax = r->ScreenYmax;
	if (r->EdgeSet&TopEdge) 
	    screenMin += 20;
	if (r->EdgeSet&BottomEdge) 
	    screenMax -= 16;

        if ((r->ScreenXmin < screenX) && (screenX < r->ScreenXmax))
    	  FillRegion(screenX, screenMin,
    	     screenMax - screenMin + 1,
    	     (sdf->type == SDF_VERTICAL_REF) ? 1 : 2);
	if (r->EdgeSet&TopEdge)
	  {
	    if (sdf->leftChild)
                DisplayText(screenX - 8, screenMin -2, sdf->leftChild, r, TRUE);
    
            barMin = max(screenX - 4, r->ScreenXmin);
            barMax = min(screenX + 5, r->ScreenXmax);
            if (barMax - barMin > 0)
                FillRegion (barMin, screenMin, 1, barMax - barMin);
	  }
      }
  }
    

/*******************************************************************/


/*
 * DrawSegment:
 * Redraws everything inside the given bounding box.
 * It assumes that the segment in question has already been cleared.
 * It also makes sure that the segment is within the window.
 */

static DrawSegment(wXmin, wXmax, wYmin, wYmax, sdf, 
				xtrans, ytrans, depth, wp,r)
short wXmin, wXmax, wYmin, wYmax, xtrans, ytrans, depth;
  register DISPLAY_RECORD_PTR sdf;
  register View_PTR wp;
  register struct SubView *r;
  {
    int locXtrans, locYtrans;
    short Sxmin, Sxmax, Symin, Symax;	/* World coords of object */
    register int   xmin, ymax;		/* Screen Coords before cliping */
    int Rxmin, Rymin, Rxmax, Rymax;	/* Screen Coords after clipping */
    DISPLAY_RECORD_PTR sdfSub;
    short edgeMask;

    if (sdf == NIL)   return;
    
    wXmin = max(wXmin, wp->WorldXmin);
    wYmin = max(wYmin, wp->WorldYmin);
    wXmax = min(wXmax, wp->WorldXmax);
    wYmax = min(wYmax, wp->WorldYmax);

    if (((wXmax - wXmin) <= 0) || ((wYmax - wYmin) <= 0))
        return;			/* Non-overlapping boxes. */
    
    while (sdf->type != SDF_SYMBOL_DEF)
      {
        if (sdf->type == SDF_SYMBOL_CALL)
    	  {
    	    locXtrans = xtrans + sdf->xmin;
    	    locYtrans = ytrans + sdf->ymin;
    	    sdfSub = sdf->leftChild;	/* points to symbol definition */
	    if (sdfSub == NIL)
		return;	/* bullet-proofing -- TRD */
    	    if ((sdfSub->xmin + locXtrans) < wXmax &&
	        (sdfSub->xmax + locXtrans) > wXmin &&
	        (sdfSub->ymin + locYtrans) < wYmax &&
	        (sdfSub->ymax + locYtrans) > wYmin) /* then we gotta expand */
	    	    if (depth < wp->expansionDepth)
	    	        DrawSegment(wXmin, wXmax, wYmin, wYmax,
				sdfSub->rightSib, locXtrans,
				locYtrans, depth + 1, wp,r);
		else
		  {	/* Draw the outline for the symbol */
		    Rxmin = XcvtToScreen(max(
			Sxmin=sdfSub->xmin + locXtrans, wXmin), wp);
		    Rxmax = XcvtToScreen(min(
			Sxmax=sdfSub->xmax + locXtrans, wXmax), wp);
		    Rymin = YcvtToScreen(min(
			Symax=sdfSub->ymax + locYtrans, wYmax), wp);
		    Rymax = YcvtToScreen(max(
			Symin=sdfSub->ymin + locYtrans, wYmin), wp);

	    	    Rxmin = max(Rxmin,r->ScreenXmin);
	    	    Rymin = max(Rymin,r->ScreenYmin);
	    	    Rxmax = min(Rxmax,r->ScreenXmax);
	    	    Rymax = min(Rymax,r->ScreenYmax);

		    edgeMask = 0;

		    if (Rxmin > r->ScreenXmin && Sxmin > wXmin)
			edgeMask += LeftEdge;
		    if (Rxmax < r->ScreenXmax && Sxmax < wXmax)
			edgeMask += RightEdge;
		    if (Rymin > r->ScreenYmin && Symax < wYmax)
			edgeMask += TopEdge;
		    if (Rymax < r->ScreenYmax && Symin > wYmin)
			edgeMask += BottomEdge;

		    if (((Rxmax - Rxmin) > 4) && ((Rymax - Rymin) > 4))
		        DrawOutline(Rxmin, Rymin, Rxmax-Rxmin,
					Rymax-Rymin, 1, edgeMask);
		    if (((Rxmax - Rxmin) > 40) && ((Rymax - Rymin) > 20))
		   	DisplayText(Rxmin+4, Rymin+20, sdfSub->leftChild, r, TRUE);
		  }
	}
    else if ((sdf->xmin + xtrans) < wXmax &&
	     (sdf->xmax + xtrans) > wXmin &&
	     (sdf->ymin + ytrans) < wYmax &&
	     (sdf->ymax + ytrans) > wYmin)	/* then we gotta expand */
	  {
	    Sxmin=sdf->xmin+xtrans;
	    Symax=sdf->ymax+ytrans;
	    Sxmax=sdf->xmax+xtrans;
	    Symin=sdf->ymin+ytrans;
	    
	    Rxmin = XcvtToScreen(max(Sxmin,wXmin), wp);
	    Rymin = YcvtToScreen(min(Symax,wYmax), wp);
	    Rxmax = XcvtToScreen(min(Sxmax,wXmax), wp);
	    Rymax = YcvtToScreen(max(Symin,wYmin), wp);

	    ymax = YcvtToScreen(Symin, wp);
	    xmin = XcvtToScreen(Sxmin, wp);

	    Rxmin = max( xmin,r->ScreenXmin);
	    Rymin = max(Rymin,r->ScreenYmin);
	    Rxmax = min(Rxmax,r->ScreenXmax);
	    Rymax = min( ymax,r->ScreenYmax);
	    if (Rxmax - Rxmin >= 0 && Rymax - Rymin >= 0 )
	      switch (sdf->type)
	        {
		case SDF_FILLED_RECTANGLE:
		    DrawRect(Rxmin, Rymin, Rxmax-Rxmin+1,
				Rymax-Rymin+1, sdf->typedata);
		    break;

		case SDF_OUTLINE:
		    /* the Rxmin, ... above give the dimensions of the
		     * rectangle to be drawn, but if it is reduced, we
		     * don't want to draw the whole outline.  Hence,
		     * we have the following monkey business to twiddle
		     * the selected edgeset:
		     */
		    edgeMask = 0;
		    if (Rxmin > r->ScreenXmin && Sxmin > wXmin)
			edgeMask += LeftEdge;
		    if (Rxmax < r->ScreenXmax && Sxmax < wXmax)
			edgeMask += RightEdge;
		    if (Rymax < r->ScreenYmax && Symin > wYmin)
			edgeMask += BottomEdge;
		    if (Rymin > r->ScreenYmin && Symax < wYmax)
			edgeMask += TopEdge;
		    DrawOutline(Rxmin, Rymin, Rxmax-Rxmin,
				Rymax-Rymin, 2, (sdf->typedata & edgeMask));
		    break;

		case SDF_HORIZONTAL_REF:
		case SDF_VERTICAL_REF:
		case SDF_SEL_HORIZ_REF:
		case SDF_SEL_VERT_REF:
		    drawRefPoint(sdf, wp, r);
		    break;

		case SDF_HORIZONTAL_LINE:
		    if (Rxmax>=Rxmin && Rymax>=Rymin)
		        FillRegion(Rxmin, Rymin, 1, Rxmax-Rxmin);
		    break;

		case SDF_VERTICAL_LINE:
		    if (Rxmax>=Rxmin && Rymax>=Rymin)
		        FillRegion(Rxmin, Rymin, Rymax-Rymin, 1);
		    break;

		case SDF_POINT:
		    FillRegion(Rxmin, Rymin, 2, 2);
		    break;

		case SDF_InternalText:
		    DisplayText( xmin, ymax, sdf->leftChild, r, FALSE);
		    break;

		case SDF_SIMPLE_TEXT:
		    DisplayText( xmin, ymax, sdf->leftChild, r, TRUE);
		    break;

		case SDF_TEXT:
		    DrawGeneralText( xmin, ymax, sdf->leftChild, r, 
		    	sdf->typedata, TRUE);
		    break;		

		case SDF_UpLine:
		case SDF_DownLine:
		    Rxmax = XcvtToScreen(Sxmax,wp);
		    Rymin = YcvtToScreen(Symax,wp);
   		    if (sdf->type==SDF_DownLine)
		        DisplayLine(xmin,Rxmax,Rymin,ymax,r);
		    else
		        DisplayLine(xmin,Rxmax,ymax,Rymin,r);
		    break;

		case SDF_RASTER:
    	  	    xmin = XcvtToScreen(Sxmin+1, wp);
		    Rymin = YcvtToScreen(Symax-1,wp);
		    if (sdf->leftChild)
		        DisplayRaster(sdf->leftChild, xmin, Rymin,
		 		wp, r, sdf->typedata);
		    break;

		case SDF_SPLINE:
		    DisplaySpline( sdf->leftChild, Sxmin, Symin,
			wp, r, sdf->typedata );
		    break;
		    
		default:
		    if (Debug) printf("Bug: Bad SDF type =%d\n",sdf->type);
		    break;
		}
	  }
    sdf = sdf->rightSib;
    if (sdf == NIL)
	return;	/* bullet-proofing -- TRD */
  }
}


/*******************************************************************/

RedrawVGT(vgt, wXmin, wXmax, wYmin, wYmax)
  short vgt;
  int wXmin, wXmax, wYmin, wYmax;
{
    short windowId;
    DISPLAY_RECORD_PTR sdf;
    register View_PTR wp = ViewTable;
    register struct SubView *r;

    if (vgt<0) return;

    wXmin = max(wXmin, -32760);
    wXmax = min(wXmax,  32760);
    wYmin = max(wYmin, -32760);
    wYmax = min(wYmax,  32760);
    
    for (windowId = 0; windowId < MAX_Views; windowId++, wp++)
        if (wp->vgt == vgt && wp->usage!=UNUSED)
          {
	    sdf = VGTtable[wp->vgt].sdfPtr;
    	    if (sdf == NIL)
    	        continue;
	    /*** if (Debug)
	      printf( "Redrawing wXmin=%d, wXmax=%d, wYmin=%d, wYmax=%d\n",
	        wXmin, wXmax, wYmin, wYmax); ***/
  	    if (EraseCursor( max( XcvtToScreen(wXmin-2,wp), wp->ScreenXmin),
	    		 min( XcvtToScreen(wXmax+2,wp), wp->ScreenXmax),
	    		 max( YcvtToScreen(wYmax+2,wp), wp->ScreenYmin),
			 min( YcvtToScreen(wYmin-2,wp), wp->ScreenYmax) ))
		continue;
	    if (wp->usage & GRAPHICS)
    	        ClearWorld(wXmin, wXmax, wYmin, wYmax, wp);
	    for (r=wp->rect;r;r=r->next)
	      if (r->bannerFlag==FALSE)
    	        DrawSegment(wXmin-1, wXmax+1, wYmin-1, wYmax+1, 
		  sdf->rightSib, 0, 0, 0, wp, r);
            Cursor();
          }
}


RedrawPartialVGT(vgt, wXmin, wXmax, wYmin, wYmax, sdf)
  short vgt;
  int wXmin, wXmax, wYmin, wYmax;
  DISPLAY_RECORD_PTR sdf;
{
	/*
	 * draw only added items in the given VGT
	 */
    short windowId;
    register View_PTR wp = ViewTable;
    register struct SubView *r;

    if (vgt<0 || sdf==NULL) return;

    wXmin = max(wXmin, -32760);
    wXmax = min(wXmax,  32760);
    wYmin = max(wYmin, -32760);
    wYmax = min(wYmax,  32760);
    
    for (windowId = 0; windowId < MAX_Views; windowId++, wp++)
        if (wp->vgt == vgt && wp->usage!=UNUSED)
          {
	    /*** if (Debug)
	      printf( "Redrawing wXmin=%d, wXmax=%d, wYmin=%d, wYmax=%d\n",
	        wXmin, wXmax, wYmin, wYmax); ***/
  	    if (EraseCursor( max( XcvtToScreen(wXmin-2,wp), wp->ScreenXmin),
	    		 min( XcvtToScreen(wXmax+2,wp), wp->ScreenXmax),
	    		 max( YcvtToScreen(wYmax+2,wp), wp->ScreenYmin),
			 min( YcvtToScreen(wYmin-2,wp), wp->ScreenYmax) ))
		continue;

	    for (r=wp->rect;r;r=r->next)
	      if (r->bannerFlag==FALSE)
    	          DrawSegment(wXmin-1, wXmax+1, wYmin-1, wYmax+1, sdf, 0, 0, 
			0, wp, r);
            Cursor();
          }
}/*******************************************************************/


/*
 * RedrawEntireView:
 * Redraws all features of the window, including
 * the outline.
 */

RedrawEntireView(windowId)
short windowId;
{
    register DISPLAY_RECORD_PTR sdf;
    register View_PTR wp = ViewTable + windowId;
    register struct SubView *r;

    if (wp->usage==UNUSED) return;
    if (EraseCursor(wp->ScreenXmin, wp->ScreenXmax, wp->ScreenYmin, wp->ScreenYmax))
        return;
    for (r=wp->rect;r;r=r->next)
      {
	if (r->bannerFlag)
	  {
	    if ( *(wp->banner) & 0200)
                DrawRect(r->ScreenXmin, r->ScreenYmin, 
    		  r->ScreenXmax - r->ScreenXmin + 1, 
		  r->ScreenYmax - r->ScreenYmin + 1,
		  BLACK );
	    else
                ClearRectangle(r->ScreenXmin, r->ScreenYmin, 
		r->ScreenYmax - r->ScreenYmin + 1,
    		r->ScreenXmax - r->ScreenXmin + 1, 0);
	  }
	else 
            ClearRectangle(r->ScreenXmin, r->ScreenYmin, 
		r->ScreenYmax - r->ScreenYmin + 1,
    		r->ScreenXmax - r->ScreenXmin + 1, 
		wp->showGrid);
        DrawOutline( r->ScreenXmin-2, r->ScreenYmin-2,
    	  r->ScreenXmax - r->ScreenXmin + 5, 
	  r->ScreenYmax - r->ScreenYmin + 5, 2, r->EdgeSet );
        sdf = VGTtable[wp->vgt].sdfPtr;
	if (r->bannerFlag)
	      DisplayBanner( wp, r);
         else
           if (sdf != NIL)
          {
            /*** if (Debug)
              printf("Redoing vgt %d\n",wp->vgt); ***/
              DrawSegment( wp->WorldXmin, wp->WorldXmax, wp->WorldYmin,
    		wp->WorldYmax, sdf->rightSib, 0, 0, 0, wp, r);
          }
     }
    Cursor();
}


InvertVGT(vgt)
  short vgt;
{
    register View_PTR wp = ViewTable;
    register struct SubView *r;
    short windowId;

    for (windowId = 0; windowId < MAX_Views; windowId++, wp++)
        if (wp->vgt == vgt && wp->usage!=UNUSED)
         {
           if (EraseCursor(wp->ScreenXmin, wp->ScreenXmax, 
	     		     wp->ScreenYmin, wp->ScreenYmax))
        	return;
           for (r=wp->rect;r;r=r->next)
           {
            InvertRegion(r->ScreenXmin, r->ScreenYmin, 
		r->ScreenYmax - r->ScreenYmin + 1,
    		r->ScreenXmax - r->ScreenXmin + 1);
           }
         Cursor();
       }
}

/*
 * RedrawBanner:
 * redraw ONLY the banner parts of a view
 */

RedrawBanner(wp)
    register View_PTR wp;
{
    register struct SubView *r;

    if (wp->usage==UNUSED) return;
    if (EraseCursor(wp->ScreenXmin, wp->ScreenXmax, wp->ScreenYmin, wp->ScreenYmax))
        return;
    for (r=wp->rect;r;r=r->next)
     if (r->bannerFlag)
      {
         if ( *(wp->banner) & 0200)
                DrawRect(r->ScreenXmin, r->ScreenYmin, 
    		  r->ScreenXmax - r->ScreenXmin + 1, 
		  r->ScreenYmax - r->ScreenYmin + 1, BLACK );
	    else
                ClearRectangle(r->ScreenXmin, r->ScreenYmin, 
		r->ScreenYmax - r->ScreenYmin + 1,
    		r->ScreenXmax - r->ScreenXmin + 1, FALSE );
        DisplayBanner( wp, r);
     }
    Cursor();
}


/*******************************************************************/


/*
 * SdfExpand:
 * Redraws the entire sdf.  It assumes that all of the
 * global window parameters have been set.
 */

SdfExpand()
{
    short windowId;

    EraseCursor(0, 1024, 0, 1024);
    DrawBackground(0, 0, 1024, 1024);
    for (windowId =0; windowId < MAX_Views; windowId++)
    	RedrawEntireView(windowId);
}


/*******************************************************************/


/*
 * XcvtToScreen:
 * Converts world x coord. to screen x coord.
 */

XcvtToScreen(x,w)
  int x;
  register View_PTR w;  
{
    return(w->Xconst+((w->zoom>=0)?((x)<<w->zoom):((x)>>(-w->zoom))));
}


/*
 * YcvtToScreen:
 * Converts world y coord. to screen y coord.
 */

YcvtToScreen(y,w)
  int y;
  register View_PTR w;  
{
    return(w->Yconst-((w->zoom>=0)?((y)<<w->zoom):((y)>>(-w->zoom))));
}



