/************************************************/
/*						*/
/*	       (C)  COPYRIGHT 1983		*/
/*		BOARD OF TRUSTEES		*/
/*	LELAND STANFORD JUNIOR UNIVERSITY	*/
/*	  STANFORD, CA. 94305, U. S. A.		*/
/*						*/
/************************************************/
 
/*
 *  fill.c
 *
 *  This file contains the routines necessary to support the filling
 *  of arbitrary polygons.  (See polygon.c for an example).  It assumes
 *  the existance of XYputLine(), which it uses to plot the edges.
 *  The entry points for the actual fill routines are XYfillInit(),
 *  XYfillLine(), and XYfillScreen().
 *
 *  The algorithm used is a straightforward (YX) parity algorithm, where
 *  line intersections for each scan line are computed, sorted, and stored.
 *  Each pair of points on a line then represents an interior region of the
 *  polygon.
 *
 *  Points are stored as a linked list of cells.  fillBuf is an array of
 *  such cells.  The x field indicates the parity of the number of line
 *  segments to the left of the clipping window which crossed that scan
 *  line.  Next points to a linked list of cells, into which each new
 *  cell is inserted.  A trailer record is appended to each list to 
 *  simplify the algorithm.
 *
 *  Jim Boyce was of great assistance designing and debugging this algorithm.
 *
 *  Update History:
 *	March 1983 -- Written by David Kaelbling.
 */
 
 
#include "Vgts.h"
#include "sdf.h"
#include <rasterops.h>
#include <Vio.h>
#include "splines.h"
 
 
/* Imports */
extern *Patterns[];		/* Array of pointers to fill patterns */
extern XYpattern();		/* Do a RasterPattern. */
extern XYputLine();		/* Return points on a line */
    
    
/* Exports */
extern XYfillInit();		/* Initializes the fill routines. */
extern XYfillLine();		/* Call once for each line segment. */
extern XYfillScreen();		/* Call after all lines are done. */
 
 
/* Local Definitions */
typedef struct cell {
    short x;
    struct cell *next;
  } CELL;
 
#define MAXSHORT 0x7FFF
 
static CELL *freelist = NULL;	/* Head of free list. */
static CELL *trailer;		/* Pointer to the trailer cell */
static CELL *fillBuf;		/* Pointer to the fill buffer */
static short oldy;		/* Var. to fix runs of points on a line */
static short xmin, xmax, ymin, ymax;	/* Clipping window coordinates */
static BOOLEAN dead;		/* Has the program died? */

/*
 *  These two internal routines will maintain the free list of
 *  allocated cells.  If no free cells are available when needed,
 *  they will be malloc'd.
 */
 
static CELL *getcell()
  {
    register CELL *tmp;
    
    /* Return an allocated cell, even if we have to make one. */
    if (tmp = freelist)
      {
	freelist = freelist->next;
      }
    else
      {
	tmp = (CELL *) malloc( sizeof(CELL) );
      }
    tmp->next = NULL;
    return( tmp );
  }
    
    
static freecell( tmp )
	register CELL *tmp;
  {
    /* Prepend this to the free list. */
    tmp->next = freelist;
    freelist = tmp;
  }

/*
 *  This routine will free all of the storage allocated by the fill routines.
 */
  
static FreeFillStorage()
  {
    register CELL *tmp, *scan;
    register short y;
    
    /* If trailer wasn't allocated, nothing worthwhile was. */
    if (trailer == NULL)
      {
	return;
      }

    /* Free all of the cells in fillBuf */
    --fillBuf;
    for (y = 0, scan = fillBuf[y].next;
	 y < ymax - ymin + 3;
	 scan = fillBuf[++y].next)
      {
	while (scan != trailer)
	  {
	    tmp = scan->next;
	    freecell( scan );
	    scan = tmp;
	  }
      }
      
    /* Free the miscellaneous */
    free( fillBuf );
    freecell( trailer );
  }
 
/*
 *  This routine will allocate the fillBuffer and initialize it.
 */
 
XYfillInit( clip )
	struct SubView *clip;
  {
    register int i;
    
    dead = FALSE;
    
    /* Clip to the physical frame buffer at least. */
    xmin = max( clip->ScreenXmin, 0 );
    xmax = min( clip->ScreenXmax, 1023);
    ymin = max( clip->ScreenYmin, 0 );
    ymax = min( clip->ScreenYmax + 1, 1023 );
     
    /* Allocate the fill buffer */
    if ((fillBuf = (CELL *) malloc(sizeof(CELL) * (ymax-ymin+3))) == NULL)
      {
	printf("XYFILLINIT:  Couldn't allocate fill buffer.\n");
	dead = TRUE;
	return( ERROR );
      }
 
    /* Allocate the trailer cell. */
    if ((trailer = getcell()) == NULL)
      {
	printf("XYFILLINIT:  Couldn't allocate trailer.\n");
	free( fillBuf );
	dead = TRUE;
	return( ERROR );
      }
 
    /* Initialize everything */
    i = ymax - ymin + 3;
    while (i--)
      {
        fillBuf[i].x = 0;
	fillBuf[i].next = trailer;
      }
    fillBuf++;		/* Fake -1 based arrays */
    trailer->x = MAXSHORT;
    trailer->next = trailer;
    
    return( 0 );
  }
 
/*
 *  This routine will insert each point passed it into the fill buffer.
 *  This is the routine passed to XYputLine() as 'Actor'.
 */
 
static XYfillPoint( xval, yval, clip, nib )
	register short xval;
	register short yval;
	struct SubView *clip;
	enum Nib nib;
  {
    register CELL **pnt;
    register CELL *new;
 
    /* If already failed, don't bother trying again. */
    if (dead)
      {
        return( ERROR );
      }

    /* Only permit one point per scan line per line */
    if (yval == oldy)
      {
	return( 0 );
      }
    oldy = yval;
 
    /* If within y bounds, insert the point into fillBuf */
    if ((yval >= ymin) && (yval <= ymax))
      {
        for (pnt = &(fillBuf[yval - ymin].next);
	     xval > (*pnt)->x;
	     pnt = &((*pnt)->next)) { }
	if ((new = getcell()) == NULL)
	  {
	    printf("XYFILLPOINT:  Can't allocate a new cell.\n");
	    FreeFillStorage();
	    dead = TRUE;
	    return( ERROR );
	  }
	new->x = xval;
	new->next = *pnt;
	*pnt = new;
      }
    return( 0 );
  }

/*
 *  This routine will take the y bounds of a line segment which lies
 *  entirely to the left of the clipping window and update the fillBuf
 *  array accordingly.
 */
 
static ClippedLine( ytop, ybot )
	register short ytop, ybot;
  {
    while (++ytop <= ybot)
      {
        fillBuf[ytop - ymin].x ^= 1;
      }
  }
 
/*
 *  This routine invokes XYputLine() for each appropriate line segment.
 *  It also initializes 'oldy' to insure that the highest point on
 *  each line segment is NOT drawn.  Also, a great deal of effort is
 *  spent clipping and discarding line segments which lie outside of
 *  the clipping window.  Lines to the left of the window simply have
 *  the parity of the number of endpoints on each scan line recorded.
 */
  
XYfillLine( xbot, ybot, xtop, ytop, clip )
	register short xtop, xbot, ytop, ybot;
	struct SubView *clip;
  {
    register float slope;
    register unsigned short computed;	/* Has slope been computed? */
    register unsigned short ytmp;
    enum Nib nib;
    
    
    /* If already busted, quit */
    if (dead)
      {
        return( ERROR );
      }

    /* Ignore horizontal lines entirely, and draw from the top down. */
    if (ybot == ytop)
      {
        return( 0 );
      }
    
    /* Get top and bottom right */
    if (ybot < ytop)
      {
	computed = ybot;
	ybot = ytop;
	ytop = computed;
	computed = xbot;
	xbot = xtop;
	xtop = computed;
      }
 
    /* Ignore lines outside y region */
    if ((ybot >= ymin) && (ytop <= ymax))
      {
        /* Clip the line if necessary */
	if ((ytop < ymin) || (ybot > ymax))
	  {
	    slope = (xbot - xtop) / (float) (ybot - ytop);
	    if (ytop < ymin)
	      {
	        xtop += (short) (slope * (ymin - ytop - 1));
		ytop = ymin - 1;
	      }
	    if (ybot > ymax)
	      {
	        xbot += (short) (slope * (ymax - ybot + 1));
		ybot = ymax + 1;
	      }
	  }
		
	/* Ignore lines off to the right, and clip others. */
	if ((xtop < xmax) || (xbot < xmax))
	  {
	    /* Compute slope only when needed. */
	    computed = 0;
    
	    /* Clip on the right */
	    if (xtop > xmax)
	      {
		slope = (ytop - ybot) / (float) (xtop - xbot);
		computed = 1;
		ytop += (short) (slope * (xmax - xtop));
		xtop = xmax;
	      }
	    else if (xbot > xmax)
	      {
		slope = (ytop - ybot) / (float) (xtop - xbot);
		computed = 1;
		ybot += (short) (slope * (xmax - xbot));
		xbot = xmax;
	      }
    
	    /* Case 1:  Line lies entirely to the left of region */
	    if ((xtop <= xmin) && (xbot <= xmin))
	      {
		ClippedLine( ytop, ybot );
	      }
	      
	    /* Case 2:  Xtop lies outside the region */
	    else if (xtop < xmin)
	      {
		if (!computed)
		  {
	            slope = (ytop - ybot) / (float) (xtop - xbot);
		    computed = 1;
		  }
                ytmp = ytop;
		ytop += (short) (slope * (xmin - xtop));
	        xtop = xmin;
		ClippedLine( ytmp, ytop );
	        oldy = ytop;
		if (xtop != xbot)
	            XYputLine( xtop,ytop, xbot,ybot, XYfillPoint, clip, nib );
		else
		    XYfillPoint( xtop, ytop, clip, nib );
	      }
	      
	    /* Case 3:  Xbot lies outside the region */
	    else if (xbot < xmin)
	      {
		if (!computed)
		  {
	            slope = (ytop - ybot) / (float) (xtop - xbot);
		    computed = 1;
		  }
		ytmp = ybot;
		ybot += (short) (slope * (xmin - xbot));
	        xbot = xmin;
		ClippedLine( ybot, ytmp );
	        oldy = ytop;
		if (ytop != ybot)
	            XYputLine( xtop,ytop, xbot,ybot, XYfillPoint, clip, nib );
		else
		    XYfillPoint( xtop, ytop, clip, nib );
	      }
	      
	    /* Case 4:  Line is entirely within the region */
	    else
	      {
	        oldy = ytop;
	        XYputLine( xtop, ytop, xbot, ybot, XYfillPoint, clip, nib );
	      }
	  }
      }
  }
 
/*
 *  This routine will output the sorted fill buffer, updating the screen.
 */
 
XYfillScreen( pattern, opaque )
	enum Pattern pattern;
	unsigned short opaque;
  {
    struct fb_raster dest;
    short inside;			/* True if already within poly */
    short y;				/* Current scan line on screen */
    register short xStart, xStop;	/* Endpoints of current fill area */
    register CELL *scan;		/* Pointer to scan down scan-line */
    
    
    /* If fillPoint died, don't bother with fillScreen */ 
    if (dead)
        return( ERROR );
    
    dest.height = 1;
    for (dest.y = y = ymin;  y <= ymax;  dest.y = y++)
      {
	inside = fillBuf[y - ymin].x;
        scan = fillBuf[y - ymin].next;
	while ((scan != trailer) || (inside))
	  {
	    /* Get the interior region, and free storage */
	    if (inside)
	      {
	        xStart = xmin;
		xStop = scan->x;
 		inside = 0;
	      }
	    else
	      {
	        xStart = scan->x;
	        scan = scan->next;
	        xStop = scan->x;
	      }
	    scan = scan->next;
  
	    /* If entirely outside clipping window, ignore. */
	    if ((xStart < xmax) && (xStop > xmin))
	      {
		/* Clip and draw the segment */
		xStop = (xStop > xmax ? xmax : xStop);
	        xStart = (xStart < xmin ? xmin : xStart);
		dest.x = xStart;
		dest.width = xStop - xStart + 1;
		XYpattern( &dest, Patterns[(int)pattern], opaque );
	      }
	  }
      }
    FreeFillStorage();
    return( 0 );
  }
