/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* event.c - Handles VGTS event queues
 *
 *
 * Bill Nowicki April 1983 
 *
 */

# include <Venviron.h>
# include <Vioprotocol.h>
# include "Vgts.h"
# include "client.h"
# include "sdf.h"	/* sdf.h because I need VGTtable[] */
# include "pad.h"

struct Event *FreeEvents = NULL;
extern short InputPad;			/* the current input pad */
extern short MouseX, MouseY, MouseButtons;
extern short Debug;			/* for extra debugging info */

/*
 * Routines that handle character events
 */

UserPut(client,c)
  register struct Client *client;
  char c;
    {
	/*
	 * Check if the last event is a Keystroke event that has not
	 * been filled up.  If not, tack another one onto the end
	 * of the queue.  Stuff the character into the last event.
	 */
      IoReply msg;
      register struct Event *e = client->eventTail;

      if (e==NULL || e->code != ReportCharacter
      		  || e->bytecount >= IO_MSG_BUFFER)
        {
		/*
		 * try to get a free Event and tack it onto the end
		 */
	  e = FreeEvents;
	  if (e==NULL) return;
	  FreeEvents = e->next;

	  if (client->eventTail)  
	    {
	      client->eventTail->next = e;
	    }
	  if (client->eventQ == NULL) client->eventQ = e;
	  client->eventTail = e;
	  e->next = NULL;
	  e->code = ReportCharacter;
	  e->bytecount = 0;
	}
      
      e->shortbuffer[e->bytecount++] = c;
    }


UserPush(client,c)
  register struct Client *client;
  {
  	/*
	 * This is the last character we are giving back to the user,
	 * so without further ado, send it to the eagerly awaiting
	 * client process.  But throw it away if a pure mouse request
	 * is outstanding.
	 */

    UserPut(client,c);
    EventToUser(client);
  }


UserCook(client,c)
    register struct Client *client;
  {
  	/*
	 * Possibly cook an input character.
	 * Drop the character if a pure mouse request is outstanding
	 */

    if (client->requestFlag == Mousereq) return;

    if (Debug) printf("before Line Edit\n");
    if (client->mode & LineBuffer && client->lineEditBuf)
      return( LineEdit(client,c) );

    if (c=='\r' && (client->mode & CR_Input) )
      c = '\n';
    if (client->mode & Echo)
      {
        if (c=='\n' && (client->mode&LF_Output))
	      	 PadPutChar('\r',TtyPadList+client->vgt);
        PadPutChar(c,TtyPadList+client->vgt);
      }
    UserPush(client,c);
  }


HandleRead(client, pid)
    register struct Client *client;
  {
	/*
	 * The user is trying to read.  See if any characters are
	 * on the event queue.  If not, set flags to wait for some.
	 */
    if ( (client->requestFlag&BlockBits) && 
          AwaitingReply(VgtsPid,client->pid) )
    	return;			/* should reply with error code!! */
    client->requestFlag |= Keyreq;
    client->pid = pid;
    EventToUser(client);
  }


PutEndEvent(client)
    register struct Client *client;
  {
	/*
	 * Put an end-of-file event on the queue. These are returned
	 * immediately if the user was waiting for anything.
	 */
    register struct Event *e;
    register int mousemode = client->mode & (ReportTransition | ReportClick);

    e = FreeEvents;
    if (e == NULL) return;
    FreeEvents = e->next;

    if (client->eventTail)  client->eventTail->next = e;
    if (client->eventQ == NULL) client->eventQ = e;
    client->eventTail = e;
    e->next = NULL;
    e->code = EOFcode + ReportCharacter + ReportClick;
    e->bytecount = 0;
    EventToUser(client);
  }



/*
 * Routines that are common to character and graphical events
 */

EventToUser(client)
    register struct Client *client;
  {
	/*
	 * If the user was requesting input, Reply with the event
	 * descriptor.
	 */
    int requestmode;
    register struct Event *e;

    requestmode = 0;
    if (client->requestFlag&Mousereq) 
    	requestmode |= ReportClick|ReportTransition;
    if (client->requestFlag&Keyreq) 
    	requestmode |= ReportCharacter;

    if ( (client->requestFlag&BlockBits)==0) return;

    e = client->eventQ;
    while (e && (!(e->code & requestmode)))
      {
	if ( (client->requestFlag & SlaveReq) &&
	    (e->code & ReportClick) )
	  {
	  	/*
		 * Here we convert the graphics event to character events.
		 * We just return from this, since ReturnToMaster eventually
		 * calls UserPush, which calls this routine recursively.
		 * For this reason, it is crucial that the SlaveReq bit
		 * be turned off before you call ReturnToMaster.
		 */

	    DropEvent(client,e);
	    client->requestFlag &= ~SlaveReq;
	    ReturnToMaster(client->vgt,e->x,e->y,e->buttons,e->fileid);
	    return;
	  }
	e = e->next;
      }

    if (e) 
      {
	  	/*
		 * We got one!! yank it off the event queue,
		 * put it on the free queue, and Reply to the client.
		 */
	register EventReq *rep;  /* variant record: IoReply */

	if (e->code & ReportCharacter)
	  {
	    rep = (EventReq *) &(e->code);
	    rep->eventcode = ReportCharacter;
	  }
    	else rep = (EventReq *) &(e->requestcode );

	DropEvent(client,e); 	/* free the one we are using */
	
	if (e->code & EOFcode)
	    rep->requestcode = END_OF_FILE;
	else
	    rep->requestcode = OK;
	rep->fileid = client->vgt;
	Reply(rep,client->pid);
        client->requestFlag &= ~BlockBits;
      }
  }


static DropEvent(client,victim)
    register struct Client *client;
    struct Event *victim;
  {
	/*
	 * Remove the given event from the the Event Queue
	 * for this client
	 */
    register struct Event *e, *last;
    
    last = NULL;
    e = client->eventQ;

    while (e && victim != e)
      {
        last = e;
        e = e->next;
      }
    if (e==NULL) return;

    if (last)
      last->next = e->next;
    else
      client->eventQ = e->next;

    if (client->eventTail==e) client->eventTail = last;
    e->next = FreeEvents;
    FreeEvents = e;
  }



/*
 * Routines that handle graphical events
 */

PutGraphicsEvent(vgt, x, y, buttons, uptrans)
    short vgt, x, y, buttons;
    register int uptrans;
  {
	/*
	 * Put a mouse event on the queue.  Should be invoked only when
	 * this VGT is open to mouse input in some form.
	 * If the client is a slave, then we put it on the master's queue.
	 */
    register struct Client *client = ClientTable + vgt;
    register struct Event *e;
    register int mousemode = client->mode & (ReportTransition | ReportClick);
    register eventcode;

    if (client->master>0)
      {
        client = ClientTable + client->master;
	mousemode = ReportClick;
      }

    e = FreeEvents;
    if (e == NULL) return;
    FreeEvents = e->next;

    if (client->eventTail)  client->eventTail->next = e;
    if (client->eventQ == NULL) client->eventQ = e;
    client->eventTail = e;
    e->next = NULL;

    eventcode = ReportTransition;
    if (uptrans) eventcode |= ReportClick;

    e->code = eventcode;
    e->x = x;  e->y = y;
    e->buttons = buttons;
    e->fileid = vgt;

    EventToUser(client);
  }



HandleGraphics(msg)
    struct EventMsg *msg;
  {
	/*
	 * The graphics input device has changed state, 
	 * and the view manager was
	 * not waiting for it.  In a requesting graphics view, or a selected
	 * requesting pad, we queue and possibly send an event.  Elsewhere,
	 * the right button invokes the view manager and the other buttons
	 * cause this VGT to be selected, if it is a pad.  Graphics VGTs
	 * cannot be selected, but the middle button always pulls a graphics
	 * view to the top.
	 */
    short vgt, user, view;
    short xw, yw, buttons = msg->buttons;
    static lastButtons = 0, gatherButtons = 0;
    int padmode;

    CursorUpdate();
    if (buttons==lastButtons) return;

    view = FindMouseClick(msg->x, msg->y, &xw, &yw);
    vgt = view < 0 ? -1 : ViewTable[view].vgt;

    lastButtons = buttons;
    gatherButtons |= buttons;

    if (vgt<0 || ClientTable[vgt].master <= 0)
      user = vgt;
    else
      user = ClientTable[vgt].master;

    padmode = user<0 ? 0 : ClientTable[user].mode;

    if (user==InputPad &&  (
	     (padmode & ReportClick) && buttons==0 ||
	     (padmode & ReportTransition) ) )
      {
	PutGraphicsEvent(vgt, xw, yw, gatherButtons, buttons==0);
	gatherButtons = 0;
	return;
      }

    if (vgt==InputPad && (padmode & ReportEscSeq))
	  {
	    if (buttons == 0 && gatherButtons != 0) 
		{
		  InterpMouseEscape(vgt, xw, yw, gatherButtons);
		  gatherButtons = 0;
		}
	    return;
	  }

    if (buttons == 0) 
     {
     		/*
		 * This is the default case, if the click was 
		 * not in a VGT associated with the current intput master.
		 */
         switch (gatherButtons)
          { 
            case RightButton:  
	        vgt = manager(InputPad);
	        break;

	    case MiddleButton:
	    case LeftButton:
	        if (user>0)
	            SelectForInput(user);
	        if (user!=vgt || vgt==0) MakePadTop(vgt);
	        break;
          }
        gatherButtons = 0;
     }
  }




HandleEventRequest(client, msg, pid)
    register struct Client *client;
    register struct EventMsg *msg;
    ProcessId pid;
  {
	/*
	 * The user has called for an event of either type.  Reporting of
	 * mouse events is controlled by the mode of the pad.  Whichever
	 * comes first, a mouse click (or transition), or a keystroke, that
	 * will be returned.
	 */
    if (msg->eventcode & ReportMotion)
	{
	  HandleGetGraphicsStatus(msg, pid, 0);
	  return;
	}
    if ((client->requestFlag&BlockBits) && 
    	  AwaitingReply(VgtsPid,client->pid) )
    	return;			/* should reply with error code!! */
    client->requestFlag |= Mousereq;
    if (msg->eventcode & ReportCharacter)
        client->requestFlag |= Keyreq;
    client->pid = pid;
    EventToUser(client);
  }


HandleGetGraphicsStatus(msg, pid, mouseFlag)
    EventReq *msg;
    int pid;
    int mouseFlag;	/* View manager is in control */
  {
    short xw, yw, topvgt;
    short vgt = msg->fileid;
    register struct Client *client = ClientTable + vgt;

	/*
	 * a direct graphics status request flushes
	 *  ALL queued events in this pad, and the controlling pad.
	 */

    FlushEvents(client);
    if (client->master>0)
        FlushEvents(ClientTable+client->master);

    topvgt = FindTopVGT(MouseX, MouseY);
    if (mouseFlag==0 && topvgt>0 && (vgt == topvgt || 
                         ClientTable[topvgt].master == vgt) )
      {
	FindMouseClick(MouseX, MouseY, &xw, &yw);
	msg->x = xw;  msg->y = yw;
	msg->buttons = MouseButtons;
	msg->fileid = topvgt;
	msg->requestcode = OK;
      }
    else msg->requestcode = END_OF_FILE;  /* mouse not in requesting VGT */
    Reply(msg, pid);
  }

 
FlushEvents(client)
    register struct Client *client;
  {
   if (client->eventQ != NULL)
      {
	client->eventTail->next = FreeEvents;
	FreeEvents = client->eventQ;
	client->eventQ = client->eventTail = NULL;
      }
}
