/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/


/*
 * FILE: draw.c
 *
 * Bill Nowicki September 1982
 *
 * This file contains all the screen drawing routines.
 * Currently that is a set of routines to manipulate rectangular 
 * regions of the screen.
 * All references to the SUN framebuffer reside in this file.
 */


# include "Vgts.h"
# include "sdf.h"
# include <rasterops.h>
# include <framebuf.h>
# include <confreg.h>
# include <text.h>
# include <sfont.h>
# include <bitmaps.h>

/*
 * SUN framebuffer specific definitions.
 */

int GXBase;

# define FontHeight 16

extern short Debug;

short ScreenLimitX, ScreenLimitY;	/* framebuffer limits */


short GridStip[] = {
			0x7FFF, 0xFFFF, 0xFFFF, 0xFFFF,
			0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
			0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
			0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF };

short Background[] = {
			0x1111, 0x2222, 0x8888, 0x4444,
			0x1111, 0x2222, 0x8888, 0x4444,
			0x1111, 0x2222, 0x8888, 0x4444,
			0x1111, 0x2222, 0x8888, 0x4444 
		     };


/*********************************************************************
 * InitDraw:
 ********************************************************************/

/*
 * Initialization routine for screen drawing routines.
 */

InitDraw()
  {
  	/*
	 * Read the configuration register and
	 * adapt to various different frame buffers.
	 */
    short config;
    
    config = emt_getconfig();
    GXBase = GXDefaultBase;
    if ( ( (struct ConfReg*)&config)->FBType )
      {
        ScreenLimitX = 1024;
	ScreenLimitY = 800;
      }
    else
      {
        ScreenLimitX = 800;
	ScreenLimitY = 1024;
      }
    ScreenEnable();
  }

ScreenEnable()
  {
    GXcontrol = GXvideoEnable;
  }

ScreenDisable()
  {
    GXcontrol = 0;
  }


/*********************************************************************
 * ClearRegion:
 ********************************************************************/

/*
 * Clears specified region of the screen.  
 */

ClearRegion(x, y, w, h)
  short x, y, w, h;
  {
    struct fb_raster window;

    window.x = x;
    window.y = y;
    window.height = h;
    window.width = w;

    GXfunction = GXset;
    RasterFill(&window);
  }


/*********************************************************************
 * FillRegion:
 ********************************************************************/

/*
 * Blackens a portion of the screen.
 */

FillRegion(x, y, h, w)
  short x, y, h, w;
  {
    struct fb_raster dst;
    dst.x = x;
    dst.y = y;
    dst.width = w;
    dst.height = h;
    GXfunction = GXclear;
    RasterFill(&dst);
  }


/*********************************************************************
 * CopyRegion:
 ********************************************************************/

/*
 * Copy specified region of the screen to specified location on the
 * screen.
 */

CopyRegion(dstX, dstY, srcX, srcY, h, w)
    short dstX, dstY, srcX, srcY, h, w;
  {
    struct fb_raster dst, src;

    dst.x = dstX;
    dst.y = dstY;
    src.x = srcX;
    src.y = srcY;
    dst.height = src.height = h;
    dst.width = src.width = w;
    GXfunction = GXcopy;
    RasterCopy(&dst,&src);
  }


/*********************************************************************
 * InvertRegion:
 ********************************************************************/

/*
 * Inverts specified region.
 */

InvertRegion(x, y, h, w)
    short x, y, h, w;
  {
    struct fb_raster dst;

    dst.x = x;
    dst.y = y;
    dst.height = h;
    dst.width = w;
    GXfunction = GXinvert;
    RasterFill(&dst);
  }


/*********************************************************************
 * XorRegion:
 ********************************************************************/

/*
 * Xors specified region with memory raster pattern.
 */

XorRegion(dstX, dstY, mStart, h, w)
    short dstX, dstY, *mStart, h, w;
  {
    struct fb_raster dst;
    struct mem_raster msrc;

    dst.x = dstX;
    dst.y = dstY;
    dst.width = msrc.width = w;
    dst.height = msrc.height = h;
    msrc.start = mStart;
    GXfunction = GXxor;
    RasterPut(&dst, &msrc);
  }


/*********************************************************************
 * DrawOutline:
 ********************************************************************/

/*
 * DrawOutline() "ors" in a black outline with absolute corner (x, y) and
 * width and height w and h, respectively.  Thickness is the thickness (in
 * pixels) of the outline.  Select is four bits in the order left, right,
 * top, and bottom for bits 0, 1, 2, and 3 respectively.  Only these parts
 * of the outline are printed.
 */

DrawOutline(x, y, w, h, thickness, select)
  short x, y, w, h, thickness, select;
{
    if (select & LeftEdge)
        FillRegion(x, y, h, thickness);
    if (select & RightEdge)
        FillRegion(x+w-thickness, y, h, thickness);
    if (select & TopEdge)
        FillRegion(x, y, thickness, w);
    if (select & BottomEdge)
        FillRegion(x, y+h-thickness, thickness, w);
}


FlashOutline(xmin, ymin, xmax, ymax, select)
  {
	/* 
	 * Invert the indicated edges
	 */
    struct fb_raster dst;
    register struct fb_raster *r = &dst;
    short clipXmin, clipYmin, clipXmax, clipYmax;

    GXfunction = GXinvert;

    ymax -= 2;		/* Since lines go inside region */

    clipXmin = max(xmin, 2);
    clipYmin = max(ymin, 2);
    clipXmax = min(xmax, ScreenLimitX-2);
    clipYmax = min(ymax, ScreenLimitY-2);
    
    if (clipXmin >= ScreenLimitX -2 ) return;
    if (clipXmax < 2 ) return;
    if (clipYmin >= ScreenLimitY -2 ) return;
    if (clipYmax < 2 ) return;

    r->width = 2;
    r->height = clipYmax - clipYmin + 1;
    r->y = clipYmin;

    if ( (select & LeftEdge) && xmin>2 && r->height>2)
      {
        r->x = xmin;
	RasterFill(r);
      }
    if ( (select & RightEdge) && xmax<ScreenLimitX-2 && r->height>2)
      {
        r->x = xmax-2;
	RasterFill(r);
      }

    r->height = 2;
    r->width = clipXmax - clipXmin + 1;
    r->x = clipXmin;

    if ( (select & TopEdge) && ymin>2 && r->width>2)
      {
        r->y = ymin;
	RasterFill(r);
      }
    if ( (select & BottomEdge) && ymax<ScreenLimitY-2 && r->width>2)
      {
        r->y = ymax;
	RasterFill(r);
      }
}



/*********************************************************************
 * ClearRectangle:
 ********************************************************************/

/*
 * Clears designated rectange with or without a grid redrawn.
 */

ClearRectangle(x, y, h, w, withGrid)
    short x, y, h, w;
    BOOLEAN withGrid;
  {
    ClearRegion(x, y, w, h);
    if (withGrid)
	DrawRectangle( x, y, w, h, GridStip);
  }


/*********************************************************************
 * DrawRectangle:
 *
 * Draw a rectangle with the indicated stipple pattern.
 ********************************************************************/

DrawRectangle(x, y, w, h, stip_ptr)
  short x, y, w, h;
  short *stip_ptr;
{
    struct fb_raster dst;

    dst.x = x;
    dst.y = y;
    dst.width = w;
    dst.height = h;
    GXfunction = GXandPattern;
    RasterPattern( &dst, stip_ptr );
}


DrawBackground(x, y, w, h)
  short x, y, w, h;
{
    struct fb_raster dst;

    dst.x = x;
    dst.y = y;
    dst.width = w;
    dst.height = h;
    GXfunction = GXcopyPattern;
    RasterPattern( &dst, Background );
}


/*******************************************************************/

# define CursorSize 32		/* number of shorts in the cursor */
# define CursorWidth 32		/* width in bits */
# define CursorHeight 16	/* height in bits */

static short WindowCursor[CursorSize] =
    {0176000, 0174000, 0170000, 0174000, 0156000, 0107000, 0003400,
     0141403, 0140603, 0141703, 0143143, 0146063, 
     0154033, 0170017, 0160007, 0140003,
     0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0
     };		/* cursor for window manager */

static short PadCursor[CursorSize] =
    {0177777, 0177777, 0140000, 0143760, 
     0143770, 0143014, 0143014, 0143014, 
     0143770, 0143760, 0143000, 0143000, 
     0143000, 0143000, 0143000, 0143000, 
     0177777, 0177777, 0, 0, 
     0000003, 0000003, 0000003, 0003003, 
     0035017, 0077037, 0143063, 0143063,
     0143063, 0077067, 0035035, 0003003
     };		/* cursor for Pad creation */

static short ExecCursor[CursorSize] = {
	0xffff,	0xffff,	0xc000,	0xc000,	0xcf80,	0xcf80,	0xcc00,	0xcc00,
	0xcc31,	0xcf3b,	0xcf1f,	0xcc0e,	0xcc0e,	0xcc1f,	0xcfbb,	0xcfb1,
	0xffff,	0xffff,	0x0,	0x0,	0x0,	0x0,	0x0,	0x0,
	0x8e1e,	0x9f3f,	0x31b1,	0x3fb0,	0x3030,	0x31b1,	0xbfbf,	0x9f1e
     };		/* cursor for Exec creation */

static short ZoomCursor[CursorSize] =
    {0176000, 0174000, 0170000, 0174000, 
     0146000, 0003000, 0177740, 0177700, 
     0000600, 0001401, 0003002, 0006006, 
     0014006, 0030006, 0077742, 0177741,
     0, 0, 0, 0, 
     0, 0, 0, 0, 
     0000000, 0101412, 0042225, 0066325, 
     0066325, 0066325, 0042225, 0101425
     };		/* cursor for Zoom mode */

static short ViewCursor[CursorSize] =
    {
	0xffff,	0xffff,	0xc000,	0xc000,	0xe010,	0xe030,	0xe036,	0xf066,
	0xf066,	0xd8c0,	0xd8c6,	0xcd86,	0xcd86,	0xc506,	0xc706,	0xc206,
	0xffff,	0xffff,	0x0,	0x0,	0x0,	0x0,	0x0,	0x0,
	0x0,	0x1c63,	0x3e63,	0x636b,	0x7f6b,	0x607f,	0x3677,	0x1c63
     };		/* cursor for view manager */

static short NormalCursor[CursorSize] =
    {0177000, 0176000, 0174000, 0174000, 0176000, 0157000, 0107400,
     03600, 01700, 0740, 0360, 0170, 074, 036, 016, 04,
     0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0
     };		/* Raster definition for the normal cursor image. */

static enum CursorStateType
  {
    Disabled,			/* don't touch it */
    Enabled,			/* move it normally when mouse moves */
    Changed,			/* rewrite it ASAP */
    Selecting,			/* XOR popup menu selections */
    WasSelecting,		/* in menu, but interrupted */
    WasChanged			/* changed, but interrupted */
  }
 CursorState = Disabled;	

static short CursorSave[CursorSize];		/* buffer saved from screen */
static BOOLEAN CursorOn = FALSE;		/* cursor is On */
short	 MouseX, MouseY;

static struct mem_raster 
	CursorRaster =	{ CursorHeight, CursorWidth, NormalCursor },
 	SaveRaster =	{ CursorHeight, CursorWidth, CursorSave };

static struct fb_raster DrawRaster =  { CursorHeight, CursorWidth, 0, 0 };
static struct fb_raster flash;

/*
 * ChangeCursor:
 *
 * Changes to the new cursor.
 */

ChangeCursor(newCursorPtr)
  short *newCursorPtr;
{
   CursorRaster.start = newCursorPtr;
   CursorState = Changed;
   Cursor();
}

SetViewCursor()
{
    ChangeCursor(ViewCursor);
}

SetZoomCursor()
{
    ChangeCursor(ZoomCursor);
}

SetPadCursor()
{
    ChangeCursor(PadCursor);
}

SetExecCursor()
{
    ChangeCursor(ExecCursor);
}

SetMainCursor()
{
    ChangeCursor(NormalCursor);
}

static mxmin,mxmax,mymin,mymax;		/* bounding box of pop-up menu */
static lastStart;			/* last entry we flashed */
static HaveToRedraw;			/* pop-up interference flag */

MenuCursor(xmin,xmax,ymin,ymax)
  {
	/*
	 * Change to menu mode
	 */
    mxmin = xmin;
    mxmax = xmax;
    mymin = ymin;
    mymax = ymax;
    lastStart = -1;
    HaveToRedraw = 0;
    CursorOn = 0;
    CursorState = Selecting;
    DoSelect();
  }

DisableCursor(dstX, dstY, srcX, srcY, h, w)
  {
  	/*
	 * called after we do a pop-up menu, to indicate that we
	 * are out of selecting mode.
	 *
	 * Returns 0 if everything went OK, true if we had to
	 * redraw everything behind the popup menu.
	 */
    int prev = HaveToRedraw;

    CursorState = Disabled;
    if (HaveToRedraw)
      {
	CopyRegion(dstX, dstY, srcX, srcY, h, w);
        RedrawScreen(mxmin,mxmax,mymin,mymax,NULL);
      }
    HaveToRedraw = 0;
    return(prev);
  }

/*
 * EraseCursor:
 * Erase the cursor image from the screen so that it doesn't
 * conflict with other drawing going on.
 *
 * The bounding box of the area being worked on is passed,
 * so we can just leave the cursor on if it is outside.
 *
 * Returns true if the area might conflict with a popup menu.
 */

EraseCursor(xmin, xmax, ymin, ymax)
  int xmin, xmax, ymin, ymax;
{
    switch (CursorState)
      {
        case Selecting:
	case WasSelecting:
	    if ( (xmin < mxmax && xmax > mxmin) 
	      && (ymin < mymax && ymax > mymin) )
	    	{
			/*
			 * Darn! we want to draw part of the screen
			 * that might overlap the pop-up menu.
			 * Set a flag and return an error.
			 */
		  HaveToRedraw = 1;
		  return(1);
		}
	    CursorState = WasSelecting;
	    break;

	case Changed:
	    CursorState = WasChanged;
	    break;

        default:
            CursorState = Disabled;
      }
    if (! CursorOn)
        return(0);
    if ( xmin>DrawRaster.x+CursorWidth || xmax<DrawRaster.x ||
    	 ymin>DrawRaster.y+CursorHeight || ymax<DrawRaster.y )
      return(0);
    CursorOn = FALSE;
    GXfunction = GXcopy;
    RasterPut( &DrawRaster, &SaveRaster ); 
    return(0);
}


/*******************************************************************/


/*
 * Cursor:
 * Redraw the cursor image on the screen.  This routine is used in
 * conjunction with EraseCursor() to prevent the cursor image from
 * interfering with other drawing that is to be done.
 * The routine also prevents the cursor from going off the screen.
 *
 * If the position has not changed since the last write,
 * it is not re-written.
 */

Cursor()
{

    if (CursorOn &&  ( CursorState==Enabled || CursorState==Disabled)
    	 && DrawRaster.x == MouseX 
	 && DrawRaster.y == MouseY)
      {
        CursorState = Enabled;
    	return;
      }

    if (CursorState==WasSelecting)
      {
        CursorState = Selecting;
        return;
      }
    if (CursorOn)
      {
        GXfunction = GXcopy;
        RasterPut( &DrawRaster, &SaveRaster );
      }
    DrawRaster.x = MouseX;
    if (DrawRaster.x < 0)  DrawRaster.x = 0;
    if (DrawRaster.x > ScreenLimitX - CursorWidth)
    	 DrawRaster.x = ScreenLimitX - CursorWidth;
    DrawRaster.y = MouseY;
    if (DrawRaster.y < 0)  DrawRaster.y = 0;
    if (DrawRaster.y > ScreenLimitY - CursorHeight)
    	 DrawRaster.y = ScreenLimitY - CursorHeight;
    if (DrawRaster.x != MouseX || DrawRaster.y != MouseY)
    	SetMouseTo(DrawRaster.x,DrawRaster.y);

    GXfunction = GXcopy;
    RasterGet( &SaveRaster, &DrawRaster );
/*
 * This can be changed for personal preference
    GXfunction = GXxor;
 */
    GXfunction = GXpaintInverted;
    RasterPut( &DrawRaster, &CursorRaster );
    CursorOn = TRUE;
    CursorState = Enabled;
}

static DoSelect()
{	
	/*
	 * select an item out of a menu.
	 * If we have not moved, do nothing.
	 * Otherwise, save the cursor area, 
	 *  then invert the menu item that we would select,
	 *  and finally Xor the cursor image.
	 */
    if (CursorOn
    	 && DrawRaster.x == MouseX 
	 && DrawRaster.y == MouseY)
	   return;

    if (CursorOn)
      {
        GXfunction = GXcopy;
        RasterPut( &DrawRaster, &SaveRaster );
      }

    flash.height = FontHeight-1;
    flash.width  = mxmax - mxmin - 2;
    flash.x = mxmin+1;
    if (MouseX >= mxmin && MouseX < mxmax &&
        MouseY >= mymin && MouseY < mymax )
	  {
	    /*
	     * flash a part of the pop-up menu
	     */
	     short start = mymin;

	     while (start+FontHeight < MouseY) start += FontHeight;
	     if (lastStart!=start)
	       {
	         GXfunction = GXinvert;
	         if (lastStart>=0)
	           {
	             flash.y = lastStart+1;
	             RasterFill( &flash );
	           }
	         flash.y = start+1;
	         RasterFill( &flash );
	         lastStart = start;
	       }
	  }
	else if (lastStart>=0)
	       {
	         flash.y = lastStart+1;
	    	 GXfunction = GXinvert;
		 RasterFill( &flash );
		 lastStart = -1;
	       }
    DrawRaster.x = MouseX;
    if (DrawRaster.x < 0)  DrawRaster.x = 0;
    if (DrawRaster.x > ScreenLimitX - CursorWidth)
    	 DrawRaster.x = ScreenLimitX - CursorWidth;
    DrawRaster.y = MouseY;
    if (DrawRaster.y < 0)  DrawRaster.y = 0;
    if (DrawRaster.y > ScreenLimitY - CursorHeight)
    	 DrawRaster.y = ScreenLimitY - CursorHeight;
    if (DrawRaster.x != MouseX || DrawRaster.y != MouseY)
    	SetMouseTo(DrawRaster.x,DrawRaster.y);
    GXfunction = GXcopy;
    RasterGet( &SaveRaster, &DrawRaster );
    GXfunction = GXpaintInverted;
    	/*
	 * The Cursor Xor's instead of OR's when you are selecting
	 * something, except for the last half line
	 */
    if (lastStart>=0 && DrawRaster.y < mymax - FontHeight/2)
    	GXfunction = GXxor;
    RasterPut( &DrawRaster, &CursorRaster );
    CursorOn = TRUE;
}


/*
 * CursorUpdate:
 * Updates the cursor image position if a cursor image is allowed on
 * the screen.
 */

CursorUpdate()
  {
    switch (CursorState)
      {
        case Changed:
        case Enabled:   Cursor();	break;
	case Selecting:	DoSelect();	break;
      }
  }


DrawLine(x,y,dx,dy)
 int x, y, dx, dy;
  {
    /*
     * Draw a vector 
     */
     GXfunction = GXclear;
     GXwidth = 1;
     Line(x,y,dx,dy);
  }


DrawGeneralText(xmin, ymin, s, r, fontNumber, paintFlag)
  short xmin, ymin;
  char *s;
  struct SubView *r;
  char fontNumber;
  {
  	/*
	 * Draw some generalized text, using Per's routine.
	 */
    static struct fb_raster temp;
    int code =  (paintFlag ? TextStdPaint : (TextStdCopy+InverseIfHigh))
	    + fontNumber;
    FontEntry *fontp, *LookupFont();

    temp.x = r->ScreenXmin;
    temp.y = r->ScreenYmin;
    temp.width = r->ScreenXmax - r->ScreenXmin + 1;
    temp.height= r->ScreenYmax - r->ScreenYmin + 1;

    if (fontp = LookupFont(fontNumber))
        ymin -= fontp->font->descent - 1;
    else return;

    if (s)
        WriteText(s, 999, xmin, ymin, code, &temp);
  }


extern FontTable Fonts;
static FontTable *junk = &Fonts;

TextSize(s,font,pxmin,pxmax,pymin,pymax)
    char *s;
    short *pxmin, *pxmax, *pymin, *pymax;
  {
	/*
	 * Calculate the width and height of a string
	 */
    int width=0, maxChars=999, code = TextStdPaint+font;
    short left = *pxmin, base = *pymin;
    FontEntry *fontp, *LookupFont();

    if (s) ScanText(s,&maxChars,&width,999,&code);
    if (fontp = LookupFont(font))
      {
        *pymin -= fontp->font->descent;
	*pymax = *pymin + fontp->font->height;
        if (Debug) printf("ymin = %d, ymax = %d\n", *pymin, *pymax);
      }
    else
      {
        if (Debug) printf("LookupFont %d returned null\n", font);
      }
    *pxmax = left + width;
  }

DisplayRaster(item, x, y, view, r, data)
	MemRaster *item;
	int x, y;
	View *view;
	register struct SubView *r;
	unsigned char data;
  {
    register struct fb_raster temp;
    int scale = view->zoom;
    int adjust;     /* so that pixels \centers/ are on grid lines */

    temp.x = r->ScreenXmin;
    temp.y = r->ScreenYmin;
    temp.width = r->ScreenXmax - r->ScreenXmin + 1;
    temp.height= r->ScreenYmax - r->ScreenYmin + 1;
    GXfunction = item->where & InverseRaster ? GXcopy : GXcopyInverted;
    if (scale < 0) scale = 0; if (scale > 4) scale = 4;
    if ((scale == 3 || scale == 4) && view->showGrid) scale |= ScaleFramed;
    adjust = (1 << (scale & 15)) >> 1;
    ShowRaster(item, x - adjust, y - adjust, scale, &temp);
  }
