/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/


/*
 * FILE: display.c
 *
 * Bill Nowicki August 1982
 *
 * This file contains various display support
 * routines for the virtual graphics terminal service.
 * popup menu, display (clipped) text in a graphics window, and
 * display a cursor.
 * These are the routines that have to know about VGTS private
 * data structures; the raster-op and draw.c routines are lower level.
 */



# include "Vgts.h"
# include <chars.h>
# include "sdf.h"


#define BLTSAVEX 800	
		/* Marks where the raster image is to be saved
		   that is overlayed by a popup menu. */
#define BLTSAVEY 800


extern short	ScreenLimitX,
		ScreenLimitY;

extern short Debug;

extern int TimesRoman12B;		/* for the pop-up menu */

short	 MouseX, MouseY, MouseButtons;

		/* Variables defining the state of the mouse device. */


/*******************************************************************/

/*
 * popup:
 * The routine popup() takes a menu and puts up (at current mouse position)
 * the appropriate pup-up menu on the screen.  It returns the number of the
 * selected menu item, or -1 if none were selected.
 * To create a menu, a
 * region of suitable size of the screen is blitted off to the invisible part,
 * and a menu is drawn over it.  A selection is taken and is returned.
 *
 * The menu is passed as a 0-terminated array of strings.
 */

short popup(menu)
  PopUpEntry menu[];
{
    short i;
    short menu_count;			/* number of strings in the menu */
    short screenX, screenY;		/* Upper left screen corner */
    short BltHeight;			/* Height of menu */
# define BltWidth 208			/* Width of menu */
    
    short saveXmin, saveXmax, saveYmin, saveYmax;
    short popup_value;
    static struct SubView fakeSubView;
    
/* First, count items in menu -- had better be <= 13 */

    menu_count = 0;
    while (menu[++menu_count].string)
	if (menu_count==13) break;

/* Now figure out where the upper-left corner goes, and the height of the
 * menu.  The menu is always located so that
 * it doesn't go off the edge of the screen.  
 * It is placed slightly farther from
 * the bottom of the screen so that it is easy to get the cursor tip into the
 * bottom selection in all cases.
 */

    BltHeight = 16*menu_count + 1;		/* +1 for bottom line */
    screenX = (MouseX - 2*BltWidth/3);
    screenY = (MouseY - BltHeight/2);
    if (screenX < 0) screenX = 0;
    if (screenY < 0) screenY = 0;
    if (screenX > ScreenLimitX - BltWidth)  screenX = ScreenLimitX - BltWidth;
    if (screenY > ScreenLimitY - BltHeight) screenY = ScreenLimitY - BltHeight;

    EraseCursor(screenX, screenX+BltWidth,
    		screenY, screenY+BltHeight);
    
    CopyRegion(BLTSAVEX, BLTSAVEY, screenX, screenY, BltHeight, BltWidth);

/* Now clear the Frame: */

    ClearRegion( screenX, screenY, BltWidth, BltHeight);

/* Now draw in the various lines */

    FillRegion(screenX, screenY, BltHeight, 1);
    FillRegion(screenX + BltWidth - 1, screenY, BltHeight, 1);
    for (i = 0; i <= menu_count; i++)
        FillRegion(screenX, screenY + i * 16, 1, BltWidth);

/*
 * The fake window is so the Display Text function does the
 * character clipping correctly.
 */
    fakeSubView.ScreenXmin = screenX;
    fakeSubView.ScreenXmax = screenX+BltWidth;
    fakeSubView.ScreenYmin = screenY;
    fakeSubView.ScreenYmax = screenY+BltHeight;

    for (i = 0; i < menu_count; i++)
# ifdef OneFont
        DisplayText(screenX+8, screenY + 16*i + (16 + 1),
		menu[i].string, &fakeSubView, TRUE );
# else OneFont
	if (TimesRoman12B)
            DrawGeneralText(screenX+8, screenY + 16*i + (16 - 1), 
		menu[i].string, &fakeSubView, Menu_Font, TRUE );
	else
            DisplayText(screenX+8, screenY + 16*i+ (16 + 1), 
		menu[i].string, &fakeSubView, TRUE );
# endif OneFont

/* OK.  The menu is up -- all we need to do is to interpret it. */
    
    MenuCursor(screenX,screenX+BltWidth,screenY,screenY+BltHeight);
    GetMouseInput();

    if (MouseX < screenX || MouseX > screenX + BltWidth ||
        MouseY < screenY || MouseY > screenY + BltHeight - 1)
    	popup_value = -1;
    else
        popup_value = menu[(MouseY - screenY - 1)/FontHeight].menuNumber;

    if (!DisableCursor(screenX, screenY, 
         BLTSAVEX, BLTSAVEY, BltHeight, BltWidth))
      {
      	/*
	 * If we did not have to redraw the world,
	 * we can just blit back what was there before.
	 */
        EraseCursor(screenX, screenX+BltWidth,
    		screenY, screenY+BltHeight);

        CopyRegion(screenX, screenY, BLTSAVEX, BLTSAVEY, BltHeight, BltWidth);
      }
    Cursor();
    return (popup_value);
}


DisplayBackground(w)
  View *w;
  {
  	/*
	 * draw in the background pattern for the indicated subViewPorts.
	 * this is trickier than it sounds, because we have to take into
	 * account any possible edges.
	 */
    register struct SubView *r;

    if (EraseCursor(w->ScreenXmin, w->ScreenXmax, w->ScreenYmin, w->ScreenYmax))
        return;
    for (r=w->rect;r;r=r->next)
      {
        int left   = (r->EdgeSet&LeftEdge)   ? 2 : 0;
        int right  = (r->EdgeSet&RightEdge)  ? 2 : 0;
        int top    = (r->EdgeSet&TopEdge)    ? 2 : 0;
        int bottom = (r->EdgeSet&BottomEdge) ? 2 : 0;

         DrawBackground( r->ScreenXmin - left, r->ScreenYmin - top, 
	     r->ScreenXmax - r->ScreenXmin+left+right+1, 
	     r->ScreenYmax - r->ScreenYmin+top+bottom+1 );
       }	
    Cursor();
  }


/*******************************************************************/


/*
 * DisplayText:
 * Draws specified string at designated location on the screen.
 * Checks for violation of screen boundaries - clip if necessary.
 * If paintFlag is true, the text is painted instead of written.
 */

DisplayText(xLeft, yBottom, string, r, paintFlag)
  int xLeft, yBottom;
  char *string;
  register struct SubView *r;
  int paintFlag;
{
    int length;
    int xRight;
    char *ch;

    if (string==NULL) return;
    length = strlen(string);
    xRight = xLeft + length*FontWidth;

    if (yBottom < r->ScreenYmin || 
        (yBottom - FontHeight)>r->ScreenYmax ||
	 xLeft  > r->ScreenXmax ||
	 xRight < r->ScreenXmin	 )
		/*
		 * Don't bother drawing it at all if it is
		 * going to be completely out of the view
		 */
        return;	

    if (yBottom > r->ScreenYmax || yBottom-FontHeight+1 < r->ScreenYmin)
      {
        if (xLeft >= ScreenLimitX) return;
        if (TimesRoman12B)
	  DrawGeneralText(xLeft, yBottom, string, r, CMR_Font, paintFlag );
	return;
      }

    if (xLeft >= r->ScreenXmin && xRight <= r->ScreenXmax)
        DrawText(xLeft, yBottom, string, length, paintFlag);
    else
      {
        if (xLeft >= r->ScreenXmin)
	  {
	    int fullChars = (r->ScreenXmax - xLeft)/FontWidth;
	    
            DrawText(xLeft, yBottom, string, fullChars, paintFlag);
	    string += fullChars;
	    xLeft += fullChars*FontWidth;
	    if (xLeft >= r->ScreenXmax || xLeft >= ScreenLimitX) return;
	  }
        if (TimesRoman12B)
            DrawGeneralText(xLeft, yBottom, string, r, CMR_Font, paintFlag );
      }
}


# define Code(x,y) (x<r->ScreenXmin ? LeftEdge : (x>r->ScreenXmax ? RightEdge : 0)) + \
   (y<r->ScreenYmin ? TopEdge : (y>r->ScreenYmax ? BottomEdge : 0)) 


# define Swap(x,y) {t=y; y=x; x=t;}

/*******************************************************************/

DisplayLine( x1, x2, y1, y2, r )
  register short x1, x2, y1, y2;
  register struct SubView *r;
  {
  	/*
	 * clips a line to a window and draws it.
	 */
    int edgeset1 = Code(x1,y1);
    int edgeset2 = Code(x2,y2);
    int t;
    
    if (x1==x2 && y1==y2) return;
    
    while (edgeset1 || edgeset2)
      {
          /*
	   * If neither part of the line is visible, return.
	   * otherwise, clip to a side determined by an edgeset,
	   * and try again.
	   */
        if (edgeset1 & edgeset2) return;
	if (edgeset1==0) 
	  {
	    Swap(edgeset1,edgeset2);
	    Swap(x1,x2);
	    Swap(y1,y2);
	  }
	if (edgeset1 & LeftEdge)
	  {
	    y1 = y1+(y2-y1)*(r->ScreenXmin-x1)/(x2-x1);
	    x1 = r->ScreenXmin;
	  }
	else if (edgeset1 & RightEdge)
	  {
	    y1 = y1+(y2-y1)*(r->ScreenXmax-x1)/(x2-x1);
	    x1 = r->ScreenXmax;
	  }
	else if (edgeset1 & TopEdge)
	  {
	    x1 = x1+(x2-x1)*(r->ScreenYmin-y1)/(y2-y1);
	    y1 = r->ScreenYmin;
	  }
	else if (edgeset1 & BottomEdge)
	  {
	    x1 = x1+(x2-x1)*(r->ScreenYmax-y1)/(y2-y1);
	    y1 = r->ScreenYmax;
	  }
	edgeset1 = Code(x1,y1);
      }
    DrawLine(x1,y1, x2-x1,y2-y1);
    
  }


DisplayBanner(w,r)
    register View_PTR w;
    register struct SubView *r;
  {
  	/*
	 * Display the banner for the given viewport with the
	 * given subviewport.
	 */

    if (TimesRoman12B)
      DrawGeneralText( w->ScreenXmin + 16, w->ScreenYmin+BannerHeight+1,
    		 w->banner, r, Menu_Font, FALSE );
    else
      DisplayText( w->ScreenXmin + 16, w->ScreenYmin+BannerHeight+1, 
    		 w->banner, r, FALSE );
  }

