/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 * Session handling code.
 * A user and workstation must signed on thru the server before accessing
 * any files.
 */

#include <wait.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vsession.h>
#include <pwd.h>
#include <server.h>
#include <debug.h>

/* Exports */
extern SystemCode TerminateSession();
extern SystemCode CreateSession();
extern SystemCode GetUserNumber();
extern unsigned char GetHostAddress();
SessionDesc *Session = NULL;
extern SystemCode ValidSessionPid();

/* Imports */
extern char *crypt(),*index();
extern ProcessId NewSessionPid, MyPid;
extern struct passwd *getpwnam();
extern Net, OldGroups;
extern SessionDesc *Session;
extern SetStringName();
extern char *GetSessionOwnerName();
extern char BlockBuffer[];
extern ProcessId KernelProcessPid;

/* Environment stuff for Process Instances */
#define MAX_ENV_VAR_LEN		256
extern char MyEnvPath[];		/* set in process.c */
char MyEnvTerm[] = "TERM=sun"; 
char MyEnvUser[ MAX_ENV_VAR_LEN + 5] = "USER=";
char MyEnvShell[ MAX_ENV_VAR_LEN + 6] = "SHELL=";
char MyEnvHome[ MAX_ENV_VAR_LEN + 5] = "HOME=";
char *MyEnvironment[] = 
  { 
    MyEnvHome, MyEnvPath, MyEnvShell, MyEnvTerm, MyEnvUser, 0
  };

SystemCode LookupServer( req, pid, segsize )
   register GetUserNumberRequest *req;
   ProcessId pid;
   unsigned segsize;
  /* 
   * Get the server pid corresponding to the specified server name.
   */
  {
#define reply	((GetUserNumberReply*)req)
    char *name;
    SystemCode r;
    unsigned char host;
    unsigned nameLength;
    unsigned namePtr;

    if( SDebug ) printf( "Lookup server request from %8x\n", pid );

    MoveLong(req->namelength,nameLength);
    MoveLong(req->username,namePtr);
    if( nameLength > 100 ) return( ILLEGAL_REQUEST );
    name = BlockBuffer;
    if( segsize < nameLength )
      {
	if( (r=MoveFrom(pid, namePtr, name, nameLength)) != OK ) return( r );
      }
    name[nameLength] = 0; /* make a string */

    if( (pid = GetServerPid(name)) == 0 ) return( NOT_FOUND );

    MoveLong( pid, reply->gid );

    return( OK );

#undef reply
  } /* LookupServer */

SystemCode CreateSession( req, pid, segsize )
    register CreateSessionRequest *req;
    ProcessId pid;
    unsigned segsize;
 /*
  * Create a new session. This involves validating the user password and
  * forking a server process to handle this session.
  */
  {
#define reply	((CreateSessionReply*)req)	/* reuse register */
    register SessionDesc *desc;
    register struct passwd *pwentry;
    register char *UserName;
    char *Password;
    SystemCode r;
    long seconds;
    int	net, i;
    unsigned NameLength;
    unsigned NamePtr;

    if( Session ) return( ILLEGAL_REQUEST );

    if( SDebug ) printf( "Create session for Pid %8x\n", pid );

    /* Retrieve the names */
    MoveLong( req->filenamelen, NameLength );
    if( NameLength > BUFFER_SIZE ) 
        return( ILLEGAL_REQUEST );

    /* Been transferred already? */
    if( segsize < NameLength )
      {
	MoveLong( req->filename, NamePtr );
	if( (r=MoveFrom(pid, NamePtr, BlockBuffer, NameLength)) != OK )  return( r );
      }

    BlockBuffer[NameLength] = 0; /* terminate password */
    UserName = &BlockBuffer[ req->filenameindex ];

    if ( SDebug ) printf( "UserName = \"%s\"\n", UserName );

    /* Find the user */
    if( (pwentry = getpwnam( UserName )) == NULL )  
        return( NOT_FOUND );

    /* Dont allow root or null password accounts to login */
    if( pwentry->pw_uid == 0 || strlen( pwentry->pw_passwd ) == 0 ) 
        return( NO_PERMISSION );

    Password = index( UserName, NULL ) + 1;

    /* Check the password */
    Password = crypt( Password, pwentry->pw_passwd ); /* Encrypt password */
    if( strcmp(Password, pwentry->pw_passwd) != 0 ) 
        return( NO_PERMISSION );

    if( SDebug ) printf( "Password validated for new session\n" );

    /* increment the pid and make sure that it doesn't conflict with
       the kernel process pid */
    if ( KernelProcessPid == ++NewSessionPid)
        NewSessionPid++;

    ReclaimInstances( 0 );	/* do some clean up */

    /* Get a new network file for this session */
    if( (net = NetOpen(NewSessionPid)) == -1 ) return( NO_SERVER_RESOURCES );

    seconds = time( 0 ); /* time of login */

    if( (desc = (SessionDesc *)AllocInstance( pid, SESSION_INSTANCE )) == NULL ) 
	return( NO_SERVER_RESOURCES );

    if ( req->sessionowner == 0 )
        desc->owner = pid;
    else
        MoveLong( req->sessionowner, desc->owner );

    if ( SDebug )
        printf( "Session owner is 0x%x\n", desc->owner );
    /* Set up instance so that it can't be read from or written to */
    desc->name = NULL;		/* sessions don't have real names */
    desc->filetype = 0;
    desc->lastblock = 0;
    desc->nextblock = 0;
    desc->blocksize = 0;
    desc->lastbytes = 0;
    desc->Vpid = NewSessionPid;

    /* Fork a session server for this new session */
    if( (desc->unixpid = fork()) == -1 )
      {
	close( net );
	reply->fileid = 0;
	FreeInstance( (FileInstance *) desc );
	return( NO_SERVER_RESOURCES );
      }

    if( desc->unixpid != 0 )
      { 
	/* PARENT main server */
	/* Close the session's network file */
        close( net );

	/* Reply to CreateInstance, can't use QueryInstance since
	   it always assigns MyPid to "fileserver" */
	MoveLong( NewSessionPid, reply->fileserver );
	reply->fileid = desc->id;  
        reply->filetype = desc->filetype;
	MoveLong(desc->lastblock,reply->filelastblock);
	MoveLong(desc->nextblock,reply->filenextblock);
	MoveLong(desc->blocksize,reply->blocksize);
	reply->filelastbytes = desc->lastbytes;

	return( OK );
      }

    /* CHILD new session server */
    sighold( PACKET_RECEIVED );

    if( SDebug ) printf( "New session created instance id 0x%x\n", desc->id );

    desc->unixpid = getpid();

    ReclaimInstances( 1 );	/* won't free SessionDesc */
    close( Net );
    if (FDebug)CheckFiles();
    Net = net;
    MyPid = NewSessionPid;

    /* Finish setting up the instance */
    desc->uid = pwentry->pw_uid;
    desc->gid = pwentry->pw_gid;
    desc->logintime = seconds;
    Session = desc;

    nice( -4 );	/* Run at high priority */

    /* Change to run as the specified user. */

    if( OldGroups ) /* Original Unix group mechanism in use. */
	setgid( pwentry->pw_gid );
    else /* Use new 4.2 style group mechanism. */
	inigrp( pwentry );

    setuid( pwentry->pw_uid );

    /* Set up the context list and change to LOGIN_CONTEXT provided */
    if ( (r = InitContexts( pwentry->pw_dir )) != OK )
      {
	if ( GDebug ) printf( "InitContexts failed code = 0x%x\n", r );
	/* Failure is caused by a bad directory, so best course of
	   action is to bail */
	exit( r );
      }

    /* Set up the Environment for this user */
    MyEnvHome[ 5 ] = NULL;
    if ( strlen( pwentry->pw_dir ) < MAX_ENV_VAR_LEN )
	strcat( MyEnvHome, pwentry->pw_dir );

    MyEnvUser[ 5 ] = NULL;
    if ( strlen( pwentry->pw_name ) < MAX_ENV_VAR_LEN )
	strcat( MyEnvUser, pwentry->pw_name );

    MyEnvShell[ 6 ] = NULL;
    if ( strlen( pwentry->pw_shell ) < MAX_ENV_VAR_LEN )
	strcat( MyEnvShell, pwentry->pw_shell );

    InitKernel( 1 );

    /* Session banners include host names now */
    sprintf( BlockBuffer, SESSION_BANNER, GetSessionOwnerName( desc->owner ) );
    SetStringName( BlockBuffer );

    return( OK );
#undef reply

  } /* CreateSession */

SystemCode ValidSessionPid( req, pid )
    KernelRequest *req;
    ProcessId pid;
  {
    ProcessId session;
    extern int ChildrenAreDead;

    /* Only the main server replies to Kernel Process messages */
    if ( Session != NULL ) return( NO_REPLY );
    
    /* Make sure we know the state of all sessions */
    if ( ChildrenAreDead )
      {
	ChildrenAreDead = 0;
        CheckSessions();
      }

    MoveLong( req->pid, session )
    if ( KDebug ) 
        printf( "ValidSessionPid: 0x%x\n", session );
    
    if ( session == MyPid || FindInstance( session, SESSION_INSTANCE ) != 0 )
        return( OK );
    else
        return( NONEXISTENT_PROCESS );
  } /* ValidSessionPid */    

CheckSessions()
  {
    struct wait status;
    int child;
    SessionDesc *desc;

    while ( (child = wait3( &status, WNOHANG|WUNTRACED, 0 )) > 0)
      {

	/* Search for all descriptors that match this pid */
        if ((desc = (SessionDesc *)FindInstance( child, PROCESS_INSTANCE )) 
	    == NULL)
          {
	    if (SDebug)
	       /* Status on a child we don't have a descriptor for. */
	        printf( "Session instance not found, UNIX pid = %d\n", child );
	    continue;
	  }

        if (SDebug) 
            printf( "Session instance 0x%x", desc->id );

	/* Only release instances if they DIE */
	if ( WIFSIGNALED( status ) || WIFEXITED( status ) )
	  {
            FreeInstance( desc );
	    if ( SDebug ) printf( " freed\n" );
	  }
	else if ( SDebug ) printf( " status is 0x%x\n", status );
      } 

  } /* CheckSessions */
