/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 * Main server code.
 *
 */

#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vsession.h>
#include <Vtime.h>
#include <signal.h>
#include <pwd.h>
#include <server.h>
#include <debug.h>

char BlockBuffer[BUFFER_SIZE+1];	/* For MoveFroms, MoveTos, etc. */

extern SystemCode CreateInstance();
extern SystemCode QueryInstance();
extern SystemCode ReadInstance();
extern SystemCode ReadAndForward();
extern SystemCode WriteInstance();
extern SystemCode ReleaseInstance();
extern SystemCode SetInstanceOwner();
extern SystemCode ModifyInstance();
extern SystemCode QueryFile();
extern SystemCode ModifyFile();
extern SystemCode RemoveFile();
extern SystemCode LookupServer();
extern SystemCode ProcessTimeout();
extern SystemCode QueueIORequest();
extern SystemCode ReadDescriptor();
extern SystemCode NameReadDescriptor();
extern SystemCode WriteDescriptor();
extern SystemCode NameWriteDescriptor();
extern SystemCode ValidSessionPid();
extern int AlarmRate;

extern ProcessId MyPid;
int PublicServer = 0;
int OldGroups = 0; /* Non-zero if using the old group protection mechanism */
char *StringName; /* Pointer to character string name used by ps */

main( argc, argv, envp )
    int argc;
    char *argv[];
    char *envp[];

  /* The V UNIX server
   *
   */
  {
    Message msg;
    register IoRequest *req = (IoRequest *) msg;
    ProcessId pid;
    unsigned segsize;
    SystemCode	reply;

    Initialize( argc, argv );

    for(;;)
      {
	segsize = BUFFER_SIZE;
	pid = ReceiveWithSegment( req, BlockBuffer, &segsize );

	/* Reset the alarm to fast rate when have an operation pending */
	if ( AlarmRate == SLOW_ALARM_RATE && req->requestcode != TIMEOUT )
	    alarm( AlarmRate = FAST_ALARM_RATE );
	    
	switch( req->requestcode )
	  {
	    case CREATE_INSTANCE:
		reply = CreateInstance( req, pid, segsize );
		break;
	    case RELEASE_INSTANCE:
		reply = ReleaseInstance( req, pid );
		break;
            case READ_INSTANCE:
		reply = ReadInstance( req, pid );
		break;
	    case READ_AND_FORWARD:
		reply = ReadAndForward( req, pid );
		break;
	    case WRITE_INSTANCE:
	    case WRITESHORT_INSTANCE:
		reply = WriteInstance( req, pid, segsize );
		break;
	    case QUERY_INSTANCE:
		reply = QueryInstance( req, pid );
		break;
	    case MODIFY_FILE:
		reply = ModifyFile( req, pid );
		break;
	    case QUERY_FILE:
		reply = QueryFile( req, pid );
		break;
	    case SET_INSTANCE_OWNER:
		reply = SetInstanceOwner( req, pid );
		break;
	    case GET_CONTEXT_ID:
		reply = GetContextId( req, pid, segsize );
		break;
	    case GET_CONTEXT_NAME:
	        reply = GetContextName( req, pid );
		break;
	    case GET_FILE_NAME:
	        reply = GetFileName( req, pid );
		break;
	    case REMOVE_FILE:
		reply = RemoveFile( req, pid, segsize );
		break;
	    case READ_DESCRIPTOR:
	        reply = ReadDescriptor( req, pid );
		break;
	    case NREAD_DESCRIPTOR:
	    	reply = NameReadDescriptor( req, pid, segsize );
		break;
	    case WRITE_DESCRIPTOR:
	    	reply = WriteDescriptor( req, pid, segsize );
		break;
	    case NWRITE_DESCRIPTOR:
		reply = NameWriteDescriptor( req, pid, segsize );
		break;
	    case LOOKUP_SERVER:
		reply = LookupServer( req, pid, segsize );
		break;
	    case GET_TIME:
		reply = GetUnixTime( req, pid );
		break;
	    case TIMEOUT:
		reply = ProcessTimeout( req, pid );
		break;
	    case QUERY_PROCESS:
		reply = ValidSessionPid( req, pid );
		break;
	    default: reply = ILLEGAL_REQUEST;
	  }
	if ( reply != OK && GDebug ) printf( "Replycode was 0x%x\n", reply );
	if( reply == NO_REPLY || reply == NOT_AWAITINGREPLY ) continue;
	req->requestcode = reply;
	Reply( req, pid );
      }
  }


Initialize( argc, argv )
   int argc;
   register char *argv[];
  /* Initialize the server, setting up the file instance descriptors.
   * Also, cause the pseudo-kernel to initialize.
   */
  {
    SystemCode r;
    extern long Debug;	/* just for setting */

    Debug = 0;
    /* Handle arguments if any */
    StringName = *argv;
    while( --argc > 0 )
      {
	++argv;
	while( **argv != '\0' )
            switch ( *((*argv)++) )
	      {
		/* real arguments */
		case 'p':	PublicServer = 1; /* answer GetPid requests */
				break;

		case 'g':	OldGroups = 1;   /* Use old (shasta) groups */
				break;

		/* Following are debugging options */
		case 'A':	Debug = -1;	/* turn on all debugging bits */
				break;
	      	case 'G':	Debug |= GENERIC_DEBUG;	
				break;
		case 'K':	Debug |= KERNEL_DEBUG;
				break;
		case 'N':	Debug |= NET_DEBUG;
				break;
		case 'I':	Debug |= INSTANCE_DEBUG;
				break;
		case 'F':	Debug |= FILE_DEBUG;
				break;
		case 'P':	Debug |= PROCESS_DEBUG;
				break;
		case 'S':	Debug |= SESSION_DEBUG;
				break;
		case 'D':	Debug |= DIRECTORY_DEBUG;
				break;
		case 'C':	Debug |= CONTEXT_DEBUG;
				break;
		case 'L':	Debug |= LOOKUP_DEBUG;
				break;
		default:
				break;
	      }
      }

    /* For some reason we cannot close stdio, this would be nice to
       supply some extra file descriptors to the client.  When stdio
       is closed funny things happen, such as, stderr output from 
       remote execution no longer works.
    */

    InitKernel( 0 ); /* Cause the kernel to initialize */
    InitInstances();
    InitIOQueue();
    /* Set LOGIN_CONTEXT and cd to it */
    if ( (r = InitContexts( PUBLIC_DIR )) != OK )
      {
	if ( GDebug )
            printf( "Error in InitContexts: Code = 0x%x\n", r );
	exit( r );
      }
    SetStringName( SERVER_BANNER );

    /* Set the logical pid map */
    SetPid( STORAGE_SERVER, GetPid(0) ); 
    SetPid( UNIX_SERVER, GetPid(0) ); 
    SetPid( TIME_SERVER, GetPid(0) ); 
  } /* Initialize */

SetStringName( str )
  register char *str;
  /* Set the command name read by the ps command */
  {
    register char *p = StringName;

    if( SDebug ) printf( "A new %s%s started\n"
    		, PublicServer ? "public " : "\0", str );
    while( *str && *p ) *p++ = *str++;
    while( *p ) *p++ = ' ';
  } /* SetStringName */


SystemCode ProcessTimeout( req, pid )
  IoRequest *req;
  ProcessId pid;
  /*
   * Increment the inactivity count.
   * Server checks every 5 timeout messages if
   * loginpid process exists and exits if it doesnt.
   * Also Check for processes which wish to report status
   * and check the queued IO requests.
   */
  {
    extern int ChildrenAreDead;
    extern KillSession();
    extern SessionDesc *Session;
    register int inactive;

    if (Session == NULL) 
      {
	if ( ChildrenAreDead )
	  {
	    ChildrenAreDead = 0;
	    CheckSessions();
	  }
	/* Reset the alarm, since we have been idle for some time */
	/* we really should have an inactivity count for the main server */
	AlarmRate = SLOW_ALARM_RATE;
        return( OK );
      }
    inactive = Session->inactivity;
    if ( ChildrenAreDead ) 
      {
	ChildrenAreDead = 0;
	CheckProcesses();
      }
     
    CheckIOQueue();

    Session->inactivity = inactive;
    ++(Session->inactivity);

    if( Session->inactivity > 5 )
      {
	if( ValidPid(Session->owner) == 0 ) 
	    KillSession();

	AlarmRate = SLOW_ALARM_RATE;	/* slow up, we're not so busy */
	Session->inactivity = 0;
      }
    return( OK );
  } /* ProcessTimeout */
