/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 * Instance routines
 */

#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vsession.h>
#include <Vtime.h>
#include <sgtty.h>
#include <signal.h>
#include <pwd.h>
#include <server.h>
#include <types.h>
#include <stat.h>
#include <debug.h>

extern FileInstance *AllocInstance();
extern FileInstance *GetInstance();
extern FreeInstance();
extern FileInstance *FindInstance();

InstanceId	Idgenerator = 0x5555; /* Random strange number to start */

FileInstance *FileInstances = NULL;

InitInstances()
    /* 
     * Initialize the instance list so that the head of the list is
     * always a null record.  This makes inserting and deleting from the
     * head of the list very simple.
     */
  {
    FileInstances = (FileInstance *) calloc( 1, sizeof(FileInstance *) );
  } /* InitInstances */


FileInstance *AllocInstance( pid, instancetype )
    ProcessId pid;
    InstanceType instancetype;

  /* Allocate a file or process instance descriptor and
   * return a pointer to the descriptor. The instance id is set
   * to the next "generation" for this descriptor.
   * Note that calloc must be used (vs. malloc), since some of the fields
   * must be zero.
   */
  {
    extern FileInstance *FileInstances;
    register FileInstance *desc;

    while( (++Idgenerator == 0) || GetInstance(Idgenerator) );

    switch ( instancetype ) 
      {
	case FILE_INSTANCE:
	    desc = (FileInstance *) calloc( 1, sizeof( FileInstance ) );
	    break;

	case PROCESS_INSTANCE:
	    desc = (FileInstance *) calloc( 1, sizeof( ProcessInstance ) );
	    break;

	case DIRECTORY_INSTANCE:
	    desc = (FileInstance *) calloc( 1, sizeof( DirectoryInstance ) );
	    break;

	case SESSION_INSTANCE:
	    desc = (FileInstance *) calloc( 1, sizeof( SessionDesc ) );
	    break;

	default:
	    /* This is an internal error */
	    if ( GDebug )
		printf( "AllocateInstance called with unknown instancetype = %d\n"
		    , instancetype );
	    return( NULL );    /* Internal error */
	    break;
      }
    if ( desc == NULL ) return( NULL );	/* calloc failed */

    desc->name = NULL;
    desc->type = instancetype;
    desc->owner = pid;
    desc->id = Idgenerator;
    desc->link = FileInstances->link;
    FileInstances->link = desc;

    return( desc );
  }

FileInstance *GetInstance( fid )
    register InstanceId fid;

  /* Locate the instance descriptor with the corresponding
   * instance id if any and return a pointer to this descriptor.
   * Return NULL if no such descriptor.
   */
  {
    extern FileInstance *FileInstances;
    register FileInstance *desc;

    desc = FileInstances->link;

    while( desc != NULL )
      {
	if( desc->id == fid ) return( desc );
	desc = desc->link;
      }
    return( NULL );
  }

FreeInstance( desc )
    FileInstance *desc;

  /* 
   * Free the specified instance descriptor.
   */
  {
    register FileInstance *ptr;

    ptr = FileInstances;

    while( ptr != NULL && ptr->link != desc ) ptr = ptr->link;

    if( ptr == NULL )
      {
	if ( GDebug ) 
	    printf( "Freeing bad instance descriptor\n" );
      }
    else
      {       
	ptr->link = desc->link;
	ptr = desc;
	/* DirectoryInstances don't alloc their names, the context cache does */
	if ( ptr->name != NULL && ptr->type != DIRECTORY_INSTANCE ) 
	    free( ptr->name );

	if ( ptr->type == PROCESS_INSTANCE && 
	     ((ProcessInstance*)ptr)->lastdata != NULL )
	         free( ((ProcessInstance*)ptr)->lastdata );
	free( ptr );
      }
  }
unsigned ReclaimInstances( ReleaseAll )
    int ReleaseAll;

 /*
  * Check that the owner of each open instance exists.
  * If it doesn't, then for instances we must close the file
  * first and for jobs we must kill the process and close the pipe.
  * Return the number of instances that were reclaimed.
  */
  {
    register int reclaimed = 0;
    register FileInstance *ptr, *prev;
    register ProcessInstance *pd;
    int upid;

    if (IDebug) 
	CheckFiles();
    prev = FileInstances;

    while( (ptr = prev->link) != NULL )
      {
	if (IDebug) printf( "Checking instance 0x%x\n", ptr );
	if( ReleaseAll || !ValidPid(ptr->owner) ) 
	  {
	    switch (ptr->type) 
	      {
		case FILE_INSTANCE:
		    if ( ptr->name != NULL ) 
		        free( ptr->name );	/* Don't free dir names */
		case DIRECTORY_INSTANCE:
		    close( ptr->unixfile );
		    prev->link = ptr->link;
		    free( ptr );
		    break;

		case PROCESS_INSTANCE:
		    /*
		     * Kill the process, if not already dead.
 		     * We must have at least one process descriptor around
		     * to "catch" the signal that should be received.
		     */
		    pd = (ProcessInstance *) ptr;
		    if ( pd->state == PD_RUNNING || pd->state == PD_STOPPED )
		      {
		        if ( close( pd->pipedesc ) == 0 )
			    pd->pipedesc = -1;	/* invalidate */
			else
			    reclaimed--;	/* nothing freed */

			upid = pd->pid;		/* save this process id */
			pd->pid = 0;		/* don't find this guy */
			if ( FindInstance( upid, PROCESS_INSTANCE ) != NULL )
			  {
			    /* Let the other instance worry about the SIGCHLD.
			       The owner of this instance no longer cares */
			    prev->link = ((FileInstance*)pd)->link;
			  }
			else
			  {
			    /* kill the process and leave the descriptor 
			       around to catch the signal */
		            killpg( getpgrp( pd->pid )
			            , ReleaseAll ? SIGKILL : SIGHUP );
		            pd->state = PD_AWAITING_DEATH;
			    pd->id = 0;		/* invalidate instanceid */
			    pd->pid = upid;
			    prev = ptr;
			  }
		      }
		    else
		      {
		        if ( pd->state == PD_AWAITING_DEATH )
		          {
		            killpg( getpgrp( pd->pid ), SIGKILL );
		            reclaimed--;
			    prev = ptr;
		          }
			else
			  {
			    /*
			     * At this point, the process is dead and
			     * we can release the instance.  If the 
			     * pipe was still open, then we have gained
			     * one more file descriptor.  
			     */
			    if ( close( pd->pipedesc ) != 0 )
			        reclaimed--;	/* nothing reclaimed */

		            prev->link = ptr->link;	/* remove from list */
			    /* 
			     * Find the sibling. if not found then this is the
			     * last reference to name, thus we may free it 
			     */
			    if ( FindInstance( pd->pid, PROCESS_INSTANCE ) 
			         == NULL )
			        free( pd->name );
			    if ( pd->lastdata != NULL )
			        free( pd->lastdata );
		            free( ptr );
			  }
		      }
		    break;

		case SESSION_INSTANCE:
		    prev = ptr;
		    reclaimed--;
		    break;

	        
		default:
		    if ( GDebug )
			printf( "ReclaimInstances , bad instancetype = %d\n"
			    , ptr->type );
		    reclaimed--;
		    prev = ptr;
		    break;
	      }

	    reclaimed++;
	  }
	else 
          {
	    prev = ptr; 
	  }
      } /* while */

    if ( IDebug ) printf( "reclaimed %d files\n", reclaimed );
    return( reclaimed );
  } /* ReclaimInstances */

FileInstance *FindInstance( id, instancetype )
    unsigned id;
    InstanceType instancetype;
/*
 * Find the id in the instance list.  If instancetype is FILE_INSTANCE,
 * then id is an UNIX fid.  If instancetype is PROCESS_INSTANCE, then 
 * id is a UNIX pid.  Returns the first descriptor matching id or NULL
 * if nothing is found.  
 */
  {
    extern SessionDesc *Session;

    switch ( instancetype )
      {
	case FILE_INSTANCE:
	case DIRECTORY_INSTANCE:
	  {
	    register FileInstance *fdesc;

	    for ( fdesc = FileInstances->link
		  ; fdesc != NULL
		  ; fdesc = fdesc->link ) 
		  if ( id ==  fdesc->unixfile && fdesc->type == instancetype ) 
		      return ( fdesc );
	    break;
	  }
	case PROCESS_INSTANCE:	/* find unix process id */
	  {
	    register ProcessInstance *pdesc;

	    for ( pdesc = (ProcessInstance *)FileInstances->link
		  ; pdesc != NULL
		  ; pdesc = pdesc->link ) 
		  if ( id ==  pdesc->pid && (pdesc->type == PROCESS_INSTANCE
		        || pdesc->type == SESSION_INSTANCE) )
		          return ( (FileInstance *) pdesc );
	    break;
	  }

	case SESSION_INSTANCE:
	  {
	    register SessionDesc *sdesc;
	    for ( sdesc = (SessionDesc *)FileInstances->link
		  ; sdesc != NULL
		  ; sdesc = sdesc->link ) 
		  if ( id ==  sdesc->Vpid && sdesc->type == SESSION_INSTANCE ) 
		      return ( (FileInstance *) sdesc );
	    break;
          }
	default:
	  {
	    if ( GDebug )
		printf( "FindInstance called with unkown instancetype = %d\n"
		    , instancetype );
	    break;
	  }
      } /* switch */

    return ( NULL );    /* not found */

  } /* FindInstance */

CheckFiles() 
  {
    register int f;
    struct stat st;

    printf( "#  links Mode  Device  Inode  LastAccess  \n");
    for( f = 0; f < 20; f++) 
      {
	if (fstat(f, &st) == 0)
	  {
	    printf( "%2d %5x %4x  %7x  %5x %s"
			,f,st.st_nlink,st.st_mode,st.st_dev,st.st_ino,ctime(&(st.st_atime)));
	  }
      }
  }
	
      
