/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 * Instance request processing routines
 */

#include <signal.h>
#include <sgtty.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vsession.h>
#include <errno.h>
#include <server.h>
#include <debug.h>

/* Exports */
extern SystemCode CreateInstance();
extern SystemCode QueryInstance();
extern SystemCode ReadInstance();
extern SystemCode ReadAndForward();
extern SystemCode WriteInstance();
extern SystemCode ReleaseInstance();
extern SystemCode SetInstanceOwner();

/* Imported routines */
extern FileInstance *AllocInstance();
extern		    FreeInstance();
extern FileInstance *FindInstance();
extern FileInstance *GetInstance();
extern SystemCode QueueIORequest();
extern SystemCode GetName();
extern KillSession(),ReclaimInstances();
extern SystemCode SetupInstance();
extern Reply();
extern SetStringName();
extern char *GetSessionOwnerName();
extern SystemCode ExecuteProgram();
extern char *FindContext();

/* Imported variables */
extern ProcessId MyPid;
extern int errno;
extern char BlockBuffer[];


SystemCode CreateInstance( req, pid, segsize )
    register CreateInstanceRequest *req;
    ProcessId pid;
    unsigned segsize;
   /* 
    * Create an instance of the specified file or call
    * ExecuteProgram if the mode is FEXECUTE.
    */
  {
    register FileInstance *desc;
    register CreateInstanceReply *reply = (CreateInstanceReply *) req;
    register SystemCode r;
    char *nameptr;
    char c;
    int	file;
    char *fileName;
    unsigned fileNameLen;
    ContextId cid;
    InstanceType InstType;
  
    /* Special creation request? */
    /* done this way to speed up most common mode: file instances */
    switch ( req->filemode & (FEXECUTE|FSESSION|FDIRECTORY) )
      {    
	case 0: /* normal files */
		InstType = FILE_INSTANCE;
		break;

	case FDIRECTORY:	/* open file as a directory */
		/* Don't let main server read directories */
		if ( Session == NULL )
		     return( ILLEGAL_REQUEST );

		if ( req->filemode != (FDIRECTORY|FREAD) )
		     return( INVALID_MODE );

		req->filemode = FREAD;		/* clear FDIRECTORY bit */
		InstType = DIRECTORY_INSTANCE;
		break;
	case FEXECUTE:
	        return ( ExecuteProgram( req, pid, segsize ) );
		break;
        case FSESSION:
                return ( CreateSession( req, pid, segsize ) );	
		break;
	default:
		return( INVALID_MODE );
		break;
      }

    /* Get the fully qualified pathname */
    if ( (r = GetName( req, pid, &segsize, &nameptr)) != OK )
        return( r );

    if( IDebug ) printf( "CreateInstance in 0x%x: \"%s\"\n", MyPid, nameptr );

    if( Session != NULL ) /* Only load files thru main server process */
      {
	if( Unauthorized( pid ) )
	   return( NO_PERMISSION ); /* Different host process */

	Session->inactivity = 0;
      }
    else if( req->filemode != FREAD ) return( NO_PERMISSION );

    if( (desc = AllocInstance( pid, InstType )) == NULL ) 
	return( NO_SERVER_RESOURCES );

retry:
    switch( req->filemode )
      {
	case FREAD:
	  {
	    file = open( nameptr, 0 );
	    if ( InstType == DIRECTORY_INSTANCE )
	        desc->filetype = READABLE+FIXED_LENGTH;
	    else
	        desc->filetype = READABLE+FIXED_LENGTH+MULTI_BLOCK;
	    break;
	  }
	case FCREATE:
	  {
	    if ( (file = creat( nameptr, DEFAULT_CREAT_MODE )) != -1) 
	      {
		close( file );
		file = open( nameptr, 2 );
		desc->filetype = READABLE+WRITEABLE+MULTI_BLOCK;
	      }
	    break;
	  }
	case FMODIFY:
	  {
	    if( (file = open( nameptr, 2)) == -1 )
	      {
		if ( (file = creat( nameptr, DEFAULT_CREAT_MODE )) != -1)
		  {
		    close( file );
		    file = open( nameptr, 2 );
		  }
	      }
	    desc->filetype = READABLE+WRITEABLE+MULTI_BLOCK;
	    break;
	  }
	case FAPPEND:
	  {
	    file = open( nameptr, 2 );
	    desc->filetype = READABLE+WRITEABLE+APPEND_ONLY+MULTI_BLOCK;
	    break;
	  }
	default: 
	    FreeInstance( desc );
	    return( INVALID_MODE );
	    break;
      }

    if( file == -1 ) /* Failed to open the file */
      {
	switch( errno )
	  {
	    case ENOENT: r = NOT_FOUND;
			 break;
	    case EPERM:
	    case EACCES:
	    case EISDIR: 
	      {
		r = NO_PERMISSION;
		break;
	      }
	    case ENFILE: /* System open file table full try to make room. */
	    case EMFILE: /* Too many open files, try to reclaim open files. */
	      {
		if( ReclaimInstances( 0 ) ) goto retry;
 		r = NO_SERVER_RESOURCES;
		break;
	      }
	    default: 
	      {
		if ( GDebug ) perror( "CreateInstance: ");
		r = INTERNAL_ERROR; 
		break;
	      }
	  }
	FreeInstance( desc );
	return( r );
      }

    if( (r = SetupInstance( file, desc, Session, nameptr )) != OK )
      {
	close( file );
	FreeInstance( desc );
	return( r );
      }
    /* Everything Ok */
    ioctl( desc->unixfile, FIOCLEX, NULL); /* Close file on successful "exec" */
    reply->fileid = desc->id;

    return( QueryInstance( (QueryInstanceRequest *)reply ) );
  } /* CreateInstance */

SystemCode QueryInstance( req )
   register QueryInstanceRequest *req;

  /* 
   * Satisfy a query instance request from a process.
   */
  {
    register CreateInstanceReply *reply = (CreateInstanceReply *) req;
    register FileInstance *desc;

    if( (desc = GetInstance(req->fileid)) == NULL ) return( INVALID_FILE_ID );

    if( IDebug ) printf( "QueryInstance: fileid 0x%x\n", req->fileid );

    if( Session != NULL ) Session->inactivity = 0;

    MoveLong(MyPid,reply->fileserver);    
    reply->filetype = desc->filetype;
    MoveLong(desc->lastblock,reply->filelastblock);
    MoveLong(desc->nextblock,reply->filenextblock);
    MoveLong(desc->blocksize,reply->blocksize);
    reply->filelastbytes = desc->lastbytes;

    return( OK );
  } /* QueryInstance */



SystemCode ReadInstance( req, pid )
   register IoRequest *req; 
   ProcessId pid;
  /* 
   * Handle a read instance request.
   */
  {
    register SystemCode	r;
#define reply	((IoReply*)req)
    int bytes, fid;
    char *ptr;
    unsigned count, totalbytes;
    register FileInstance *filedesc;
#define procdesc	((ProcessInstance*)filedesc)
    register char *buffer;

    MoveLong(req->bytecount, count);
    if ( count == 0 ) return( OK );
    req->bytecount = 0;

    if( (filedesc = GetInstance(req->fileid)) == NULL ) 
        return( INVALID_FILE_ID);

    if( filedesc->filetype & READABLE == 0 ) return( NOT_READABLE );

    if( Session != NULL ) Session->inactivity = 0;

    MoveLong(req->blocknumber,bytes);
    if( IDebug ) printf("ReadInstance fileid 0x%4x, block %d, bytes %d\n",
			    req->fileid, bytes, count );

    switch( filedesc->type ) 
      {
	case FILE_INSTANCE:
	  {
	    bytes *=  (filedesc->blocksize); 

	    if( lseek(filedesc->unixfile, bytes, 0) == -1 ) 
	        return( BAD_BLOCK_NO );

	    fid = filedesc->unixfile;
	    buffer = BlockBuffer;
	    break;
	  }
	case PROCESS_INSTANCE:
	  {
	    buffer = procdesc->lastdata;
	    if ( bytes == filedesc->nextblock - 1 )
	      {
		if ( PDebug ) printf( "ReRead of Block #%d\n", bytes );
	        MoveLong( procdesc->lastdatasize, reply->bytecount );
	        if( procdesc->lastdatasize <= IO_MSG_BUFFER )
	          { /* send inside of message */
	            swab( buffer, req->shortbuffer
		    	 , procdesc->lastdatasize+1 );
		    r = procdesc->lastreply;
	          }
		/*
		 * NOTE: we are banking on the fact that procdesc->blocksize
		 * is not greater than MAX_APPENDED_SEGMENT.  It would be
 		 * ridiculous to have it different anyway
		 */
	        else if( req->requestcode != READ_AND_FORWARD )
	          {
		    MoveLong( req->bufferptr, ptr );
	            reply->replycode = procdesc->lastreply;
	            ReplyWithSegment( reply, pid
		    	, buffer, ptr, procdesc->lastdatasize );
	            r = NOT_AWAITINGREPLY;
		  }
		else
		  {
		    MoveLong( req->bufferptr, ptr );
	            r = MoveTo( pid, ptr, buffer, procdesc->lastdatasize );
		  }
  		return( r );
	      } 	    
	    else if ( bytes != procdesc->nextblock ) 
	      {
		if (PDebug) 
		    printf( "Bad block number, got %d, expected %d\n"
		            , bytes, procdesc->nextblock);
		return( BAD_BLOCK_NO );
	      }

	    /* We are reading the correct block */
	    fid = procdesc->pipedesc;
	    ioctl( fid, FIONREAD, &bytes ); /* how much available */

	    if ( bytes <= 0 ) 
		switch ( ((ProcessInstance *)procdesc)->state )
		  {
		    case PD_DEAD:
		    case PD_KILLED:
		        return ( END_OF_FILE );
			break;
		    case PD_STOPPED:
		    case PD_RUNNING:
			/* No bytes now, so hang request until data avail*/
			MoveLong( count, req->bytecount );  /* restore req */
			return( QueueIORequest( req, pid ) );
			break;
		    default:
			if ( GDebug ) 
			    printf( "ReadInstance, unknown state = %d\n"
				    , ((ProcessInstance *)procdesc)->state );
			return( INTERNAL_ERROR );
			break;
		  } /* switch */
	    
	    /* Don't attempt read more than what is in the pipe. */
	    count = bytes < count ? bytes : count;
	    break;
	  }
	case DIRECTORY_INSTANCE:
	    /* NOTE: bytes is really block number */
	    return( ReadDirectory( req, pid, filedesc, bytes, count ) );
	    break;
	default:
	    return( INTERNAL_ERROR );
	    break;
      } /* switch ( instance type ) */

    /* From this point on, all instances are alike */
    MoveLong( req->bufferptr, ptr );
    totalbytes = 0;

    while( totalbytes < count )
      {
	bytes = count - totalbytes;
	bytes = (bytes < BUFFER_SIZE) ? bytes : BUFFER_SIZE;
	bytes = read( fid, buffer, bytes );

	if ( bytes < 0 ) 
	  {
	    if ( GDebug ) perror("ReadInstance:");
	    r = DEVICE_ERROR;
	    break;
	  } 
	if( bytes == 0 )
	  {
	     r = END_OF_FILE;
	     break;
	  }
	if( bytes <= IO_MSG_BUFFER && totalbytes == 0 )
	  { /* Concatenate onto message */
	    r = OK;
	    swab(buffer,req->shortbuffer,bytes+1);
	  }
	else if( (bytes <= MAX_APPENDED_SEGMENT) && (Session != NULL)
		&& req->requestcode != READ_AND_FORWARD )
	  { /* Session check prevents the boot service from using this */
	    totalbytes += bytes;
	    MoveLong(totalbytes, reply->bytecount);
	    reply->replycode = ((totalbytes < count) ? END_OF_FILE : OK);
	    ReplyWithSegment( reply, pid, buffer, ptr, bytes );
	    r = NOT_AWAITINGREPLY;
	  } 
	else
	  {
	    r = MoveTo( pid, ptr, buffer, bytes );
	    ptr += bytes;
	  }
	if( r != OK ) break;
	totalbytes += bytes;
      }

    /* this should be a generalized stream check, not just for PROCESS's */
    if ( filedesc->type == PROCESS_INSTANCE )
      {
	/* Set the reply code, check for ReplyWithSegment */
        procdesc->lastreply = r == NOT_AWAITINGREPLY ? reply->replycode : r;
	if ( procdesc->lastreply == OK || procdesc->lastreply == END_OF_FILE )
	  {
	    procdesc->nextblock++;
            procdesc->lastdatasize = totalbytes;
	    /* data should be in the right place already */
	  }
	else
	    procdesc->lastdatasize = 0;
      }

    if ( r == OK || r == END_OF_FILE )
        MoveLong( totalbytes, reply->bytecount );

    return( r );

#undef procdesc
#undef reply
  } /* ReadInstance */

SystemCode ReadAndForward( req, pid )
    IoRequest *req;
    ProcessId pid;

  /* Read the specified amount and then forward back requestor back
   * to original sender. Used initially for efficient program loading.
   */
  {
    SystemCode r;
    ProcessId forwarder;

    if( (forwarder = Forwarder( pid )) == 0) return( BAD_STATE );

    if( IDebug ) printf("ReadAndForward: forwarder 0x%8x fileid 0x%4x\n",
				forwarder, req->fileid );

    if( (r = ReadInstance( req, pid )) == NO_REPLY ) return( NO_REPLY );

    /* If r == NOT_AWAIT... then a ReplyWithSegment was done in ReadInst */
    /* Stuff in the replycode ??? */
    req->blocknumber = (unsigned) r == NOT_AWAITINGREPLY ? OK : r; 

    Forward( req, pid, forwarder );
    return( NOT_AWAITINGREPLY );
  } /* ReadAndForward */



SystemCode WriteInstance( req, pid, segsize )
    register IoRequest *req; 
    ProcessId pid; 
    unsigned segsize;

  /* Handle a write instance request to a file
   */
  {
#define reply ((IoReply*)req)
    SystemCode r;
    unsigned count, bytes, ptr, totalbytes; 
    int fid;
    register FileInstance *desc;
#define procdesc	((ProcessInstance*)desc)

    MoveLong(req->bytecount,count);
    req->bytecount = 0;		/* Return zero if an error */

    if( (desc = GetInstance(req->fileid)) == NULL ) return( INVALID_FILE_ID );

    if( desc->filetype & WRITEABLE == 0 ) 
	return( NOT_WRITEABLE );

    if( Session != NULL ) 
	Session->inactivity = 0;

    totalbytes = 0;
    MoveLong( req->bufferptr, ptr );
    MoveLong( req->blocknumber, bytes );

    if( IDebug ) printf("WriteInstance fileid 0x%4x, block %d, bytes %d\n",
				req->fileid, bytes, count );
    switch ( desc->type ) 
      {
	case FILE_INSTANCE:
	  {
	    bytes *= desc->blocksize;
	    if( lseek(desc->unixfile, bytes, 0) == -1 ) return( BAD_BLOCK_NO );
	    fid = desc->unixfile;
	    break;
	  }
	case PROCESS_INSTANCE:
	  {
	    if ( bytes == desc->lastblock - 1 )
	      {
		if ( PDebug )
		    printf( "Repeat WriteInstance block %d\n", bytes );
		MoveLong( procdesc->lastdatasize, reply->bytecount );
	        return( procdesc->lastreply );
 	      }
	    if ( bytes != desc->lastblock ) 
	      {
		if (PDebug) 
		    printf( "Bad block number, got %d, expected %d\n"
		            , bytes, desc->lastblock );
		return( BAD_BLOCK_NO );
	      }
	    switch ( ((ProcessInstance *)desc)->state )
	      {
		case PD_DEAD:
		case PD_KILLED:
		    return ( END_OF_FILE );
		    break;
		case PD_STOPPED:
		case PD_RUNNING:
		    break;
		default:
		    if ( GDebug ) 
			printf( "WriteInstance, unknown state = %d\n"
				, ((ProcessInstance *)desc)->state );
		    return( INTERNAL_ERROR );
		    break;
	      } /* switch */
	    
	    /* If we attempt to write more than the pipe can hold, the
	       server will hang.  In this case, hang request until data
	       is available. */
	    fid = ((ProcessInstance *)desc)->pipedesc;
	    ioctl( fid, FIONREAD, &bytes );
	    if ( count > (MAX_BYTES_IN_PIPE - bytes) )
		return( QueueIORequest( req, pid ) );
	    break;
	  }
	default:
	    if ( GDebug )
	        printf( "WriteInstance unknown instance type, 0x%x\n"
			 , desc->type );
	    return( INTERNAL_ERROR );
	    break;
      } /* switch */

    while( totalbytes < count )
      {
	bytes = count - totalbytes;
	bytes = (bytes < BUFFER_SIZE) ? bytes : BUFFER_SIZE;

	r = OK;
	if( count <= IO_MSG_BUFFER ) /* Get data from message */
	  {
	    swab( req->shortbuffer, BlockBuffer, bytes+1 );
	  }
	else if( bytes > segsize )
	  {
	    if( (r = MoveFrom(pid,ptr+segsize,&BlockBuffer[segsize],bytes-segsize)) != OK )
		break;
	    segsize = 0;
	    ptr += bytes;
	  }
	bytes = write( fid, BlockBuffer, bytes );
        if( bytes == 0 )
	  {
	    r = DEVICE_ERROR;
	    break;
	  }
	totalbytes += bytes;
      }
    MoveLong( totalbytes ,req->bytecount );
    /* this test should really be for streams and not just PROCESS_INSTANCES */
    if ( desc->type == PROCESS_INSTANCE )
      {
        if ( (procdesc->lastreply = r) == OK )
	  {
	    procdesc->lastdatasize = totalbytes;
	    procdesc->lastblock++;
	  }
	else
	    procdesc->lastdatasize = 0;
      }
    return( r );
#undef reply
#undef procdesc
 } /* WriteInstance */



SystemCode ReleaseInstance( req, pid )
   IoRequest *req; ProcessId;

  /*
   * Release the instance but first perform the necessary clean up
   * actions as follows (only if requestor has permission):
   *		FILE_INSTANCE:  Close the file 
   *		DIRECTORY_INSTANCE: Close the file
   *		PROCESS_INSTANCE: Send hup signal to process free descriptor
   *				  only if there is another one for this
   *				  process.  Otherwise must keep desc around
   *				  to make sure the process gets killed eg,
   *				  it is ignoring SIGHUP.
   *		SESSION_INSTANCE: Kill all processes and close all instances
   *				  then exit.
   */
  {
    register FileInstance *desc;


    if( (desc = GetInstance(req->fileid)) == NULL ) return( INVALID_FILE_ID );


    if( IDebug ) printf( "ReleaseInstance on fileid 0x%x\n", req->fileid );

    if( Session != NULL ) Session->inactivity = 0;

    switch ( desc->type ) 
      {
	case DIRECTORY_INSTANCE:
	case FILE_INSTANCE:
	  {
    	    if( desc->owner != pid ) return( NO_PERMISSION );
	    close( desc->unixfile );
            FreeInstance( desc );
	    break;
	  }
	case PROCESS_INSTANCE:
	  {
	    register ProcessInstance *pdesc;
	    register long upid;

    	    if( desc->owner != pid ) return( NO_PERMISSION );
	    pdesc = (ProcessInstance *) desc;
	    close( pdesc->pipedesc );
	    pdesc->pipedesc = -1;	/* invalidate */

	    upid = pdesc->pid;
	    pdesc->pid = 0;	/* Mark this desc so that it won't be found */
	    if( (desc = FindInstance( upid, PROCESS_INSTANCE )) == NULL )
	      {
		if ( PDebug ) printf( "pid %d no sibling,", upid );
		if ( pdesc->state == PD_DEAD || pdesc->state == PD_KILLED )
		  {
		    if ( PDebug ) printf( "freeing \n" );
	            FreeInstance( pdesc );
		  }
		else
		  {
		    if ( PDebug ) printf( "awaiting death\n" );
	            pdesc->state = PD_AWAITING_DEATH;
		    killpg( getpgrp( upid ), SIGHUP );
		    pdesc->pid = upid;		/* save pid for SIGCHLD */
		    pdesc->id = 0;		/* instance no longer valid */
		  }
	      }
	    else
	      {
		if (PDebug) printf( "pid %d freeing not killing\n", upid );
		pdesc->name = NULL;	/* don't free the name */
	        FreeInstance( pdesc );	/* other guy will kill the process,
					   and free the name */
	      }
	    break;
          }
        case SESSION_INSTANCE:
	  {
            /* Check requestor is on same network host as session creator */
            if( Unauthorized(pid) ) 
	        return( NO_PERMISSION );

            if( SDebug ) 
	        printf( "Terminating session: Pid %8x instance id 0x%x\n"
		        , MyPid, Session->id );

   	    /* Have to reply before the fall */
            ((IoReply*)req)->replycode = OK;
            Reply( req, pid );

	    KillSession();
	    break;
	  }

      } /* switch */

    return( OK );
  } /* ReleaseInstance */

SystemCode SetInstanceOwner( req, pid )
   IoRequest *req;

  /*
   * Set the owner of the instance to pid in request iff the requestor
   * is the current owner.
   */
  {
    register FileInstance *desc;

    if( (desc = GetInstance(req->fileid)) == NULL ) return( INVALID_FILE_ID );

    if( desc->owner != pid ) return( NO_PERMISSION );

    if( Session != NULL ) Session->inactivity = 0;

    MoveLong( req->instanceowner, desc->owner );
    
    if ( IDebug )
        printf( "SetInstanceOwner id - 0x%x and owner pid 0x%x\n", desc->id
		, desc->owner );

    if ( desc->type == SESSION_INSTANCE )
      {
        /* Change the session name to reflect the change in owner */
        sprintf( BlockBuffer, SESSION_BANNER, GetSessionOwnerName( desc->owner ) );
        SetStringName( BlockBuffer );
      }
    return( OK );
  } /* SetInstanceOwner */

