/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 *
 * UNIX-specific file access routines, etc.
 */

#include <types.h>
#include <stat.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vsession.h>
#include <Vnaming.h>
#include <Vdirectory.h>
#include <server.h>
#include <debug.h>

extern SystemCode GetName();
extern char *ParseName();
extern FileInstance *GetInstance();
extern SystemCode SetupInstance();
extern char *FindContext();
extern SetDebugMode();

SystemCode SetupInstance( file, desc, session, Name )
    int file;
    register FileInstance *desc;
    SessionDesc *session;
    char *Name;

    /* Check it is a disk file or directory and not a device.
     * Setup file instance descriptor for further access.
     */
  {
    struct stat buf;
    register SystemCode r;
    unsigned short mode;
    ContextId cid;

    fstat( file, &buf );
    mode = buf.st_mode;

    if( session == NULL )
      {
	if( !(mode&(S_IREAD >> 6)))
	  return( NO_PERMISSION);  
      }

    if( (mode &= S_IFMT) != S_IFREG && mode != S_IFDIR )
	return( NO_PERMISSION );

    if ( desc->type == DIRECTORY_INSTANCE )
      {
	/* Directory bit was set, so check if it is a directory */
        if ( mode != S_IFDIR )
	    return( INVALID_MODE );

	/* NOTE: have to be able to cd to it, inorder to contruct Descriptors
	 	 so it is best to try and put it in the context cache.
         */
	if ( (r = InsertContextNode( Name, strlen( Name ), &cid )) != OK )
	    return( r );
	
        desc->name = FindContext( cid );	/* should be there */
	((DirectoryInstance *)desc)->contextid = cid;
	desc->blocksize = sizeof( UnixFileDescriptor );
        desc->lastblock = buf.st_size / sizeof( UnixFileDescriptor );
        desc->lastbytes = 0;	/* should always be zero */
        desc->nextblock = 0;
        desc->unixfile = file; /* Remember the unix file descriptor */
      }
    else
      {
        /* Set up the file instance descriptor as required */
        desc->name = ParseName( malloc( strlen( Name ) + 2 ), Name, 0 );

        /* Block size picked to be most efficient for UNIX and network */
        desc->blocksize = BLOCK_SIZE;
        desc->lastblock = buf.st_size / BLOCK_SIZE;
        desc->lastbytes = buf.st_size % BLOCK_SIZE;
        desc->nextblock = 0;
        desc->unixfile = file; /* Remember the unix file descriptor */
      }

    return( OK );

  } /* SetupInstance */


SystemCode RemoveFile( req, pid, segsize )
  CreateInstanceRequest *req;
  ProcessId pid;
  unsigned segsize;
  {
    extern SessionDesc *Session;
    extern char BlockBuffer[];
    extern int errno;
    register SystemCode r;
    unsigned fileName,fileNameLen;
    char *NamePtr;
  
    if( Session == NULL ) return( NO_PERMISSION );
 
    if( Unauthorized(pid) ) return( NO_PERMISSION );

    Session->inactivity = 0;

    if ( (r = GetName( req, pid, &segsize, &NamePtr )) != OK )
      {
	((ContextReply *)req)->contextid = 0;
        return( r );
      }

    if( FDebug ) printf( "RemoveFile: \"%s\"\n", NamePtr );

    if( unlink(NamePtr) == -1 ) return( NOT_FOUND );

    return( OK );
  } /* RemoveFile */

SystemCode QueryFile( req, pid )
   IoRequest *req; ProcessId pid;

  /* Handle a query file request for the named file.
   */
  {
    register FileInstance *desc;

    if( (desc = GetInstance(req->fileid)) == NULL ) return( INVALID_FILE_ID );
    
    return( REQUEST_NOT_SUPPORTED );
  } /* QueryFile */

SystemCode ModifyFile( req, pid )
   IoRequest *req; ProcessId pid;

  /* 
   * Handle a modify file request.
   * Only valid for SessionDescriptors at present.
   */
  {
    register FileInstance *desc;
    long debug;

    if( (desc = GetInstance(req->fileid)) == NULL ) return( INVALID_FILE_ID );
    
    if ( FDebug ) 
         printf( "ModifyFile: id 0x%x, pid 0x%x\n"
	 	  , req->fileid, pid );

    if ( desc->type == SESSION_INSTANCE )
      {
	/* change to the requested debug mode */
        MoveLong( req->blocknumber, debug );
	SetDebugMode( debug );
        return( OK );
      }
    return( REQUEST_NOT_SUPPORTED );

  } /* ModifyFile */


