/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 *
 * Implements the directory information requests: READ & WRITE _DESCRIPTOR
 * Also handles read instance on DIRECTORY_INSTANCEs.
 */

#include <types.h>
#include <stat.h>
#include <pwd.h>
#include <grp.h>
#include <dir.h>
#include <errno.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vsession.h>
#include <Vnaming.h>
#include <Vdirectory.h>
#include <server.h>
#include <debug.h>

extern int errno;
extern SystemCode GetName();
extern char *ParseName();
extern FileInstance *GetInstance();
extern SystemCode SetupInstance();
extern struct passwd *getpwuid();
extern struct group *getgrgid();
extern SystemCode ReadDirectory();
extern UnixFileDescriptor *ConstructDescriptor();
extern SystemCode ChangeContext();
extern SystemCode GetDescriptor();
extern SystemCode ModifyDescriptor();
extern char *rindex();
extern char *GetUnixUserName(), *GetUnixGroupName();

extern char BlockBuffer[];

UnixFileDescriptor *ConstructDescriptor( st, name )
    register struct stat *st;
    register char *name;
    
/*
 * Take the stat structure and name and create a descriptor
 * entry and return a pointer to a UnixFileDescriptor.  Name 
 * has to be in the directory entry form (ie. no '/'s)
 */
  {
    register UnixFileDescriptor *Udesc = (UnixFileDescriptor *)BlockBuffer;

    Udesc->descriptortype = RealMakeLong( UNIXFILE_DESCRIPTOR );
    if ( DDebug )
        printf( "desctype = 0x%x\n", Udesc->descriptortype );

    /* don't swab, since it is in the segment */
    strncpy( Udesc->name, name, UnixFileNameLen );

    MoveShort( st->st_dev, Udesc->st_dev );
    MoveShort( st->st_ino, Udesc->st_ino );
    MoveShort( st->st_mode, Udesc->st_mode );
    MoveShort( st->st_nlink, Udesc->st_nlink );
    MoveShort( st->st_uid, Udesc->st_uid );
    MoveShort( st->st_gid, Udesc->st_gid );
    MoveShort( st->st_rdev, Udesc->st_rdev );
    RealMoveLong( st->st_size, Udesc->st_size );
    RealMoveLong( st->st_atime, Udesc->st_atime );
    RealMoveLong( st->st_mtime, Udesc->st_mtime );
    RealMoveLong( st->st_ctime, Udesc->st_ctime );
    
    if ( (name = GetUnixUserName( st->st_uid )) != NULL )
	strncpy( Udesc->ownername, name, UnixUserNameLen );
    else
        *((short *)(Udesc->ownername)) = 0;	/* guarrantee right byte */
   
    if ( (name = GetUnixGroupName( st->st_gid )) != NULL )
	strncpy( Udesc->groupname, name, UnixUserNameLen );
    else
        *((short*)(Udesc->groupname)) = 0;

    if ( DDebug )
      {
        printf( "owner %8s, group %8s, name %s\n", Udesc->ownername
		, Udesc->groupname, Udesc->name );
	printf( "st_mode = 0%o, Udesc->st_mode 0%o\n", st->st_mode, Udesc->st_mode );
      }
    
    return( Udesc );
    
  } /* ConstructDescriptor */
    
SystemCode ModifyDescriptor( Name, Descriptor )
    char *Name;
    UnixFileDescriptor *Descriptor;
/*
 * Update the directory entry for Name to look like Descriptor.
 * This only means to modify those fields which can be changed
 * in UNIX. 
 */
  {
    if ( DDebug )
        printf( "ModifyDescriptor: mode 0%o\n", Descriptor->st_mode&~S_IFMT );

    if ( chmod( Name, Descriptor->st_mode&~S_IFMT ) != 0 )
        switch ( errno )
	  {
	    case ENOENT:
	    	return( NOT_FOUND );
		break;
	    default:		/* should be more specific */
	        return( NO_PERMISSION );
		break;
	  }
    else
        return( OK );
  } /* ModifyDescriptor */


SystemCode GetDescriptor( req, pid, segsize, Descriptor )
    DescriptorRequest *req;
    ProcessId pid;
    unsigned *segsize;
    UnixFileDescriptor *Descriptor;
/*
 * Retrieve the descriptor from the process.
 */
  {
    unsigned length;
    char *ptr;
    SystemCode r;
    register UnixFileDescriptor *dp1, *dp2;
    
    MoveLong( req->segmentlen, length );
    if( length > BUFFER_SIZE
        ||  length < ( req->dataindex + sizeof( UnixFileDescriptor ) ) ) 
        return( BAD_ARGS );

    /* Was the name sent with the packet? */
    if ( *segsize < length )
      { /* no, have to retrieve from remote host */
	MoveLong( req->segmentptr, ptr );

	if( ( r = MoveFrom( pid, ptr, BlockBuffer, length )) != OK)
		return( r );
	*segsize = length;
      }
   
    dp1 = Descriptor;
    dp2 = (UnixFileDescriptor *)&BlockBuffer[ req->dataindex ];
    
    RealMoveLong( dp2->descriptortype, dp1->descriptortype );
    if ( DDebug )
        printf( "GetDescriptor: desctype = 0x%x\n", dp2->descriptortype );

    /* don't swab, since it is in the segment */
    strncpy( dp1->name, dp2->name, UnixFileNameLen );
    strncpy( dp1->ownername, dp2->ownername, UnixUserNameLen );
    strncpy( dp1->groupname, dp2->name, UnixUserNameLen );

    MoveShort( dp2->st_dev, dp1->st_dev );
    MoveShort( dp2->st_ino, dp1->st_ino );
    MoveShort( dp2->st_mode, dp1->st_mode );
    MoveShort( dp2->st_nlink, dp1->st_nlink );
    MoveShort( dp2->st_uid, dp1->st_uid );
    MoveShort( dp2->st_gid, dp1->st_gid );
    MoveShort( dp2->st_rdev, dp1->st_rdev );
    RealMoveLong( dp2->st_size, dp1->st_size );
    RealMoveLong( dp2->st_atime, dp1->st_atime );
    RealMoveLong( dp2->st_mtime, dp1->st_mtime );
    RealMoveLong( dp2->st_ctime, dp1->st_ctime );
    
    return( OK );
    
  } /* GetDescriptor */

SystemCode ReadDirectory( req, pid, desc, block, bytes )
    register IoRequest *req;
    ProcessId pid;
    register DirectoryInstance *desc;
    register unsigned block;
    unsigned bytes;
/*
 * Read a directory entry and format it as a UnixFileDescriptor.
 */
  {
    struct dir dentry;
    struct stat entrystat;
    UnixFileDescriptor *Udesc;
    char *ptr;
#define reply	((IoReply*)req)

    block *= sizeof( struct dir );

    if ( lseek( desc->unixfile, block, 0 ) == -1 )
        return( BAD_ARGS );	/* shouldn't happen unless block < 0 */
   
    
    if ( (block = read( desc->unixfile, &dentry, sizeof( struct dir ) )) < 0 )
        return( DEVICE_ERROR );
    
    if ( block == 0 )
        return( END_OF_FILE );
    
    if ( block != sizeof( struct dir ) )
        return( INTERNAL_ERROR );
    
    /* change to the context so that we may interpret the name
       of the directory entry */
    if ( (block = (unsigned)ChangeContext( desc->contextid )) != OK )
        return( (SystemCode) block );
    
    if ( stat( dentry.d_name, &entrystat ) != 0 )
        switch( errno )
	  {
	    case ENOENT: /* only reason for failure (inode == 0) */
	    		 /* easier to check this way */
		Udesc = (UnixFileDescriptor*)BlockBuffer;
		Udesc->descriptortype = RealMakeLong( EMPTY_DESCRIPTOR );
		break;
	    default:
	    	return( INTERNAL_ERROR );
		break;
	  }
    else
        Udesc = ConstructDescriptor( &entrystat, dentry.d_name );

    /* where to put it */
    MoveLong( req->bufferptr, ptr );

    /* Everything is okay */
    reply->replycode = OK;
    MoveLong( bytes, reply->bytecount );

    /* Using ReplyWithSegment could change if descriptor changes */
    ReplyWithSegment( reply, pid, (char *)Udesc, ptr, bytes );
    return( NOT_AWAITINGREPLY );

#undef reply
  } /* ReadDirectory */

/* DESCRIPTOR REQUEST HANDLERS */
SystemCode ReadDescriptor( req, pid )
    DescriptorRequest *req;
    ProcessId pid;
/*
 * Read the descriptor for the file specified by the fileid
 * in the request.
 */
  {
    register FileInstance *desc;
    char *ptr, *ptr2;
    struct stat FileStat, NameStat;
    UnixFileDescriptor *Udesc;
#define reply	((DescriptorReply*)req)

    if( Session == NULL ) 
        return( ILLEGAL_REQUEST );

    if( (desc = GetInstance(req->fileid)) == NULL ) 
        return( INVALID_FILE_ID );

    if( desc->type != FILE_INSTANCE && desc->type != DIRECTORY_INSTANCE ) 
        return( REQUEST_NOT_SUPPORTED );

    /* got enough room in the segment */
    MoveLong( req->segmentlen, ptr );
    if ( req->dataindex + sizeof( UnixFileDescriptor ) > (long)ptr )
        return( BAD_BUFFER );

    Session->inactivity = 0;

    /* Should always be able to obtain status on open files */
    if ( fstat( desc->unixfile, &FileStat ) != 0 )
	return( INTERNAL_ERROR );
    /* Do a consistency check on the validity of the name and
       the instance */
    if ( stat( desc->name, &NameStat ) != 0 
	 || NameStat.st_ino != FileStat.st_ino 
         || NameStat.st_dev != FileStat.st_dev )
        return( NOT_FOUND );

    /* have to make a directory entry equivalent name ie.
       "xyzzy" not "/zz/yy/xyzzy" */
    if ( *(ptr = rindex( desc->name, '/' ) + 1) == '\0' )
      {
	/* Directory instance name */
	ptr--;		/* skip '/' */
	while( ptr > desc->name && *--ptr != '/' );
	ptr++;
      }
        
    /* get equivalent of directory entry for the instance name */
    if ( DDebug ) printf( "Constructing descriptor for %s\n", ptr );
   
    Udesc = ConstructDescriptor( &NameStat, ptr );

    /* where to put it */
    MoveLong( req->segmentptr, ptr );
    ptr += req->dataindex;
    
    reply->replycode = OK;
    
    /* Using ReplyWithSegment could change if descriptor changes */
    ReplyWithSegment( reply, pid, (char *)Udesc, ptr, sizeof( UnixFileDescriptor ) );
    return( NOT_AWAITINGREPLY );
#undef reply

  } /* ReadDescriptor */

SystemCode NameReadDescriptor( req, pid, segsize )
    DescriptorRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Return the descriptor for the name in the request.
 */
  {
    register SystemCode r;
    char *NamePtr;
    char *ptr,*ptr2;
    struct stat NameStat;
    UnixFileDescriptor *Udesc;
#define reply	((DescriptorReply*)req)

    /* we do this check so that we don't have to explicitly check the
       permissions on all the directories down the path */
    if( Session == NULL ) 
        return( ILLEGAL_REQUEST );
 
    /* got enough room in the segment */
    MoveLong( req->segmentlen, ptr );
    if ( req->dataindex + sizeof( UnixFileDescriptor ) > (long)ptr )
        return( BAD_BUFFER );

    Session->inactivity = 0;

    if ( (r = GetName( req, pid, &segsize, &NamePtr )) != OK )
      {
        return( r );
      }

    if( DDebug ) printf( "NameReadDescriptor: \"%s\"\n", NamePtr );

    if ( stat( NamePtr, &NameStat ) != 0 )
	return( NOT_FOUND );

    /* make the name so that it will look like a directory entry */
    ptr = ParseName( (char*)malloc( strlen( NamePtr ) + 2 ), NamePtr, 0 );

    /* check for the special instance of the name being '/' */
    if ( *(ptr2 = rindex( ptr, '/' ) + 1) == '\0' )
        ptr2--;
    
    Udesc = ConstructDescriptor( &NameStat, ptr2 );
    
    free( ptr );	/* free the name that we allocated */

    /* where to put it */
    MoveLong( req->segmentptr, ptr2 );
    ptr2 += req->dataindex;
    
    reply->replycode = OK;
    
    /* Using ReplyWithSegment could change if descriptor changes */
    ReplyWithSegment( reply, pid, (char *)Udesc, ptr2, sizeof( UnixFileDescriptor ) );

    return( NOT_AWAITINGREPLY );
#undef reply
  } /* NameReadDescriptor */

SystemCode WriteDescriptor( req, pid, segsize )
    DescriptorRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Read the descriptor for the file specified by the fileid
 * in the request.
 */
  {
    register FileInstance *desc;
    struct stat FileStat, NameStat;
    UnixFileDescriptor Descriptor;
    register SystemCode r;

    if ( Session == NULL ) 
        return( ILLEGAL_REQUEST );

    if( (desc = GetInstance(req->fileid)) == NULL ) 
        return( INVALID_FILE_ID );

    if( desc->type != FILE_INSTANCE && desc->type != DIRECTORY_INSTANCE ) 
        return( REQUEST_NOT_SUPPORTED );

    Session->inactivity = 0;

    /* Should always be able to obtain status on open files */
    if ( fstat( desc->unixfile, &FileStat ) != 0 )
	return( INTERNAL_ERROR );

    /* Do a consistency check on the validity of the name and
       the instance */
    if ( stat( desc->name, &NameStat ) != 0 
	 || NameStat.st_ino != FileStat.st_ino 
         || NameStat.st_dev != FileStat.st_dev )
        return( NOT_FOUND );

    if ( (r = GetDescriptor( req, pid, &segsize, &Descriptor )) != OK )
        return( r );

    return( ModifyDescriptor( desc->name, &Descriptor ) );

  } /* WriteDescriptor */

SystemCode NameWriteDescriptor( req, pid, segsize )
    DescriptorRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Get the descriptor specified in the request and modify
 * the entries of the named file.
 */
  {
    register SystemCode r;
    char *NamePtr;
    UnixFileDescriptor Descriptor;

    /* we do this check so that we don't have to explicitly check the
       permissions on all the directories down the path */
    if( Session == NULL ) return( NO_PERMISSION );
 
    Session->inactivity = 0;

    if ( (r = GetName( req, pid, &segsize, &NamePtr )) != OK )
        return( r );

    if( DDebug ) printf( "NameWriteDescriptor: \"%s\"\n", NamePtr );

    if ( (r = GetDescriptor( req, pid, &segsize, &Descriptor )) != OK )
        return( r );

    return( ModifyDescriptor( NamePtr, &Descriptor ) );

  } /* NameWriteDescriptor */

