/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * V Team Server
 *
 * Marvin Theimer, 5/83
 *
 * This server manages the loading and termination of teams.  It also keeps
 * track of relevant information concerning running teams and provides an
 * exception handler of last resort in the form of a postmortem debugger.
 * Information concerning the status of teams running under the team server
 * can be obtained by reading the (logical) team server directory.
 *
 * TPM 8/18/83 - Redid directory access code.
 * TPM 8/26/83 - Lowered process priorities to sane values (were both 0),
 *	and changed to use LoadTeamFromNamedFile to eliminate duplicated
 *	code and buffer space.
 * Marvin Theimer, 9/29/83
 *	Changed load request to send root messages for both the program's
 *	io and the postmortem debugger's io.
 */


#include <Venviron.h>
#include <Vprocess.h>
#include <Vteams.h>
#include <Vioprotocol.h>
#include <Vexceptions.h>
#include <Vnaming.h>
#include <Vdirectory.h>
#include <chars.h>


#define FALSE 0
#define TRUE 1

#define TeamTimeTick 0		/* Request code used by the timer process to
				   signal a tick to the teamserver. */
#define TimerInterval 1		/* Delay period between timer messages. */

#define MaxTeams 64		/* Maximum number of running teams that can be
				   handled by the team server. */
#define MaxArgsLength 128	/* Maximum size in bytes of the parameters
				   passed in the segment accompanying a load
				   request. */

#define DirectoryInstId	0x4321	/* Instance id used for directory */

#define PostDebugger "[public]debug"

#define DebuggerTeamPriority 29


typedef struct 			/* Table entry record for a team that has
				   been loaded by the team server. */
  {
    ProcessId ownerPid;		/* Pid of process owning the loaded team. 
				   0 if this entry is unused.  The
				   owner has the right to request destruction
				   of the loaded team.  Also, if the owner
				   dies, then all teams owned by him are 
				   killed.  Teams that are to be owned by
				   the "system" are given an ownerPid equal
				   to the pid of the teamserver. */
    ProcessId rootPid;		/* Pid of root process of loaded team. */
    ProcessId debuggerPid;	/* Pid of postmortem debugger invoked for this
				   team if it has caused an exception.  0 if
				   no postmortem debugger has been invoked. */
    char *name;			/* File name team was loaded from. */
    ProcessId loadServer;	/* Pid of file instance from which to load 
				   team. */
    InstanceId loadFile;	/* Fileid of file instance from which to load
				   team. */
    RootMessage rtMsg;		/* Root message for program to be loaded. */
    RootMessage debugRtMsg;	/* Root message for postmortem debugger. */
  } TeamsEntry;

TeamsEntry Teams[MaxTeams] = {0};
ProcessId TeamServerPid;


SystemCode LoadTeamSpace();
SystemCode TerminateTeams();
SystemCode InvokeDebugger();
SystemCode PerformMaintenance();
static SystemCode CreateInstance();
static SystemCode QueryInstance();
static SystemCode ReadInstance();
static SystemCode NReadDescriptor();
static SystemCode ReadDescriptor();

extern char *malloc();
extern ProcessId LoadTeamFromFileInstance();
extern ProcessId LoadTeamFromNamedFile();
extern SystemCode RegisterTeamServer();

extern RootMessage *RootMsg;


/*
 * InitTeamServer:
 * Starts up the team server if it doesn't already exist.
 */

InitTeamServer(newProcess)
    int newProcess;		/* True if TeamServer should as a separate
				   process. */
  {
    ProcessId timerPid;
    int TeamServer(), TeamTimerProcess();

    if (ValidPid(GetPid(TEAM_SERVER, LOCAL_PID)))
	return;

    if (newProcess)
      {
	TeamServerPid = Create(3, TeamServer, 4000);
	if (TeamServerPid == 0)
	  {
	    printf("ERROR - Failed to create team server process.\n");
	    exit();
	  }
	if (!Ready(TeamServerPid, 0))
	  {
	    printf("ERROR - Failed to ready team server process.\n");
	    exit();
	  }
      }
    else
	TeamServerPid = GetPid(ACTIVE_PROCESS, LOCAL_PID);
    timerPid = Create(10, TeamTimerProcess, 500);
    if (timerPid == 0)
      {
        printf("ERROR - Failed to create a team server process.\n");
	exit();
      }
    if (!Ready(timerPid, 0))
      {
        printf("ERROR - Failed to ready a team server process.\n");
	exit();
      }
    if (!newProcess)
	TeamServer();
  }


/*
 * TeamServer:
 * Top level of the team server.
 */

TeamServer()
  {
    Message msg;
    LoadTeamRequest *reqMsg = (LoadTeamRequest *) msg;
    LoadTeamReply *repMsg = (LoadTeamReply *) msg;
    ProcessId reqPid;
    ProcessId exceptionServerPid;
    RegHandlerRequest *regReq = (RegHandlerRequest *) msg;

    /* Register team server under its local logical id. */
    SetPid(TEAM_SERVER, TeamServerPid, LOCAL_PID);
    
#ifdef HOSTSERVER
    /* Register team server with the host server.  List it as available
       for remote task execution. */
    RegisterTeamServer("no user", 1);
#endif HOSTSERVER

    /* Register the team server as the exception handler of last resort. */
    exceptionServerPid = GetPid(EXCEPTION_SERVER, LOCAL_PID);
    if (exceptionServerPid)
      {
	regReq->requestcode = REGISTER_HANDLER;
	regReq->regFlag = 1;
	regReq->handlerPid = GetPid(ACTIVE_PROCESS, LOCAL_PID);
	regReq->regPid = regReq->handlerPid;
	exceptionServerPid = Send(msg, exceptionServerPid);
	if (exceptionServerPid == 0)
	  {
	    exit();
	  }
      }

    /* Loop on requests for service. */
    while (TRUE)
      {
	reqPid = Receive(reqMsg);
	switch (reqMsg->requestcode)
	  {
	    case CREATE_INSTANCE:
		repMsg->replycode = CreateInstance(reqMsg, reqPid);
		break;
	    case READ_INSTANCE:
		repMsg->replycode = ReadInstance(reqMsg, reqPid);
		break;
	    case QUERY_INSTANCE:
		repMsg->replycode = QueryInstance(reqMsg, reqPid);
		break;
	    case RELEASE_INSTANCE:
	    case SET_INSTANCE_OWNER:
		repMsg->replycode = OK;
		break;
	    case READ_DESCRIPTOR:
		repMsg->replycode = ReadDescriptor(reqMsg, reqPid);
		break;
	    case NREAD_DESCRIPTOR:
		repMsg->replycode = NReadDescriptor(reqMsg, reqPid);
		break;
	    case TeamLoadRequest:
	        repMsg->replycode = LoadTeamSpace(reqMsg, reqPid);
		break;
	    case TERMINATE_TEAM:
	        repMsg->replycode = TerminateTeams(reqMsg, reqPid);
		break;
	    case EXCEPTION_REQUEST:
	        repMsg->replycode = InvokeDebugger(reqMsg, reqPid);
		break;
	    case TeamTimeTick:
		repMsg->replycode = PerformMaintenance();
		break;
	    default:
	        repMsg->replycode = REQUEST_NOT_SUPPORTED;
		break;
	  }
	if (repMsg->replycode != NO_REPLY)
	  {
	    Reply(repMsg, reqPid);
	  }
      }
  }


/*
 * LoadTeamSpace:
 * Loads a team from an already opened file which is specified by a file
 * instance (server, fileid).  Replies to the requestor with the processId
 * of the team's root process and also forwards the team to the requestor so
 * that the he can complete setting up the team (e.g. arguments).
 */

SystemCode LoadTeamSpace(reqMsg, reqPid)
    LoadTeamRequest *reqMsg;
    ProcessId reqPid;
  {
    LoadTeamReply *repMsg = (LoadTeamReply *) reqMsg;
    int i;
    SystemCode error;
    char buffer[MaxArgsLength];
    char *curPtr, *t;
    int len;
    short nameLength;
    char *name;
    ProcessId newPid;

    /* Find an unused entry in the Teams table. */
    for (i = 0; i < MaxTeams; i++)
      {
	if (Teams[i].ownerPid == 0)
	    break;
      }
    if (i == MaxTeams)
        return(NO_SERVER_RESOURCES);
    
    /* Get parameters associated with the loaded team. */
    if (reqMsg->bytecount > MaxArgsLength)
        return(BAD_ARGS);
    error = MoveFrom(reqPid, buffer, reqMsg->bufferptr, reqMsg->bytecount);
    if (error != OK)
	return(error);
    curPtr = buffer;		/* Points to file name. */
    nameLength = strlen(curPtr);
    name = malloc(nameLength);
    if (name == NULL)
        return(NO_MEMORY);
    strcpy(name, curPtr);	/* Assumes that file name is terminated with
				   a \0. */
    curPtr += nameLength + 1;	/* Points past \0 */
    
    /* Try to create and load a team from the specified file instance. */
    newPid = LoadTeamFromFileInstance(reqMsg->fileserver, reqMsg->fileid, 
    		&error);
    if (newPid == NULL)
      {
        free(name);
        return(error);
      }
    
    /* Initialize Teams table entry. */
    if (reqMsg->noOwner)
      {
	Teams[i].ownerPid = GetPid(ACTIVE_PROCESS, LOCAL_PID);
      }
    else
      {
	Teams[i].ownerPid = reqPid;
      }
    Teams[i].rootPid = newPid;
    Teams[i].debuggerPid = 0;
    Teams[i].name = name;
    Teams[i].loadServer = reqMsg->fileserver;
    Teams[i].loadFile = reqMsg->fileid;
    t = (char *) &(Teams[i].rtMsg);
    len = sizeof(RootMessage);
    while (len-- > 0)
	*t++ = *curPtr++;
    t = (char *) &(Teams[i].debugRtMsg);
    len = sizeof(RootMessage);
    while (len-- > 0)
	*t++ = *curPtr++;
    
    /* Reply to the requestor with the team's root process. */
    repMsg->replycode = OK;
    repMsg->rootPid = newPid;
    Reply(repMsg, reqPid);
    
    /* Forward the team to the requestor so that he can complete setting it
       up. */
    Forward(repMsg, newPid, reqPid);
				/* The contents of repMsg is unimportant. */

    return(NO_REPLY);
  }


/*
 * TerminateTeams:
 * Destroys the specified team(s).  Also destroys any associated postmortem
 * debuggers.  If the termination
 * request is for a postmortem debugger, then the associated team it was
 * invoked on is also destroyed.
 */

SystemCode TerminateTeams(reqMsg, reqPid)
    ExitTeamRequest *reqMsg;
    ProcessId reqPid;
  {
    int i;

    if (reqMsg->requestType == SpecificTeam)
      {
	/* Find the Teams table entry for the specified team.  
	   It will be either
	   a "regular" team or a postmortem debugger team associated with a 
	   regular team entry. */
	for (i = 0; i < MaxTeams; i++)
	    if ((Teams[i].ownerPid != 0) &&
		    ((reqMsg->rootPid == Teams[i].rootPid) || 
		    (reqMsg->rootPid == Teams[i].debuggerPid)))
		break;
	if (i == MaxTeams)
	  {
	    return(NOT_FOUND);
	  }
	
	/* Terminate and reclaim the teams for the table entry. */
	TerminateSingleTeam(i);
      }
    else
	/* Terminate resp. all teams or all teams of the specified 
	   requestor. */
      {
	for (i = 0; i < MaxTeams; i++)
	  {
	    if ((reqMsg->requestType == AllTeams) ||
 	    	    (reqPid == Teams[i].ownerPid))
	      {
	        /* Terminate and reclaim the teams for this table entry. */
		TerminateSingleTeam(i);
	      }
	  }
      }
       
    return(OK);
  }


/*
 * TerminateSingleTeam:
 * Terminate the team and associated postmortem debugger (if any) for the
 * designated Teams table entry and then reclaim the table entry.
 */

TerminateSingleTeam(i)
    int i;			/* Index of Teams table entry. */
  {
    /* Destroy the "regular" team specified in the table entry. */
    DestroyProcess(Teams[i].rootPid);

    /* If the team has an associated postmortem debugger then
       destroy that too. */
    if (Teams[i].debuggerPid != 0)
      {
	DestroyProcess(Teams[i].debuggerPid);
      }

    /* Release the team's program image file instance */
    ReleaseInstance(Teams[i].loadServer, Teams[i].loadFile, OK);

    /* Return the space for the name to free space. */
    free(Teams[i].name);

    /* Remove the table entry. */
    Teams[i].ownerPid = 0;
  }


/*
 * InvokeDebugger:
 * Invokes the postmortem debugger to handle a team that has incurred an
 * exception.
 */

SystemCode InvokeDebugger(reqMsg, reqPid)
    LoadTeamRequest *reqMsg;
    ProcessId reqPid;
  {
    int i;
    ProcessId debuggerPid;
    SystemCode error;
    char lineBuf[MaxArgsLength];
    char *args[10];
    char **argv;

    /* Find the Teams table entry for the team incurring the exception. */
    for (i = 0; i < MaxTeams; i++)
	if ((Teams[i].ownerPid != 0) && SameTeam(Teams[i].rootPid, reqPid))
	    break;
    if (i == MaxTeams)		/* Exception occurred in the postmortem
				   debugger! */
      {
        /* Find which postmortem debugger is the culprit. */
	for (i = 0; i < MaxTeams; i++)
	    if ((Teams[i].ownerPid != 0) && 
	    	SameTeam(Teams[i].debuggerPid, reqPid))
	        break;
	if (i == MaxTeams)
	    exit();		/* Should never get an exception from an
				   unregistered team!! */
	
	/* Print standard exception information. */
	PrintStdExceptInfo(Teams[i].debugRtMsg.stdoutserver, 
		Teams[i].debugRtMsg.stdoutfile,
		reqMsg, reqPid, "Exception in postmortem debugger");
	   
	/* Destroy both the Postmortem debugger and the team it was invoked
	   on. */
	TerminateSingleTeam(i);
	   
	return(NO_REPLY);
      }

    /* Load the postmortem debugger from a prespecified file. */
    debuggerPid = LoadTeamFromNamedFile(PostDebugger, &error);
    if (debuggerPid == NULL)
      {
          /*
	   * Print an error message and the standard exception information. 
	   */
	strcpy(lineBuf,"Error loading postmortem debugger: ");
	strcat(lineBuf,ErrorString(error));
	PrintStdExceptInfo(Teams[i].debugRtMsg.stdoutserver, 
		Teams[i].debugRtMsg.stdoutfile,
		reqMsg, reqPid, 
		lineBuf);
	   
	/* Terminate the team causing the exception. */
	TerminateSingleTeam(i);
	
	return(NO_REPLY);
      }
      
    /* Set up the arguments for the debugger and set its team priority. */
    strcpy(lineBuf, PostDebugger);
    strcat(lineBuf, " -p");
    strcat(lineBuf, " ");
    strcat(lineBuf, Teams[i].name);
    ParseLine(lineBuf, args);
    argv = args;
    SetUpArguments(debuggerPid, argv);
    SetTeamPriority(debuggerPid, DebuggerTeamPriority);
    
    /* Enter debugger into the Teams table entry of the team
       incurring the exception. */
    Teams[i].debuggerPid = debuggerPid;
    
    /* Set the debugger running and forward the exception message to it. */
    Reply(&(Teams[i].debugRtMsg), debuggerPid);
    Forward(reqMsg, reqPid, debuggerPid);
    
    return(NO_REPLY);
  }


/*
 * CreateInstance:
 * Permit reading the teamserver's directory.  Nothing is actually
 *   created here, since the teamserver maintains a single, permanent
 *   instance.
 */

static SystemCode CreateInstance(reqMsg, reqPid)
    CreateInstanceRequest *reqMsg;
    ProcessId reqPid;
  {
    CreateInstanceReply *repMsg = (CreateInstanceReply *) reqMsg;

    repMsg->fileid = DirectoryInstId;
    return(QueryInstance(reqMsg, reqPid));
  }


/*
 * QueryInstance:
 * Return parameters of the directory instance.
 */

static SystemCode QueryInstance(reqMsg, reqPid)
    QueryInstanceRequest *reqMsg;
    ProcessId reqPid;
  {
    QueryInstanceReply *repMsg = (QueryInstanceReply *) reqMsg;

    if (reqMsg->fileid != DirectoryInstId)
      {
	return(INVALID_FILE_ID);
      }
    repMsg->fileserver = GetPid(ACTIVE_PROCESS, LOCAL_PID);
    repMsg->blocksize = sizeof(TeamDescriptor);
    repMsg->filetype = READABLE|FIXED_LENGTH;
    repMsg->filelastbytes = repMsg->blocksize;
    repMsg->filelastblock = MaxTeams-1;
    repMsg->filenextblock = 0;
    return(OK);
  }


/*
 * ReadInstance:
 * Read entries from the teamserver's directory.
 */

static SystemCode ReadInstance(reqMsg, reqPid)
    IoRequest *reqMsg;
    ProcessId reqPid;
  {
    unsigned len, block;
    IoReply *repMsg = (IoReply *) reqMsg;
    TeamDescriptor desc;
    SystemCode error;

    len = reqMsg->bytecount;
    block = reqMsg->blocknumber;
    repMsg->bytecount = 0;

    if (len > sizeof(TeamDescriptor))
	return(BAD_BYTE_COUNT);

    if (reqMsg->fileid != DirectoryInstId)
	return(INVALID_FILE_ID);

    if (block >= MaxTeams)
	return(END_OF_FILE);

    /* Set up descriptor to return */
    BuildDescriptor(&desc, block);

    /* Do the actual data movement and reply-to-read. */
    repMsg->bytecount = len;
    if (len > MAX_APPENDED_SEGMENT)
      {
	error = MoveTo(reqPid, reqMsg->bufferptr, &desc, len);
	if (error != OK)
	  {
	    repMsg->bytecount = 0;
	    return (error);
	  }
	else if (block == MaxTeams-1 && len == sizeof(TeamDescriptor))
	    return (END_OF_FILE);  /* successful read to EOF */
	else
	    return (OK);
      }
    else
      {
	if (block == MaxTeams-1 && len == sizeof(TeamDescriptor))
	    repMsg->replycode = END_OF_FILE;  /* successful read to EOF */
	else
            repMsg->replycode = OK;

	ReplyWithSegment(repMsg, reqPid, &desc, reqMsg->bufferptr, len);
        return(NO_REPLY);
      }
  }


/*
 * NReadDescriptor:
 * Read the descriptor of a single team, specified by name.  For this
 *   server, the contextid is the team's root pid, while the string
 *   name is ignored.  Contextid DEFAULT_CONTEXT (0) specifies the
 *   context directory's descriptor.
 */

static SystemCode NReadDescriptor(reqMsg, reqPid)
    register DescriptorRequest *reqMsg;
    ProcessId reqPid;
  {
    register DescriptorReply *repMsg = (DescriptorReply *) reqMsg;
    TeamDescriptor desc;
    unsigned len = sizeof(TeamDescriptor);
    int i;

    /* Check if the context directory is being named */
    if (reqMsg->namecontextid == DEFAULT_CONTEXT)
      {
	reqMsg->fileid = DirectoryInstId;
	return (ReadDescriptor(reqMsg, reqPid));
      }

    if (reqMsg->segmentlen - reqMsg->dataindex < len)
	return (BAD_BUFFER);

    /* Find the team being referenced */
    for (i = 0; i < MaxTeams; i++)
      {
	if ((Teams[i].ownerPid != 0) &&
	        (Teams[i].rootPid == reqMsg->namecontextid)) 
	    break;
      }
    if (i == MaxTeams) return (NOT_FOUND);

    /* Build a descriptor for it */
    BuildDescriptor(&desc, i);

    /* Do the actual data movement.  Since len is constant,
     *   this code seems like overkill. */
    if (len > MAX_APPENDED_SEGMENT)
      {
	return(MoveTo(reqPid, reqMsg->segmentptr + reqMsg->dataindex,
	    &desc, len));
      }
    else
      {
        repMsg->replycode = OK;

	ReplyWithSegment(repMsg, reqPid, &desc, 
	    reqMsg->segmentptr + reqMsg->dataindex, len);
        return(NO_REPLY);
      }
  }    
    


/*
 * ReadDescriptor:
 * Read the descriptor of a single file, specified by instance id.
 *   The only "file" implemented by the team server is its directory.
 *   It doesn't really have a descriptor, so we return an EmptyDescriptor.
 */

static SystemCode ReadDescriptor(reqMsg, reqPid)
    register DescriptorRequest *reqMsg;
    ProcessId reqPid;
  {
    register DescriptorReply *repMsg = (DescriptorReply *) reqMsg;
    EmptyDescriptor desc;
    unsigned len = sizeof(EmptyDescriptor);
    int i;

    if (reqMsg->segmentlen - reqMsg->dataindex < len)
	return (BAD_BUFFER);

    if (reqMsg->fileid != DirectoryInstId)
	return (INVALID_FILE_ID);

    desc.descriptortype = EMPTY_DESCRIPTOR;

    /* Do the actual data movement.  Since len is constant,
     *   this code seems like overkill. */
    if (len > MAX_APPENDED_SEGMENT)
      {
	return(MoveTo(reqPid, reqMsg->segmentptr + reqMsg->dataindex,
	    &desc, len));
      }
    else
      {
        repMsg->replycode = OK;

	ReplyWithSegment(repMsg, reqPid, &desc, 
	    reqMsg->segmentptr + reqMsg->dataindex, len);
        return(NO_REPLY);
      }
  }    
    

/*
 * BuildDescriptor:
 * Creates a TeamDescriptor record that can be handed back to a client.
 */

BuildDescriptor(desc, index)
    TeamDescriptor *desc;
    int index;
  {
    if (Teams[index].ownerPid == 0)
      {
	desc->descriptortype = EMPTY_DESCRIPTOR;
      }
    else
      {
	desc->descriptortype = TEAM_DESCRIPTOR;
	strncpy(desc->fileName, Teams[index].name, TeamFileNameLength);
	desc->ownerPid = Teams[index].ownerPid;
	desc->rootPid = Teams[index].rootPid;
	desc->fileserver = Teams[index].loadServer;
	desc->fileid = Teams[index].loadFile;
      }
  }

/*
 * PerformMaintenance:
 * Does periodic maintenance functions such as destroying teams owned by
 * deceased owners and reclaiming Teams table entries for vanished teams.
 */

SystemCode PerformMaintenance()
  {
    int i, flag = 1;

    /* Get rid of teams whose owners have gone away. */
    while (flag)
      {
        flag = 0;
	for (i = 0; i < MaxTeams; i++)
	  {
	    if (Teams[i].ownerPid != 0)
	      {
		/* Garbage-collect if the owner has died or the team 
		   has somehow
		   vanished by unspecified means (such as a manual 
		   DestroyProcess). */
		if ((!ValidPid(Teams[i].ownerPid)) ||
		    (!ValidPid(Teams[i].rootPid)))
		  {
		    TerminateSingleTeam(i);
		    flag = 1;
		  }
	      }
	  }
      }

    return(OK);
  }


/*
 * PrintStdExceptInfo:
 * Prints an error message and then calls the standard exception handler
 * routine.
 */

PrintStdExceptInfo(fileServer, fileid, reqMsg, reqPid, errMsg)
    ProcessId fileServer;
    InstanceId fileid;
    ExceptionRequest *reqMsg;
    ProcessId reqPid;
    char *errMsg;
  {
    File *outFile;
    SystemCode error;

    /* Open the output file instance. */
    outFile = OpenFile(fileServer, fileid, FAPPEND, &error);
    if (outFile == NULL)
      {
	outFile = stdout;
      }

    /* Print the error message if specified. */
    if (errMsg)
      {
	fprintf(outFile, errMsg);
	Flush(outFile);
      }

    /* Call the standard exception handler routine to print out the
       exception information. */
    StandardExceptionHandler(reqMsg, reqPid, outFile);

    /* Close the output file instance unless it is stdout . */
    if (outFile != stdout)
	Close(outFile);
  }




/*
 * TeamTimerProcess:
 * Periodically wakes up and sends a msg to the main team server process
 * so that it can perform various maintenance functions.
 */

TeamTimerProcess()
  {
    Message msg;
    IoRequest *req = (IoRequest *) msg;

    while (1)
      {
	Delay(TimerInterval, 0);
	req->requestcode = TeamTimeTick;
	Send(msg, TeamServerPid);
      }
  }
