/*
 * localname.c
 *
 * V-System Local Name Server
 * Copyright 1983 (c) All Rights Reserved.
 * David Cheriton, Tim Mann
 *
 * The local name server provides mappings from locally defined context
 *   names to contexts, where a context is specified by (name-server, 
 *   context-id).  The server also provides symbolic names for the devices 
 *   implemented by the V kernel device server.  The context and device
 *   directories are made available as files.
 */

#include "Vnaming.h"
#include "Vdirectory.h"
#include "localname.h"
#include "Vioprotocol.h"
#include "Vdevicetypes.h"

/* Functions defined */
ProcessId InitLocalNameServer();
static Initialize();
static LocalNameServer();
static SystemCode DefaultRequest();
static SystemCode CreateInstance_();
static SystemCode QueryInstance();
static SystemCode ReadInstance();
static SystemCode ReleaseInstance();
static SystemCode SetInstanceOwner();
static SystemCode AddContextName();
static SystemCode DeleteContextName();
static SystemCode GetContextId();
static SystemCode GetContextName();
static SystemCode GetFileName();
static SystemCode ReadDescriptor();
static SystemCode NReadDescriptor();
static SystemCode MapName();
static DirectoryEntry *FindEntry();
static SystemCode MoveInString();
static SystemCode ScanField();

static DirectoryEntry ContextDirectory[CONTEXT_DIRECTORY_SIZE];
static DirectoryEntry DeviceDirectory[DEVICE_DIRECTORY_SIZE];

static DirectoryEntry *ContextDirDescriptor;
static DirectoryEntry *DeviceDirDescriptor;

static char NameBuffer[MAX_PATH_LENGTH+1];
static char *LastComponent; /* ptr to last component of name in NameBuffer */

static ProcessId LocalNameServerPid;


ProcessId InitLocalNameServer()
  {
    /*
     * Create and start the local name server.
     */
    ProcessId pid;

    if (ValidPid(pid = GetPid(NAME_SERVER, LOCAL_PID))) return (pid);
    pid = Create(NAME_SERVER_PRIORITY, LocalNameServer, NAME_SERVER_STACK);
    LocalNameServerPid = pid;
    if (pid != 0)
      {
	SetPid(NAME_SERVER, pid, LOCAL_PID);
	Ready(pid, 0);
      }
    return (pid);
  }


static LocalNameServer()
  {
    /*
     * Base function of the local name server.
     */
    Message msg;
    register ContextRequest *request = (ContextRequest *) msg;
    register ContextReply *reply = (ContextReply *) msg;
    register ProcessId pid;
    register SystemCode r;

    Initialize();

    while (1)
      {
	pid = Receive(request);
	
	switch (request->requestcode)
	  {
	    case ADD_CONTEXT_NAME:
		r = AddContextName(request, pid);
		break;

	    case DELETE_CONTEXT_NAME:
		r = DeleteContextName(request, pid);
		break;

	    case GET_CONTEXT_ID:
		r = GetContextId(request, pid);
		break;

	    case GET_CONTEXT_NAME:
		r = GetContextName(request, pid);
		break;

	    case GET_FILE_NAME:
	        r = GetFileName(request, pid);
		break;

	    case CREATE_INSTANCE: 
		r = CreateInstance_(request, pid);
		break;

	    case QUERY_INSTANCE:
		r = QueryInstance(request, pid);
		break;

	    case READ_INSTANCE:
		r = ReadInstance(request, pid);
		break;

	    case SET_INSTANCE_OWNER:
	    case RELEASE_INSTANCE: 
		r = OK;
		break;

	    case READ_DESCRIPTOR:
		r = ReadDescriptor(request, pid);
		break;

	    case NREAD_DESCRIPTOR:
	        r = NReadDescriptor(request, pid);
		break;

	    default:
		r = DefaultRequest(request, pid);
		break;
	  }

	if (r != NO_REPLY)
	  {
	    reply->replycode = r;
	    Reply(reply, pid);
	  }
      }
  }


static Initialize()

  /* Add standard contexts to the context directory,
   * and perform other server initialization.
   */
  {

    register DirectoryEntry *entry;

    entry = &ContextDirectory[0];
    entry->c.descriptortype = LOCALNAME_DESCRIPTOR;
    strcpy(entry->c.name, ".");
    entry->c.entrytype = IMMUTABLE;
    entry->c.serverpid = LocalNameServerPid;
    entry->c.contextid = CONTEXT_DIRECTORY;

    entry++;
    entry->c.descriptortype = LOCALNAME_DESCRIPTOR;
    strcpy(entry->c.name, "context");
    entry->c.entrytype = IMMUTABLE;
    entry->c.serverpid = LocalNameServerPid;
    entry->c.contextid = CONTEXT_DIRECTORY;
    ContextDirDescriptor = entry;

    entry++;
    entry->c.descriptortype = LOCALNAME_DESCRIPTOR;
    strcpy(entry->c.name, "device");
    entry->c.entrytype = IMMUTABLE;
    entry->c.serverpid = LocalNameServerPid;
    entry->c.contextid = DEVICE_DIRECTORY;
    DeviceDirDescriptor = entry;

    entry++;
    entry->c.descriptortype = LOCALNAME_DESCRIPTOR;
    strcpy(entry->c.name, "public");
    entry->c.entrytype = IMMUTABLE+LOGICAL_PID;
    entry->c.serverpid = STORAGE_SERVER;
    entry->c.contextid = PUBLIC_CONTEXT;

    entry++;
    entry->c.descriptortype = LOCALNAME_DESCRIPTOR;
    strcpy(entry->c.name, "internet");
    entry->c.entrytype = IMMUTABLE+LOGICAL_PID;
    entry->c.serverpid = INTERNET_SERVER;
    entry->c.contextid = DEFAULT_CONTEXT;

    entry++;
    entry->c.descriptortype = LOCALNAME_DESCRIPTOR;
    strcpy(entry->c.name, "vgts");
    entry->c.entrytype = IMMUTABLE+LOGICAL_PID;
    entry->c.serverpid = VGT_SERVER;
    entry->c.contextid = DEFAULT_CONTEXT;

    entry++;
    entry->c.descriptortype = LOCALNAME_DESCRIPTOR;
    strcpy(entry->c.name, "teams");
    entry->c.entrytype = IMMUTABLE+LOGICAL_PID;
    entry->c.serverpid = TEAM_SERVER;
    entry->c.contextid = DEFAULT_CONTEXT;

    entry++;
    entry->c.descriptortype = LOCALNAME_DESCRIPTOR;
    strcpy(entry->c.name, "hosts");
    entry->c.entrytype = IMMUTABLE+LOGICAL_PID;
    entry->c.serverpid = HOST_SERVER;
    entry->c.contextid = DEFAULT_CONTEXT;

    entry = &DeviceDirectory[0];
    entry->c.descriptortype = LOCALNAME_DESCRIPTOR;
    strcpy(entry->c.name, ".");
    entry->c.entrytype = IMMUTABLE;
    entry->c.serverpid = LocalNameServerPid;
    entry->c.contextid = DEVICE_DIRECTORY;

    entry++;
    entry->d.descriptortype = DEVICE_DESCRIPTOR;
    strcpy(entry->d.name, "serial0");
    entry->d.entrytype = IMMUTABLE+SET_UNSPECIFIED0;
    entry->d.devicetype = SERIAL;
    entry->d.unspecified0 = 0;

    entry++;
    entry->d.descriptortype = DEVICE_DESCRIPTOR;
    strcpy(entry->d.name, "serial1");
    entry->d.entrytype = IMMUTABLE+SET_UNSPECIFIED0;
    entry->d.devicetype = SERIAL;
    entry->d.unspecified0 = 1;

    entry++;
    entry->d.descriptortype = DEVICE_DESCRIPTOR;
    strcpy(entry->d.name, "console");
    entry->d.entrytype = IMMUTABLE;
    entry->d.devicetype = CONSOLE;

    entry++;
    entry->d.descriptortype = DEVICE_DESCRIPTOR;
    strcpy(entry->d.name, "ethernet");
    entry->d.entrytype = IMMUTABLE;
    entry->d.devicetype = ETHERNET;

    entry++;
    entry->d.descriptortype = DEVICE_DESCRIPTOR;
    strcpy(entry->d.name, "mouse");
    entry->d.entrytype = IMMUTABLE;
    entry->d.devicetype = MOUSE;

  }


static SystemCode AddContextName(request, pid)
    ContextRequest *request;
    ProcessId pid;
  {
    SystemCode r;
    DirectoryEntry *contextptr;
    int search, index;

    if ( (r=MapName(request, pid, &contextptr)) != NOT_FOUND )
	return ( (r == OK) ? DUPLICATE_NAME : r );
    if (null(LastComponent)) return (BAD_ARGS);

    /* If not simply a context name then context was not found (?) */
    if (NameBuffer[request->nameindex] != '\0') return (NOT_FOUND);

    /* Can only add names to context directory */
    if (request->namecontextid != CONTEXT_DIRECTORY) 
	return (ILLEGAL_REQUEST);

    /* Find a free directory entry */
    index = -1;
    for (search = 0; search < CONTEXT_DIRECTORY_SIZE; ++search)
      {
	if (ContextDirectory[search].c.descriptortype == EMPTY_DESCRIPTOR)
	  {
	    index = search;
	    break;
	  }
      }
    if (index == -1) return (NO_SERVER_RESOURCES);

    /* Enter the new context in the context directory */
    contextptr = &ContextDirectory[index];
    contextptr->c.descriptortype = LOCALNAME_DESCRIPTOR;
    strcpy(contextptr->c.name, LastComponent);
    contextptr->c.entrytype = request->entrytype & ~LOCAL_MASK;
    contextptr->c.serverpid = request->serverpid;
    contextptr->c.contextid = request->contextid;
    contextptr->c.instanceid = request->instanceid;
    contextptr->c.otherinfo = request->otherinfo;

    return (OK);
  }


static SystemCode DeleteContextName(request, pid)
    ContextRequest *request;
    ProcessId pid;
  {
    SystemCode r;
    DirectoryEntry *contextptr;
    register ContextReply *reply = (ContextReply *) request;

    if ( (r = MapName(request, pid, &contextptr)) != OK) return (r);
    if (null(LastComponent)) return (BAD_ARGS);

    if (contextptr->c.entrytype & IMMUTABLE) 
	return (ILLEGAL_REQUEST);

    /* Return old information about this name to the caller */
    reply->serverpid = contextptr->c.serverpid;
    reply->contextid = contextptr->c.contextid;
    reply->instanceid = contextptr->c.instanceid;
    reply->otherinfo = contextptr->c.otherinfo;
    reply->entrytype = contextptr->c.entrytype;
    
    /* Free up the directory entry */
    contextptr->c.descriptortype = EMPTY_DESCRIPTOR;

    return (OK);
  }


static SystemCode GetFileName(request, pid)
    ContextRequest *request;
    ProcessId pid;
  {
  /*
   * Return a string name for the given file instance, and the context
   *   in which the string name is to be interpreted.  The latter is
   *   always (LocalNameServerPid, ANY_CONTEXT) if the instance
   *   exists.
   */
    register ContextReply *reply = (ContextReply *) request;
    SystemCode r;
    char *name;
    int length;

    if (request->instanceid == CONTEXT_DIRECTORY_INSTANCE)
	name = "[context]";
    else if (request->instanceid == DEVICE_DIRECTORY_INSTANCE)
        name = "[device]";
    else
	return (INVALID_FILE_ID);

    /* Found a name for this file -- return it */
    length = strlen(name);
    if (length > request->namelength) length = request->namelength;

    if (r = MoveTo(pid, request->nameptr, name, length))
	return (r);

    reply->serverpid = LocalNameServerPid;
    reply->contextid = ANY_CONTEXT;  /* this is an "absolute" name */
    reply->entrytype = 0;
    reply->otherinfo = 0;
    reply->namelength = length;

    return (OK);
  }


static SystemCode GetContextName(request, pid)
    ContextRequest *request;
    ProcessId pid;
  {
  /*
   * Return a string name for the given (server, contextid) pair, if 
   *   we have one registered, and return the (server, contextid) pair
   *   in which the string name is to be interpreted.  The latter is
   *   always (LocalNameServerPid, ANY_CONTEXT) if the name is
   *   recognized at all.
   */
    register ContextReply *reply = (ContextReply *) request;
    SystemCode r;
    DirectoryEntry *contextptr;
    char namebuffer[MAX_NAME_LENGTH+3];
    int length, found;

    found = 0;

    /* Look first for an entry with the SESSION bit set */
    contextptr = &ContextDirectory[0];  /* entry zero is "." */
    while (++contextptr < &ContextDirectory[CONTEXT_DIRECTORY_SIZE])
      {
	if ( (contextptr->c.descriptortype == LOCALNAME_DESCRIPTOR) &&
	     (contextptr->c.entrytype & LOGICAL_PID) == 0 &&
	     (contextptr->c.entrytype & SESSION) &&
	      contextptr->c.serverpid == request->serverpid &&
	     (request->contextid == ANY_CONTEXT ||
	      contextptr->c.contextid == request->contextid) )
	  {
	    found++;
	    break;
	  }
      }

    /* If not found, look for one without */
    if (!found)
      {
	contextptr = &ContextDirectory[0];  /* entry zero is "." */
	while (++contextptr < &ContextDirectory[CONTEXT_DIRECTORY_SIZE])
	  {
	    if ( (contextptr->c.descriptortype == LOCALNAME_DESCRIPTOR) &&
		 (contextptr->c.entrytype & LOGICAL_PID) == 0 &&
		  contextptr->c.serverpid == request->serverpid &&
	         (request->contextid == ANY_CONTEXT ||
		  contextptr->c.contextid == request->contextid) )
	      {
		found++;
		break;
	      }
	  }
      }

    if (!found) return (NOT_FOUND);

    /* Found a name for this context -- return it */
    namebuffer[0] = '[';
    namebuffer[1] = '\0';
    strcat(namebuffer, contextptr->c.name);
    strcat(namebuffer, "]");
    length = strlen(namebuffer);
    if (length > request->namelength) length = request->namelength;

    if (r = MoveTo(pid, request->nameptr, namebuffer, length))
	return (r);

    reply->serverpid = LocalNameServerPid;
    reply->contextid = ANY_CONTEXT;
    reply->entrytype = contextptr->c.entrytype;
    reply->instanceid = contextptr->c.instanceid;
    reply->otherinfo = contextptr->c.otherinfo;
    reply->namelength = length;

    return (OK);
  }


static SystemCode CreateInstance_(request, pid)
    CreateInstanceRequest *request;
    ProcessId pid;
  {
    SystemCode r;
    DirectoryEntry *entry;
    CreateInstanceReply *reply = (CreateInstanceReply *) request;
    ProcessId serverpid;

    /* Lookup next component and forward if more to name */
    if ( (r=MapName(request, pid, &entry)) != OK ) return (r);

    if ((entry->c.descriptortype) == LOCALNAME_DESCRIPTOR)
      {
	if (entry->c.entrytype & LOGICAL_PID)
	    serverpid = GetPid(entry->c.serverpid, ANY_PID);
	else
	    serverpid = entry->c.serverpid;

	/* If not a context implemented by this server then forward. */
	if (serverpid != LocalNameServerPid)
	  {
	    request->contextid = entry->c.contextid;
	    Forward(request, pid, serverpid);
	    return (NO_REPLY);
	  }
	/* Perform a create instance on context or device directory,
	 *   which are the only files implemented by the context server.
	 * The FDIRECTORY bit should be set by the client, but we don't
	 *   insist on it.
	 */
	if (entry->c.contextid == CONTEXT_DIRECTORY)
	    reply->fileid = CONTEXT_DIRECTORY_INSTANCE;
	else /* entry->c.contextid == DEVICE_DIRECTORY */
	    reply->fileid = DEVICE_DIRECTORY_INSTANCE;

	return (QueryInstance(request, pid));
      }
    else /* (entry->d.descriptortype == DEVICE_DESCRIPTOR) */
      {
	/* Insert non-symbolic naming information in the message
	 *   and forward it off to the kernel device server
	 */
	request->type = entry->d.devicetype;
	if (entry->d.entrytype & SET_UNSPECIFIED0)
	    request->unspecified[0] = entry->d.unspecified0;
	Forward(request, pid, GetPid(DEVICE_SERVER, LOCAL_PID));
	return (NO_REPLY);
      }

  }


static SystemCode QueryInstance(request, pid)
    QueryInstanceRequest *request;
    ProcessId pid;
  {
    register QueryInstanceReply *reply = (QueryInstanceReply *) request;

    if (request->fileid == CONTEXT_DIRECTORY_INSTANCE)
        reply->filelastblock = CONTEXT_DIRECTORY_SIZE-1;
    else if (request->fileid == DEVICE_DIRECTORY_INSTANCE)
	reply->filelastblock = DEVICE_DIRECTORY_SIZE-1;
    else
        return (INVALID_FILE_ID);

    reply->fileserver = LocalNameServerPid;
    reply->blocksize = sizeof(DirectoryEntry);
    reply->filetype = READABLE | FIXED_LENGTH;
    reply->filelastbytes = reply->blocksize;
    reply->filenextblock = 0;

    return (OK);
  }


static SystemCode ReadInstance(request, pid)
    IoRequest *request;
    ProcessId pid;
  {
    SystemCode r;
    unsigned index, size, dirsize;
    DirectoryEntry *directory;
    register IoReply *reply = (IoReply *) request;

    if (request->fileid == CONTEXT_DIRECTORY_INSTANCE)
      {
	directory = ContextDirectory;
	dirsize = CONTEXT_DIRECTORY_SIZE;
      }
    else if (request->fileid == DEVICE_DIRECTORY_INSTANCE)
      {
	directory = DeviceDirectory;
	dirsize = DEVICE_DIRECTORY_SIZE;
      }
    else
      {
	reply->bytecount = 0;
	return (INVALID_FILE_ID);
      }

    if ( (index = request->blocknumber) >= dirsize)
      {
	reply->bytecount = 0;
	return (END_OF_FILE);
      }

    if ( (size = request->bytecount) > sizeof(DirectoryEntry))
      {
	reply->bytecount = 0;
	return (BAD_BYTE_COUNT);
      }

    if ( (r=MoveTo(pid, request->bufferptr, &directory[index], size)) == OK )
	reply->bytecount = size;
    else
	reply->bytecount = 0;

    if (index == dirsize-1 && 
            reply->bytecount == sizeof(DirectoryEntry))
        return (END_OF_FILE);   /* successful read to EOF */
    else
	return (r);
  }


static SystemCode GetContextId(request, pid)
    ContextRequest *request;
    ProcessId pid;
  {
    SystemCode r;
    ContextReply *reply = (ContextReply *) request;
    DirectoryEntry *contextptr;

    /* Lookup context and forward if more to name */
    if ( (r=MapName(request, pid, &contextptr)) != OK ) return (r);

    /* Make sure what we found was a context */
    if ((contextptr->c.descriptortype) != LOCALNAME_DESCRIPTOR) 
	return (ILLEGAL_REQUEST);

    /* Insert information in the reply message */
    if (contextptr->c.entrytype & LOGICAL_PID)
	reply->serverpid = GetPid(contextptr->c.serverpid, ANY_PID);
    else
        reply->serverpid = contextptr->c.serverpid;

    reply->contextid = contextptr->c.contextid;
    reply->instanceid = contextptr->c.instanceid;
    reply->otherinfo = contextptr->c.otherinfo;
    reply->entrytype = contextptr->c.entrytype;

    return (OK);
  }


static SystemCode NReadDescriptor(request, pid)
    DescriptorRequest *request;
    ProcessId pid;
  {
    SystemCode r;
    DirectoryEntry *entry;
    ProcessId serverpid;

    /* Lookup next component and forward if more to name */
    if ( (r=MapName(request, pid, &entry)) != OK ) return (r);

    if ((entry->c.descriptortype) == LOCALNAME_DESCRIPTOR)
      {
	if (entry->c.entrytype & LOGICAL_PID)
	    serverpid = GetPid(entry->c.serverpid, ANY_PID);
	else
	    serverpid = entry->c.serverpid;

	/* If not a context implemented by this server then forward. */
	if (serverpid != LocalNameServerPid)
	  {
	    request->namecontextid = entry->c.contextid;
	    Forward(request, pid, serverpid);
	    return (NO_REPLY);
	  }
      }

    /* Perform an NReadDescriptor on a device or locally-implemented
     *   directory */

    if (request->segmentlen - request->dataindex < sizeof(DirectoryEntry))
	return (BAD_BUFFER);

    return(MoveTo(pid, request->segmentptr + request->dataindex,
		entry, sizeof(DirectoryEntry)));

  }


static SystemCode ReadDescriptor(request, pid)
    DescriptorRequest *request;
    ProcessId pid;
  {
    DirectoryEntry *descriptor;

    /* The only locally implemented files are the context directory
     *   and the device directory */

    if (request->fileid == CONTEXT_DIRECTORY_INSTANCE)
	descriptor = ContextDirDescriptor;
    else if (request->fileid == DEVICE_DIRECTORY_INSTANCE)
	descriptor = DeviceDirDescriptor;
    else
	return (INVALID_FILE_ID);

    if (request->segmentlen - request->dataindex < sizeof(DirectoryEntry))
	return (BAD_BUFFER);

    return(MoveTo(pid, request->segmentptr + request->dataindex,
		descriptor, sizeof(DirectoryEntry)));

  }
   

static SystemCode MapName(request, pid, entry)
    NameRequest *request;
    ProcessId pid;
    DirectoryEntry **entry;
  /* 
   * Map to a directory entry, or forward the request to
   *   the associated server.
   */
  {
    SystemCode r;
    char *field;
    DirectoryEntry *directory;
    unsigned dirsize;
    ProcessId serverpid;

    /* Parse pathname */
    r = MoveInString(request, pid);
    if (r != OK) return (r);

    /* Handle '[' escape to root context */
    if (NameBuffer[request->nameindex] == '[')
      {
	request->namecontextid = CONTEXT_DIRECTORY;
	request->nameindex++;
      }

    /* Handle case of client trying to map a name in ANY_CONTEXT leniently */
    if (request->namecontextid == ANY_CONTEXT)
	request->namecontextid = CONTEXT_DIRECTORY;

    if ( (r=ScanField(request, pid, &field)) != OK )
	return (r);
	
    LastComponent = field;	

    if (request->namecontextid == CONTEXT_DIRECTORY)
      {
	directory = ContextDirectory;
	dirsize = CONTEXT_DIRECTORY_SIZE;
      }
    else if (request->namecontextid == DEVICE_DIRECTORY)
      {
	directory = DeviceDirectory;
	dirsize = DEVICE_DIRECTORY_SIZE;
      }
    else 
	return (NOT_FOUND);

    /* A null field means "the current context" */
    if (null(field))
      {
	*entry = directory;  /* the first entry points to itself */
	return (OK);
      }

    *entry = FindEntry(field, directory, dirsize);
    if (*entry == NULL) return (NOT_FOUND);

    /* Forward the request if there is
     *  more to the pathname than the context */
    if (NameBuffer[request->nameindex] != '\0')
      {
	if (((*entry)->c.descriptortype) != LOCALNAME_DESCRIPTOR) 
	    return (NOT_FOUND);

	request->namecontextid = (*entry)->c.contextid;
	if ((*entry)->c.entrytype & LOGICAL_PID)
	    serverpid = GetPid((*entry)->c.serverpid, ANY_PID);
	else
	    serverpid = (*entry)->c.serverpid;

	if (Forward(request, pid, serverpid) == 0)
	    return (NOT_FOUND); 	/* Name server has died */
	else
            return (NO_REPLY);
      }

    return (OK);
  }


static SystemCode DefaultRequest(request, pid)
    NameRequest *request;
    ProcessId pid;
  {
    SystemCode r;
    DirectoryEntry *contextptr;

    /* If the request is not forwarded, it is intended for this server,
     *  but the request code was not recognized.
     */
    if ( (r=MapName(request, pid, &contextptr)) == OK )
	r = ILLEGAL_REQUEST;

    return (r);
  }


static DirectoryEntry *FindEntry(name, directory, size)
    char *name;
    DirectoryEntry directory[];
    int size;
  {
    /*
     * Find the given name in the given directory
     */
    SystemCode r;
    int index;

    for (index = 0; index < size; index++)
      {
	if ( directory[index].c.descriptortype != EMPTY_DESCRIPTOR &&
	     strcmp(name, directory[index].c.name) == 0 )
		return (&directory[index]);
      }
    return (NULL);
  }


static SystemCode MoveInString(request, pid)
    NameRequest *request;
    ProcessId pid;
  {
    register SystemCode r;

    /* Move in string argument */
    if (request->namelength > MAX_PATH_LENGTH) return (BAD_ARGS);
    r = MoveFrom(pid, NameBuffer, request->nameptr, request->namelength);
    if (r != OK) return (r);
    NameBuffer[request->namelength] = '\0';  /* Null-terminate name */

    return (OK);
  }


static SystemCode ScanField(request, client, field)
    NameRequest *request;
    ProcessId client;
    char **field;
  {
    /* Scan the next field in a pathname and advance nameindex */
    register char *p;
    int guard;

    p = &NameBuffer[request->nameindex];
    *field = p;
    for (guard = 0; guard < MAX_NAME_LENGTH; guard++)
      {
	if (*p == ']' || *p == '\0') break;
	p++;
      }
    if (*p != '\0') *p++ = '\0';
    request->nameindex = p - NameBuffer;
    if (guard >= MAX_NAME_LENGTH) return (BAD_ARGS);

    return (OK);
  }
