/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer,  5/83
 */


/*
 * Higher-level support routines for the TCP part of the
 * network server.
 */


#include "net.h"
#include "iptcp.h"


/*
 * Imported routines
 */

extern PktBuf AllocBuf();
extern PktBuf DeQueue();
extern Boolean In();
extern SystemCode SendIp();

/*
 * Forward function declarations
 */

SystemCode MoveTcpSegment();
SystemCode TryToSendToIp();




/*
 * SelectIss:
 * Selects an initial sequence number for a new connection.
 */

Bit32 SelectIss()
  {
    int     clicks;		/* Number of time interrupts since last
				   second. */
    Bit32 iss;			/* Initial sequence number. */

    iss = (Bit32) GetTime(&clicks);
    return (iss + (Bit32) clicks);
  }




/*
 * SendSynSegment:
 * Issue a Syn segment to IP.
 *
 * Currently no data is sent with this segment.
 */

SystemCode SendSynSegment(pTcb, iss, ackFlag, ack)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    Bit32 iss;			/* Initial sequence number. */
    Boolean ackFlag;		/* Inicates if should send an Ack. */
    Bit32 ack;			/* Ack number to send. */
  {
    SystemCode     statusCode;
    PktBuf packet;
    TcpPktPtr p;
    Bit16 *v;
    int i;

    packet = AllocBuf();
    if (packet == Null)
	return(NO_MEMORY);
    InitPacketBuffer(pTcb->localPort, pTcb->foreignSocket.port,
    	    StandardByteDataOffset + 8,
	    StandardByteDataOffset + 8, packet, iss, ack,
	    ackFlag, False, False, True, False, 
	    pTcb->rcvWnd - pTcb->delRcvWnd);

    p = TcpPkt(packet);
    p->data[0] = MaxSegSizeOption;
    p->data[1] = (Bit8) 4;
    v = (Bit16 *) &(p->data[2]);
    *v = (Bit16) MaxPacketDataLength;
    p->data[4] = EndOptionList;
    p->data[5] = (Bit8) 0;
    p->data[6] = (Bit8) 0;
    p->data[7] = (Bit8) 0;

    statusCode = SendIp(pTcb, packet, pTcb->foreignSocket.host);
    if (statusCode == OK)
      {
	pTcb->iss = iss;
	pTcb->sndUna = iss;
	pTcb->sndNxt = iss + 1;
	pTcb->state = SynSent;
	EnQueue(&(pTcb->q[RetransQueue]), packet);
	packet->unspecified[0] = 0;
				/* Init. retransmission number. */
	StartTimer(pTcb, RetransTimeout, LenRetransTimeout);
      }
    else
      {
	FreeBuf(packet);
      }
    return(statusCode);
  }




/*
 * QSend:
 * Either queues a segment in the send queue of the connection or queues
 * a null segment with the fin bit set.  Then tries to send packets
 * to the Ip level if possible.
 * NOTE: The current implementation always sets the psh bit due to the nature
 * of the V I/O protocol.
 */

SystemCode QSend(pTcb, movePid, fin, rqMsg)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    ProcessId movePid;		/* Process id of client whose data is
				   to be sent. */
    Boolean fin;		/* Indicates if fin bit should be set in
				   last packet. */
    IoRequest *rqMsg;		/* Client's write request msg. */
  {
    SystemCode statusCode;
    PktBuf packet;
    TcpPktPtr p;
    char *t, *f;
    int i, spaceLeft;

    if ((FreeBufferCount <= 1) && (TotalBufferCount >= MaxBufAllocNum - 1))
      {
	return(RETRY);
      }
    packet = AllocBuf();
    if (packet == NULL)
      {
	return(RETRY);
      }
    InitPacketBuffer(pTcb->localPort, pTcb->foreignSocket.port, 
    		StandardByteDataOffset,	StandardByteDataOffset, packet,
		pTcb->sndNxt, pTcb->rcvNxt,
		True, True, False, False, False, 
		pTcb->rcvWnd - pTcb->delRcvWnd);
    EnQueue(&(pTcb->q[SendQueue]), packet);

    if (fin)
      {
	p = TcpPkt(packet);
	p->hdr.fin = 1;
	pTcb->sndNxt += 1;
      }
    else
      {
	if (rqMsg->bytecount <= IO_MSG_BUFFER)
	  { 			/* NOTE: We are assuming that IO_MSG_BUFFER
				   is always smaller than
				   pTcb->maxPacketDataLength. */
	    t = (char *) (packet->dataptr + packet->length);
	    f = (char *) rqMsg->shortbuffer;
	    for (i = 0; i < rqMsg->bytecount; i++)
		*t++ = *f++;
            packet->length += rqMsg->bytecount;
	  }
	else if (rqMsg->bytecount <= pTcb->maxPacketDataLength)
	  {
	    statusCode = MoveFrom(movePid, (packet->dataptr + packet->length),
		    rqMsg->bufferptr, rqMsg->bytecount);
	    if (statusCode != OK)
	      {
		return(statusCode);
	      }
	    packet->length += rqMsg->bytecount;
	  }
	else			/* User sent too much data! */
	    return(BAD_ARGS);
        pTcb->sndNxt += rqMsg->bytecount;
      }
    if (pTcb->state >= Estab)
        statusCode = TryToSendToIp(pTcb);
    return(statusCode);
  }




/*
 * TryToSendToIp:
 * Tries to send packets from the specified queue to IP.
 *
 * NOTE: Assumes that the psh bit is always set.  If this assumption should
 * ever change, then Clark's Silly Window Syndrome avoidance scheme should
 * be incorporated here; namely only send a packet if the usable window
 * size is at least 25% of the offered window size.
 */

SystemCode TryToSendToIp(pTcb)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
  {
    SystemCode statusCode;
    PktBuf  packet, pkt;
    TcpPktPtr p;

    StopTimer(pTcb, SndWndTimeout);
				/* Stop the timer on the assumption that we
				   will have a large enough window to send a 
				   packet. */
    statusCode = OK;
    while (!Empty(pTcb->q[SendQueue]))
      {
        packet = pTcb->q[SendQueue].head;
        p = TcpPkt(packet);
        if (!In(pTcb->sndUna, p -> hdr.sequenceNumber +
    		DLength(packet), pTcb->sndUna + pTcb->sndWnd))
	  {			/* Send window is too small to send
				           packet. */
	    if (Empty(pTcb->q[RetransQueue]))
		StartTimer(pTcb, SndWndTimeout, LenSndWndTimeout);
				/* Start a timer to periodically query the
				   remote host to see if it's rcv window has
				   grown. */
	    break;
	  }
        if (pTcb->sndUp != NULL)
          {
	    p -> hdr.urgentPointer = pTcb->sndUp - p -> hdr.sequenceNumber;
	    p -> hdr.urg = True;
	    if (In(p -> hdr.sequenceNumber, pTcb->sndUp,
			p->hdr.sequenceNumber + DLength(packet)))
	      {
		pTcb->sndUp = NULL;
	      }
          }
        else
	    p -> hdr.urg = False;
        p -> hdr.acknowledgementNumber = pTcb->rcvNxt;
        statusCode = SendIp(pTcb, packet, pTcb->foreignSocket.host);
        if (statusCode == OK)
          {
	    pkt = DeQueue(&(pTcb->q[SendQueue]));
	    EnQueue(&(pTcb->q[RetransQueue]), pkt);
	    pkt->unspecified[0] = 0;
				/* Init. retransmission number. */
          }
        else
	    break;
      }
    StartTimer(pTcb, RetransTimeout, LenRetransTimeout);
    return(statusCode);
  }




/*
 * ServiceRcvRequest:
 * Reassemble queued incoming segments into user's receive buffer.
 * Set notification for urgent data if this is the case.
 * Remove processed incoming segments from Segments Received queue.
 * Reply to user's Read call.
 *
 * Note: packet->dataptr and packet->length now refer only to the
 * actual Tcp data to be sent to the user.  Adjustments for overlapping
 * sequence numbers, etc. have already been made.
 * Assumes that a Read call is outstanding and that bytes are
 * available for receiving.
 */

ServiceRcvRequest(pTcb)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
  {
    register int len;
    Message msg;
    IoReply *repMsg = (IoReply *) msg;
    SystemCode sCode;			/* Return code from MoveTo. */

	len = Min(pTcb->rcvByteCnt, pTcb->q[SegQueue].head->length);
	if (pTcb->rcvUrgFlag)
	    sCode = UrgentData;
	else
	  {
	    switch (pTcb->state)
	      {
	        case Closed:
	        case LastAck:
	        case Closing:
		case TimeWait:
		    sCode = END_OF_FILE;
		    break;
		default:
	            sCode = OK;
		    break;
	      }
	  }
	ReplyToRead(sCode, pTcb->rcvId, pTcb->q[SegQueue].head, 
		pTcb->rcvQ, len);
	pTcb->rcvQ = NULL;

	len = pTcb->q[SegQueue].head->length;
	if (pTcb->rcvUp <= (pTcb->rcvNxt - (Bit32)pTcb->segQBytesAvail))
	    pTcb->rcvUrgFlag = False;

	/* Keep around the last block read for handling duplicate reads. */
	if (pTcb->lastReadBuf != NULL)
	  {
	    FreeBuf(pTcb->lastReadBuf);
	  }
	pTcb->lastReadBuf = DeQueue(&(pTcb->q[SegQueue]));
	NetInstTable[pTcb->netId].filenextblock++;

        pTcb->segQBytesAvail -= len;
	pTcb->rcvWnd += len;
	pTcb->delRcvWnd += len;
	if (pTcb->delRcvWnd > RcvWindowCutoff)
	  {
	    pTcb->delRcvWnd = 0;
	    SendAckSegment(pTcb);
				/* Notify remote side of connection of the
				   new window size. */
	  }
  }




/*
 * QSegment:
 * Place incoming segment buffer on segment or save Queue, depending on its
 * relation to rcvNxt.  If the segment begins beyond rcvNxt then store it on
 * the save queue for later processing.
 * If the segment can be processed then:
 * Make dataptr and length of packet buffer refer only to actual
 * Tcp data for user.
 * Adjust for overlapping data by changing dataptr and length
 * of packet buffer.
 * Update Tcb state to reflect accepted segment.
 * If flag then honor a
 * Read call if one is outstanding and enough data is available.
 * Acknowledge data received back to remote side.
 */

QSegment(pTcb, curPkt, flag)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
    Boolean flag;		/* Indicates if should honor Receive
				   requests. */
  {
    int     diff;		/* Computation variable. */
    PktBuf     packet, pkt;
    Bit32 seqNum;

    if (curPkt->segLen == 0)
      {
	DiscardSegment(curPkt->segAdr, NULL);
	return;			/* Don't queue an empty segment. */
      }

    packet = curPkt->segAdr;
    if (!In((pTcb->rcvNxt - curPkt->segLen), 
	    TcpPkt(packet)->hdr.sequenceNumber, 
	    pTcb->rcvNxt))	/* Check if rcvNxt < segSeq */
      {
        pTcb->numOutOfOrderPkts++;
        if (InternetDebug)
	  {
	    printf("Tcp segment has arrived which starts beyond rcvNxt.\n");
	  }

	if ((FreeBufferCount < 1) && (TotalBufferCount >= MaxBufAllocNum))
	    DiscardSegment(curPkt->segAdr, "Not enough buffers.");	
				/* Can't afford the packet buffer. */

        /* The segment starts beyond rcvNxt.  Store it for later processing
	   on the save queue.  Store in sorted order of sequenceNumber. */
	packet->unspecified[0] = curPkt->segLen;
	packet->unspecified[1] = curPkt->segDataOffset;
	seqNum = TcpPkt(packet)->hdr.sequenceNumber;
	if ((Empty(pTcb->q[SaveQueue])) || 
		(In(TcpPkt(pTcb->q[SaveQueue].tail)->hdr.sequenceNumber,
			seqNum, (pTcb->rcvNxt + pTcb->rcvWnd))))
	  {
	    EnQueue(&(pTcb->q[SaveQueue]), packet);
	  }
	else if (In(pTcb->rcvNxt, seqNum,
		TcpPkt(pTcb->q[SaveQueue].head)->hdr.sequenceNumber))
	  {
	    /* Insert packet at head of queue. */
	    packet->next = pTcb->q[SaveQueue].head;
	    pTcb->q[SaveQueue].head = packet;
	  }
	else
	  {
	    /* Find where in the queue the packet belongs.  We already 
	       know that it isn't at the head or tail of it - so no special
	       cases must be handled. */
	    for (pkt = pTcb->q[SaveQueue].head; pkt != NULL; pkt = pkt->next)
	      {
	        if (In(TcpPkt(pkt)->hdr.sequenceNumber, seqNum,
			(pTcb->rcvNxt + pTcb->rcvWnd)))
		  {
		    packet->next = pkt->next;
		    pkt->next = packet;
		    break;
		  }
	      }
	  }
	return;
      }
    diff = (int) (pTcb->rcvNxt - TcpPkt(packet)->hdr.sequenceNumber);
				/* Size of prefix to discard. */
    packet->dataptr += curPkt->segDataOffset + diff;
    packet->length = curPkt->segLen - diff;
				/* The tcpLength and dataOffset fields
				   will be used as ptrs into the data
				   field and don't take into account the
				   header length anymore. */
    pTcb->segQBytesAvail +=  packet->length;
    pTcb->rcvWnd -= packet->length;
    pTcb->rcvNxt += packet->length;
    EnQueue(&(pTcb->q[SegQueue]), packet);
    StartTimer(pTcb, AckTimeout, LenAckTimeout);
				/* Clark's Silly Window Syndrome algorithm
				   would check the psh bit here to avoid
				   processing an extra timeout interrupt.
				   For the current implementation this
				   doesn't yield any benefit. */
    if (!Empty(pTcb->q[SaveQueue]))
	ProcessSavedSegments(pTcb);
				/* Check to see if this new segment allows
				   any of the saved segments to be processed.
				   */
    if (flag && (pTcb->rcvQ != Null))
        if (pTcb->segQBytesAvail > 0)
    	    ServiceRcvRequest(pTcb);
  }




/*
 * ProcessSavedSegments:
 * Processes any segments in the SaveQueue that can be relative
 * to the current value of rcvNxt.
 */

ProcessSavedSegments(pTcb)
    TcpTcbPtr pTcb;
  {
    PktBuf packet;
    int diff;

    if (InternetDebug)
      {
	printf("About to process saved Tcp segments.\n");
      }
    while (!Empty(pTcb->q[SaveQueue]))
      {
    	packet = pTcb->q[SaveQueue].head;
        if (!In((pTcb->rcvNxt - pTcb->rcvWnd),
		TcpPkt(packet)->hdr.sequenceNumber,
		pTcb->rcvNxt))
	  {
	    return;
	  }
	packet = DeQueue(&(pTcb->q[SaveQueue]));
	/* The following is the same basic code (no curPkt avail.) as in
	   QSegment. */
	diff = (int) (pTcb->rcvNxt - TcpPkt(packet)->hdr.sequenceNumber);
				/* Size of prefix to discard. */
	if (diff >= packet->unspecified[0])
				/* Note: unspecified[0] = curPkt->segLen */
	  {
	    FreeBuf(packet);	/* Already processed this data during
				   retransmission. */
	    continue;
	  }
	packet->dataptr += packet->unspecified[1] + diff;
				/* Note: unspecified[1] = 
				           curPkt->segDataOffset */
	packet->length = packet->unspecified[0] - diff;
	pTcb->segQBytesAvail += packet->length;
	pTcb->rcvWnd -= packet->length;
	pTcb->rcvNxt += packet->length;
	EnQueue(&(pTcb->q[SegQueue]), packet);
      }
  }




/*
 * RemoveAckedSendSegments:
 * Removes acked send segments from the retransmission queue.
 *
 * NOTE: only entire segments are removed from the queue.
 */

RemoveAckedSendSegments(pTcb)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
  {
    PktBuf packet, pkt;
    SystemCode statusCode;
    int del;

    for (packet = pTcb->q[RetransQueue].head; packet != Null;
	    packet = pTcb->q[RetransQueue].head)
      {
        if ((TcpPkt(packet)->hdr.syn) || (TcpPkt(packet)->hdr.fin))
	    del = 1;
	else
	    del = 0;
	if (In(pTcb->iss, (TcpPkt(packet))->hdr.sequenceNumber + 
		DLength(packet) + del, pTcb->sndUna))
	  {
	    pkt = DeQueue(&(pTcb->q[RetransQueue]));
	    FreeBuf(pkt);
	  }
	else
	    break;
      }

    if (pTcb->state >= Estab)
	statusCode = TryToSendToIp(pTcb);

    if (Empty(pTcb->q[RetransQueue]))
	StopTimer(pTcb, RetransTimeout);
    else
	StartTimer(pTcb, RetransTimeout, LenRetransTimeout);

    if ((statusCode != OK) && pTcb->waitSignal)
	NotifyUser(pTcb, statusCode);
  }




/*
 * NotifyUser:
 * if WaitSignal is outstanding then reply to it with statusCode.
 * Assumes that a WaitSignal is outstanding.
 */

NotifyUser(pTcb, statusCode)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    SystemCode statusCode;		/* Status code to return to user. */
  {
    Message msg;

    ((IoReply *)msg)->replycode = statusCode;
    pTcb->wtSignalId = Reply(msg, pTcb->wtSignalId);
    pTcb->waitSignal = False;
  }




/*
 * SendControlSegment:
 * Sends a Tcp control packet to IP.
 * Control packet here means no data is sent.
 * Uses spare buffer set to ensure that the packet can be sent.
 */

SendControlSegment(pTcb, sPort, fPort, fHost, seq, ackFlag, ack, rst, fin)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    Bit16 sPort;		/* source port. */
    Bit16 fPort;		/* destination port. */
    Bit32 fHost;		/* destination host. */
    Bit32 seq, ack;		/* Sequence and ack number to use. */
    Boolean ackFlag;		/* Indicates whether to send an ack. */
    Boolean rst;		/* Value of rst bit field in packet. */
    Boolean fin;		/* Value of fin bit field in packet. */
  {
    struct pbuf Spare;		/* Spare packet buffer used to guarantee that
				   a control segment can be sent. */
    PktBuf SparePacketBuffer = &Spare;
    SystemCode statusCode;

    InitPacketBuffer(sPort, fPort, 
    	    StandardByteDataOffset, StandardByteDataOffset,
	    SparePacketBuffer, seq, ack,
	    ackFlag, False, rst, False, fin, 
	    pTcb->rcvWnd - pTcb->delRcvWnd);

    statusCode = SendIp(pTcb, SparePacketBuffer, fHost);
				/* Status returned doesn't matter. */
  }




/*
 * SendAckSegment:
 * Send an Ack segment to IP.
 */

SendAckSegment(pTcb)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
  {
    SendControlSegment(pTcb, pTcb->localPort, pTcb->foreignSocket.port,
		pTcb->foreignSocket.host,
		pTcb->sndNxt, True, pTcb->rcvNxt, False, False);
  }
