/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer,  5/83
 */


/*
 * Contains top-level routines pertaining to the TCP part of the 
 * network server.
 */


#include "net.h"
#include "iptcp.h"


/*
 * Constant definitions
 */

#define DefaultLocalPort 100	/* Default local port identifier.  This plus
				   the netinstance index is used as the local
				   port id if none is specified by the
				   client. */


/*
 * Imported routines
 */

extern char *SafeMalloc();
extern PktBuf DeQueueSafe();
extern SystemCode QueryInstance();
extern Bit32 SelectIss();
extern SystemCode SendSynSegment();
extern SystemCode QSend();
extern Boolean NoAccess();
extern SystemCode VerifySecurityAndPrecedence();
extern Bit16 CheckSum();
extern TcpTcbPtr AllocateTcb();
extern SystemCode SendIp();
extern PktBuf DeQueue();




/*
 * InitTcp:
 * Global Tcp initialization routine.
 * Dummy routine in this implementation scheme.
 */

InitTcp()
  {
  }




/*
 * CreateTcpConnection:
 * Called from the main InternetServer process to create and initialize a
 * Tcp connection and its associated process. 
 */

SystemCode CreateTcpConnection(msg, eventPid)
    Message msg;
    ProcessId eventPid;
  {
    int TcpProcess();
    ProcessId tcpPid;
    int netId;		/* Index of a network instance. */
    TcpParms1 *ct = (TcpParms1 *)
		((CreateInstanceRequest *)msg)->unspecified;

    netId = FindWhichConnection(True, ct->localPort,
		ct->foreignPort,
		ct->foreignHost);
    if (netId != -1)
        if (ValidPid(NetInstTable[netId].ownerPid))
            return(BUSY);
	else
	    DeallocateTcb(NetInstTable[netId].tcbId, BAD_STATE);
    if (NoAccess())
	return(NO_PERMISSION);
    tcpPid = Create(2, TcpProcess, TcpStackSize);
    if (tcpPid == 0)
      {
	if (InternetDebug)
	  {
	    printf("Couldn't Create TcpProcess.\n");
	  }
        return(NO_MEMORY);
      }
    if (!Ready(tcpPid, 0))
      {
	printf("Couldn't Ready TcpProcess.\n");
	ExitInternetServer();
      }
    Forward(msg, eventPid, tcpPid);
    return(NO_REPLY);
  }



/*
 * CreateTcpInstance:
 * Processes user CreateInstance request.
 */

SystemCode CreateTcpInstance(pTcb, msg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    Message msg;
    ProcessId eventPid;
  {
    SystemCode returnCode;
    int netId;		/* Index of network instance. */
    TcpParms1 *ct = (TcpParms1 *)
		((CreateInstanceRequest *)msg)->unspecified;

    switch (pTcb->state)
      {
	case Closed: 
	      {
		netId = AllocNetInst(TCPtype, eventPid, GetPid(0, LOCAL_PID),
			MaxPacketDataLength, pTcb);
		if (netId == -1)
		  {
		    returnCode = NO_SERVER_RESOURCES;
		    break;
		  }

		pTcb->netId = netId;
		if (ct->localPort)
		    pTcb->localPort = ct->localPort;
		else
		    pTcb->localPort = DefaultLocalPort +
		    		(Bit16) (GetTime(0) % 1000);
		pTcb->foreignSocket.port = ct->foreignPort;
		pTcb->foreignSocket.host = ct->foreignHost;
		pTcb->originalForeignSocket = pTcb->foreignSocket;
		pTcb->active = ct->active;
		pTcb->prc = ct->precedence;
		pTcb->security = ct->security;
		pTcb->rcvId = eventPid;

		if ((returnCode = VerifySecurityAndPrecedence(pTcb)) != OK)
		  {
		    DeallocateTcb(pTcb, returnCode);
		    break;
		  }
		if (!pTcb->active)
		  {
		    pTcb->state = Listen;
		    pTcb->passiveOpen = True;
		    returnCode = OK;
		    break;
		  }
		if (pTcb->foreignSocket.port == Null16)
		  {
		    returnCode = BAD_ARGS;
		    DeallocateTcb(pTcb, BAD_ARGS);
		    break;
		  }
		returnCode = SendSynSegment(pTcb, SelectIss(), False, Null32);
		if (returnCode != OK)
		  {
		    DeallocateTcb(pTcb, returnCode);
		  }
		break;
	      }
	case Listen: 
	      {
		if (!ct->active)
		  {
		    returnCode = BAD_ARGS;
		    break;
		  }
		if (ct->foreignPort == Null16)
		  {
		    returnCode = BAD_ARGS;
		    break;
		  }

		pTcb->foreignSocket.port = ct->foreignPort;
		pTcb->foreignSocket.host = ct->foreignHost;
		pTcb->originalForeignSocket = pTcb->foreignSocket;
		pTcb->active = ct->active;
		pTcb->prc = ct->precedence;
		pTcb->security = ct->security;
		pTcb->rcvId = eventPid;
	        /* No options being treated as yet. */

		returnCode = SendSynSegment(pTcb, SelectIss(), False, Null32);
		break;
	      }
	default: 
	      {
		returnCode = BUSY;
		break;
	      }
      }
    if (returnCode == OK)
	returnCode = NO_REPLY;
    return(returnCode);
  }




/*
 * ReleaseTcpInstance:
 * Processes both the regular close and also the abort form of 
 * releasing a TCP connection.
 * Connection is "TCP" closed on Read instance close.  Write instance close
 * is a no-op.
 */

SystemCode ReleaseTcpInstance(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    SystemCode returnCode, CloseCall(), AbortCall();
    int indx = rqMsg->fileid >> 1;

    if (rqMsg->fileid & 1)
	NetInstTable[indx].inUse &= ~ READ_INST;
    else
	NetInstTable[indx].inUse &= ~ WRITE_INST;

    if ((pTcb->instState & TCP_CONN_RELEASED) || (!(rqMsg->fileid & 1)))
	return(OK);		/* Do nothing if
				   connection already released or write
				   instance being released. */
    pTcb->instState |= TCP_CONN_RELEASED;

    if (rqMsg->releasemode == REL_STANDARD)
	returnCode = CloseCall(eventPid, pTcb);
    else
	returnCode = AbortCall(pTcb, ABORTED);
    return(returnCode);
  }




/*
 * WriteTcpInstance:
 * Processes user Write request.
 */

SystemCode WriteTcpInstance(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    SystemCode returnCode;
    register int indx = rqMsg->fileid >> 1;

    if (rqMsg->fileid & 1)
      {
	return(NOT_WRITEABLE);
      }
    if (rqMsg->bytecount > NetInstTable[indx].blocksize)
      {
	return(BAD_BYTE_COUNT);
      }
    if (rqMsg->blocknumber == (NetInstTable[indx].filelastblock-1))
				/* This is a duplicate message - ignore it. */
      {
	return(OK);
      }
    if (rqMsg->blocknumber != (NetInstTable[indx].filelastblock))
      {
	return(BAD_BLOCK_NO);
      }

    NetInstTable[indx].filelastblock++;
    NetInstTable[indx].filelastbytes = rqMsg->bytecount;

    switch (pTcb->state)
      {
	case Listen: 
	      {
		if (pTcb->foreignSocket.port == Null16)
		  {
		    returnCode = BAD_ARGS;
		    break;
		  }
		pTcb->active = True;
		returnCode = SendSynSegment(pTcb, SelectIss(), False, Null32);
		if (returnCode == OK)
		  {
		    if (pTcb->sndUrgBit)
		      {
			pTcb->sndUp = pTcb->sndNxt + rqMsg->bytecount;
		      }
		    returnCode = QSend(pTcb, eventPid, False, rqMsg);
		  }
		break;
	      }
	case SynSent: 
	case SynRcvd: 
	case Estab: 
	case CloseWait: 
	      {
		if (pTcb->sndUrgBit)
		  {
		    pTcb->sndUp = pTcb->sndNxt + rqMsg->bytecount;
		  }
		returnCode = QSend(pTcb, eventPid, False, rqMsg);
		break;
	      }
	default: 
	      {
		returnCode = BAD_STATE;
		break;
	      }
      }
    return(returnCode);
  }




/*
 * ReadTcpInstance:
 * Processes user Read request.
 */

SystemCode ReadTcpInstance(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    register int indx = rqMsg->fileid >> 1;

    if (!(rqMsg->fileid & 1))
      {
	return(NOT_READABLE);
      }
    if (rqMsg->bytecount > NetInstTable[indx].blocksize)
      {
	return(BAD_BYTE_COUNT);
      }
    if ((rqMsg->blocknumber == (NetInstTable[indx].filenextblock - 1)) &&
    	(pTcb->lastReadBuf != NULL))
      {
	ReplyToRead(OK, eventPid, pTcb->lastReadBuf, rqMsg->bufferptr,
		Min(rqMsg->bytecount, pTcb->lastReadBuf->length));
	return(NO_REPLY);
      }
    if (rqMsg->blocknumber != NetInstTable[indx].filenextblock)
      {
	return(BAD_BLOCK_NO);
      }

    pTcb->rcvId = eventPid;
    pTcb->rcvQ = rqMsg->bufferptr;
    pTcb->rcvByteCnt = rqMsg->bytecount;

    switch (pTcb->state)
      {
	case Listen: 
	case SynSent: 
	case SynRcvd: 
	    break;		/* Read only handled when state is
				   Estab or further; so wait until then. 
				*/
	case Estab: 
	case FinWait1: 
	case FinWait2: 
	      {
		if (pTcb->segQBytesAvail > 0)
		    ServiceRcvRequest(pTcb);
		break;
	      }
	case CloseWait: 
	      {
		if (pTcb->segQBytesAvail == 0)
		  {
		    ReplyToRead(END_OF_FILE, pTcb->rcvId, NULL, NULL, 0);
		    pTcb->rcvQ = NULL;
		  }
		else
		    ServiceRcvRequest(pTcb);
		break;
	      }
	case Closed:
	      {
		ReplyToRead(END_OF_FILE, pTcb->rcvId, NULL, NULL, 0);
		pTcb->rcvQ = NULL;
		break;
	      }
	default: 
	      {
		ReplyToRead(BAD_STATE, pTcb->rcvId, NULL, NULL, 0);
		pTcb->rcvQ = NULL;
		break;
	      }
      }
    return(NO_REPLY);
  }




/*
 * QueryTcpInstance:
 * QueryInstance for a Tcp connection.  Does a network query instance.
 */

SystemCode QueryTcpInstance(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;
    QueryInstanceRequest *rqMsg;
    ProcessId eventPid;
  {
    SystemCode replyCode;

    replyCode = QueryInstance(rqMsg);
    return(replyCode);
  }




/*
 * QueryTcpFile:
 * Returns state information about the specified TCP connection.
 */

SystemCode QueryTcpFile(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    QueryFileRequest *rqMsg;
    ProcessId eventPid;
  {
    TcpParms1 *qt1 = (TcpParms1 *) rqMsg->unspecified;
    TcpParms2 *qt2 = (TcpParms2 *) rqMsg->unspecified;

    switch (rqMsg->whichParms)
      {
	case 0:
	    qt1->localPort = pTcb->localPort;
	    qt1->foreignPort = pTcb->foreignSocket.port;
	    qt1->foreignHost = pTcb->foreignSocket.host;
	    qt1->active = pTcb->active;
	    qt1->precedence = pTcb->prc;
	    qt1->security = pTcb->security;
	    return(OK);
	case 1:
	    if (pTcb->rcvUrgFlag)
	        qt2->rcvUrgFlag = True;
	    else
	        qt2->rcvUrgFlag = False;
	    qt2->sndUrgFlag = pTcb->sndUrgBit;
	    qt2->bytesAvail = pTcb->segQBytesAvail;
	    qt2->state = pTcb->state;
	    return(OK);
	case 2:
	    pTcb->wtSignalId = eventPid;
	    pTcb->waitSignal = True;
	    return(NO_REPLY);
	default:
	    return(BAD_ARGS);
      }
  }




/*
 * ModifyTcpFile:
 * Processes requests to modify or query the state parameters of
 * the designated TCP connection.
 */

SystemCode ModifyTcpFile(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    ModifyFileRequest *rqMsg;
    ProcessId eventPid;
  {
    TcpParms2 *qt2 = (TcpParms2 *) rqMsg->unspecified;
    switch (rqMsg->whichParm)
      {
	case ModTcpUrgFlag:
	    pTcb->sndUrgBit = ! pTcb->sndUrgBit;
	    break;
	default:
	    return(BAD_ARGS);
      }
    return(OK);
  }




/*
 * RcvTcp:
 * Processes incoming TCP segments handed up from the IP level.
 */

RcvTcp(pTcb, packet)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    PktBuf packet;
  {
    Bit32 ipsrc, ipdst;
    CurPktRec CurrentPacket;	/* Holds information about current 
				   incoming TCP segment. */
    CurPktPtr curPkt = &CurrentPacket;
    TcpPktPtr p;
    int     i;

    ipsrc = (Bit32) packet->unspecified[0];
    ipdst = (Bit32) packet->unspecified[1];
    p = TcpPkt(packet);
    /*
     * Check for bad checksums and too long packets.
     */
    if ((p->hdr.checkSum != CheckSum(p, packet->length, ipsrc,
	    ipdst, (Bit16) TCPprot)) || 
	    (packet->length > (MaxPacketDataLength + StandardByteDataOffset)))
      {
	if (InternetDebug)
	  {
	    printf("**** BadCheckSum or Packet too long ****\n");
	  }
	FreeBuf(packet);
	return;
      }

    curPkt->foreignPort = p->hdr.sourcePort;
    curPkt->foreignHost = ipsrc;
    curPkt->segAdr = packet;
    curPkt->segLen = packet->length - (int) (TcpPkt(curPkt->segAdr) -> 
						hdr.dataOffset * 4);
    curPkt->segPrc = 0;			/* Currently not set. */
    curPkt->segSecurity = 0;		/* Currently not set. */
    curPkt->segDataOffset = TcpPkt(curPkt->segAdr) -> hdr.dataOffset * 4;
    SegmentArrives(pTcb, curPkt);
  }




/*
 * NextTcpTimeout:
 * Returns the time of the next timeout for the specified connection.
 */

int NextTcpTimeout(pTcb)
    TcpTcbPtr pTcb;
  {
    int i, min;

    min = pTcb->timers[0];
    for (i = 1; i < NumTcpTimeouts; i++)
	if (min > pTcb->timers[i])
	    min = pTcb->timers[i];
    return(min);
  }




/*
 * TcpTimeout:
 * Handles TCP timeout events.
 */

TcpTimeout(pTcb, msg, eventPid)
    TcpTcbPtr     pTcb;
    Message msg;
    ProcessId eventPid;
  {
    PktBuf packet;
    TcpPktPtr p;
    IoRequest *rqMsg = (IoRequest *) msg;

    if (pTcb->instState & TCP_CONN_CLOSED)
				/* Connection already closed. */
        return;

    if (msg[1] == InvalidConnOwner)
      {
        rqMsg->fileid = pTcb->netId << 1;
	rqMsg->releasemode = REL_ABORT;
	ReleaseTcpInstance(pTcb, rqMsg, eventPid);
	rqMsg->fileid += 1;
	ReleaseTcpInstance(pTcb, rqMsg, eventPid);
	return;
      }
    else
      {
	if (pTcb->timers[AckTimeout] <= CurrentTime) 
	      {
		SendControlSegment(pTcb, pTcb->localPort,
				pTcb->foreignSocket.port,
				pTcb->foreignSocket.host, pTcb->sndNxt,
				True, pTcb->rcvNxt, False, False);
		StopTimer(pTcb, AckTimeout);
	      }
	if ((pTcb->timers[RetransTimeout] <= CurrentTime) &&
	    (!(Empty(pTcb->q[RetransQueue]))))
	      {
	        pTcb->numRetransTimeouts++;
		if (InternetDebug)
		  {
		    printf("XXXXXXXXXX Retransmission XXXXXXXXXX\n");
		  }
		packet = pTcb->q[RetransQueue].head;
		packet->unspecified[0] += 1;
		if (packet->unspecified[0] > MaxRetransTimeouts)
		  {
		    DeallocateTcb(pTcb, TIMEOUT);
		    return;	/* Return to avoid processing other timeouts
				   after connection is already closed. */
		  }
		else
		  {
		    TcpPkt(packet)->hdr.acknowledgementNumber = pTcb->rcvNxt;
		    SendIp(pTcb, packet, pTcb->foreignSocket.host);
		    StartTimer(pTcb, RetransTimeout, LenRetransTimeout);
		  }
	      }
	if (pTcb->timers[TmeWaitTimeout] <= CurrentTime) 
	      {
		DeallocateTcb(pTcb, TIMEOUT);
		return;		/* Return immediately so that if any timeout
				   processing is ever done after this one it
				   will not be done on a closed connection. */
	      }
	if (pTcb->timers[SndWndTimeout] <= CurrentTime)
	  {
	    if (InternetDebug)
	      {
		printf(" --- SndWndTimeout occurred ---\n");
	      }
	    if (!Empty(pTcb->q[SendQueue]))
	      {
		packet = pTcb->q[SendQueue].head;
		p = TcpPkt(packet);
		if (pTcb->sndUp != NULL)
		  {
		    p->hdr.urgentPointer = pTcb->sndUp - 
		    			p->hdr.sequenceNumber;
		    p->hdr.urg = True;
		    if (In(p->hdr.sequenceNumber, pTcb->sndUp,
		    		p->hdr.sequenceNumber + DLength(packet)))
		      {
		        pTcb->sndUp = NULL;
		      }
		  }
		else
		    p->hdr.urg = False;
		p->hdr.acknowledgementNumber = pTcb->rcvNxt;
		SendIp(pTcb, packet, pTcb->foreignSocket.host);
	      }
	    StartTimer(pTcb, SndWndTimeout, LenSndWndTimeout);
	  }
      }
  }




/*
 * CloseCall:
 * Processes normal closing of a TCP connection.
 */

SystemCode CloseCall(eventPid, pTcb)
    ProcessId eventPid;
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
  {
    SystemCode returnCode;

    switch (pTcb->state)
      {
	case Listen: 
	case SynSent: 
	      {
		DeallocateTcb(pTcb, ABORTED);
		returnCode = OK;
		break;
	      }
	case SynRcvd: 
	      {
		if ((Empty(pTcb->q[SendQueue])) &&
			(Empty(pTcb->q[SegQueue])) &&
			(Empty(pTcb->q[RetransQueue])))
				/* No sends issued and no pending data. */
		    pTcb->state = FinWait1;
		returnCode = QSend(pTcb, eventPid, True, NULL);
				/* Won't actually get sent until state
				   Estab. */
		break;
	      }
	case Estab: 
	      {
		pTcb->state = FinWait1;
		returnCode = QSend(pTcb, eventPid, True, NULL);
		break;
	      }
	case FinWait1: 
	case FinWait2: 
	      {
		returnCode = OK;
		break;
	      }
	case CloseWait: 
	      {
		pTcb->state = LastAck;
		returnCode = QSend(pTcb, eventPid, True, NULL);
		break;
	      }
	default: 
	      {
		returnCode = BAD_STATE;
		break;
	      }
      }
    return(returnCode);
  }




/*
 * AbortCall:
 * Processes abort form of closing a TCP connection.
 */

SystemCode AbortCall(pTcb, status)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    SystemCode status;
  {
    switch (pTcb->state)
      {
	case SynRcvd: 
	case Estab: 
	case FinWait1: 
	case FinWait2: 
	case CloseWait: 
	      {
		SendControlSegment(pTcb, pTcb->localPort,
			pTcb->foreignSocket.port,
			pTcb->foreignSocket.host,
			pTcb->sndNxt, False, Null32, True, False);
		DeallocateTcb(pTcb, status);
		break;
	      }
	default: 
	      {
		DeallocateTcb(pTcb, status);
		break;
	      }
      }
    return(OK);
  }




/*
 * TcpProcess:
 * Tcp connection process.  Handles all actions for this connection after
 * its initial creation.
 */

TcpProcess()
  {
    struct TcbType tcb;
    TcpTcbPtr pTcb = &tcb;
    Message  msg;
    IoRequest *rqMsg = (IoRequest *)msg;
    IoReply *repMsg = (IoReply *)msg;
    ProcessId eventPid;
    PktBuf packet;

    InitTcpConnection(pTcb);
    while (True)
      {
        while ((packet = DeQueueSafe(&(pTcb->readerQueue))) != NULL)
	  {
	    if (!(pTcb->instState & TCP_CONN_CLOSED))
				/* Handle packets only if connection is not
				   yet closed. */
		RcvTcp(pTcb, packet);
	    else
		FreeBuf(packet);
	  }

	if ((pTcb->instState & TCP_CONN_CLOSED) &&
		(NetInstTable[pTcb->netId].inUse == 0))
	  {
	    DeallocNetInst(pTcb->netId);
	    Destroy(0);		/* Commit suicide. */
	  }

	pTcb->receiveBlocked = True;
	eventPid = Receive(msg);
	pTcb->receiveBlocked = False;
	    switch (rqMsg->requestcode)
	      {
	        case CREATE_INSTANCE: 
				/* This is either the initial request
				   forwarded to the newly created connection
				   process or a request to change the 
				   connection state from passive to active. */
		    repMsg->replycode = 
			CreateTcpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case WRITE_INSTANCE:
	        case WRITESHORT_INSTANCE:
		    repMsg->replycode = 
			WriteTcpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case READ_INSTANCE:
		    repMsg->replycode = 
			ReadTcpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case RELEASE_INSTANCE:
		    repMsg->replycode = 
			ReleaseTcpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case QUERY_FILE:
		    repMsg->replycode = 
			QueryTcpFile(pTcb, rqMsg, eventPid);
		    break;
	        case MODIFY_FILE:
		    repMsg->replycode = 
			ModifyTcpFile(pTcb, rqMsg, eventPid);
		    break;
	        case QUERY_INSTANCE:
		    repMsg->replycode = 
		    	QueryTcpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case NetPktRcvd:
		    repMsg->replycode = OK;
		    break;
	        case NetTimeout:
		    TcpTimeout(pTcb, rqMsg, eventPid);
		    repMsg->replycode = OK;
		    break;
		default:
		    repMsg->replycode = REQUEST_NOT_SUPPORTED;
		    break;
	      }
	if (repMsg->replycode != NO_REPLY)
	  {
	    eventPid = Reply(msg, eventPid);
	  }
      }
  }




/*
 * PrintTcpConnectionState:
 * Diagnostic routine that prints out information regarding the current
 * state of the specified Tcp connection.
 */

PrintTcpConnectionState(pTcb)
    TcpTcbPtr pTcb;
  {
    int i, n;
    PktBuf pkt;

    printf("Net instance descriptor index: %d\n", pTcb->netId);
    printf("Connection state: %d     Connection release state: %d\n", 
    	    pTcb->state, pTcb->instState);
    if (pTcb->receiveBlocked)
        printf("Connection is receiveBlocked.\n");
    if (pTcb->readerQueue.head->next->pkt)
        printf("Non-empty readerQueue.\n");
    if (pTcb->rcvId)
        printf("Process %x is waiting on a Read.\n", pTcb->rcvId);
    printf("Number of bytes avail. for user read: %d\n", 
    	    pTcb->segQBytesAvail);
    if (!Empty(pTcb->q[SendQueue]))
      {
        n = 0;
	for (pkt = pTcb->q[SendQueue].head; pkt != NULL; pkt = pkt->next)
	    n++;
	printf("%d packets in the SendQueue.\n", n);
      }
    if (!Empty(pTcb->q[RetransQueue]))
      {
        n = 0;
	for (pkt = pTcb->q[RetransQueue].head; pkt != NULL; pkt = pkt->next)
	    n++;
	printf("%d packets in the RetransQueue.\n", n);
      }
    if (!Empty(pTcb->q[SegQueue]))
      {
        n = 0;
	for (pkt = pTcb->q[SegQueue].head; pkt != NULL; pkt = pkt->next)
	    n++;
	printf("%d packets in the SegQueue.\n", n);
      }
    if (!Empty(pTcb->q[SaveQueue]))
      {
        n = 0;
	for (pkt = pTcb->q[SaveQueue].head; pkt != NULL; pkt = pkt->next)
	    n++;
	printf("%d packets in the SaveQueue.\n", n);
      }
    printf("CurrentTime: %d     Timeouts: ", CurrentTime);
    for (i = 0; i < NumTcpTimeouts; i++)
      {
	printf("%d  ", pTcb->timers[i]);
      }
    printf("\n");
    printf("iss: %x    sndUna: %x    sndNxt: %x    sndWnd: %x\n",
    	    pTcb->iss, pTcb->sndUna, pTcb->sndNxt, pTcb->sndWnd);
    printf("irs: %x    rcvNxt: %x    rcvWnd: %x    delRcvWnd: %x\n",
    	    pTcb->irs, pTcb->rcvNxt, pTcb->rcvWnd, pTcb->delRcvWnd);
    printf("\n");
    printf("Number of retransmissions: %d\n", pTcb->numRetransTimeouts);
    printf("Number of out of order packets: %d\n", pTcb->numOutOfOrderPkts);
  }
