/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer,  5/83
 */


/*
 * Routines for the protocol-independent part of the internet server.
 */


#include "net.h"


/*
 * Variable declarations
 */

PktBuf PacketFreeList;
				/* Packet buffer free list header. */
int NumAllocCalls, NumDeallocCalls;
int AllocSem = 0, AllocSemCount = 0;
				/* Semaphor used for buffer allocation calls.
				   */


/*
 * Imported routines and tables
 */

extern char *malloc();
extern InitPnet();
extern RcvPnet();
extern InitIp();
extern RcvIp();
extern InitTcp();
extern IpReassemblyTimeout();

extern struct FuncBlock FuncTable[];
extern int NetLevelProtocol[];

/*
 * Forward function declarations
 */

Boolean InvalidFileid();
SystemCode QueryInstance();
Boolean AddBuf();
PktBuf AllocBuf();
PktBuf DeQueueSafe();




/*
 * InternetServer:
 * Main routine of the internet server.
 * Accepts CreateInstance requests for nonexistent connection instances
 * and QueryInstances.  CreateInstance results in the creation of a separate
 * connection process that will handle all further interactions for that
 * connection.  QueryInstance should normally be handled by the individual
 * connection processes; however the request is also handled here so that
 * clients can "reconnect" to a connection.
 */

InternetServer()
  {
    Message  msg;
    IoRequest *rqMsg = (IoRequest *)msg;
    IoReply *repMsg = (IoReply *)msg;
    ProcessId eventPid;
    PktBuf packet;

    while (True)
      {
	eventPid = Receive(msg);
	    switch (rqMsg->requestcode)
	      {
	        case CREATE_INSTANCE: 
		    repMsg->replycode = 
		        FuncTable[((CreateInstanceRequest *)msg)->type].
			CreateConnection(rqMsg, eventPid);
		    break;
	        case QUERY_INSTANCE:
		    repMsg->replycode = QueryInstance(rqMsg);
		    break;
	        case WRITE_INSTANCE:
	        case WRITESHORT_INSTANCE:
	        case READ_INSTANCE:
	        case RELEASE_INSTANCE:
	        case QUERY_FILE:
	        case MODIFY_FILE:
	        case NetPktRcvd:
	        case NetTimeout:
		    repMsg->replycode = MODE_NOT_SUPPORTED;
		    break;
		default:
		    repMsg->replycode = REQUEST_NOT_SUPPORTED;
		    break;
	      }
	if (repMsg->replycode != NO_REPLY)
	  {
	    eventPid = Reply(msg, eventPid);
	  }
      }
  }




/*
 * InitInternetServer:
 * Initialize the InternetServer.
 * Initializes global data structures and starts the helper processes
 * running.
 */

InitInternetServer(qFlag, localFlag, debugFlag)
    int qFlag;			/* Set up query process for runtime
    				   diagnostics if qFlag is true. */
    int localFlag;		/* True if internetserver should be local. */
    int debugFlag;		/* True if debug output should be printed. */
  {
    int     QueryProcess(), TimeoutTimer(), CleanupTimer(), RcvPnet();
    Process_id InternetServerPid;
				/* Process id of internet server process. */
    Process_id queryId, timer1Id, readerId;
    int i;

    InternetDebug = debugFlag;	/* Debugging flag. */

    /*
     * Register the server.
     */
    InternetServerPid = GetPid(0, LOCAL_PID);
    if (localFlag)
        SetPid(INTERNET_SERVER, InternetServerPid, LOCAL_PID);
    else
        SetPid(INTERNET_SERVER, InternetServerPid, ANY_PID);
    /*
     * Initialize the internet server data structures.
     */
    for (i = 0; i < MAX_NET_INST; i++)
      {
	NetInstTable[i].inUse = 0;
	NetInstTable[i].next = i + 1;
      }
    NetInstTable[MAX_NET_INST - 1].next = -1;
    NetInstFree = 0;
    NetInstHead = -1;
    /* 
     * Initialize the various modules.
     */
    InitParms();
    InitBufMgr();
    InitPnet();
    InitIp();
    InitTcp();
    /*
     * Initialize helper processes.
     */
    if (qFlag)			/* Set up a query process for runtime
				   diagnostics. */
      {
        queryId = Create(2, QueryProcess, 1500);	/* 3000 works */
	if (queryId == 0)
	  {
	    printf("Error in creation of an internet helper process\n");
	    ExitInternetServer();
	  }
	if (!Ready(queryId, 0))
	  {
	    printf("Error in readying of an internet helper process\n");
	    ExitInternetServer();
	  }
      }
    TimerPid = Create(1, TimeoutTimer, 500);	/* 1000 works */
    timer1Id = Create(3, CleanupTimer, 500);	/* 1000 works */
    readerId = Create(0, RcvPnet, 2500);	/* 3000 works */
    if ((TimerPid == 0) || (timer1Id == 0) || (readerId == 0))
      {
	printf("Error in creation of an internet helper process\n");
	ExitInternetServer();
      }
    if (!Ready(TimerPid, 0) || !Ready(timer1Id, 0) || !Ready(readerId, 0))
      {
	printf("Error in readying of an internet helper process\n");
	ExitInternetServer();
      }

    InternetServer();
  }




/*** GENERAL SUPPORT ROUTINES ***/




/*
 * QueryProcess:
 * Process which allows inspection of the internal state of the 
 * Internet server.
 */

QueryProcess()
  {
    char c;
    int a, i;
    PktBuf pkt;

    Resynch(stdin);
    while (1)
      {
        printf("\nEnter command:  "); 
        c = getc(stdin);
	while (getc(stdin) != '\n')
	    ;
        switch (c)
          {
            case 'd':
	        if (InternetDebug)
		    InternetDebug = 0;
		else
		    InternetDebug = 1;
	        printf("InternetDebug set to %d\n", InternetDebug);
	        break;
	    case 'n':
	        printf("List of active network instances:\n");
	        for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
		  {
		    printf("\nnetId: %d,  protocol: %d, blocksize: %d\n", 
				i, NetInstTable[i].prot, 
				NetInstTable[i].blocksize);
		    printf("inUse: %d,  Connection handler process: %x\n",
				NetInstTable[i].inUse, NetInstTable[i].pid);
		  }
		break;
	    case 'p':
	        printf("    Enter connection instance index:  ");
		i = 0;
		for (c = getc(stdin); c != '\n'; c = getc(stdin))
		  {
		    i = i * 10 + (c - '0');
		  }
		if ((i < 0) || (i >= MAX_NET_INST) ||
		    (NetInstTable[i].inUse == 0))
		    printf("Error - bad connection index specified.\n");
		else if (NetInstTable[i].prot != TCPtype)
		    printf("Error - non-Tcp connection specified.\n");
		else
		    PrintTcpConnectionState(NetInstTable[i].tcbId);
	        break;
	    case 'f':
		printf("FreeBufferCount: %d\n",	FreeBufferCount);
		printf("TotalBufferCount: %d\n", TotalBufferCount);
	        printf("NumAllocCalls: %d\n",NumAllocCalls);
		printf("NumDeallocCalls: %d\n",	NumDeallocCalls);
	        printf("NetInst free list:    ");
	        for (i = NetInstFree; i != -1; i = NetInstTable[i].next)
		  {
		    printf("%d  ", i);
		  }
		printf("\n");
		break;
	    case 's':
		printf("AllocSemCount: %d\n", AllocSemCount);
		break;
	    case 'r':
	        printf("CurrentTime: %d,    NextTimeout: %d\n",
			CurrentTime, NextTimeout);
		printf("MaxInt: %d\n", MaxInt);
		break;
	    case 'e':
		PrintPnetStatus();
		break;
	    case 'x':
	        ExitInternetServer();
	    default:
	        printf("Unknown request character\n");
		break;
          }
      }
  }




/*
 * InvalidFileid:
 * Checks if the fileid received in a message is reasonable.
 */

Boolean InvalidFileid(rqMsg)
    IoRequest *rqMsg;
  {
    int indx = rqMsg->fileid >> 1;

    switch (rqMsg->requestcode)
      {
	case CREATE_INSTANCE:
	case NetPktRcvd:
	case NetTimeout:
	    return(False);
	case WRITE_INSTANCE:
	case WRITESHORT_INSTANCE:
	case READ_INSTANCE:
	case QUERY_INSTANCE:
	case RELEASE_INSTANCE:
	case QUERY_FILE:
	case MODIFY_FILE:
	    if ((indx < 0) || (indx >= MAX_NET_INST) ||
	    		(NetInstTable[indx].inUse == 0))
		return(True);
	    else
		return(False);
	default:
	    return(True);
      }
  }




/*
 * QueryInstance:
 * Returns the state of the specified network connection instance.
 */

SystemCode QueryInstance(rqMsg)
    QueryInstanceRequest *rqMsg;
  {
    QueryInstanceReply *repMsg = (QueryInstanceReply *) rqMsg;
    int indx = rqMsg->fileid >> 1;

    if (InvalidFileid(rqMsg))
	return(BAD_ARGS);
    repMsg->fileserver = NetInstTable[indx].pid;
    repMsg->blocksize = NetInstTable[indx].blocksize;
    if (rqMsg->fileid & 1)	/* Read instance */
      {
	repMsg->filetype = READABLE + STREAM + VARIABLE_BLOCK;
	/* NOTE: lastbytes and lastblock have no meaning for a read 
	   instance. */
	repMsg->filelastbytes = repMsg->blocksize;
	repMsg->filelastblock = NetInstTable[indx].filenextblock + 1;
	repMsg->filenextblock = NetInstTable[indx].filenextblock;
      }
    else
      {
	repMsg->filetype = WRITEABLE + STREAM + VARIABLE_BLOCK;
	repMsg->filelastbytes = NetInstTable[indx].filelastbytes;
	repMsg->filelastblock = NetInstTable[indx].filelastblock;
	/* NOTE: nextblock has no meaning for a write instance. */
	repMsg->filenextblock = NetInstTable[indx].filelastblock;
      }
    return(OK);
  }




/*
 * AllocNetInst:
 * Allocates a network instance descriptor.
 * Returns its index or -1 to signal none left.
 *
 * NOTE: AllocNetInst and DeallocNetInst do NOT lock the descr. lists
 * while modifying them.  This only works as long as ALL calls to these
 * routines are made from processes of the same priority.
 */

int AllocNetInst(prot, ownerPid, pid, blocksize, tcbId)
    int prot;
    ProcessId ownerPid, pid;
    int blocksize;
    unsigned tcbId;
  {
    int i;

    if (NetInstFree == -1)
      {
	return(-1);
      }
    i = NetInstFree;
    NetInstFree = NetInstTable[i].next;
    NetInstTable[i].next = NetInstHead;
    NetInstHead = i;

    NetInstTable[i].inUse = READ_INST | WRITE_INST;
    NetInstTable[i].ownerPid = ownerPid;
    NetInstTable[i].prot = prot;
    NetInstTable[i].pid = pid;
    NetInstTable[i].tcbId = tcbId;
    NetInstTable[i].blocksize = blocksize;
    NetInstTable[i].filelastbytes = 0;
    NetInstTable[i].filelastblock = 0;
    NetInstTable[i].filenextblock = 0;

    ActivateNetProtocol(NetLevelProtocol[prot]);
    return(i);
  }




/*
 * DeallocNetInst:
 * Deallocates a network connection.
 *
 * NOTE: AllocNetInst and DeallocNetInst do NOT lock the descr. lists
 * while modifying them.  This only works as long as ALL calls to these
 * routines are made from processes of the same priority.
 */

DeallocNetInst(indx)
    int indx;
  {
    int i;
    int netProt;

    if (InternetDebug)
      {
	printf("------DeallocNetInst: %d------\n", indx);
      }

    NetInstTable[indx].inUse = 0;
    /* Deallocate the connection descriptor. */
    if (indx == NetInstHead)
      {
	NetInstHead = NetInstTable[indx].next;
      }
    else
      {
	for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
	  {
	    if (indx == NetInstTable[i].next)
	      {
		break;
	      }
	  }
	NetInstTable[i].next = NetInstTable[indx].next;
      }

    /* Check if a network-level protocol can be deactivated. */
    netProt = NetLevelProtocol[NetInstTable[indx].prot];
    NetInstTable[indx].next = NetInstFree;
    NetInstFree = indx;
    for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
      {
	if (netProt == NetLevelProtocol[NetInstTable[i].prot])
	  {
	    break;
	  }
      }
    if (i == -1)
      {
	DeactivateNetProtocol(netProt);
      }
  }




/*
 * ReplyToRead:
 * Replies to a user client's read request.
 */

ReplyToRead(replycode, pid, packet, bufferptr, len)
    SystemCode replycode;	/* Replycode to send to reader. */
    ProcessId pid;		/* Process id of reader. */
    PktBuf packet;		/* Packet containing data to return to
				   reader.  NULL if no data to return. */
    char *bufferptr;		/* Address of reader's buffer. */
    int len;			/* Length of data to return. */
  {
    Message msg;
    IoReply *repMsg = (IoReply *) msg;
    SystemCode status;
    int i;
    char *t, *f;

    repMsg->replycode = replycode;
    /*
     * Move data into reader's buffer if there is any.
     */
    if (packet == NULL)
	repMsg->bytecount = 0;
    else
      {
        status = OK;
	repMsg->bytecount = len;
	
	if( len > MAX_APPENDED_SEGMENT )
	  {
	    status = MoveTo(pid, bufferptr, packet->dataptr, len);
	  }
	else if( len > IO_MSG_BUFFER )
	  {
	    ReplyWithSegment( msg, pid, packet->dataptr, bufferptr, len );
	    return;
	  }
	else if (len > 0)
	  {
	    t = (char *) (repMsg->shortbuffer);
	    f = (char *) (packet->dataptr);
	    for (i = 0; i < len; i++)
		*t++ = *f++;
	  }
	if (status != OK)
	  {
	    repMsg->bytecount = 0;
	    repMsg->replycode = status;
	  }
      }
    pid = Reply(msg, pid);
  }




/*
 * PacketCheck:
 * Checks to see if a packet has overwritten its boundaries.
 */

PacketCheck(packet)
    PktBuf packet;
  {
    if (packet->dataptr < packet->data)
      {
	printf("DING DING DING DING ---- packet->dataptr too small!!!\n");
      }
    if ((packet->dataptr + packet->length) > (packet->data + MAXPBUFSIZE))
      {
	printf("DING DING DING DING ---- packet length too long!!!\n");
      }
  }




/*
 * ExitInternetServer:
 * Exits the internetserver.
 */

ExitInternetServer()
  {
    exit();
  }




/*** ROUTINES RELATING TO DATA MOVEMENT ***/




/*
 * MoveData:
 * Moves len bytes from 'from' to 'to' in the same address space.
 */

MoveData(to, from, len)
    char *to, *from;
    int len;
  {
    int i;

    for (i = 0; i < len; i++)
	*to++ = *from++;
  }




/*** ROUTINES RELATING TO BUFFER MANAGEMENT ***/




/*
 *  InitBufMgr:
 *  Initialise the buffer manager data structures.
 */

InitBufMgr ()
  {
    PacketFreeList = NULL;
    FreeBufferCount = 0;
    TotalBufferCount = 0;
    NumAllocCalls = NumDeallocCalls = 0;
  }




/*
 * ReturnBuf:
 * Return a packet buffer to free memory.
 */

ReturnBuf()
  {
    PktBuf packet;

    packet = AllocBuf();
    if (packet != NULL)
      {
	free(packet);
	TotalBufferCount -= 1;
      }
  }





/*
*  AllocBuf:
*    Allocate a buffer (if available), and return a pointer to it.
*    If no buffers available, return NULL.
*/

PktBuf AllocBuf()
  {
    PktBuf packet;
    int i;

    while (AllocSem == 1)
      {
	AllocSemCount += 1;
	Delay(0,1);
      }
    AllocSem = 1;
    if (PacketFreeList == NULL)
      {
	for (i = 0; i < BufAllocNum; i++)
	  {
	    if (TotalBufferCount > MaxBufAllocNum)
		break;
	    packet = (PktBuf) malloc(sizeof(struct pbuf));
	    if (packet == NULL)
	      {
	        if (InternetDebug)
		  {
		    printf("malloc returned NULL.\n");
		  }
	        break;
	      }
	    packet->next = PacketFreeList;
	    PacketFreeList = packet;
	    FreeBufferCount += 1;
	    TotalBufferCount += 1;
	  }
        if (PacketFreeList == NULL)
	  {
	    if (InternetDebug)
	      {
		printf("RAN OUT OF PACKET BUFFERS!\n");
	      }
	    AllocSem = 0;
	    return(NULL);
	  }
      }
    NumAllocCalls += 1;
    FreeBufferCount -= 1;
    packet = PacketFreeList;
    PacketFreeList = PacketFreeList->next;
    AllocSem = 0;
    packet->length = 0;
    packet->next = NULL;
    packet->dataptr = packet->data;
    return(packet);
  }




/*
 * DeallocBuf:
 * Deallocate a buffer.
 */

DeallocBuf(pkt)
    PktBuf pkt;
  {
    NumDeallocCalls += 1;
    FreeBufferCount += 1;
    while (AllocSem == 1)
      {
	AllocSemCount += 1;
	Delay(0,1);
      }
    AllocSem = 1;
    pkt->next = PacketFreeList;
    PacketFreeList = pkt;
    AllocSem = 0;
  }


/*** SAFE QUEUEING ROUTINES ***/


/*
 * InitSafeQueue:
 */

InitSafeQueue(q, ringBufs)
    RingQueue *q;
    RingBufRec ringBufs[];
  {
    int i;

    for (i = 1; i < MAX_RING_BUFS; i++)
      {
	ringBufs[i].pkt = Null;
	ringBufs[i].next = &(ringBufs[i-1]);
      }
    ringBufs[0].pkt = Null;
    ringBufs[0].next = &(ringBufs[MAX_RING_BUFS - 1]);
    q->head = &(ringBufs[0]);
    q->tail = q->head;
  }


/*
 * EnQueueSafe:
 */

Boolean EnQueueSafe(pkt, q)
    PktBuf pkt;
    RingQueue *q;
  {
    if (q->tail->next->pkt)
	return(False);
    else
      {
	q->tail = q->tail->next;
	q->tail->pkt = pkt;
	return(True);
      }
  }


/*
 * DeQueueSafe:
 */

PktBuf DeQueueSafe(q)
    RingQueue *q;
  {
    PktBuf pkt;

    if (q->head->next->pkt)
      {
        q->head = q->head->next;
	pkt = q->head->pkt;
	q->head->pkt = Null;
	return(pkt);
      }
    else
	return(Null);
  }




/*** ROUTINES RELATING TO TIMEOUTS ***/




/*
 * TimeoutTimer:
 * Delays until the next timeout or until awoken.  Checks to see if a 
 * timeout has occurred and send a msg to the appropriate process.
 */

TimeoutTimer()
  {
    Message msg;
    IoRequest *rqMsg = (IoRequest *) msg;
    int i, t;
    int secs, clicks;

    Time0 = GetTime(&clicks);
    CurrentTime = clicks;
    NextTimeout = MaxInt;

    while (True)
      {
        t = NextTimeout - CurrentTime;
	Delay(0, t);
	secs = GetTime(&clicks) - Time0;
	CurrentTime = CTime(secs, clicks);
	NextTimeout = MaxInt;

	for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
	  {
	        t = FuncTable[NetInstTable[i].prot].NextTimeout(
			NetInstTable[i].tcbId);
				/* t contains the earliest timeout for this
				   connection. */
		if (t <= CurrentTime)
		  {
		    rqMsg->requestcode = NetTimeout;
		    msg[1] = ConnTimeout;
		    Send(msg, NetInstTable[i].pid);
		  }
		t = FuncTable[NetInstTable[i].prot].NextTimeout(
			NetInstTable[i].tcbId);
				/* Have to do it again to get updated timeout
				   values from processing timeouts. */
		if (InternetDebug && (t <= CurrentTime))
		  {
		    printf(" ++++ WARNING - Bad minimum timout values!\n");
		  }
		if (t < NextTimeout)
		    NextTimeout = t;
	  }
      }
  }




/*
 * CleanupTimer:
 * Delays Timer1Interval clicks and then checks to see if the owners of
 * all the connections are still valid.  If not, then a message is sent
 * to the associated connection process notifying it of the fact.
 * Also trickles a packet buffer back to free memory if too many are
 * currently allocated and free.
 */

CleanupTimer()
  {
    Message msg;
    IoRequest *rqMsg = (IoRequest *) msg;
    int i;

    while (True)
      {
	Delay(0, Timer1Interval);

	if (FreeBufferCount > BufAllocMax)
	    ReturnBuf();

	for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
	  {
	    if (!ValidPid(NetInstTable[i].ownerPid))
	      {
		rqMsg->requestcode = NetTimeout;
		msg[1] = InvalidConnOwner;
		Send(msg, NetInstTable[i].pid);
	      }
	  }
      }
  }




/*** ROUTINES RELATING TO QUEUEING ***/


/*
 * InitQueue:
 */

InitQueue(q)
    Queue *q;
  {
    q->head = NULL;
    q->tail = NULL;
  }


/*
 * EnQueue:
 */

EnQueue(q, rec)
    Queue *q;
    PktBuf rec;
  {
    rec->next = NULL;
    if (q->head == NULL)
        q->head = rec;
    else
        q->tail->next = rec;
    q->tail = rec;
  }


/*
 * DeQueue:
 */

PktBuf DeQueue(q)
    Queue *q;
  {
    PktBuf rec;

    if (q->head == NULL)
        rec = NULL;
    else
      {
        rec = q->head;
	q->head = q->head->next;
	rec->next = NULL;
	if (q->head == NULL)
	    q->tail = NULL;
      }
    return(rec);
  }
