/*
 * create.c
 *
 * DESCRIPTION
 * Four library functions for use with the V kernel.  They provide
 *  a convenient interface to the kernel's process creation mechanism.
 *
 * EXPORTS
 * pid = Create(priority, fn, ss)	--Create a process
 * Ready(pid, n, a1, ..., an)		--Start it running
 * Destroy(pid)				--Destroy it and free stack
 * Suicide()				--Same as Destroy(0)
 *
 * IMPORTS
 * malloc, various kernel primitives
 *
 * AUTHOR
 * Tim Mann
 *
 * HISTORY
 * 03/12/82 TPM - Initial version
 * 04/01/83 TPM - Added Destroy and made Suicide public
 */

#include "../include/Vio.h"
#include "../include/Vprocess.h"

char *LastStackAllocated;

/* Create:
 *
 * Creates a process executing the function fn, with stack size
 *   ss.  The process is NOT started at this time.  The function
 *   fn may take arguments; these are set up by Ready at the time
 *   a process is started.
 * Returns the pid of the created process, or zero if the process
 *   cannot be created or there is no memory left for the stack.
 */
ProcessId Create(priority, fn, ss)
int (*fn)(), ss;
short priority;
  {
    ProcessId child;
    char *stackLimit;
    long *stackTop;
    Processor_state childState;
    PerProcessArea *childPerProcess;

    ss += ss & 1;	/* Round ss up to next even size */
    LastStackAllocated = stackLimit = (char *) malloc(ss);
    if (stackLimit == NULL) return (0);
    stackTop = (long *) (stackLimit + ss);

    child = CreateProcess(priority, 0, 0);

    if (ReadProcessState(child, &childState) == 0) return (0);
    childState.USER_STACK_POINTER = (Unspec) stackTop;
    childState.perProcess = (Unspec *) stackLimit;
    childState.perProcessLoc = (Unspec **) &PerProcess;
    childState.pc = (Unspec) fn;
    if (WriteProcessState(child, &childState) == 0) return (0);

    /* Initially, give child the same standard io, etc., as its creator */
    childPerProcess = (PerProcessArea *) childState.perProcess;
    *childPerProcess = *PerProcess;
    childPerProcess->stackSize = ss;

    return (child);
  }


/* Ready:
 *
 * Sets up arguments on a newly created process's stack, and adds
 *  the process to the ready queue by replying to it.  Returns zero
 *  if the process couldn't be replied to.
 * The first argument is the process id, the second is a count of
 *  the number of arguments to be passed to the child process, and
 *  the third and following contain the arguments.
 */
Ready(child, nargs, firstarg)
ProcessId child;
short nargs;
long firstarg;
  {
    Processor_state childState;
    long *stackPtr;
    int Suicide();
    Message msg;
    long *argp;
    
    if (ReadProcessState(child, &childState) == 0) return (0);
    stackPtr = (long *) childState.USER_STACK_POINTER;

    /* Stack up the arguments */
    argp = &firstarg + nargs;
    for ( ; nargs>0 ; nargs--)
      {
	*--stackPtr = *--argp;
      }

    /* Arrange for the child to destroy itself
     *  if it falls off the end of its code */
    *--stackPtr = (long) Suicide;

    childState.USER_STACK_POINTER = (Unspec) stackPtr;
    if (WriteProcessState(child, &childState) == 0) return (0);

    /* Reply to it */
    if (Reply(msg, child) == 0) return(0);

    return (child);

  }

Destroy(pid)
    ProcessId pid;
  {
    Processor_state state;

    /* Free stack if on the same team */
    if (SameTeam(0, pid))
      {
	if (ReadProcessState(pid, &state) == 0) return (0);
	free(state.perProcess);
      }

    DestroyProcess(pid);
  }


Suicide()
  {
    Destroy(0);
  }
