/*
 * V  - Copyright (c) 1982 by David Cheriton, Tim Mann
 * All rights reserved.
 *
 * Team file loading function
 */

#include <Vioprotocol.h>
#include <Vprocess.h>
#include <b.out.h>

extern File *Open();

#define TRUE 1
#define FALSE 0
#define NUL '\0'
#define CHUNK 256
#define BUFFERSIZE 1024
#define DEFAULT_STACK 4000

ProcessId LoadTeam( filename, priority, stacksize, errcode )
    char *filename;
    unsigned priority;
    int stacksize;
    SystemCode *errcode;
  /*
   * Create a new team with the specified priority, and load the program 
   *   contained in the specified file into it.
   *
   * The file must be in b.out format. The b.out header is checked for a 
   *   valid "magic number."
   * The file is loaded into the new team and the bss segment is zeroed.
   * The number of bytes specified by stacksize is allocated
   *   at the end of the team space as a stack, unless stacksize
   *   is -1, in which case 4000 bytes are allocated.
   * If errors occur during loading, NULL is returned, and a system reply
   *   code describing the error is placed in errcode if it is not null.
   *
   * Team layout:
   *  |kernel|text|data|uninitialized data(bss)|stack space|
   */

  {
    File *teamfile;
    int  rsize, bytes, blocksize;
    char *bssptr, *readend, *loadptr, *teamendptr, *GetTeamSize(),
	*SetTeamSize();
    short *databuffer = (short *) malloc(BUFFERSIZE);
    register char *buffer = (char *) databuffer;
    short *zeroes = (short *) calloc(BUFFERSIZE/sizeof(short), sizeof(short));
    ProcessId teampid;
    struct bhdr header;
    Processor_state teamstate;
    SystemCode r;
#ifdef TIMES
    int seconds, startclicks, endclicks;
#endif

    if (errcode == NULL) errcode = &r;
    if( stacksize == -1 ) stacksize = DEFAULT_STACK;

    if ( (teampid = CreateTeam(priority, 0, 0)) == NULL )
      {
	*errcode = NO_PDS;	/* could also be illegal priority, but ... */
        free(databuffer);
	free(zeroes);
	return (NULL);
      }

#ifdef DEBUG
    printf("Open File \n");
#endif
    teamfile = Open(filename, FREAD | FBLOCK_MODE, errcode);
    if( *errcode || (teamfile == NULL) )
      {
	DestroyProcess(teampid);
        free(databuffer);
	free(zeroes);
	return (NULL);
      }

#ifdef DEBUG
    printf("Read file.\n");
#endif
#ifdef TIMES
    seconds = GetTime(&startclicks);
#endif
    bytes = 0;
    blocksize = teamfile->blocksize;

    /* Read first block of file to get b.out header */
    rsize = Read(teamfile, buffer, blocksize);
#ifdef DEBUG
    printf("   bytes read = %d.\n",rsize);
#endif    
    if( rsize < sizeof(struct bhdr) )
      { 
	Close(teamfile);
	DestroyProcess(teampid);
	*errcode = END_OF_FILE;
        free(databuffer);
	free(zeroes);
	return (NULL);
      }
    bytes += rsize;

    /* Copy b.out header to header buffer */
    header = *(struct bhdr *) buffer;
    
    /* Check the header has the legal magic number */
#ifdef DEBUG
    printf("Magic number is %o should be %o.\n",header.fmagic, FMAGIC);
    printf("Text\tData\tBss\tSymbols\n");
    printf("%x\t%x\t%x\t%x\n",header.tsize,header.dsize
		,header.bsize,header.ssize);
#endif
    if( header.fmagic != FMAGIC )
      {
	Close(teamfile);
	DestroyProcess(teampid);
	*errcode = BAD_STATE;
        free(databuffer);
	free(zeroes);
	return (NULL);
      }
    /* Calculate the required sizes and pointers */
    loadptr = GetTeamSize( teampid ); /* Point to start loading */
    /* Beginning of bss area in team */
    readend = bssptr = loadptr + header.tsize + header.dsize;
    /* Pointer to one beyond last byte to be init in read */

    /* Total amount of memory allocated to team */
    teamendptr = readend + header.bsize + stacksize;
	
    /* Set size of destination team */
    if( SetTeamSize(teampid, teamendptr) != teamendptr )
      {
	Close(teamfile);
	DestroyProcess(teampid);  /* don't need to set to null if destroying */
	*errcode = NO_MEMORY;
        free(databuffer);
	free(zeroes);
	return (NULL);
      }

    /* Move remainder of first data block to team code segment */
    buffer += sizeof(struct bhdr);
    rsize -= sizeof(struct bhdr);
    /* In case amount to read is less than that in one block */
    if( loadptr+rsize > readend ) rsize = (int) (readend - loadptr);

    if( (*errcode = MoveTo(teampid,loadptr,buffer,rsize)) != OK )
      {
	Close(teamfile);
	DestroyProcess(teampid);
        free(databuffer);
	free(zeroes);
	return(NULL);
      }
    buffer = (char *) databuffer;
    loadptr += rsize;
    teamfile->block = 1; /* Set to next block number */

    /* Load rest of file */
#ifdef SLOWLOAD
    while( loadptr < readend )
      {
	bytes = (int) (readend - loadptr);
	bytes = (bytes < blocksize) ? bytes : blocksize;

	if( (rsize = Read( teamfile, buffer, bytes)) < bytes )
	  {
	    Close(teamfile);
	    errcode = teamfile->lastexception;
	    DestroyProcess(teampid);
            free(databuffer);
	    free(zeroes);
	    return (NULL);
	  }
#ifdef DEBUG
	printf("    Returned from Read.  bytes = %d.\n",rsize);
#endif
#ifdef DEBUG
	printf("MoveTo(%8x,%6x,%6x,%6x)\n",teampid,loadptr,buffer,rsize);
#endif
	if( (*errcode = MoveTo(teampid,loadptr,buffer,rsize)) != OK )
	  {
	    Close(teamfile);
	    DestroyProcess(teampid);
            free(databuffer);
	    free(zeroes);
	    return(NULL);
	  }
	loadptr += rsize;
	teamfile->block++; /* move to next file block */
      }
#else
    /* Fast team load using the forwarding trick */
     {
	Message msg;
	IoRequest *req = (IoRequest *) msg;

	req->requestcode = READ_AND_FORWARD;
	req->fileid = teamfile->fileid;
	req->blocknumber = 1;
	req->bufferptr = loadptr;
	req->bytecount = (unsigned) (readend - loadptr);
	Forward( req, teampid, teamfile->fileserver );
	ReceiveSpecific( req, teampid );
	*errcode = (SystemCode) (req->blocknumber>>16);
				 /* Special place for replycode */
	if( *errcode != OK )
	  {
	    Close(teamfile);
	    DestroyProcess(teampid);
            free(databuffer);
	    free(zeroes);
	    return( NULL );
	  }
	loadptr = readend;
      }
#endif
    Close(teamfile);
#ifdef TIMES
    seconds = GetTime(&endclicks) - seconds;
    printf("Time to Read file= %d Milliseconds",((endclicks - startclicks) +
						(seconds * 100))*10 );
    bytes = header.tsize + header.dsize;

    printf(" Bytes read = %d => Bytes per sec = %d\n", bytes,
		(bytes * 100)/((endclicks - startclicks) + (seconds * 100)) );
#endif

    /* Now zero the bss area. */
#ifdef DEBUG
	printf("Zero bss segment\n");
#endif DEBUG

    loadptr = bssptr;

    for( bytes = header.bsize; bytes > 0; bytes -= CHUNK)
      {
	if( (*errcode=MoveTo(teampid,loadptr,zeroes,
		((bytes <= CHUNK) ? bytes : CHUNK))) != OK) 
	  {
	    DestroyProcess(teampid);
            free(databuffer);
	    free(zeroes);
	    return(NULL);
	  }
	loadptr += CHUNK;
      }

    /* Set pc and stack pointer for new team */
    ReadProcessState(teampid, &teamstate);
    teamstate.USER_STACK_POINTER = teamendptr - 2*sizeof(long);  /* Put two long zeros on stack */
    if( (*errcode=MoveTo(teampid,teamstate.USER_STACK_POINTER,zeroes,2*sizeof(long))) != OK )
      {
	DestroyProcess(teampid);
        free(databuffer);
	free(zeroes);
	return(NULL);
      }
    teamstate.pc = (Unspec) header.entry;
    WriteProcessState(teampid, &teamstate);

#ifdef DEBUG
    printf("Loading done\n");
#endif
    free(databuffer);
    free(zeroes);
    return(teampid);
  }

