/*
 * Distributed V Kernel - Copyright (c) 1982 by David Cheriton, Willy Zwaenepoel
 *
 * Inter kernel communication routines
 *
 */

#include "process.h"
#include "naming.h"
#include "../../libc/include/Vikc.h"
#include "../../libc/include/Vioprotocol.h"

#define RETRANSMISSION_PERIOD	(256/MAX_PROCESSES)
#define	MAXTRANS		(MAX_RETRANSMISSIONS*RETRANSMISSION_PERIOD)

unsigned short 	CurrentSeqNo = 1;
int numRetrans = 0;
extern kPacket	*kPacketSave;
ProcessBlock ProcBlock;

/* Kernel packet handler */

extern Process_id
    pRemoteSend(),
    pRemoteReply(),
    pRemoteForward(),
    pRemoteReceiveSpecific(),
    pBreathOfLife(),
    pRemoteGetPid(),
    pRemoteGetPidReply(),
    pRemoteMoveFromReq(),
    pRemoteMoveFromRep(),
    pRemoteMoveToReq(),
    pRemoteMoveToRep(),
    pNAck(),
    pBadPacketType();

typedef Process_id (*PFPID)(); /* pointer to a function returning a pid */

PFPID RemoteTrapTable[16] =	/* has to agree with the remote opcodes */
  {
    pBadPacketType,
    pRemoteSend,
    pRemoteReply,
    pRemoteForward,
    pRemoteReceiveSpecific,
    pBreathOfLife,
    pRemoteGetPid,
    pRemoteGetPidReply,
    pRemoteMoveFromReq,
    pRemoteMoveFromRep,
    pRemoteMoveToReq,
    pRemoteMoveToRep,
    pNAck,
    pBadPacketType,
    pBadPacketType,
    pBadPacketType
  };

Process_id NonLocal( pid ) Process_id pid;
  /*
   * A pid is local if its first 16 bits are equal to those of the logical
   * host number. 
   */
  {
    extern unsigned LogicalHostNumber;
    register unsigned host;

    host = pid & 0xffff0000;
    if( (host == LogicalHostNumber) || (pid == 0) )
      return( 0 );
    else
      return( 1 );
  }    

/* Inter process communication */

Process_id NonLocalSend ( msg, pid ) register Unspec *msg; Process_id pid;

  {

    extern Process *Pd_bundle[], *Active, *Readyq_head;
    register Process *pd, *prev;
    unsigned append;
    Process_id GetReply();
    int WriteKernelPacket();

    /* Adjust process state */
    pd = Active;
    pd->forwarder = ( Process * ) pd->pid;
    pd->blocked_on = pid;
    pd->seqNo = CurrentSeqNo++;
    Removeready();
    pd->state = AWAITING_REPLY;
    /* Send off packet */
    if( (pd->dataSegmentPro & READ_ACCESS) &&
	(pd->dataSegmentSize > 0 ) )
      {
	if( pd->dataSegmentSize >= MAX_APPENDED_SEGMENT )
	    append = MAX_APPENDED_SEGMENT;
	else
	    append = pd->dataSegmentSize;
	WriteKernelPacket(remoteSend,pd->pid,pid,0,append,0,
			  NULL,pd->seqNo,msg,pd->dataSegmentPtr);
      }
    else
	WriteKernelPacket(remoteSend,pd->pid,pid,0,0,NULL,
			  NULL,pd->seqNo,msg,NULL);
    pd->numTrans = 1;
    /* Arrange to finish up with reply after unblocking */
    pd->finish_up = (Unspec (*)()) GetReply;
    pd->returnMessage = (Unspec *) msg;
  }

Process_id NonLocalReceiveSpecific ( msg, pid )
register Unspec *msg; Process_id pid;

  {
    extern Process *Pd_bundle[], *Active, *Readyq_head;
    register Process *pd, *prev;
    Process_id GetMessage();
    int WriteKernelPacket();

    /* Adjust process state */
    pd = Active;
    pd->blocked_on = pid;
    pd->seqNo = CurrentSeqNo++;
    Removeready();
    pd->state = RECEIVE_BLKED;
    /* Send kernel packet */
    WriteKernelPacket(remoteReceiveSpecific,pd->pid,pid,0,0,
		      NULL,NULL,pd->seqNo,NULL,NULL);
    pd->numTrans = 1;
    /* Arrange to finish up with message after unblocking */
    pd->finish_up = (Unspec (*)()) GetMessage;
    pd->returnMessage = (Unspec *) msg;
  }

/*
Process_id NonLocalReply ( msg, pid ) register Unspec *msg; Process_id pid;
  {
    register Process *pd;
    register long *msgarray;
    extern Process *Pd_bundle[], *Active;
    int WriteKernelPacket();
    Process *GetAlien(), *DestroyAlien();

    if( (pd=GetAlien(pid)) == NULL ) return;
    msgarray = (long *) pd->msg;
    if( msgarray[OPCODE_INDEX] & IDEMPOTENT_MASK )
	DestroyAlien( pd->pid );
    else
      {
        pd->state = REPLIED_TO;
     	pd->numTrans = RETRANSMISSION_PERIOD + 2;
	Copy_in_msg( pd->msg, msg );
      }
    WriteKernelPacket(remoteReply,pd->blocked_on,pid,0,0,NULL,
		      NULL,pd->seqNo,msg,NULL);
  }
*/

Process_id NonLocalReplyWithSegment( msg, pid, src, dest, bytes )
Unspec *msg; Process_id pid; unsigned char *src, *dest; unsigned bytes;
  {
    int WriteKernelPacket();
    Process *GetAlien(), *DestroyAlien();
    extern Process *Active;
    register Process *pd;
    long *msgarray;

    if( (pd = GetAlien(pid)) == NULL )
	return( 0 );
    if( bytes == 0 )
      {
	msgarray = (long *) pd->msg;
	if( msgarray[OPCODE_INDEX] & IDEMPOTENT_MASK )
	    DestroyAlien( pd->pid );
	else
	  {
	    pd->state = REPLIED_TO;
	    pd->numTrans = RETRANSMISSION_PERIOD + 2;
	    Copy_in_msg( pd->msg, msg );
	  }
   	WriteKernelPacket(remoteReply,pd->blocked_on,pid,0,0,NULL,
			  NULL,pd->seqNo,msg,NULL);
      }
    else
      {
	DestroyAlien( pd->pid );
	/* Currently only one packet allowed */
	if( bytes > MAX_APPENDED_SEGMENT )
	    return( 0 );
	else
	    WriteKernelPacket(remoteReply,pd->blocked_on,pid,0,bytes,0,
			      dest,pd->seqNo,msg,src);
      }
  }

Process_id NonLocalForward1 ( msg, from_pid, to_pid )
register Unspec *msg; Process_id from_pid, to_pid;

/* from_pid is non-local, to_pid unknown */

  {
    register Process *pd;
    register long *msgarray;
    extern Process *Active;
    int WriteKernelPacket();
    Process *GetAlien(), *DestroyAlien();

    if( (pd=GetAlien(from_pid)) == NULL ) return;
    msgarray = (long *) pd->msg;
    if( msgarray[OPCODE_INDEX] & IDEMPOTENT_MASK )
	DestroyAlien( pd->pid );
    else
      {
	pd->blocked_on = to_pid;
	pd->forwarder = (Process *) Active->pid;
        pd->state = FORWARDED_TO;
     	pd->numTrans = RETRANSMISSION_PERIOD + 2; /* Make sure */
	Copy_in_msg( pd->msg, msg );
      }
    WriteKernelPacket(remoteForward,to_pid,from_pid,Active->pid,0,NULL,
		      NULL,pd->seqNo,msg,NULL);
  }

Process_id NonLocalForward2 ( msg, from_pid, to_pid )
register Unspec *msg; Process_id from_pid, to_pid;

  /*  from_pid local, to_pid non local */

  {
    extern Process *Pd_bundle[], *Active;
    register Process *sender;
    int WriteKernelPacket();

    /* Adjust process state */
    sender = Map_to_pd ( from_pid );
    Copy_in_msg ( sender->msg, msg );
    sender->blocked_on = to_pid;
    sender->state = AWAITING_REPLY;
    sender->seqNo = CurrentSeqNo++;
    /* Send packet */
    WriteKernelPacket(remoteSend,from_pid,to_pid,sender->forwarder,0,NULL,
		      NULL,sender->seqNo,msg,NULL);
    sender->numTrans = 1;
  }
  
Process_id NonLocalMoveFrom ( srcpid, src, dst, bytes )
Process_id srcpid; register Unspec *src, *dst; register unsigned bytes;

  {
    extern Process *Active, *Pd_bundle[], *Readyq_head;
    register Process *pd;
    int WriteKernelPacket();
    Process *FindAlien();

    /* Reset counter in alien */
    if( (pd=FindAlien(srcpid)) == NULL ) return;
    pd->numTrans = MAXTRANS;
    /* Adjust process state */
    pd = Active;
    pd->dataSegmentPtr = (Unspec *) dst;
    pd->dataSegmentSize = bytes;
    pd->dataExpected = (Unspec *) dst;
    pd->remoteSegmentPtr = (Unspec *) src; /* remember for retransmission */
    pd->blocked_on = srcpid;
    pd->seqNo = CurrentSeqNo++;
    Removeready();
    pd->state = MOVEFROM_BLKED;
    /* Send packet */
    WriteKernelPacket(remoteMoveFromReq,pd->pid,srcpid,0,bytes,dst,
		      src,pd->seqNo,NULL,NULL);
    pd->numTrans = 1;
  }
  
Process_id NonLocalMoveTo ( dstpid, dst, src, bytes )
Process_id dstpid; Unspec *dst; Unspec *src; unsigned bytes;

  {
    extern Process *Active, *Moveq_head, *Moveq_tail;
    register Process *pd;
    unsigned noPackets;
    Process *FindAlien();

    /* Reset counter in alien */
    if( (pd=FindAlien(dstpid)) == NULL ) return;
    pd->numTrans = MAXTRANS;
    /* Adjust process state */
    pd = Active;
    pd->remoteSegmentPtr = dst;
    pd->remoteSegmentSize = bytes;
    pd->dataSegmentPtr = src;
    pd->blocked_on = dstpid;
    pd->seqNo = CurrentSeqNo++;
    Removeready();
    pd->state = MOVETO_BLKED;
    /* Send packets */
    if( Moveq_head == NULL )
      {
	if( (noPackets = (bytes-1)/MAX_APPENDED_SEGMENT + 1) == 1 )
	  {
	    WriteKernelPacket(remoteMoveToReq,pd->pid,dstpid,0,bytes,bytes,
			      dst,pd->seqNo,NULL,src);
	  }
	else
	  {
	    Moveq_tail = Moveq_tail->link = pd;
	    pd->link = NULL;
	    pd->moveIndex = 1;
	    WriteKernelPacket(remoteMoveToReq,pd->pid,dstpid,
			      0,MAX_APPENDED_SEGMENT,bytes,
			      dst,pd->seqNo,NULL,src);
	  }
      }
    else
      {
	Moveq_tail = Moveq_tail->link = pd;
	pd->moveIndex = 0;
      }
    pd->numTrans = 1;
  }

Process_id pBadPacketType()
  {
    return;
  }

Process_id pRemoteSend()
  {
    extern Process *Pd_bundle[], *Readyq_head;
    extern Process_id Kernel_Process_Pid;
    register Process *receiver, *pd, *prev;
    register kPacket *kp = kPacketSave;
    Process *CreateAlien(), *FindAlien();

    if((receiver=Map_to_pd(kp->dstPid))->pid!=kp->dstPid)
      {
	if( kp->dstPid == Kernel_Process_Pid )
	    goto L;
	if( kp->length != 0 )
	    DiscardDataPacket( kp->length );
	if( !NonLocal( kp->dstPid ) ) 
	    WriteKernelPacket(nAck,kp->dstPid,kp->srcPid,0,0,NULL,
			      NULL,kp->sequenceNo,NULL,NULL);
	return;
      }
L:
    /* Duplicate check */
    if( (pd=FindAlien(kp->srcPid)) != NULL )
      {
	if( pd->seqNo == kp->sequenceNo )
	  {
	    /* Duplicate */
	    if( kp->length != 0 )
	    	DiscardDataPacket( kp->length );
	    if( pd->state == REPLIED_TO )
	      {
		pd->numTrans = RETRANSMISSION_PERIOD + 2;
	    	WriteKernelPacket( remoteReply,pd->blocked_on,pd->pid,0,0,NULL,
			   	   NULL,pd->seqNo,pd->msg,NULL );
	      }
	    else if( pd->state == FORWARDED_TO )
	      {
		pd->numTrans = RETRANSMISSION_PERIOD + 2;
	    	WriteKernelPacket( remoteForward,pd->blocked_on,pd->pid,pd->forwarder,0,NULL,
			 	   NULL,pd->seqNo,pd->msg,NULL );
	      }
	    else
	      {
		pd->numTrans = MAXTRANS;
    	    	WriteKernelPacket( breathOfLife,kp->dstPid,kp->srcPid,0,0,NULL, 
			 	   NULL,kp->sequenceNo,NULL,NULL );
	      }
	    return;
	  }
	else
	  {
	    /* A new send from this pid, destroy old alien */
	    DestroyAlien( pd->pid );
	  }
      }

    if ( ( pd = CreateAlien() ) == NULL ) 
      {
	WriteKernelPacket( breathOfLife,kp->dstPid,kp->srcPid,0,0,NULL,
			   NULL,kp->sequenceNo,NULL,NULL );
	return;
      }

    if( kp->dstPid == Kernel_Process_Pid )
      {
	SendKernel( pd, kp->msg );
	return;
      }
    if( receiver->state == RECEIVE_BLKED &&
	(receiver->blocked_on == 0 || receiver->blocked_on == pd->pid ) )
      {
	pd->state = AWAITING_REPLY;
	receiver->blocked_on = pd->pid;
	receiver->last_sender = pd;
	if( receiver->recBufSize == 0 )
	  {
	    if( kp->length != 0 )
		DiscardDataPacket( kp->length );
	  }
	else
	  {
	    SetProcessContext( receiver );
	    if( kp->length == 0 )
		*(receiver->recBufSizePtr) = 0;
	    else if( kp->length <= receiver->recBufSize )
	      {
		ReadDataPacket( kp->length, receiver->recBufPtr );
		*(receiver->recBufSizePtr) = kp->length;
	      }
	    else /* length > receiver->recBufSize */
	      {
		ReadDataPacket( receiver->recBufSize, receiver->recBufPtr );
		DiscardDataPacket(kp->length - receiver->recBufSize);
		/* *recBufSizePtr has the right value */
	      }
	    SetProcessContext( Active );
	  }
	Addready( receiver );
      }
    else
      {
	pd->link = NULL;
	pd->state = SEND_BLKED;
	receiver->msg_queue_end = (receiver->msg_queue_end)->link = pd;
	if( kp->length != 0 )
	    DiscardDataPacket( kp->length );
      }

  }
  
Process_id pRemoteReply()
  {
    extern Process *Pd_bundle[], *Readyq_head, *Active;
    register Process *sender;
    register kPacket *kp = kPacketSave;

    if(((sender=Map_to_pd(kp->dstPid))->pid!=kp->dstPid )||
       ( sender->state != AWAITING_REPLY ) ||
       ( sender->blocked_on != kp->srcPid ) ||
       ( sender->seqNo != kp->sequenceNo ) )
      return;
    sender->state = READY;
    Copy_in_msg ( sender->msg, kp->msg );
    if( kp->length != 0 )
      {
	if( ((sender->dataSegmentPro & WRITE_ACCESS) == 0 ) ||
	    ((unsigned)sender->dataSegmentPtr > (unsigned)kp->remoteaddress)||
	    ((unsigned)kp->remoteaddress+kp->length >
	     (unsigned)sender->dataSegmentPtr+sender->dataSegmentSize) )
	  {
	    DiscardDataPacket( kp->length );
	  }
	else
	  {
	    SetProcessContext( sender );
	    ReadDataPacket( kp->length, kp->remoteaddress );  
	    SetProcessContext( Active );
	  }
      }
    Addready ( sender );
    return;
  }


Process_id pRemoteReceiveSpecific()
  {
    extern Process *Pd_bundle[];
    register Process *pd;
    register kPacket *kp = kPacketSave;
    int WriteKernelPacket();

    if ( (pd=Map_to_pd(kp->dstPid))->pid != kp->dstPid )
      {
	if( !NonLocal( kp->dstPid ) )
	    WriteKernelPacket(nAck,kp->dstPid,kp->srcPid,0,0,NULL,
			      NULL,kp->sequenceNo,NULL,NULL);
	return;
      }
    WriteKernelPacket ( breathOfLife,kp->dstPid,kp->srcPid,0,0,NULL, 
			NULL,kp->sequenceNo,NULL,NULL );
  }

Process_id pBreathOfLife()
  {
    extern Process *Pd_bundle[];
    register kPacket *kp = kPacketSave;
    register Process *sender;

    if ( ( (sender=Map_to_pd(kp->dstPid))->pid != kp->dstPid ) ||
	 ( sender->seqNo != kp->sequenceNo ) )
      return;
    sender->numTrans = 0;
  }

 
Process_id pRemoteForward()
  {
    Process *pd, *sender, *receiver;
    register kPacket *kp = kPacketSave;
    int WriteKernelPacket();
    /* kp->srcPid is to_pid, kp->dstPid is from_pid */

    if ( ( (sender=Map_to_pd(kp->dstPid))->pid != kp->dstPid ) ||
	 ( sender->seqNo != kp->sequenceNo ) )
      return;

    if ( (sender->state != AWAITING_REPLY) ||
         (sender->blocked_on != kp->forwarder ) )
      return;

    Copy_in_msg ( sender->msg, kp->msg );

    if ( NonLocal(kp->srcPid) )
      {
        sender->blocked_on = kp->srcPid;
        sender->state = AWAITING_REPLY;
        sender->forwarder = ( Process * ) kp->forwarder;
	sender->seqNo = CurrentSeqNo++;
        WriteKernelPacket(remoteSend,kp->dstPid,kp->srcPid,0,0,NULL,
			  NULL,sender->seqNo,kp->msg,NULL);
	sender->numTrans = 1;
      }
    else
      {
        if ( (receiver=Map_to_pd(kp->srcPid))->pid != kp->srcPid )
	  {
	    /* Forwarding to a nonexistent local process */
	    sender->blocked_on = kp->srcPid;
	    ((IoReply *)sender->msg)->replycode = NONEXISTENT_PROCESS;
	    Addready(sender);
	    return;
	  }

        if ( receiver->state == RECEIVE_BLKED &&
	     (receiver->blocked_on==0 || receiver->blocked_on==kp->dstPid) )
	  {
	    sender->blocked_on = kp->srcPid;
	    sender->state = AWAITING_REPLY;
	    receiver->blocked_on = kp->dstPid;
	    receiver->last_sender = sender;
	    if (receiver->recBufSize != 0)
	      {
		SetProcessContext(receiver);
		*(receiver->recBufSizePtr) = 0;
	      }
	    Addready ( receiver );
  	  }
        else
	  {
	    sender->link = NULL;
	    sender->blocked_on = kp->srcPid;
	    sender->state = SEND_BLKED;
	    receiver->msg_queue_end = (receiver->msg_queue_end)->link
				    = sender;
          }
        return;
      }
  }

Process_id pRemoteMoveFromReq()
  {
    Process *pd, *active;
    extern Process *Active;
    register kPacket *kp = kPacketSave;
    int WriteKernelPacket();
    short i, noFullPackets;

    if ( (pd=Map_to_pd(kp->dstPid))->pid != kp->dstPid ) 
      {
	if( !NonLocal( kp->dstPid ) )
	    WriteKernelPacket(nAck,kp->dstPid,kp->srcPid,0,0,NULL,
			      NULL,kp->sequenceNo,NULL,NULL);
        return;
      }
    if ( (pd->state != AWAITING_REPLY) ||
         (pd->blocked_on != kp->srcPid) ||
	 ((pd->dataSegmentPro & READ_ACCESS) == 0) ||
         ((unsigned)pd->dataSegmentPtr > (unsigned)kp->remoteaddress) ||
         ((unsigned)kp->remoteaddress+kp->length > 
	  (unsigned)(pd->dataSegmentPtr)+pd->dataSegmentSize) ) 
      {
        return;
      }
    pd->numTrans = 0;	/* Reset transmissions counter */
    SetProcessContext( pd );
    noFullPackets = kp->length / MAX_APPENDED_SEGMENT;
    for( i=0; i<noFullPackets; i++ )
      {
        WriteKernelPacket(remoteMoveFromReply,kp->dstPid,kp->srcPid,
			  0,MAX_APPENDED_SEGMENT,kp->length,
			  kp->localaddress,kp->sequenceNo,NULL,kp->remoteaddress);
        kp->localaddress = (Unspec *)((unsigned)(kp->localaddress) + MAX_APPENDED_SEGMENT);
        kp->remoteaddress = (Unspec *)((unsigned)(kp->remoteaddress) + MAX_APPENDED_SEGMENT);
      }
    if( kp->length % MAX_APPENDED_SEGMENT != 0 )
      {
        WriteKernelPacket(remoteMoveFromReply,kp->dstPid,kp->srcPid,
			  0, kp->length%MAX_APPENDED_SEGMENT,kp->length,
			  kp->localaddress,kp->sequenceNo,NULL,kp->remoteaddress);
      }
    SetProcessContext( Active ); /* Reset context to active process */	  
  }
   
Process_id pRemoteMoveFromRep()
  {
    extern Process *Active;
    Process *pd, *active;
    register short words;
    register Unspec *dst;
    register kPacket *kp = kPacketSave;
    int ReadDataPacket(), DiscardDataPacket();

    /* kp->remoteaddress is the localaddress */
    active = Active;
    if ( (pd=Map_to_pd(kp->dstPid))->pid != kp->dstPid ) 
      {
        DiscardDataPacket( kp->length );
        return;
      }
    if ( (pd->state != MOVEFROM_BLKED) ||
         (pd->blocked_on != kp->srcPid) ||
	 (pd->seqNo != kp->sequenceNo) ||
         (pd->dataSegmentPtr > kp->remoteaddress) ||
         ((unsigned)kp->remoteaddress+ kp->length >
	  (unsigned)(pd->dataSegmentPtr)+pd->dataSegmentSize) )
      {
        DiscardDataPacket( kp->length );
        return;
      }

    if( pd->dataExpected == kp->remoteaddress )
      {
        SetProcessContext( pd );
        ReadDataPacket( kp->length, kp->remoteaddress );
        SetProcessContext( Active );
        pd->dataExpected = (Unspec *)( (unsigned)pd->dataExpected + kp->length );
        if( (unsigned)pd->dataExpected == 
	     (unsigned)pd->dataSegmentPtr + pd->dataSegmentSize )
          {
	    SetReturnValue( pd, OK );
	    Addready( pd );
	    return;
          }
      }
    else if( pd->dataExpected < kp->remoteaddress )
      {
        DiscardDataPacket( kp->length );
        /* Wait for the timer to get up and retransmit */
        return;
      }
    else if( pd->dataExpected > kp->remoteaddress )
      {
        DiscardDataPacket( kp->length );
        return;
      } 
  }

Process_id pRemoteMoveToReq()
  {
    extern Process *Active;
    Process *pd, *active;
    register kPacket *kp = kPacketSave;
    int ReadDataPacket(), DiscardDataPacket();
    /* kp->remoteaddress is a local address,
       kp->localaddress is the total length */

    active = Active;
    if ( (pd=Map_to_pd(kp->dstPid))->pid != kp->dstPid ) 
      {
        DiscardDataPacket( kp->length );
        return;
      }
    if ( (pd->state != AWAITING_REPLY) ||
         (pd->blocked_on != kp->srcPid) ||
	 ((pd->dataSegmentPro & WRITE_ACCESS) == 0) ||
         (pd->dataSegmentPtr > kp->remoteaddress) ||
         ((unsigned)kp->remoteaddress+kp->length > 
	  (unsigned)(pd->dataSegmentPtr)+pd->dataSegmentSize) )
      {
        DiscardDataPacket( kp->length );
        return;
      }
      pd->numTrans = 0; /* Reset transmission counter */
    /* Sequencing check */
    if( pd->dataExpected==NULL )
      {
        SetProcessContext( pd );
        ReadDataPacket( kp->length, kp->remoteaddress );
        SetProcessContext( Active );
        pd->dataExpected = (Unspec *)((unsigned)kp->remoteaddress+kp->length);
        pd->remoteSegmentPtr = kp->remoteaddress;
        pd->remoteSegmentSize = (unsigned) kp->localaddress;
        /* Finish check */
        if((unsigned)pd->dataExpected == 
	   (unsigned)pd->remoteSegmentPtr + (unsigned)kp->localaddress)
	  {
	    pd->dataExpected = NULL;
	    WriteKernelPacket(remoteMoveToReply,kp->dstPid,kp->srcPid,
			      0,kp->localaddress,pd->remoteSegmentPtr,
			      NULL,kp->sequenceNo,NULL,NULL);
  	  }
        return;
      }
    else if( pd->dataExpected == kp->remoteaddress )
      {
        SetProcessContext( pd );
        ReadDataPacket( kp->length, kp->remoteaddress );
        SetProcessContext( Active );
        pd->dataExpected = (Unspec *)((unsigned)kp->remoteaddress+kp->length);
        /* Finish check */
        if((unsigned)pd->dataExpected ==
	   (unsigned)pd->remoteSegmentPtr + (unsigned)kp->localaddress)
	  {
	    pd->dataExpected = NULL;
	    WriteKernelPacket(remoteMoveToReply,kp->dstPid,kp->srcPid,
			      0,kp->localaddress,pd->remoteSegmentPtr,
			      NULL,kp->sequenceNo,NULL,NULL);
  	  }
        return;
      }
    else if( pd->dataExpected < kp->remoteaddress )
      {
        DiscardDataPacket( kp->length );
        /* do nothing for now */
        return;
      }
    else if( pd->dataExpected > kp->remoteaddress )
      {
        DiscardDataPacket( kp->length );
        return;
      }
  }

Process_id pRemoteMoveToRep()
  {
    extern Process *Readyq_head, *Pd_bundle[];
    register kPacket *kp = kPacketSave;
    Process *pd;
    /* kp->remoteaddress is a local address */

    if( (pd=Map_to_pd(kp->dstPid))->pid != kp->dstPid )
      {
	if( !NonLocal( kp->dstPid ) )
	    WriteKernelPacket(nAck,kp->dstPid,kp->srcPid,0,0,NULL,
			      NULL,kp->sequenceNo,NULL,NULL);
	return;
      }
    if( (pd->state != MOVETO_BLKED) ||
	(pd->seqNo != kp->sequenceNo) ||
        (pd->blocked_on != kp->srcPid) ) 
      {
	return;
      }
    if( (pd->remoteSegmentPtr == kp->localaddress) &&
        (pd->remoteSegmentSize == kp->length) )
      {
        SetReturnValue( pd, OK ); /* Successful */
        Addready( pd );
      }
    return;
  }

Process_id pNAck()
  {
    register Process *pd;
    register kPacket *kp = kPacketSave;
    IoReply *iorep;

    if( ( (pd=Map_to_pd(kp->dstPid))->pid != kp->dstPid ) ||
	( pd->state == READY ) ||		/* I am not quite sure */
	( pd->blocked_on != kp->srcPid ) ||
	( pd->seqNo != kp->sequenceNo ) )
	return;
    SetReturnValue( pd, NONEXISTENT_PROCESS ); /* for Moves */
    iorep = ( IoReply * ) (pd->msg);
    iorep->replycode = NONEXISTENT_PROCESS;  /* for Sends */
    Addready( pd );
  }

/* Stuff related to naming */

int NonLocalGetPid ( logical_id ) register unsigned logical_id;

  {
    extern Process *Active, *Pd_bundle[], *Readyq_head;
    register Process *pd;
    int WriteKernelPacket(); Process_id GetReply();

    pd = Active;
    pd->blocked_on = logical_id; /* Preserved here for retransmission */
    pd->seqNo = CurrentSeqNo++;
    Removeready();
    pd->state = GETPID_BLKED;
    /* The destination pid in WriteKernelPacket has to be zero to make
       pretty damn sure that the packet is broadcast */
    WriteKernelPacket(remoteGetPid,pd->pid,0,logical_id,0,NULL,
		      NULL,pd->seqNo,NULL,NULL);
    pd->numTrans = 1;
    pd->finish_up = (Unspec (*)()) GetReply;
    /* Make sure only the blocked_on field will be returned by GetReply */
    pd->returnMessage = NULL;
  }

Process_id pRemoteGetPid()
  {
    extern Logical_id_entry Logical_id_map[];
    Process_id pid;
    register Process *pd;
    register kPacket *kp = kPacketSave;
    int WriteKernelPacket();
    /* kp->forwarder is logical_id */

    if ( ( kp->forwarder > MAX_LOGICAL_ID ) ||
	 ( kp->forwarder == ACTIVE_PROCESS ) ||
	 ( kp->forwarder == DEVICE_SERVER ) ||
	 ( (pid = Logical_id_map[kp->forwarder].pid) == 0 ) ||
	 ( Logical_id_map[kp->forwarder].scope == LOCAL_PID ) ||
	 ( !NonLocal( kp->srcPid ) ) ||
	 ( (pd=Map_to_pd(pid))->pid != pid ) )
      return;

    /* Note that GetPid should not return a nAck because other
	hosts might have the server running */
    WriteKernelPacket(remoteGetPidReply,pid,kp->srcPid,kp->forwarder,0,NULL,
		      NULL,kp->sequenceNo,NULL,NULL);
  }

Process_id pRemoteGetPidReply()
  {
    extern Process *Readyq_head;
    register Process *sender;
    register kPacket *kp = kPacketSave;
    /* kp->forwarder is the logical id */

    if ( ( (sender=Map_to_pd(kp->dstPid))->pid != kp->dstPid ) ||
	 ( sender->state != GETPID_BLKED ) ||
	 ( sender->blocked_on != kp->forwarder ) ||
	 ( sender->seqNo != kp->sequenceNo ) )
      return;
    /* Avoid retransmission now by making sender appear ready */
    sender->state = READY;
    sender->blocked_on = kp->srcPid; /* for GetReply to return */
    Addready( sender );
  }

/* Retransmission routines */

int Retransmit ( pd ) Process *pd;

  {
    extern int numRetrans;
    int RetransmitData();
    IoReply *iorep;

    if ( pd->numTrans == MAXTRANS )
      {
        /* Timed out */
	SetReturnValue( pd, KERNEL_TIMEOUT );
	iorep = ( IoReply * ) (pd->msg);
	iorep->replycode = KERNEL_TIMEOUT;
	/* Begin hack */
	if( pd->state == GETPID_BLKED ) pd->blocked_on = 0;
	/* End hack */
        Addready ( pd );
	return;
      }
    if( ( pd->numTrans != RETRANSMISSION_PERIOD ) &&
	( pd->numTrans != 2*RETRANSMISSION_PERIOD ) )
      {
        pd->numTrans++;
	return;
      }
    numRetrans++; 
    switch( pd->state )
      {
	case AWAITING_REPLY:
	  WriteKernelPacket(remoteSend,pd->pid,pd->blocked_on,pd->forwarder,0,NULL,
			    NULL,pd->seqNo,pd->msg,NULL);
          pd->numTrans++;
	  break;
	case RECEIVE_BLKED:
	  WriteKernelPacket(remoteReceiveSpecific,pd->pid,pd->blocked_on,
			    0,0,NULL,
			    NULL,pd->seqNo,NULL,NULL);
	  pd->numTrans++;
	  break;
	case GETPID_BLKED:
	  WriteKernelPacket(remoteGetPid,pd->pid,0,pd->blocked_on,0,NULL,
			    NULL,pd->seqNo,NULL,NULL);
	  pd->numTrans++;
	  break;
	case MOVEFROM_BLKED:
	  WriteKernelPacket(remoteMoveFromReq,pd->pid,pd->blocked_on,
			    0,pd->dataSegmentSize,pd->dataSegmentPtr,
			    pd->remoteSegmentPtr,pd->seqNo,NULL,NULL);
	  pd->numTrans++;
	  break;
	case MOVETO_BLKED:
	  RetransmitData( pd );
	  break;
      }
  }

int RetransmitData( pd ) Process *pd;

{
  extern Process *Active;
  short noFullPackets, i;
  Unspec *src;
  Unspec *dst;

  noFullPackets = pd->remoteSegmentSize / MAX_APPENDED_SEGMENT;
  dst = pd->remoteSegmentPtr;
  src = pd->dataSegmentPtr;
  SetProcessContext( pd );
  for( i=0; i<noFullPackets; i++ )
    {
      WriteKernelPacket(remoteMoveToReq,pd->pid,pd->blocked_on,
		        0,MAX_APPENDED_SEGMENT,pd->remoteSegmentSize,
		        dst,pd->seqNo,NULL,src);
      dst = (Unspec *)((unsigned)dst + MAX_APPENDED_SEGMENT);
      src = (Unspec *)((unsigned)src + MAX_APPENDED_SEGMENT);
    }
  if( pd->remoteSegmentSize % MAX_APPENDED_SEGMENT != 0 )
    {
      WriteKernelPacket( remoteMoveToReq,pd->pid,pd->blocked_on,
		  	 0,pd->remoteSegmentSize%MAX_APPENDED_SEGMENT,pd->remoteSegmentSize,
		 	 dst,pd->seqNo,NULL,src );
    }
  SetProcessContext( Active );
  pd->numTrans++;
}

/* Alien handlers */

Process *FindAlien( pid ) register Process_id pid;
  {
    extern Process *Ad_bundle[];
    register Process *pd;

    for( pd = Ad_bundle[pid&(MAX_PROCESSES-1)]; 
	 pd != NULL; 
	 pd = pd->brother )
      {
	if( pd->pid == pid ) 
	    return( pd );
      }
    return( NULL );
  }

Process *GetAlien( pid ) Process_id pid;
  {
    extern Process *Ad_bundle[];
    register Process *pd;

    for( pd = Ad_bundle[pid&(MAX_PROCESSES-1)]; 
	 pd != NULL; 
	 pd = pd->brother )
      {
	if( pd->pid == pid && pd->state == AWAITING_REPLY ) 
	    return( pd );
      }
    return( NULL );
  }

Process *CreateAlien()
  {
    extern Process *Ad_bundle[], *Free_pds, *Last_free_pd;
    register unsigned index;
    register Process *pd, *prev, *help;
    Process *save;
    register kPacket *kp = kPacketSave;

    if( (pd=Free_pds) == NULL ) return( NULL ); /* No free aliens */
    if( (Free_pds=pd->link) == NULL )	/* Last free alien */
      	Last_free_pd = ( Process * ) &Free_pds;

    index = kp->srcPid & (MAX_PROCESSES-1);

    if( (pd->old_pid&(MAX_PROCESSES-1)) == index )
      {
	/* Potential problem of local validation. Choose another pd */
	save = pd;
	if( (pd=Free_pds) == NULL )
	  {
	    Last_free_pd = (Last_free_pd->link = save);
	    save->link = NULL;
	    return( NULL );
	  }
	if( (Free_pds=pd->link) == NULL )
	  Last_free_pd = ( Process * ) &Free_pds;
	/* Return the previous one to the free list */
	Last_free_pd = (Last_free_pd->link = save);
	save->link = NULL;
      }
  
    prev = Ad_bundle[index];
    if( prev == NULL )
      {
	Ad_bundle[index] = pd;
      }
    else
      {
	for(help=prev->brother;help!=NULL;help=help->brother)
	  {
	    prev = help;
	  }
	prev->brother = pd;
      }	

    pd->numTrans = MAXTRANS;
    pd->link = NULL;
    pd->pid = kp->srcPid;
    pd->blocked_on = kp->dstPid;
    pd->brother = NULL;
    pd->seqNo = kp->sequenceNo;
    Copy_in_msg( pd->msg, kp->msg );

    return( pd );
  }

Process *DestroyAlien( pid ) Process_id pid;
  {
    extern Process *Ad_bundle[], *Free_pds, *Last_free_pd;
    register unsigned index;
    register Process *pd, *prev, *receiver;

    index = pid & (MAX_PROCESSES-1);
    prev = Ad_bundle[index];

    if( (pd=prev)->pid == pid )
      {
	pd->pid = 0;
	Ad_bundle[index] = pd->brother;
      }
    else
      {    
	for(pd=prev->brother;pd!=NULL;prev=pd, pd=pd->brother)
	  if( pd->pid == pid )
	    {
	      prev->brother = pd->brother;
	      pd->pid = 0;
	      break;
	    }
      }
    if( pd == NULL )
	return( NULL );
    else
      {
	/* Remove alien from message queue if necessary */
	if( (pd->state == SEND_BLKED) &&
	    ( (receiver=Map_to_pd(pd->blocked_on))->pid == pd->blocked_on ) )
	  {
            /* Remove from sender's queue */
            prev = (Process *) &(receiver->msg_queue);
            while( prev->link != pd ) prev = prev->link;
            if( (prev->link = pd->link) == NULL )
                receiver->msg_queue_end =
                    (Process *) &(receiver->msg_queue);
	  }
	Last_free_pd = (Last_free_pd->link = pd);
	pd->link = NULL;
    	return( pd );
      }
  }

