/*
 * Distributed V Kernel - Copyright (c) 1982 by David Cheriton, Willy Zwaenepoel
 *
 * Kernel Ethernet driver
 */

#include "../../libc/include/Venviron.h"
#include "../../libc/include/Vethernet.h"
#include "../../libc/include/Vikc.h"

#include "interrupt.h"
#include "ether3meg.h"
#include "../mi/dm.h"
#define WORDS_PER_PACKET (sizeof(kPacket)/2+2)/* control words in a kernel packet */

typedef Process_id (*PFPID)();

/* Imports */
extern Process *Map_pid();
extern SystemCode NotSupported();
extern DeviceInstance *GetDevice();

/* Exports */

extern SystemCode EnetCreate();
extern SystemCode EnetRead();
extern SystemCode EnetWrite();
extern SystemCode EnetQuery();
extern SystemCode EnetRelease();
extern SystemCode EnetCheckRequest();
extern SystemCode EnetReadPacket();
extern SystemCode EnetPowerup();

unsigned char	EnetHostNumber;		/* physical ethernet address */
DeviceInstance	*EthernetInstance = NULL;	/* ptr to DeviceInstance for Ethernet */
ProcessId	EnetWriter;		/* Non-zero if a writer using Ethernet */
int		EnetReceiveMask = ENET_DEFAULT_MASK;	/* addresses to listen for */
short		EnetStatus;		/* Current status settings */
int		EnetBadCounts = 0;	/* Number of bad count values */
short		EnetCollisions = 0;	/* Number of collision errors */
short		EnetOverflows = 0;	/* Queue overflow errors */
short		EnetCRCerrors = 0;	/* Packets with bad CRC's */
short		EnetSyncErrors = 0;	/* Receiver out of sync */
short		EnetTimeouts = 0;	/* Transmitter timeouts */
int		EnetValidPackets = 0;
char		kPacketArea[sizeof(kPacket)+4]; /* 4 bytes fo net header */
					/* Save area for kernel packets */
EnetBlock	*EnetPacket  = (EnetBlock *) kPacketArea;
kPacket 	*kPacketSave = (kPacket *) &kPacketArea[4]
;
					/* Pointer to kernel packet area */
int	 	NetAlarm = 0;

unsigned short GetLogicalHost()

  /* Return the logical host of this workstation */
  {
    unsigned char GetGenerationNumber();

    EnetHostNumber = ( char ) EIAddress;
    return( EnetHostNumber<<8 | GetGenerationNumber() );
  }

/* Macro expansion to interrupt-invoked C call to Ethernetinterrupt */
Call_inthandler(EnetInterrupt)

SystemCode EnetCreate( req, desc ) CreateEnetRequest *req;
				DeviceInstance *desc;
  /* Create an instance for the Ethernet interface.
   */
 {
    int	receiveMask;
    register CreateInstanceReply *reply = (CreateInstanceReply *) req;

    if( req->filemode != FCREATE ) return( MODE_NOT_SUPPORTED );

    if( EthernetInstance != NULL && Map_pid(EthernetInstance->owner) ) return( BUSY );

    /*
     * Initialize the device instance descriptor, can assume all fields
     * are zero except for owner and id.
     */

    desc->readfunc = EnetRead;
    desc->writefunc = EnetWrite;
    desc->modifyfunc = NotSupported;
    desc->queryfunc = EnetQuery;
    desc->releasefunc = EnetRelease;
    desc->reader = 0;
    desc->writer = 0;

    EthernetInstance = desc; /* Record the instance for interrupt routine */

    desc->type = (READABLE+WRITEABLE+VARIABLE_BLOCK+STREAM);
    desc->blocksize = ENET_MAX_PACKET;

    EnetReceiveMask = req->receivermask;

    /* Initialize the interface */
    return( EnetPowerup() );
  }
    
SystemCode EnetPowerup()

  /*  Powerup and initialize the Ethernet Interface Board.
   * Done when a new device instance of the ethernet is created.
   */
  {
    extern int Asm_EnetInterrupt();
    register int (**intvec)() = (int(**)()) 0x64; /* Level 1 */
    register int count;
    
    /* Plug interrupt location */
    *intvec = Asm_EnetInterrupt;

    /* Get hardwired address of this host */
    EnetHostNumber = ( char ) EIAddress;
    EnetStatus = EnetInit + EnetIntDisable + EnetIntLevel1 +
		(EnetReceiveMask&ENET_PROMISCUOUS ? 0 : EnetNoFilterData);

    EIStatus = EnetStatus; /* Write to interface status register */
    for( count = 0; count < 256; count++ ) EIAddress = count;

    /*  Setup for self and broadcast. */

    EnetStatus &= ~EnetNoFilterData;
    EIStatus = EnetStatus;

    if( EnetReceiveMask & ENET_BROADCAST ) EIAddress = 0;

    if( EnetReceiveMask & ENET_SELF ) EIAddress = EnetHostNumber;

    /* Turn off initialization and enable interrupts */
    EnetStatus &= ~(EnetInit + EnetIntDisable);
    EIStatus = EnetStatus;
    
    EnetFlushReceiver();

    return( OK );
 }

SystemCode EnetRelease( req, desc )
    IoRequest *req;
    DeviceInstance *desc;
   /*
    * Release the ethernet instance.
    */
  {
    desc->owner = 0; /* Free the descriptor */
    /* It is assumed that the reader will unblock on next packet
     * if currently blocked.
     */
    return( OK );
  }

SystemCode EnetRead( req, desc ) IoRequest *req; DeviceInstance *desc;

   /* Handle a read instance request for the Ethernet interface.
    */
  {
    extern Process *Active;
    SystemCode	r;

    /*
     * Reply for RETRY if already a waiting reader process.
     * This effectively provides busy-wait queuing for reading.
     */
    if( ValidPid(desc->reader) ) return( RETRY );

    if( (r=EnetCheckRequest(req)) != OK ) return( r );

    /* Check if packet waiting in receiver FIFO.
       We will disable the receiver interrupt here to prevent an interrupt
       from happening during the transfer of the packet or during the
       time between Active is set AWAITING_INT and when it is off the
       readyQ ( done in dm.c SendDevice ) 
    */

    EnetStatus |= EnetRxIntDisable;
    EIStatus = EnetStatus;

    if( EnetReadPacket(req, desc, Active) == OK ) return( OK );

    /* Wait for a packet to arrive */
    desc->reader = Active->pid;
    Active->state = AWAITING_INT;

    disable;
    EnetStatus &= ~EnetRxIntDisable;
    EIStatus = EnetStatus;
    return( NO_REPLY );
  }

SystemCode EnetWrite( req, desc ) IoRequest *req; DeviceInstance *desc;

   /* Handle a write instance request to the Ethernet interface.
    */
  {
    extern	ProcessId EnetWriter;
    extern	Process *Active;
    IoReply	*reply = (IoReply *) req;
    SystemCode r;
    int words;
    register int lwords;
    register short *p;
    register unsigned bytes;
    register char *buffer;

    if( (r = EnetCheckRequest(req)) != OK ) return( r );

    /*
     * Reply for RETRY if currently writing.
     * This effectively provides busy-wait queuing for writing.
     */
    if( EnetWriter ) return( RETRY );

    if( (bytes = req->bytecount) <= IO_MSG_BUFFER )
	buffer = (char *) req->shortbuffer;
    else
	buffer = req->bufferptr;

    EnetWriter = Active->pid;
  
    /* Transfer a packet to the Ethernet Interface Board. */

    bytes = bytes + ( req->bytecount & 1); /* make it even */

    p = (short *) buffer;

    EIData0 = words = bytes >> 1;	/* length in words */
    /* Write packet to interface */
    lwords = words>>1;
    ++lwords;
    while( --lwords ) {EIData0 = *p++;EIData0 = *p++;}
    if( words&1 ) EIData0 = *p++;

    /* Dont wait for the interrupt. No point */

    return( OK );
  }

SystemCode EnetQuery( req, desc ) QueryEnetReply *req; DeviceInstance *desc;

  {
    req->EthernetHostNumber = EnetHostNumber;
    req->NumCollisions = EnetCollisions;
    req->NumOverflows = EnetOverflows;
    req->NumCRCErrors = EnetCRCerrors;
    req->NumSyncErrors = EnetSyncErrors;
    req->NumTimeOuts = EnetTimeouts;
    req->NumValidPackets = EnetValidPackets;

    return( OK );
  } 


SystemCode EnetCheckRequest( req ) register IoRequest *req;

  /* Check that the read or write request is has a legitmate buffer, etc.
   */
  {
    register unsigned count;
    register SystemCode r;

    /*  Check length */
    count = req->bytecount;

    if( count <= IO_MSG_BUFFER ) return( OK );

    req->bytecount = 0; /* To be left zero if a check fails */
    if( count > ENET_MAX_PACKET )
	r = BAD_BYTE_COUNT;
    else  /* Make sure data pointer is valid.
	   *  Check that on a word boundary and not in the kernel area.
	   */
    if( (BadUserPtr(req->bufferptr)) ||
    (Active->team->team_space.size < (req->bufferptr + count)) ||
    ((int) req->bufferptr) & 1 )
	r = BAD_BUFFER;
    else
      {
	req->bytecount = count;
	r = OK;
      }
    return( r );
  }

EnetInterrupt()

   /* Handle an interrupt from the ethernet interface.
    */
  {
    extern DeviceInstance *EthernetInstance;
    extern ProcessId EnetWriter;
    extern Process *Moveq_head, *Moveq_tail, *Active;
    register Process *pd;
    register IoRequest *req;
    register IoReply *reply;
    register DeviceInstance *desc;
    register short transmitterack;
 
    desc = EthernetInstance;

    if( EIStatus&EnetRxInt && (EnetStatus&EnetRxIntDisable) == 0 )
      {  /* Receiver interrupt */
	if( desc )
	  {
	    if( (pd = Map_pid(desc->reader)) == NULL )
	      {
		if( Map_pid(desc->owner) == NULL )
		  {
		    desc->owner = 0; /* Free the descriptor */
		    EthernetInstance = NULL;
		  }
	    	else
		  {
		    EnetStatus |= EnetRxIntDisable;
		    EIStatus = EnetStatus;
		    NetAlarm = ENET_READER_TIMEOUT;
		    return;
		  }
		desc = NULL; 
	      }
	    else req = (IoRequest *) pd->msg;
	  }
	if( EnetReadPacket(req, desc, pd) == OK )
	  {
	    reply = (IoReply *) req;
	    reply->replycode = OK;
	    desc->reader = 0;
	    Addready( pd );
	  }
      }
    else if( EIStatus & EnetTxInt ) /* Transmitter interrupt */
      {
	transmitterack = EIAddress;
	EnetWriter = 0; 
	/* Remote block transmission */
	if( Moveq_head != NULL )
	  {
	    register Process *pd = Moveq_head;
	    unsigned help, noPackets;
	    noPackets=
(pd->remoteSegmentSize-1-MAX_APPENDED_SEGMENT*pd->moveIndex)/MAX_APPENDED_SEGMENT + 1;
	    if( noPackets == 1 )
	      {
		help = pd->remoteSegmentSize % MAX_APPENDED_SEGMENT;
		if( help == 0 ) help = MAX_APPENDED_SEGMENT;
		SetProcessContext( pd );
		WriteKernelPacket(remoteMoveToReq,pd->pid,pd->blocked_on,
				  0,help, pd->remoteSegmentSize,
		(unsigned)pd->remoteSegmentPtr+pd->moveIndex*MAX_APPENDED_SEGMENT,
		pd->seqNo,NULL,
		(unsigned)pd->dataSegmentPtr+pd->moveIndex*MAX_APPENDED_SEGMENT);
		SetProcessContext( Active );
		if( (Moveq_head=pd->link) == NULL )
		    Moveq_tail = (Process *) &Moveq_head;
	      }
	    else
	      {
		SetProcessContext( pd );
		WriteKernelPacket(remoteMoveToReq,pd->pid,pd->blocked_on,
		0,MAX_APPENDED_SEGMENT,pd->remoteSegmentSize,
		(unsigned)pd->remoteSegmentPtr+pd->moveIndex*MAX_APPENDED_SEGMENT,
		pd->seqNo,NULL,
		(unsigned)pd->dataSegmentPtr+pd->moveIndex*MAX_APPENDED_SEGMENT);
		SetProcessContext( Active );
		pd->moveIndex++;
	      }
	  }
	if( EIStatus & EnetTimeout )++EnetTimeouts;
      }
    EIStatus = EnetStatus; /* Update status as changed above */
  }

SystemCode EnetReadPacket( req, desc, pd )
  IoRequest *req;
  DeviceInstance *desc;
  Process *pd;
   /*  Physically transfer a packet from the Ethernet
    *  Interface Board into the reading process's buffer.
    */
  {
    extern Process *Active;
    extern PFPID RemoteTrapTable[];
    register unsigned	words, readwords, doublelwords;
    register short	*p, *save;
    short		status;
    IoReply		*replymsg = ( IoReply * ) req;
    SystemCode		ret;
    
    /* Check if anything in the receiver FIFO queue */
    if( EIStatus & EnetRxInt == 0 ) return( END_OF_FILE );

    status = EIData0;
    if( status & EnetQueueEmpty )
				/* This was in reality the lookahead word.
				   The next word is the real status word. */
      {
	status = EIData0;
      }

    /* Find a valid error-free packet */
    while( !(status & EnetQueueEmpty) )
      {
	words = (status & EnetCountMask) - 1;
	if( (status & (EnetCollision|EnetOverflow|EnetCRCerror)) == 0 && words <= (ENET_MAX_PACKET>>1)
		&& words > 8 )

	  {
	    /* A valid packet */

	    /* Get the first eight words of the packet and figure out
	       if it is a non-alias process kernel packet */
	    save = ( short * ) EnetPacket;
	    *save++ = EIData0; /* SRC and DEST */
	    *save++ = EIData0; /* Ether type */
	    *save++ = EIData0; /* Kernel packet type */
	    *save++ = EIData0; /* Sequence number */
	    *save++ = EIData0; /* srcPid */
	    *save++ = EIData0;
	    *save++ = EIData0; /* dstPid */
	    *save++ = EIData0;
	    
	    if( EnetPacket->EtherType != KERNEL_PACKET ||
		(kPacketSave->dstPid & REMOTE_ALIAS_PROCESS) )
	      {
		/* First case: it is not a kernel packet */
		/* and not to a remote alias process */
		if( desc == NULL ) /* Ethernet device not open */
		  {
		    /* Discard packet */
		    words -= 7;
		    while( --words ) status = EIData0;
		    ret = NO_REPLY;
		    goto readchecksum;
		  }
		/* First, fill in the bytecount in the reply */
	    	if ( ( words << 1 ) < req->bytecount )
		  req->bytecount = words << 1;
	    	/* Now decide where to put the data */
		SetProcessContext( pd );
	    	if( req->bytecount <= IO_MSG_BUFFER ) 
		  p = (short *) req->shortbuffer;
	    	else
		  p = (short *) req->bufferptr;
		/* Stick the first eight words where they belong */
		save = ( short * ) EnetPacket;
		*p++ = *save++;
		*p++ = *save++;
		*p++ = *save++;
		*p++ = *save++;
		*p++ = *save++;
		*p++ = *save++;
		*p++ = *save++;
		*p++ = *save++;
		/* And now the rest */
	    	readwords = (req->bytecount >> 1) + (req->bytecount & 1);
	    	if( words <= readwords )
	      	  {
		    words -= 7;
		    while( --words ) *p++ = EIData0;
	      	  }
	    	else /* A larger packet than requested */
	      	  {
		    words -= readwords;
		    readwords  -= 7;
		    while( --readwords ) *p++ = EIData0;
		    ++words;
		    while( --words ) status = EIData0; /* Discard extra words */
	      	  }
		ret = OK;
		SetProcessContext( Active );
	      }
	    else
	      {
		/* Second case : a kernel packet */
		/*
		 * Note that this only reads the control portion of the packet.
		 * The data portion, if there, is read by ReadDataPacket().
		 */
		doublelwords = ((WORDS_PER_PACKET-4)>>2);
		while(--doublelwords) 
		  {
		    *save++ = EIData0;*save++ = EIData0;
		    *save++ = EIData0;*save++ = EIData0;
		  }
		*save++ = EIData0; *save = EIData0;
		(*RemoteTrapTable[(kPacketSave->packetType)&0xf])();
		ret = NO_REPLY;
	      }
readchecksum:
	    /* Check that we have not hit the end of the receiver buffer */
	    if( !(EIStatus & EnetRxInt) )
	      {
		/* We have read past the end of the receiver buffer */
		++EnetSyncErrors;
		return( DEVICE_ERROR );
	      }
	    status = EIData0;	/* Read and discard CRC checksum */
	    ++EnetValidPackets;
	    return( ret );
	  }
	/* Otherwise we discard this packet */

	if( status & EnetCollision ) ++EnetCollisions;
	if( status & EnetOverflow ) ++EnetOverflows;
	if( status & EnetCRCerror ) ++EnetCRCerrors;
	words += 2;
	while( --words ) status = EIData0;

	status = EIData0;	/* Read the status word for the next packet.
				   */
      }

    return( END_OF_FILE );
  }


EnetFlushReceiver()

  /* Flush receiver buffer just in case of problems.
   */
  {
    register unsigned count;
    register short status;

    for( count = 0; (count < 0x4000) && (EIStatus & EnetRxInt); ++count )
      status = EIData0;
    if( count == 0x4000 ) Kabort( "Ethernet interface failure" );
  }

/* Routines specific to the distributed version */

int WriteKernelPacket( type, srcPid, dstPid, forwarder, length, localaddress,
		       remoteaddress, sequenceNo, msg, data )
short type; Process_id srcPid, dstPid, forwarder; unsigned length;
Unspec *localaddress, *remoteaddress; unsigned short sequenceNo;
unsigned short *msg; unsigned char *data;

  {
    extern ProcessId EnetWriter;
    register short i;
    register short words, lwords;
    register short buffer;
    short transmitterack;
    register short *shortdata = ( short * ) data;

    if( EnetWriter )
      {
        while( (EIStatus & EnetTxInt) == 0 ) ;
        transmitterack = EIAddress;
      }
    EnetWriter = 1;
    if( data == NULL )
	EIData0 = WORDS_PER_PACKET;
    else
      {
	words = (short) ( ( length >> 1 ) + ( length & 1 ) );
	EIData0 = (short)(WORDS_PER_PACKET + words);
      }
    EIData0 = (short)(((dstPid&0xFF000000)>>16) + EnetHostNumber);
    EIData0 = KERNEL_PACKET;
    EIData0 = type;
    EIData0 = sequenceNo;
    EIData0 = (short)(srcPid>>16);
    EIData0 = (short)(srcPid);
    EIData0 = (short)(dstPid>>16);
    EIData0 = (short)(dstPid);
    EIData0 = (short)(forwarder>>16);
    EIData0 = (short)(forwarder);
    EIData0 = 0;			/* user number */
    EIData0 = 0;
    EIData0 = (short)(length>>16);
    EIData0 = (short)(length);
    EIData0 = (short)((unsigned)localaddress>>16);
    EIData0 = (short)(localaddress);
    EIData0 = (short)((unsigned)remoteaddress>>16);
    EIData0 = (short)(remoteaddress);
    i = 5;
    if( msg == NULL )
        while( --i ) 
	  {
	    EIData0 = 0;EIData0 = 0;
	    EIData0 = 0;EIData0 = 0;
	  }
    else
	while( --i ) 
	  {
	    EIData0 = *msg++;EIData0 = *msg++;
	    EIData0 = *msg++;EIData0 = *msg++;
	  }
    if( data == NULL )
	return;
    else
      {
	lwords = words>>2;
	++lwords;
	if( ((unsigned)data & 1) == 0 )
      	  {
	    while(--lwords) 
	      {
	    	EIData0 = *shortdata++;EIData0 = *shortdata++;
	    	EIData0 = *shortdata++;EIData0 = *shortdata++;
	      }
	    if( words & 2 ) {EIData0 = *shortdata++;EIData0 = *shortdata++;}
	    if( words & 1 ) EIData0 = *shortdata++;
      	  }
        else
          {
	    while(--lwords)
	      {
	    	buffer = *data++;
	    	EIData0 = buffer<<8 | *data++; 
	    	buffer = *data++;
	    	EIData0 = buffer<<8 | *data++; 
	    	buffer = *data++;
	    	EIData0 = buffer<<8 | *data++; 
	    	buffer = *data++;
	    	EIData0 = buffer<<8 | *data++; 
	      }
	    if( words & 2 )
	      {
	    	buffer = *data++;
	    	EIData0 = buffer<<8 | *data++;
	    	buffer = *data++;
	    	EIData0 = buffer<<8 | *data++;
	      }
	    if( words & 1 )
	      {
	    	buffer = *data++;
	    	EIData0 = buffer<<8 | *data++;
	      }
          }
    	return;
      }
  }

int ReadDataPacket ( bytes, dst )
unsigned bytes; register unsigned char *dst;

  {
    register int words, lwords;
    register short *shortdst = (short *) dst;
    register short buffer;

    words = bytes >> 1;
    lwords = words >> 2;
    ++lwords;
    if( ((unsigned)dst & 1) == 0 )
      {
	while (--lwords) 
	  {
	    *shortdst++ = EIData0;*shortdst++ = EIData0;
	    *shortdst++ = EIData0;*shortdst++ = EIData0;
	  }
 	if( words & 2 ) {*shortdst++ = EIData0;*shortdst++ = EIData0;}
	if( words & 1 ) *shortdst++ = EIData0;
      } 
    else
      {
	while (--lwords)
	  {
	    buffer = EIData0;
	    *dst++ = (unsigned char)(buffer>>8);
	    *dst++ = (unsigned char)(buffer);
	    buffer = EIData0;
	    *dst++ = (unsigned char)(buffer>>8);
	    *dst++ = (unsigned char)(buffer);
	    buffer = EIData0;
	    *dst++ = (unsigned char)(buffer>>8);
	    *dst++ = (unsigned char)(buffer);
	    buffer = EIData0;
	    *dst++ = (unsigned char)(buffer>>8);
	    *dst++ = (unsigned char)(buffer);
	  }
	if( words & 2 )
	  {
	    buffer = EIData0;
	    *dst++ = (unsigned char)(buffer>>8);
	    *dst++ = (unsigned char)(buffer);
	    buffer = EIData0;
	    *dst++ = (unsigned char)(buffer>>8);
	    *dst++ = (unsigned char)(buffer);
	  }
	if( words & 1 )
	  {
	    buffer = EIData0;
	    *dst++ = (unsigned char)(buffer>>8);
	    *dst++ = (unsigned char)(buffer);
	  }
      }
    if( bytes & 1 )
      {
	if( ((unsigned)dst & 1) != 0 )
	    *dst = (unsigned char) (EIData0>>8);
	else
	  {
	    dst = (unsigned char *) shortdst;
	    *dst = (unsigned char) (EIData0>>8);
	  }
      }
    return( OK );
  }

int DiscardDataPacket ( bytes ) unsigned bytes;

  {
    register int words, lwords;
    register short status;

    words = (bytes>>1) + (bytes&1);
    lwords = words>>2;
    ++lwords;
    while (--lwords) 
      {
	status = EIData0;status = EIData0;
	status = EIData0;status = EIData0;
      }
    if( words & 2 ) {status = EIData0;status = EIData0;}
    if( words & 1 ) status = EIData0;
    return( OK );
  }

int NetCheck()
  {
    extern Process *Pd_bundle[];

    if( EthernetInstance == NULL )
      {
	/* Read packet out of interface and re-enable receiver interrupt */
	EnetReadPacket( NULL, NULL, NULL );
	EnetStatus &= ~EnetRxIntDisable;
	EIStatus = EnetStatus;
	return;
      }
    if( Map_pid( EthernetInstance->reader ) != NULL )
      {
	/* Reader is present: return immediately */
	return;
      }
    if( Map_pid( EthernetInstance->owner ) == NULL )
      {
	/* The owner has gone away: free the descriptor, read out of
	   interface and re-enable the receiver interrupt */
	EthernetInstance->owner = NULL;
	EthernetInstance = NULL;
	EnetReadPacket( NULL, NULL, NULL );
	EnetStatus &= ~EnetRxIntDisable;
	EIStatus = EnetStatus;
	return;
      }
    /* The owner is still there, reader not present: read out of interface
       and reset the alarm clock */
    ++EnetOverflows;
    EnetReadPacket( NULL, NULL, NULL );
    NetAlarm = ENET_READER_TIMEOUT;
  }	  
