/*
 * Use the sun as terminal to the serial lines (like a serial telnet).
 * No arguments will result in a connection to serial 1.
 * Otherwise the first character in the first argument indicates
 * which line is to be used (ie. 0 or 1)
 *
 * RJN 8/25/83 - Author.
 * TPM 8/26/83 - Juggled priorities some more, keeping same relative order.
 */
#include <Vioprotocol.h>
#include <Vserial.h>
#include <Vgts.h>

# define ReaderStack 500	/* small stack for reader process */
# define WriterStack 500
# define MIN( A, B )	((A) < (B) ? (A) : (B))

extern ProcessId GetPid();


/* The circular buffers */
# define BUFF_SIZE	4096
char InBuffer[BUFF_SIZE];
char OutBuffer[BUFF_SIZE];
char *InHead = &InBuffer[0], *InTail = &InBuffer[0];
char *OutHead = &OutBuffer[0], *OutTail = &OutBuffer[0];
char *InEnd = &InBuffer[BUFF_SIZE], *OutEnd = &OutBuffer[BUFF_SIZE];

WriterProcess( Head, Tail, End, To )
    register char **Head, **Tail;
    char *End;
    register File *To;
  /*
   * take the characters from the tail of the queue and output until
   * tail meets the Head or the End.  When the buffer is empty,
   * Delay for a while and try again.
   */
  {
    register int i;
    register char *TmpHead;
    Message msg;
    register unsigned EndLessBuffSize = (unsigned)End - BUFF_SIZE;
    register unsigned BlockSize = To->blocksize;
    
    do
      {
        while( *Head == *Tail ) Delay( 1, 0 );

	if ( *Head < *Tail )
	  {
	    /* go to end first */
	    while( *Tail < End )
	      {
	        i = Write( To, *Tail, MIN( End - *Tail, BlockSize ) );
		To->block++;
	        if ( i <= 0 ) break;
		*Tail += i;
	      }
	    *Tail = (char*)EndLessBuffSize;
	  }
        /* clean up the rest */
	while( *Tail < (TmpHead = *Head) )
          {
	    i = Write( To, *Tail, MIN( TmpHead - *Tail, BlockSize ) );
	    To->block++;
	    if ( i <= 0 ) break;
	    *Tail += i;
	  }
      }
    while( To->lastexception == OK );
    
    PrintError( To->lastexception, "WriterProcess" );
    
    msg[0] = 0;
    Send( msg, Creator( 0 ) );
	
  } /* WriterProcess */

ReaderProcess( Head, Tail, End, From, WriterPid )
    register char **Head, **Tail;
    char *End;
    register File *From;
    ProcessId WriterPid;
  /*
   * Read characters from From and put in the circular buffer.
   * After reading some characters, WakeUp the WriterProcess.
   * Have to make sure that there is enough room to read in
   * From->blocksize worth of bytes.  No real check for overflow
   * so the buffer will get overwritten in this event.
   */
  {
    register int i;
    register char *TmpTail;
    char junk;
    Message msg;
    register unsigned EndLessBuffSize = (unsigned)End - BUFF_SIZE - 1;
    register unsigned BlockSize = From->blocksize;
   
    do
      {
	if ( *Tail <= *Head )
	  {
	    /* go to end first */
	    while ( *Head < End )
	      {
	        i = Read( From, *Head, MIN( BlockSize, End - *Head ) );
		From->block++;
	        if ( i <= 0 ) break;
		*Head += i;
	        Wakeup( WriterPid );
	      }
	    *Head = (char*)EndLessBuffSize;
	  }
        while ( *Tail == (*Head + 1) )
	  {
	    /* dump on the floor or nearest waste receptacle */
	    Wakeup( WriterPid );
	    Delay( 0, 1 );
	  }
	if ( *Head == (char*)EndLessBuffSize ) *Head++;
	
        /* Have to account for the start */
	while ( (*Head + 1) < (TmpTail = *Tail) )
	  {
	    i = Read( From, *Head, MIN( BlockSize, TmpTail - *Head ) );
	    From->block++;
	    if ( i <= 0 ) break;
	    *Head += i;
	    Wakeup( WriterPid );
          }
      }
    while( From->lastexception == OK );
    
    PrintError( From->lastexception, "ReaderProcess" );
    
    msg[0] = 0;
    Send( msg, Creator( 0 ) );
	
  } /* ReaderProcess */

main( argc, argv )
    int argc;
    char **argv; 
 {
    SystemCode err;
    ProcessId InRdr,InWrtr,OutRdr,OutWrtr;
    File *serialread, *serialwrite;
    Message msg;
    CreateSerialInstanceRequest req;
    SystemCode error;

    /*
     * Open an instance to the serial device.
     */
    req.requestcode = CREATE_INSTANCE;
    req.filename = "Serial";
    req.filenamelen = 6;
    req.filemode = FCREATE;
    if ( argc == 1 )
         req.linenum = 1;
    else
         switch ( argv[1][0] )
	   {
	     case '0':
	         req.linenum = 0;
		 break;
	     case '1':
	         req.linenum = 1;
		 break;
             default:
	         printf( "usage: %s [line #]\n", argv[0] );
		 exit( BAD_ARGS );
		 break;
  	   }

    req.type = SERIAL;
    serialwrite = _Open(&req, FCREATE, 
			GetPid(DEVICE_SERVER, LOCAL_PID), &error);
    if (error != OK || (serialwrite == NULL))
      {
	printf("ERROR - Failed to open serial write instance: %s\n", ErrorString( error ) );
	exit( error );
      }
    
    serialread = OpenFile( GetPid(DEVICE_SERVER, LOCAL_PID), 
    		serialwrite->fileid + 1, FREAD, &error );
    if (error != OK || (serialread == NULL))
      {
	printf("ERROR - Failed to open serial read instance: %s\n", ErrorString( error ) );
	exit( error );
      }
    

    SetTeamPriority( GetPid( ACTIVE_PROCESS, LOCAL_PID ), 0 );
    InRdr = Create( 2, ReaderProcess, ReaderStack );
    InWrtr = Create( 34, WriterProcess, WriterStack ); /* don't hog CPU */
    OutRdr = Create( 1, ReaderProcess, ReaderStack );
    OutWrtr = Create( 1, WriterProcess, WriterStack );

    if ( stdout->fileserver == GetPid( VGT_SERVER, LOCAL_PID ) )
        ModifyPad( stdout,0 );

    Ready( InRdr, 5, &InHead, &InTail, InEnd, serialread, InWrtr );
    Ready( InWrtr, 4, &InHead, &InTail, InEnd, stdout );

    Ready( OutRdr, 5, &OutHead, &OutTail, OutEnd, stdin, OutWrtr );
    Ready( OutWrtr, 4, &OutHead, &OutTail, OutEnd, serialwrite );

    Receive( msg );
    DestroyProcess( InRdr );
    DestroyProcess( InWrtr );
    DestroyProcess( OutRdr );
    DestroyProcess( OutWrtr );
    Flush( stdout );

  } /* main */
