/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 */

/* This file, commands.c, is part of the V debugger, Vdb.  It contains the
 * loop for processing commands and several of the higher level routines
 * which implement commands.  This is the heart of the control structure
 * of Vdb.
 *
 * Routines included here:  CommandLoop, IncrememtPc, DecrementPc,
 * DisplayMemLoc, GetValToCompare, PrintHelp, and QueryDebugger.
 */


#include <b.out.h>
#include <Vio.h>
#include <Venviron.h>
#include "Vdb.h"

extern char GetMemByte();
extern long *FindRegister();
extern enum DbCmds ParseToGetCommand();
char *IncrementDot(), *DecrementDot();
extern ExitDebugger(), SetBreakpoint(), RemoveBreakpoint(), PrintBreakpoints(),
	GetMemLong(), PutMem(), TypeOut(), DisplayMemLoc(), ReadInputLine(),
	ParseToGetArg(), ParseForSymbol(), StackDump(),
	printnum(), ValidInstrPc(), ValidAddress();


/* Values used in the search command are "sticky"--remain the same
 * until changed by the user, and thus must be saved when control is
 * returned to the user program.
 */
long mask = ~0;			/* search mask */
long searchvalue;		/* search value */
char *lowlimit, *hilimit;	/* search limits */

char InputBuffer[80];		/* Buffer used to hold user's last input
				   line. */

char *DebuggerSymbolNames[18] =
  {
    "%d0 ", "%d1 ", "%d2 ", "%d3 ", "%d4 ", "%d5 ", "%d6 ", "%d7 ",
    "%a0 ", "%a1 ", "%a2 ", "%a3 ", "%a4 ", "%a5 ", "%fp ", "%sp ",
    "%pc ", "%sr "
  };




/*
 * Process commands until we get a "go" or "continue".
 */
CommandLoop(bpno)
    short *bpno;		/* Used for index of breakpt. set at
				   current pc, if any.  0 if none set. */
  {
    char c, go = 0;
    short nbytes = 0;
    short inc;
    char *adrs;
    char depType;
    long test;
    char searchmode;
    short bpnum;
    enum DbCmds cmd;
    int len, i;
    long arg1, arg2, arg3, arg4;
    int flag1, flag2, flag3, flag4;
    char type1;
    char symbol[5];
    struct sym *symadr;
    long *regadr;
    long savValue;
    SystemCode error;
    char *saveDot, *startDot;
    long tmask;				/* temporary mask used in search */
    char disassembledCode[MAX_CODE_LINE_LENGTH];

    SsFlag = 0;				/* out of single step mode */
    SsSbrFlag = 0;
    strcpy( disassembledCode, OutputBuffer.buf );
    TempTypeOutMode = 0;

    while (!go)
      {
	putchar('.');
	Flush(stdout);

	len = ReadInputLine(InputBuffer);
	if( len == -1 )
	  {
	    printf("Hit EOF on input too many times\n");
	    ExitDebugger();
	  }

  	cmd = ParseToGetCommand(InputBuffer, len);
	flag1 = ParseToGetArg(InputBuffer, 1, &arg1);
	type1 = SymType;
	flag2 = ParseToGetArg(InputBuffer, 2, &arg2);
	flag3 = ParseToGetArg(InputBuffer, 3, &arg3);
	flag4 = ParseToGetArg(InputBuffer, 4, &arg4);
	if ((flag1 == -1) || (flag2 == -1) || (flag3 == -1) || (flag4 == -1))
	  {
	    printf("Invalid symbol in arguments.\n");
	    continue;
	  }

	switch(cmd)
	  {
	    case NoCmd:
		break;
	    case DisplayNextCmd:
	    case DisplayPrevCmd:
	    case DisplayPtrCmd:
		if (flag1)
		  {
		    Dot = (char *) arg1;
/*		    putchar('\n');		*/
		    if (cmd == DisplayPtrCmd)
		      {
			saveDot = Dot;
			Dot = (char *) GetMemLong(Dot, &error);
			if( error != OK )
			  {
			    printf( "Value of Dot is illegal address.\n" );
			    Dot = saveDot;
			  }
		      }
		  }

		else
		  {
		    switch (cmd)
		      {
			case DisplayNextCmd:

			    Dot = IncrementDot(Dot, nbytes);
			    break;
			case DisplayPrevCmd:
			    Dot = DecrementDot(Dot, nbytes);
			    /* The default decrement of nbytes is a little
			     * weird, but as good as anything else. */
			    break;
			case DisplayPtrCmd:
			    saveDot = Dot;
			    Dot = (char *) GetMemLong(Dot, &error);
			    if( error != OK )
			     {
				printf("Content of Dot is illegal address.\n" );
			        Dot = saveDot;
			      }
			    break;
		      }
		  }

		putchar('\r');
		for (i = 0; i < 70; i++)
		    putchar(' ');
		putchar('\r');
		putchar('.');
		nbytes = DisplayMemLoc();
		if (nbytes == -1)
		  {
		    printf("Illegal address specified.\n");
		    nbytes = 0;
		  }
		break;

	    case DisplayNextPageCmd:
	    case DisplayPrevPageCmd:
		if (flag1)
		    Dot = (char *) arg1;
		if (flag2)
		    len = arg2;
		else
		    len = PAGE_LENGTH;

		if( cmd == DisplayPrevPageCmd )
		  {
		    startDot = Dot;
		    nbytes = 4;		/* Default typeout length.*/
		    for( i = 1; i < len; i++ )
		      {
			startDot = DecrementDot( startDot, nbytes );
		      }
		    Dot = startDot;
		  }

		nbytes = DisplayMemLoc();
		if (nbytes == -1)
		  {
		    printf("Illegal address specified.\n");
		    break;
		  }

		for (i = 1; i < len; i++)
		  {
		    Dot = IncrementDot(Dot, nbytes);
		    nbytes = DisplayMemLoc();
		    if (nbytes == -1)
		      {
			printf("Illegal address specified");
			break;
		      }
		  }

		if( cmd == DisplayPrevPageCmd )
		  Dot = startDot;

		break;

	    case DisplayEqCmd:
	        if (flag1)
		    printnum(arg1);
		else
		    printnum(Dot);
		putchar('\n');
		break;

	    case DisplayRegCmd:
		for (i = 0; i < 8; i++)
		  {
		    printf("%s", DebuggerSymbolNames[i]);
		    inc = printnum(SavedRegisters[i]);
		    while (inc++ < 25)
			putchar(' ');
		    printf("%s", DebuggerSymbolNames[i+8]);
		    printnum(SavedRegisters[i+8]);
		    putchar('\n');
		  }
	        putchar('\n');
		printf("%s", DebuggerSymbolNames[16]);
		printnum(ProgramCounter);
		putchar('\n');
		printf("%s", DebuggerSymbolNames[17]);
		printnum(StatusRegister);
		putchar('\n');
		break;


	    case ReplaceCmd:
		if (flag1)
		  {
		    if (flag3)
			depType = GiveTypeOutMode(arg3);
		    else
		      {
			SetTypeOutMode(arg1);
			depType = ActualTypeOutMode;
		      }
		    if( depType == STRTYPE )
		      {
			printf("Cannot replace strings, just characters.\n");
		      }
		    else
		      {
			if (flag2)
		            nbytes = PutMem(depType, arg1, arg2);
			else
			    nbytes = PutMem(depType, arg1, 0);
			if (nbytes == -1)
			    printf("Illegal address specified for replacement.\n");
		      }
		  }
		else
		    printf("No memory location specified for replacement.\n");
		break;

	    case ReplaceRegCmd:
		ParseForSymbol(InputBuffer, symbol, 5);
		regadr = FindRegister(symbol);
		if (regadr)
		  {
		    if (flag2)
		      if( ( regadr == (long *)&ProgramCounter ) && !ValidInstrPc( arg2 ) )
			printf( "Invalid address specified for pc.\n" );
		      else
			*regadr = arg2;
		    else
			*regadr = 0;
		  }
		else
		    printf("No register specified for replacement.\n");
		break;


	    case GoCmd:
	        if (flag1)
		    if (ValidInstrPc(arg1))
		        ProgramCounter = (short *) arg1;
		    else
		      {
		        printf("Invalid starting address specified.\n");
			break;
		      }
		go = 1;
		break;

	    case GoBrkptCmd:
	        if (cbptr)	/* Inside a break point? */
		    if (flag1)
		        cbptr->count = arg1;
		    else
			cbptr->count = 1;
		go = 1;
		break;


	    case SingleStepSbrCmd:
	        SsSbrFlag = 1;
		/* NO break */

	    case SingleStepCmd:
	        SsFlag = 1;
		if (flag1)
		    SsCount = arg1;
		else
		    SsCount = 1;
		go = 1;
		break;


	    case BpCmd:
	        if (flag1)
				/* Remove or set a breakpoint. */
		  {
		    if (flag2)
		      {
			if ((arg2 <= 1) || (arg2 >= BPMAX))
			  {
			    printf("Invalid breakpoint number specified.\n");
			    break;
			  }
		      }
		    else
			arg2 = ANY;
		    if (arg1 != 0)
				/* Set a breakpoint. */
		      {
			if (!ValidInstrPc(arg1))
			  {
			    printf("Invalid instruction address given.\n");
			    break;
			  }
			bpnum = SetBreakpoint(arg2, arg1);
			if (arg1 == (long) ProgramCounter)
			    *bpno = bpnum;
				/* Signal that a breakpoint was set at the
				   current pc. */
		      }
		    else	/* Remove a breakpoint. */
		      {
		        RemoveBreakpoint(arg2);
			if ((arg2 == ANY) || (arg2 == *bpno))
			    *bpno = 0;
		      }
		  }
		else		/* Display breakpoints. */
		    PrintBreakpoints();
		break;


	    case SearchCmd:
	        if (flag1)
		  {
		    searchvalue = arg1;
		    if (flag4)
			searchmode = GiveTypeOutMode(arg4);
		    else
			searchmode = type1;
		  }

		if (flag2)
		  if( ValidAddress( arg2 ) )
		    lowlimit = (char *) arg2;
		  else
		    {
		      printf( "Lower limit is invalid address.\n" );
		      break;
		    }

		if (flag3)
		  if( ValidAddress( arg3 ) )
		    hilimit = (char *) arg3;
		  else
		    {
		      printf( "Upper limit is invalid address.\n" );
		      break;
		    }

		inc = ( searchmode == CHARTYPE || 
			searchmode == BYTETYPE ||
			searchmode == STRTYPE)?    1: 2;
		tmask = mask;

		if (searchmode == STRTYPE)
		  {
		    savValue = searchvalue;
		    searchvalue = (long) (*((char *) searchvalue));
				/* searchvalue points to user-specified
				   string - get the first byte. */
		    tmask = 0xff;
				/* Mask to look just at a byte. */
		  }

		for ( adrs = lowlimit; adrs < hilimit; adrs += inc)
		  {
		    test = GetValToCompare( searchmode, adrs);
		    if ( ( test & tmask ) == ( searchvalue & tmask ) )
		      {
		        if (searchmode == 0)
			  {
			    SetTypeOutMode(adrs);
			    TypeOut(ActualTypeOutMode, adrs);
			  }
			else if (searchmode == STRTYPE)
			  {
			    int len;
			    char *memAdr;
			    char *strPtr;

			    len = strlen(savValue);
			    memAdr = adrs + 1;
			    strPtr = (char *) (savValue + 1);
			    for (i = 1; i < len; i++)
				if (*strPtr++ != GetMemByte(memAdr++,&error)
					|| error != OK )
				    break;
			    if (i == len)
				TypeOut(searchmode, adrs);
			    else
				continue;
			  }
			else
			    TypeOut( searchmode, adrs );
			putchar('\n');
		      }
		  }
		if (searchmode == STRTYPE)
		    searchvalue = savValue;
		break;


	    case MaskCmd:
	        if (flag1)
		    mask = arg1;
		else
		    mask = 0;
		break;


	    case IRadixCmd:
	        if (flag1)
		    if( arg1 > 1 && arg1 < 26L )	/* possible radix ? */
			InputRadix = (short) arg1;
		    else
			InputRadix = DEFRADIX;
		else
		    InputRadix = DEFRADIX;
		break;

	    case ORadixCmd:
	        if (flag1)
		    if( arg1 > 1 && arg1 < 26L )	/* possible radix ? */
			OutputRadix = (short) arg1;
		    else
			OutputRadix = DEFRADIX;
		else
		    OutputRadix = DEFRADIX;
		break;


	    case TmpTypeCmd:
	        if (flag1)
		  {
	            TempTypeOutMode = GiveTypeOutMode(arg1);
		    if ((TempTypeOutMode == STRTYPE) && flag2)
			StringLength = arg2;
		  }
		else
		    TempTypeOutMode = GiveTypeOutMode(0);

		break;

	    case TypeCmd:
	        if (flag1)
		  {
	            TypeOutMode = GiveTypeOutMode(arg1);
		    if ((TypeOutMode == STRTYPE) && flag2)
			StringLength = arg2;
		  }
		else
		    TypeOutMode = GiveTypeOutMode(0);
		TempTypeOutMode = 0;
		break;


	    case StackTraceCmd:
		if( flag1 )
		  StackDump( disassembledCode, arg1 );
		else
		  StackDump( disassembledCode, 0 );
		break;


	    case OffsetCmd:
		if( flag1 && arg1 > 0 )
		  LargestOffset = arg1;
		else
		  LargestOffset = DEFOFFSET;
		break;

	    case StopProcessCmd:
		StopProcessFlag = !StopProcessFlag;
		if( StopProcessFlag )
		  printf("Only the process with the exception will halt.\n");
		else
		  printf("The whole team will halt on an exception. \n");
		break;


	    case HelpCmd:
		PrintHelp();
		break;


	    case QuitCmd:
		ExitDebugger();
		break;

	    case StatsCmd:
		QueryDebugger();
		break;


	    case BadCmd:
	    default:
		printf("Command not understood.\n");
		break;
	  }
      }
  }



/*
 * Here to increment pc.
 */
char *IncrementDot( pc, inc )
	register char *pc;
	register int inc;
  {
    return( (char *) pc + inc );
  }


/*
 * Function to decrement pc. Returns pc - amount decremented.
 *	 Notice that decrementing over instructions is troublesome
 * because instructions can be 2 to 12 bytes long.  The routine
 * attempts to find the longest possible instruction that could
 * have ended at pc, unless it discovers an orb instruction (opcode 00).  In
 * that case, it continues the search, returning the length of the orb
 * only if no shorter instruction could end at pc.
 */

char *DecrementDot( pc, dec )
	register char *pc;
	register int dec;
  {
    register int nwords = 0, i;
    int legal;
    register short *pos = (short *) (wordadr( pc ) - 14);
    SystemCode error;
    int orflag = 0;
    int orwords = 0;

    if ((ActualTypeOutMode == INSTTYPE) && (dec > 0))
      {
	for( i = 6; i > 0; i--)
	  {
	    if( ValidAddress( ++pos ) )
	      {
		nwords = dasm(pos, Symfile, &legal);
						/* attempt to disassemble */
	    	if( (nwords == i) && legal )
		  if( GetMemByte( pos, &error ) == 0 )
		    {
		      /* Don't settle for or instruction if you can avoid it.*/
		      orflag = 1;
		      orwords = nwords;
		    }
		  else
		    return( (char *) pos );
	      }
	  }
	if( orflag ) return( (char *)(pos - orwords) );
      }

    return((char *) pc - dec );
  }



/*
 * Here to get data from memory for search comparison
 * NOTE: for type STRTYPE this routine just returns the first byte stored at
 * pc so that an initial comparison can be made.
 */

long GetValToCompare( type, pc )
	char *pc;
	short type;
  {
    SystemCode error;		/* pc has already been checked for validity */

    switch (type)
      {
	case 0:			/* Default search mode */
	case LONGTYPE:
	    return((long)GetMemLong( pc, &error));
	case CHARTYPE:
	case BYTETYPE:
	    return((long)GetMemByte( pc, &error));
	case WORDTYPE:
	case INSTTYPE:
	    return((long) GetInstrWord( pc, &error ));
	case STRTYPE:
	    return((long)GetMemByte( pc, &error));
      }
  }


/*
 * Print command summary.
 */

PrintHelp()
  {
    char c;

    printf("\nDebugger Commands:\n");
    printf(" expression/\n");
    printf(" expression\\    Set dot to expression and display its contents.\n");
    printf(" expression@    Set dot to expression and display the contents\n");
    printf("                of its contents.\n");
    printf(" expression=    Display value of expression.\n");
    printf(" /              Increment dot and display contents of that address.\n");
    printf(" \\              Decrement dot and display contents of that address.\n");
    printf(" @              Set dot to contents it points to and then print new contents.\n");
    printf(" =              Print value of dot.\n");
    printf(" expr,nlines,n  Display the nlines instructions which begin at expr.\n");
    printf("                Default expr is dot; default nlines is 24.\n");
    printf(" expr,nlines,p  Display the nlines instructions which end at expr.\n");
    printf("                Default expr is dot; default nlines is 24.\n");
    printf(" d              Display the contents of the registers.\n");
    printf(" s              Print a trace of subroutine calls.\n");
    printf(" expr,brknum,b  Set breakpoint brknum (2 <= brknum <= 15) at expr.\n");
    printf("                If expr is 0, clear breakpoint brknum; if omitted,\n");
    printf("                print all breakpoints.  If brknum omitted, first\n");
    printf("                available breakpoint is used.\n");
    printf(" expr,g         Execute starting at expr until exception, breakpoint,\n");
    printf("                or termination. If expr omitted, start at current pc.\n");
    printf(" expr,gb        Go past current breakpoint expr times.\n");
    printf(" expr,x         Execute the next expr instructions.\n");
    printf(" expr,y         Like x but subroutine calls are single instructions.\n");
    printf("\nType a character for more commands:");
    getchar(c);
    putchar('\n');

    printf(" type,t         Temporarily set typeout mode to type.  Type can be:\n");
    printf("                'c'--character; 'h'--hex; 'w'--word; 'l'--long;\n");
    printf("                's',strLength--string; or 'i'--instruction.  Apostrophes\n");
    printf("                are considered part of the type specification.\n");
    printf(" type,tt        Permanently set typeout mode to type.\n");
    printf(" expr1,expr2,type,r\n");
    printf("                Replace contents of expr1 with expr2.  Expr2 has type type.\n");
    printf(" register,expr,rr\n");
    printf("                Replace contents of register with expr.  Registers\n");
    printf("                are referred to as %%a0-%%a7,%%d0-%%d7,%%fp,%%sp,%%pc, or %%sr.\n");
    printf(" base,ir        Set input radix to base.\n");
    printf(" base,or        Set output radix to base.\n");
    printf(" displacement,of  Set maximum offset from a symbol to displacement.\n");
    printf(" sp             Switch from stopping the whole team at an exception to\n");
    printf("                just stopping the process with the exception, or vice versa.\n");
    printf(" pattern,lowlimit,highlimit,type,f\n");
    printf("                Find pattern in the address range lowlimit to highlimit-1.\n");
    printf("                Pattern's type is type, and a mask determines how much\n");
    printf("                of it is significant:\n");
    printf(" expr,m         Set find's mask to expr. -1,m forces a complete match;\n");
    printf("                f,m matches only the last four bits; 0,m matches anything.\n");
    printf(" h              This help command.\n");
    printf(" w              Internal debugger statistics.\n");
    printf(" q              Exit the debugger and the program. \n");
    printf("\nFurther information can be found in /ng/berglund/help/Vdb.press on diablo.\n");
  }



/*
 * Print internal statistics.
 */

QueryDebugger()
  {
    QueryCache();
    printf("CurExceptionPid %x  Origin %x  DebugeeTeamSize %x\n",
	CurExceptionPid, Origin, DebugeeTeamSize );
    printf("ProgramCounter %x\n", ProgramCounter );
    QueryMemBuffer();
  }
