
/*
 * Perform a live autopsy on the current state of the V system running on
 * this workstation.  Print out mostly all information about a process that is 
 * returned by QueryProcessState.  The user may explicitly ask for a complete
 * dump of a process' information.  
 */

#include <Vprocess.h>

#define INTRO_HDR	"Process Information for Host 0x%x:\n\n"
#define FORMAT_HDR1	"                  Team     Priority"
#define FORMAT_HDR2	"Process  Parent   Root     Net Team   State"
#define FORMAT		"0x%04x   0x%04x  0x%04x    %3d  %3d   "
#define PID_MASK	0xffff
/* main:
 *
 * If no arguments do a general display of all processes that can be
 * found by chasing down the tree in a depth first manor.  If the case
 * of arguments, expect that the user has typed in some pid's and wishes
 * to find out their status.  There isn't much error checking done 
 * on the user's input.
 */

char *ProgramName;
main( argc, argv)
  int argc;
  char *argv[];
  {
    register ProcessId headpid;
    ProcessId pid;
    ProcessId GetPid(), Creator();

    headpid = GetPid( ACTIVE_PROCESS, LOCAL_PID )&(~PID_MASK);

    ProgramName = *argv;
    if ( argc == 1)
      {
	/* Do a general process dump */
	printf( INTRO_HDR, headpid >> 16 );
	puts( FORMAT_HDR1 );
	puts( FORMAT_HDR2 );

	/* first find the head of the show */
	headpid = 0;
	while( (pid = Creator(headpid)) != 0 ) headpid = pid;

	/* how's that for a procedure name */
	PrintMeAndMySonOrBrother( headpid );
      }
    else
      while ( --argc )
	{
	  /* should do some nice user interface stuff here,
	     but I have no time for that now */
	  sscanf( *++argv, "%x", &pid );

	  /* allow the user to leave the host part of the pid blank */
	  if ( pid <= PID_MASK) pid |= headpid; 
	  PrintProcess( pid );
	}
  } /* main */

PrintMeAndMySonOrBrother( pid )
    register ProcessId pid;
  /* Do a depth first search of the process tree */
  {
    ProcessBlock pb;

    /* PrintMe */
    if ( QueryProcessState( pid, &pb ) != pid)
      {
	printf( "Invalid Process Identifier 0x%x\n", pid );
	return;
      }

    printf(FORMAT, pid&PID_MASK, pb.father&PID_MASK, pb.team_root&PID_MASK
		, pb.priority, pb.team_priority );
    PrintState( &pb );

    if ( pb.son != 0)
      {
	PrintMeAndMySonOrBrother( pb.son );
      }
    
    if ( pb.brother != 0 )
      {
	PrintMeAndMySonOrBrother( pb.brother );
      }

  } /* PrintMeAndMySonOrBrother */


PrintState( pb)
    register ProcessBlock *pb;
  /*
   * Print out the state of the process in form other than hex digits.
   * The cases are self explanatory.
   * Wishlist:
   *	-- Do all the fields in the ProcessBlock
   */
  {

    switch (pb->state)
      {
	case READY:
	    printf("Ready");
	    break;

	case RECEIVE_BLKED:
	    printf("Receive blocked");
	    if (pb->blocked_on == 0)
		printf(", nonspecific");
	    else
	        printf(" on %x",pb->blocked_on);
	    break;

	case AWAITING_REPLY:
	    printf("Awaiting reply from %x",pb->blocked_on);
	    break;

	case SEND_BLKED:
	    printf("Send blocked on %x",pb->blocked_on);
	    break;

	case AWAITING_INT:
	    printf("Awaiting interrupt");
	    break;

	case DELAYING:
	    printf("Delaying, %d clicks left",pb->blocked_on);
	    break;

	case GETPID_BLKED:
	    printf("Remote GetPid, logical id %d",pb->blocked_on);
	    break;

	case MOVEFROM_BLKED:
	    printf("Remote MoveFrom, from %x",pb->blocked_on);
	    break;

	case MOVETO_BLKED:
	    printf("Remote MoveTo, to %x",pb->blocked_on);
	    break;

	case REPLIED_TO:
	    printf("Alien, replied to by %x", pb->blocked_on);
	    break;
	    
	case FORWARDED_TO:
	    printf("Alien, forwarded to %x", pb->blocked_on);
	    break;

	default:
	    printf("Unknown state %d", pb->state );
	    break;

      }

    putchar('\n');

  } /* PrintState */


PrintProcess( pid )
    ProcessId pid;
  /*
   * Print out everything that is in the process block on the poor
   * fool's screen.  If he can decipher it all the better.
   */
  {
    ProcessBlock processblock;
    register ProcessBlock *pb = &processblock;

    if ( QueryProcessState( pid, pb ) != pid )
      {
	printf( "Invalid Process Identifier 0x%x\n", pid );
	return;
      }

    printf( "Information about Process 0x%x\n", pid );
    /* Convert Process specific info to a non-sexist format */
    printf( 
    "Process: Link 0x%x, Parent 0x%x,\n\t Sibling 0x%x, Child 0x%x\n"
    , pb->link, pb->father, pb->brother, pb->son );

    /* team fields */
    printf( "Team: Root 0x%x, Priority %d, UserPtr 0x%x, Size 0x%x\n"
	    , pb->team_root, pb->team_priority, pb->team_user, pb->team_size );

    /* Not sure what this stuff does, but regurgitate it anyway */
    printf( "Msg: Forwarded by 0x%x, Queue Head 0x%x, Queue Tail 0x%x\n"
		, pb->forwardedby, pb->msg_queue, pb->msg_queue_end );

    printf( "State: " );
    PrintState( pb );
    putchar('\n' );

  } /* PrintProcess */
