/* @(#) idprom.h 1.1@(#) Solbourne id 9/21/93 23:57:56 */
/*
 * Copyright 1988 Solbourne Computer, Inc.
 * All rights reserved.
 */

/*
 * Structure declaration for ID prom in CPU board
 *
 * Each board on Kbus contains a slot addressable identification prom.
 * Slot addressability is provided thru accesses to IO space 1. The
 * address 0x1YAAAAAA in IO space 1 addresses the board in slot Y.
 * Slots are numbered starting at slot 1 thru slot 7. Each board 
 * position also has an alternate slot address which is equal to the slot 
 * number plus 8, i.e. slot #9 is the same as slot #1. The first n bytes
 * of the idprom contain a manditory device identification field. The 
 * remainder of the idprom for each board contains optional information
 * about the board. The rom is defined at every 8th byte in the address 
 * space with the 1st idprom byte being at some byte between byte 0 and byte 7.
 * This header file describes the manditory portion of the idprom.
 */

#define IDPROM_OFFSET	8
#define MAXDEVIDSTR	129

#ifndef LOCORE
struct dev_id {
	char		dichar;			/* one char in desc string */
	unsigned char	unused[IDPROM_OFFSET-1];
};

struct idprom {
	unsigned char	optinfo:1;		/* ==1 if opt info in prom */
	char		major_type:7;		/* board major type char */
	unsigned char	unused1[IDPROM_OFFSET-1];
	unsigned char	minor_type;		/* board minor type number */
	unsigned char	unused2[IDPROM_OFFSET-1];
	char		revlevel1;		/* 1st char for rev level */
	unsigned char	unused3[IDPROM_OFFSET-1];
	char		revlevel2;		/* 2nd char for rev level */
	unsigned char	unused4[IDPROM_OFFSET-1];
	unsigned char	offset1;		/* byte of offset to next hdr */
	unsigned char	unused5[IDPROM_OFFSET-1];
	unsigned char	offset2;		/* byte of offset to next hdr */
	unsigned char	unused6[IDPROM_OFFSET-1];
	unsigned char	offset3;		/* byte of offset to next hdr */
	unsigned char	unused7[IDPROM_OFFSET-1];
	unsigned char	offset4;		/* byte of offset to next hdr */
	unsigned char	unused8[IDPROM_OFFSET-1];
	unsigned char	size1;			/* byte of size of idprom */
	unsigned char	unused9[IDPROM_OFFSET-1];
	unsigned char	size2;			/* byte of size of idprom */
	unsigned char	unused10[IDPROM_OFFSET-1];
	unsigned char	size3;			/* byte of size of idprom */
	unsigned char	unused11[IDPROM_OFFSET-1];
	unsigned char	size4;			/* byte of size of idprom */
	unsigned char	unused12[IDPROM_OFFSET-1];
	unsigned char	cksum;			/* 2's complement checksum */
	unsigned char	unused13[IDPROM_OFFSET-1];
	struct dev_id	string[MAXDEVIDSTR];	/* string describing device */
};

#define IDINDEX(field)		((int)(&((struct idprom *)0)->field))
#endif !LOCORE

/*
 * major board type values 
 */
#define SYSTEM_BOARD	'S'
#define SYSTEM_BOARD_BINFO	's'
#define GRAPHICS_BOARD	'G'
#define COPROC_BOARD	'C'
#define PROC_BOARD	'P'
#define MEMORY_BOARD	'M'
#define TY_BOARD	'T'
#define ID_MASK		0x7f	/* mask for major board type */

/*
 * minor board type values
 */
#define GRAPHICS_MONO	0x00
#define GRAPHICS_COLOR	0x01
#define GRAPHICS_CG2	0x02
#define MEMORY_16MB	0x00
#define PROC_FUJI	0x00
#define PROC_CYPRESS	0x01
#define SYSTEM_P0	0x00
#define	COPROC_KBUS_CTR	0	/* kbus counter board */
#define	COPROC_HIPPI	25	/* kcpz: hippi: antel board */

/*
 * offsets to various fields of the manditory portion of the idprom
 */
#define ID_OFF_MAJOR	0x0000
#define ID_OFF_MINOR	0x0008
#define ID_OFF_REVLEV1	0x0010
#define ID_OFF_REVLEV2	0x0018
#define ID_OFF_OFFSET1	0x0020
#define ID_OFF_OFFSET2	0x0028
#define ID_OFF_OFFSET3	0x0030
#define ID_OFF_OFFSET4	0x0038
#define ID_OFF_SIZE1	0x0040
#define ID_OFF_SIZE2	0x0048
#define ID_OFF_SIZE3	0x0050
#define ID_OFF_SIZE4	0x0058
#define ID_OFF_CKSUM	0x0060
#define ID_OFF_STRING	0x0068

/*
 * define the board rev as an integer for comparisons (lower case chars)
 */
#define	BOARD_REV(a,b)	((((a & 0x5F)) << 8) | (b & 0x5F))

#ifndef LOCORE
/*
 * Take the 4 bytes describing the offset to the next header and
 * make an offset. This is an offset from the beginning of the idprom to
 * the next header.
 */
#define IDPROM_NHDR_OFFSET(idp)	\
		(((idp)->offset1 << 24) | ((idp)->offset2 << 16) | \
		((idp)->offset3 << 8)  | ((idp)->offset4 << 0))

/*
 * Take the 4 bytes describing the size of the idprom and make a size.
 */
#define IDPROM_SIZE(idp)	\
		(((idp)->size1 << 24) | ((idp)->size2 << 16) | \
		((idp)->size3 << 8)  | ((idp)->size4 << 0))


/*
 * System board revision information (starting on DA boards... )
 * This is maintained in a PAL on the I/O board (seperate from eeprom/idprom).
 * If board has PAL then binfo major_type == SYSTEM_BOARD_BINFO.
 */
#define ID_OFF_SYSTEM_BINFO	0x4000	
#define	SYSTEM_BINFO_PN		6
struct system_binfo {
	unsigned char	major_type;
#define			sw_key		major_type
	unsigned char	unused1[IDPROM_OFFSET-1];
	struct pn {
		unsigned char	c;
		unsigned char	unused1[IDPROM_OFFSET-1];
	}		pn[SYSTEM_BINFO_PN];
	char		revlevel1;		/* 1st char real rev level */
	unsigned char	unused2[IDPROM_OFFSET-1];
	char		revlevel2;		/* 2nd char real rev level */
	unsigned char	unused3[IDPROM_OFFSET-1];
};

/*
 * System board optional idprom information
 */
#define S_SPARE1	12
#define S_SPARE2	16
#define S_SPARE3	2
#define SERIALNO_SIZE	16
#define MFGDATE_SIZE	6
#define ENETADDR_SIZE	6

#define OEM_SIZE	1
#define BANNER_SIZE	64 
#define COPYRIGHT_SIZE	64
#define LOGOBITMAP_SIZE	512


struct system_optinfo {
	unsigned char	hostid1;	/* unique host identifier */
	unsigned char	unused1[IDPROM_OFFSET-1];
	unsigned char	hostid2;	/* unique host identifier */
	unsigned char	unused2[IDPROM_OFFSET-1];
	unsigned char	hostid3;	/* unique host identifier */
	unsigned char	unused3[IDPROM_OFFSET-1];
	unsigned char	hostid4;	/* unique host identifier */
	unsigned char	unused4[IDPROM_OFFSET-1];
	unsigned char	spare1[S_SPARE1 * IDPROM_OFFSET];
	struct dev_id	serial[SERIALNO_SIZE];	/* serial number string */
	unsigned char	spare2[S_SPARE2 * IDPROM_OFFSET];
	struct dev_id	enetaddr[ENETADDR_SIZE];/* ethernet address */
	unsigned char	spare3[S_SPARE3 * IDPROM_OFFSET];
	struct dev_id	mfgdate[MFGDATE_SIZE];	/* manufacture date */

	unsigned char	oem;			/* OEM flag */
	unsigned char	unused5[IDPROM_OFFSET-1];
	struct dev_id	banner[BANNER_SIZE]; /* banner string */
	struct dev_id	copyright[COPYRIGHT_SIZE]; /* copyright string */
	unsigned char	logobitmap[LOGOBITMAP_SIZE * IDPROM_OFFSET];

};

/*
 * Concatenate the bytes describing the hostid.
 */
#define IDPROM_HOSTID(soi)	\
		(((soi)->hostid1 << 24) | ((soi)->hostid2 << 16) | \
		((soi)->hostid3 << 8)  | ((soi)->hostid4 << 0))

/*
 * Frame buffer capabilities table -- this table contains:
 *	one fb_cap_header struct
 *	res_count fb_sizes structs
 *	one fb_cap_trailer structs
 */
struct fb_cap_header {
	unsigned char	fb_type;	/* frame buffer type */
	unsigned char	unused1[IDPROM_OFFSET-1];
	unsigned char	no_of_planes;	/* number of planes */
	unsigned char	unused2[IDPROM_OFFSET-1];
	unsigned char	access_size;	/* maximum access size */
	unsigned char	unused3[IDPROM_OFFSET-1];
	unsigned char	res_count;	/* number of resolutions */
	unsigned char	unused4[IDPROM_OFFSET-1];
};
struct fb_cap_trailer {
	struct dev_id	white_value[4];	/* white value for frame buffer */
	struct dev_id	black_value[4];	/* black value for frame buffer */
	struct dev_id	fb_offset[4];	/* offset in bytes of frame buffer */
	struct dev_id	kb_offset[4];	/* offset in bytes of keyboard */
	struct dev_id	reserved[28];
};

/*
 * combine the bytes to make values
 */
#define COMBINE_TWO(ptr, field)\
		((unsigned)((((ptr)->field[0].dichar & 0xff) << 8) |\
		(((ptr)->field[1].dichar & 0xff) << 0)))
#define COMBINE_FOUR(ptr, field)\
		((unsigned)((((ptr)->field[0].dichar & 0xff) << 24) |\
		(((ptr)->field[1].dichar & 0xff) << 16) |\
		(((ptr)->field[2].dichar & 0xff) << 8) |\
		(((ptr)->field[3].dichar & 0xff) << 0)))
#define FB_CAP_TABLE_WHITE_VALUE(ptr)	COMBINE_FOUR(ptr, white_value)
#define FB_CAP_TABLE_BLACK_VALUE(ptr)	COMBINE_FOUR(ptr, black_value)
#define FB_CAP_TABLE_FB_OFFSET(ptr)	COMBINE_FOUR(ptr, fb_offset)
#define FB_CAP_TABLE_KB_OFFSET(ptr)	COMBINE_FOUR(ptr, kb_offset)

/*
 * FB Sizes data structure format
 */
struct fb_sizes {
	struct dev_id	x_res[2];	/* X resolution */
	struct dev_id	y_res[2];	/* Y resolution */
	struct dev_id	x_offset[2];	/* X offset */
	struct dev_id	fb_size[4];	/* size of frame buffer */
	struct dev_id	itbl_offset[4];	/* initialization table offset */
	struct dev_id	font_offset[4];	/* font table offset */
};

/*
 * combine the bytes to make values in the fb_sizes struct
 */
#define FB_SIZES_X_RES(ptr)		COMBINE_TWO(ptr, x_res)
#define FB_SIZES_Y_RES(ptr)		COMBINE_TWO(ptr, y_res)
#define FB_SIZES_X_OFFSET(ptr)		COMBINE_TWO(ptr, x_offset)
#define FB_SIZES_SIZE(ptr)		COMBINE_FOUR(ptr, fb_size)
#define FB_SIZES_ITBL_OFFSET(ptr)	COMBINE_FOUR(ptr, itbl_offset)
#define FB_SIZES_FONT_OFFSET(ptr)	COMBINE_FOUR(ptr, font_offset)
#endif !LOCORE
