/* @(#) socketvar.h 1.2@(#) Solbourne id 10/4/93 12:24:05 */
/*
 * Copyright 1990 Solbourne Computer, Inc.
 * All rights reserved.
 */

/*	@(#)socketvar.h 2.17 89/05/08 SMI; from UCB 7.3 12/30/87 */

/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */

#ifndef _sys_socketvar_h
#define _sys_socketvar_h

#ifdef	solbourne
#include "sys/osmp_lock.h"
#include "sys/debug.h"
#endif	solbourne
/*
 * Kernel structure per socket.
 * Contains send and receive buffer queues,
 * handle on protocol and pointer to protocol
 * private data and error information.
 */
struct socket {
	short	so_type;		/* generic type, see socket.h */
	short	so_options;		/* from socket call, see socket.h */
	short	so_linger;		/* time to linger while closing */
	short	so_state;		/* internal state flags SS_*, below */
	caddr_t	so_pcb;			/* protocol control block */
	struct	protosw *so_proto;	/* protocol handle */
/*
 * Variables for connection queueing.
 * Socket where accepts occur is so_head in all subsidiary sockets.
 * If so_head is 0, socket is not related to an accept.
 * For head socket so_q0 queues partially completed connections,
 * while so_q is a queue of connections ready to be accepted.
 * If a connection is aborted and it has so_head set, then
 * it has to be pulled out of either so_q0 or so_q.
 * We allow connections to queue up based on current queue lengths
 * and limit on number of queued connections for this socket.
 */
	struct	socket *so_head;	/* back pointer to accept socket */
	struct	socket *so_q0;		/* queue of partial connections */
	struct	socket *so_q;		/* queue of incoming connections */
	short	so_q0len;		/* partials on so_q0 */
	short	so_qlen;		/* number of connections on so_q */
	short	so_qlimit;		/* max number queued connections */
	short	so_timeo;		/* connection timeout */
	u_short	so_error;		/* error affecting connection */
	short	so_pgrp;		/* pgrp for signals */
	u_long	so_oobmark;		/* chars to oob mark */
/*
 * Variables for socket buffering.
 */
	struct	sockbuf {
		u_long	sb_cc;		/* actual chars in buffer */
		u_long	sb_hiwat;	/* max actual char count */
		u_long	sb_mbcnt;	/* chars of mbufs used */
		u_long	sb_mbmax;	/* max chars of mbufs to use */
		u_long	sb_lowat;	/* low water mark (not used yet) */
		struct	mbuf *sb_mb;	/* the mbuf chain */
		struct	proc *sb_sel;	/* process selecting read/write */
		short	sb_timeo;	/* timeout (not used yet) */
		short	sb_flags;	/* flags, see below */
	} so_rcv, so_snd;
#define	SB_MAX		(64*1024)	/* max chars in sockbuf */
#ifdef	solbourne
					/* nfs sockets may have more... */
#endif	solbourne
#define	SB_LOCK		0x01		/* lock on data queue  */
#define	SB_WANT		0x02		/* someone is waiting to lock */
#define	SB_WAIT		0x04		/* someone is waiting for data/space */
#define	SB_SEL		0x08		/* buffer is selected */
#define	SB_COLL		0x10		/* collision selecting */
/*
 * Hooks for alternative wakeup strategies.
 * These are used by kernel subsystems wishing to access the socket
 * abstraction.  If so_wupfunc is nonnull, it is called in place of
 * wakeup any time that wakeup would otherwise be called with an
 * argument whose value is an address lying within a socket structure.
 */
	struct wupalt	*so_wupalt;
#ifdef	solbourne
	slock	so_lock;
#endif	solbourne
};

struct wupalt {
	int	(*wup_func)();		/* function to call instead of wakeup */
	caddr_t	wup_arg;		/* argument for so_wupfunc */
	/*
	 * Other state information here, for example, for a stream
	 * connected to a socket.
	 */
};

/*
 * Socket state bits.
 */
#define	SS_NOFDREF		0x001	/* no file table ref any more */
#define	SS_ISCONNECTED		0x002	/* socket connected to a peer */
#define	SS_ISCONNECTING		0x004	/* in process of connecting to peer */
#define	SS_ISDISCONNECTING	0x008	/* in process of disconnecting */
#define	SS_CANTSENDMORE		0x010	/* can't send more data to peer */
#define	SS_CANTRCVMORE		0x020	/* can't receive more data from peer */
#define	SS_RCVATMARK		0x040	/* at mark on input */

#define	SS_PRIV			0x080	/* privileged for broadcast, raw... */
#define	SS_NBIO			0x100	/* non-blocking ops */
#define	SS_ASYNC		0x200	/* async i/o notify */
#define SS_PIPE			0x400	/* pipe behavior for POSIX & SVID */


/*
 * Macros for sockets and socket buffering.
 */

/* how much space is there in a socket buffer (so->so_snd or so->so_rcv) */
#define	sbspace(sb) \
    (MIN((int)((sb)->sb_hiwat - (sb)->sb_cc),\
	 (int)((sb)->sb_mbmax - (sb)->sb_mbcnt)))

/* do we have to send all at once on a socket? */
#define	sosendallatonce(so) \
    ((so)->so_proto->pr_flags & PR_ATOMIC)

/* can we read something from so? */
#define	soreadable(so) \
    ((so)->so_rcv.sb_cc || ((so)->so_state & SS_CANTRCVMORE) || \
	(so)->so_qlen || (so)->so_error)

/* can we write something to so? */
#define	sowriteable(so) \
    (sbspace(&(so)->so_snd) > 0 && \
	(((so)->so_state&SS_ISCONNECTED) || \
	  ((so)->so_proto->pr_flags&PR_CONNREQUIRED)==0) || \
     ((so)->so_state & SS_CANTSENDMORE) || \
     (so)->so_error)

/* adjust counters in sb reflecting allocation of m */
#define	sballoc(sb, m) { \
	(sb)->sb_cc += (m)->m_len; \
	(sb)->sb_mbcnt += MSIZE; \
	if ((m)->m_off > MMAXOFF) \
		(sb)->sb_mbcnt += MCLBYTES; \
}

/* adjust counters in sb reflecting freeing of m */
#define	sbfree(sb, m) { \
	(sb)->sb_cc -= (m)->m_len; \
	(sb)->sb_mbcnt -= MSIZE; \
	if ((m)->m_off > MMAXOFF) \
		(sb)->sb_mbcnt -= MCLBYTES; \
}

#ifdef	solbourne
/* set lock on sockbuf sb */
#define sblock(so, sb) { \
	int s_; \
	SO_LOCK_ASSERT(so); \
	SO_SPL(so, s_ = splimp()); \
	while ((sb)->sb_flags & SB_LOCK) {  \
		(sb)->sb_flags |= SB_WANT; \
		SO_SLEEP(so, (caddr_t)&(sb)->sb_flags, PZERO+1); \
	} \
	(sb)->sb_flags |= SB_LOCK; \
	SO_SPL(so, (void) splx(s_)); \
}

/* release lock on sockbuf sb */
#define sbunlock(so, sb) { \
	int s_; \
	SO_LOCK_ASSERT(so); \
	SO_SPL(so, s_ = splimp()); \
	(sb)->sb_flags &= ~SB_LOCK; \
	if ((sb)->sb_flags & SB_WANT) { \
		(sb)->sb_flags &= ~SB_WANT; \
		if ((so)->so_wupalt) { \
			(*(so)->so_wupalt->wup_func)(so, \
				(caddr_t)&(sb)->sb_flags, \
				(so)->so_wupalt->wup_arg); \
		} else \
			wakeup((caddr_t)&(sb)->sb_flags); \
	} \
	SO_SPL(so, (void) splx(s_)); \
}
#else	solbourne
/* set lock on sockbuf sb */
#define	sblock(so, sb) { \
	while ((sb)->sb_flags & SB_LOCK) { \
		(sb)->sb_flags |= SB_WANT; \
		(void) sleep((caddr_t)&(sb)->sb_flags, PZERO+1); \
	} \
	(sb)->sb_flags |= SB_LOCK; \
}

/* release lock on sockbuf sb */
#define	sbunlock(so, sb) { \
	(sb)->sb_flags &= ~SB_LOCK; \
	if ((sb)->sb_flags & SB_WANT) { \
		(sb)->sb_flags &= ~SB_WANT; \
		if ((so)->so_wupalt) \
			(*(so)->so_wupalt->wup_func)(so, \
			(caddr_t)&(sb)->sb_flags, \
				(so)->so_wupalt->wup_arg);\
		else \
			wakeup((caddr_t)&(sb)->sb_flags); \
	} \
}
#endif	solbourne

#define	sorwakeup(so)	sowakeup((so), &(so)->so_rcv)
#define	sowwakeup(so)	sowakeup((so), &(so)->so_snd)

#ifdef KERNEL
struct	socket *sonewconn();
#endif

#ifdef	solbourne
#ifdef	DOMAIN_LOCK
#ifdef KERNEL
extern slock domain_lock;
#endif KERNEL
#define SO_LOCK(so, vmflag) /* Note: a macro that MIGHT SET vmflag */ \
	if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
		simple_lock(&(so)->so_lock); \
	} else { \
		if (holding_lock(&domain_lock)) \
			(vmflag) = -1; \
		else { \
			simple_lock(&domain_lock); \
			vmflag = 1; \
		} \
	}

#define SO_UNLOCK(so, vmflag) \
	if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
		simple_unlock(&(so)->so_lock); \
	} else { \
		if ((vmflag) >= 0) { \
			simple_unlock(&domain_lock); \
			vmflag = -1; \
		} \
	}
#else	DOMAIN_LOCK
#define SO_LOCK(so, vmflag) /* Note: a macro that MIGHT SET vmflag */ \
	if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
		simple_lock(&(so)->so_lock); \
	} else { \
		grab_vm_context(vmflag); \
	}

#define SO_UNLOCK(so, vmflag) \
	if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
		simple_unlock(&(so)->so_lock); \
	} else { \
		drop_vm_context(vmflag); \
	}
#endif	DOMAIN_LOCK

#if NCPUS > 1
#ifdef	ASS
#   define SO_LOCK_ASSERT(so) \
	if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
		ASSERT_LOCK(&(so)->so_lock); \
	} else { \
		ASSERT_LOCK(&domain_lock); \
	}
#else	ASS
#   define SO_LOCK_ASSERT(so)
#endif	ASS
#else
#   define SO_LOCK_ASSERT(so)
#endif

#define SO_SPL(so, statement) \
	if (!((so)->so_proto->pr_flags & PR_NOSPL)) { \
		(statement); \
	}

#ifdef	DOMAIN_LOCK
#define SO_SLEEP(so, chan, pri) \
	u.u_procp->p_sock_sleep = 1; \
	if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
		sleep_and_unlock(chan, pri, &(so)->so_lock); \
		simple_lock(&(so)->so_lock); \
	} else { \
		sleep_and_unlock(chan, pri, &domain_lock); \
		simple_lock(&domain_lock); \
	} \
	u.u_procp->p_sock_sleep = 0;
#else	DOMAIN_LOCK
#define SO_SLEEP(so, chan, pri) \
	u.u_procp->p_sock_sleep = 1; \
	if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
		sleep_and_unlock(chan, pri, &(so)->so_lock); \
		simple_lock(&(so)->so_lock); \
	} else { \
		sleep_and_unlock(chan, pri, (slock_t)NULL); \
	} \
	u.u_procp->p_sock_sleep = 0;
#endif	DOMAIN_LOCK

#ifdef	DOMAIN_LOCK
#define	SO_MGET(so, mp, canwait, type)	{ \
	MGET(mp, 0, type); \
	if (((mp) == 0) && (canwait)) { \
		if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
			simple_unlock(&(so)->so_lock); \
		} else { \
			simple_unlock(&domain_lock); \
		} \
		MGET(mp, canwait, type); \
		if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
			simple_lock(&(so)->so_lock); \
		} else { \
			simple_lock(&domain_lock); \
		} \
	} \
	}
#else	DOMAIN_LOCK
#define	SO_MGET(so, mp, canwait, type)	{ \
	MGET(mp, 0, type); \
	if (((mp) == 0) && (canwait)) { \
		if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
			simple_unlock(&(so)->so_lock); \
		} \
		MGET(mp, canwait, type); \
		if ((so)->so_proto->pr_flags & PR_PARALLEL) { \
			simple_lock(&(so)->so_lock); \
		} \
	} \
	}
#endif	DOMAIN_LOCK
#endif	solbourne
#endif /*!_sys_socketvar_h*/
