; I/O System for 86-DOS version 1.10 and later. Revised 12-3-81.

; Assumes a CPU Support card at F0 hex for character I/O,
; with disk drivers for North Star controllers.

; Select whether the auxiliary port is the Support Card parallel port
; or on channel 1 of a Multiport Serial card addressed at 10H.
PARALLELAUX:	EQU	1
SERIALAUX:	EQU	0

; Select whether the printer is connected to the Support card parallel
; output port (standard) or channel 0 of a Multiport Serial card
; addressed at 10H.
PARALLELPRN:	EQU	1
SERIALPRN:	EQU	0

; If the Multiport Serial was chosen for either the auxiliary or the
; printer, select the baud rate here. Refer to Multiport Serial manual
; page 11 to pick the correct value for a given baud rate.
PRNBAUD:EQU	7		; 1200 baud
AUXBAUD:EQU	0FH		; 19200 baud

DOSLEN:	EQU	28
DOSSECT:EQU	30+2+2+8

	ORG	0
	PUT	100H

BASE:	EQU	0F0H
SIOBASE:EQU	10H
STAT:	EQU	BASE+7
DATA:	EQU	BASE+6
DAV:	EQU	2
TBMT:	EQU	1
SERIAL:	EQU	SERIALPRN+SERIALAUX
STCDATA:EQU	BASE+4		; Ports for 9513 Timer chip.
STCCOM:	EQU	BASE+5

	IF	SERIALAUX
AUXSTAT:EQU	SIOBASE+3
AUXDATA:EQU	SIOBASE+2
	ENDIF

	IF	PARALLELAUX
AUXSTAT:EQU	BASE+13
AUXDATA:EQU	BASE+12
	ENDIF

	IF	SERIALPRN
PRNSTAT:EQU	SIOBASE+1
PRNDATA:EQU	SIOBASE+0
	ENDIF

	IF	PARALLELPRN
PRNSTAT:EQU	BASE+13
PRNDATA:EQU	BASE+12
	ENDIF

	JMP	INIT
	JMP	STATUS
	JMP	INP
	JMP	OUTP
	JMP	PRINT
	JMP	AUXIN
	JMP	AUXOUT
	JMP	READ
	JMP	WRITE
	JMP	DSKCHG
	JMP	SETDATE
	JMP	SETTIME
	JMP	GETTIME

INIT:
	MOV	AL,0FFH		;Mask all interrupts
	OUT	BASE+3		;Send mask to slave
	XOR	AX,AX
	MOV	SS,AX
	MOV	SP,400H		;Set stack just below I/O system
	PUSH	CS
	POP	DS

; Initialize time-of-day clock.

	MOV	SI,STCTAB
	MOV	CX,4		;Initialize 4 registers
INITSTC:
	LODB
	OUT	STCCOM		;Select register to initialize
	LODB
	OUT	STCDATA
	LODB
	OUT	STCDATA
	LOOP	INITSTC

	IF	SERIAL
	MOV	CX,4
SERINIT:
	LODB
	OUT	SIOBASE+1
	OUT	SIOBASE+3
	LOOP	SERINIT
	LODB			;Baud rate for channel 0
	OUT	SIOBASE+8
	LODB			;Baud rate for channel 1
	OUT	SIOBASE+9
	ENDIF

; Load 86-DOS

	PUSH	DS
	MOV	AX,DOSSEG	; Set segment register for loading DOS.
	MOV	DS,AX
	XOR	BX,BX		; Offset in DOSSEG is zero.
	MOV	AL,BL		; Drive 0.
	MOV	CX,DOSLEN
	MOV	DX,DOSSECT
	CALL	READ,40H
	POP	DS

	MOV	SI,INITTAB
	CALL	0,DOSSEG
	MOV	DX,100H
	MOV	AH,26		;Set DMA address
	INT	21H
	MOV	CX,[6]		;Get size of segment
	MOV	BX,DS		;Save segment for later
;DS must be set to CS so we can point to the FCB
	MOV	AX,CS
	MOV	DS,AX
	MOV	DX,FCB		;File Control Block for COMMAND.COM
	MOV	AH,15
	INT	21H		;Open COMMAND.COM
	OR	AL,AL
	JNZ	COMERR		;Error if file not found
	XOR	AX,AX
	MOV	[FCB+33],AX	;Set 4-byte Random Record field to
	MOV	[FCB+35],AX	;   beginning of file
	INC	AX
	MOV	[FCB+14],AX	;Set record length field
	MOV	AH,39		;Block read (CX already set)
	INT	21H
	JCXZ	COMERR		;Error if no records read
	TEST	AL,1
	JZ	COMERR		;Error if not end-of-file
;Make all segment registers the same
	MOV	DS,BX
	MOV	ES,BX
	MOV	SS,BX
	MOV	SP,5CH		;Set stack to standard value
	XOR	AX,AX
	PUSH	AX		;Put zero on top of stack for return
	MOV	DX,80H
	MOV	AH,26
	INT	21H		;Set default transfer address (DS:0080)
	PUSH	BX		;Put segment on stack
	MOV	AX,100H
	PUSH	AX		;Put address to execute within segment on stack
	RET	L		;Jump to COMMAND

COMERR:
	MOV	DX,BADCOM
	MOV	AH,9		;Print string
	INT	21H
	EI
STALL:	JP	STALL

STCTAB:	DB	17H		;Select master mode register
	DW	84F3H		;Enable time-of-day
	DB	1		;Counter 1 mode register
	DW	0138H
	DB	2
	DW	0038H
	DB	3
	DW	0008H		;Set counter 3 to count days

	IF	SERIAL
	DB	0B7H, 77H, 4EH, 37H, PRNBAUD, AUXBAUD
	ENDIF

BADCOM:	DB	13,10,"Error in loading Command Interpreter",13,10,"$"
FCB:	DB	1,"COMMAND COM"
	DS	25

GETTIME:
	MOV	AL,0A7H		;Save counters 1,2,3
	OUT	STCCOM
	MOV	AL,0E0H		;Enable data pointer sequencing
	OUT	STCCOM
	MOV	AL,19H		;Select hold 1 / hold cycle
	OUT	STCCOM
	CALL	STCTIME		;Get seconds & 1/100's
	XCHG	AX,DX
	CALL	STCTIME		;Get hours & minutes
	XCHG	AX,CX
	IN	STCDATA
	MOV	AH,AL
	IN	STCDATA
	XCHG	AL,AH		;Count of days
	JP	POINTSTAT

STCTIME:
	CALL	STCBYTE
	MOV	CL,AH
STCBYTE:
	IN	STCDATA
	MOV	AH,AL
	SHR	AH
	SHR	AH
	SHR	AH
	SHR	AH
	AND	AL,0FH		;Unpack BCD digits
	AAD			;Convert to binary
	MOV	AH,AL
	MOV	AL,CL
	RET

SETTIME:
	PUSH	CX
	PUSH	DX
	CALL	LOAD0		;Put 0 into load registers to condition timer
	MOV	AL,43H		;Load counters 1 & 2
	OUT	STCCOM
	POP	DX
	POP	CX
	CALL	LOAD
	MOV	AL,43H
	OUT	STCCOM		;Load counters 1&2
	CALL	LOAD0
	MOV	AL,27H		;Arm counters 1,2,3
	OUT	STCCOM
	JP	POINTSTAT

LOAD0:
	XOR	CX,CX
	MOV	DX,CX
LOAD:
	MOV	AL,09		;Counter 1 load register
	CALL	OUTDX
	MOV	AL,0AH		;Counter 2 load register
	MOV	DX,CX
OUTDX:
	OUT	STCCOM		;Select a load register
	MOV	AL,DL
	CALL	OUTBCD
	MOV	AL,DH
OUTBCD:
	AAM			;Convert binary to unpacked BCD
	SHL	AH
	SHL	AH
	SHL	AH
	SHL	AH
	OR	AL,AH		;Packed BCD
	OUT	STCDATA
	RET

SETDATE:
	XCHG	AX,DX		;Put date in DX
	MOV	AL,0BH		;Select Counter 3 load register
	OUT	STCCOM
	XCHG	AX,DX
	OUT	STCDATA
	MOV	AL,AH
	OUT	STCDATA
	MOV	AL,44H		;Load counter 3
	OUT	STCCOM
POINTSTAT:
	PUSH	AX
	MOV	AL,1FH		;Point to status register
	OUT	STCCOM		;   so power-off glitches won't hurt
	POP	AX
	RET	L

STATUS:
	IN	STAT
	AND	AL,DAV
	RET	L

INP:
	IN	STAT
	AND	AL,DAV
	JZ	INP
	IN	DATA
	AND	AL,7FH
	RET	L

OUTP:
	PUSH	AX
OUTLP:
	IN	STAT
	AND	AL,TBMT
	JZ	OUTLP
	POP	AX
	OUT	DATA
	RET	L

PRINT:
	PUSH	AX
PRINLP:
	IN	PRNSTAT
	AND	AL,TBMT
	JZ	PRINLP
	POP	AX
	OUT	PRNDATA
	RET	L

AUXIN:
	IN	AUXSTAT
	AND	AL,DAV
	JZ	AUXIN
	IN	AUXDATA
	RET	L

AUXOUT:
	PUSH	AX
AUXLP:
	IN	AUXSTAT
	AND	AL,TBMT
	JZ	AUXLP
	POP	AX
	OUT	AUXDATA
	RET	L
;
; North Star disk controller addresses.
;
DSKSEG:	EQU	0FE80H		; `F' is for extended address modification.
WRTADR:	EQU	200H
CMMND:	EQU	300H
DRVSEL:	EQU	1H
WRTSEC:	EQU	4H
STPOFF:	EQU	8H
STPON:	EQU	9H
NOP:	EQU	10H
RSETSF:	EQU	14H
STPOUT:	EQU	1CH
STPIN:	EQU	1DH
BSTAT:	EQU	20H
RDBYTE:	EQU	40H
MOTOR:	EQU	80H
;
; Status bits.
;
TK0:	EQU	01H		; Track 0 bit.
WP:	EQU	02H		; Write protect bit.
BDY:	EQU	04H		; Data body (sync byte) found.
WRT:	EQU	08H		; Write bytes status flag.
MO:	EQU	10H		; Motor on.
SF:	EQU	80H		; Indicates sector hole was detected.
;
; Delay times in sectors for various disk functions.
;
MOTORD:	EQU	31		; Motor up-to-speed time (1 second).
HEADD:	EQU	14		; Head-load settle time.  Actually, the head
				;  doesn't require this much time to settle,
				;  but this much time is required to
				;  synchronize the sector counter.
STEPD:	EQU	2		; Step time.  One or two only.
				;  1 -> 20mS, 2 -> 40mS.
;
; Various numbers of things.
;
NSECT:	EQU	10		; 10 North Star sectors per track.
NTRACK:	EQU	35		; 35 tracks on standard SA-400 drive.
ERRLIM:	EQU	10		; Number of soft errors.
;
; READ and WRITE functions.
; AL = drive number.
; CX = Number of sectors to transfer.
; DX = Logical record number.
; DS:BX = Transfer address.
;
READ:	
	MOV	AH,1		; AH = 1 to read.
	JP	READWRITE
WRITE:
	MOV	AH,0		; AH = 0 to write.
READWRITE:
	CMP	DX,350		; See if too large a sector number is requested
	JB	SECTOROK	; Jump if OK.
	MOV	AL,0CH		; Error type C, "data error".
	STC			; Set CY flag to indicate error.
	RET	L		; Quit immediatly.
SECTOROK:
	MOV	SI,BX		; Transfer address to SI & DI.
	MOV	DI,BX
	UP			; Set direction flag for autoincrement.
	PUSH	ES		; Store extra segment.
	MOV	BX,DS		; Put data segment in extra segment.
	MOV	ES,BX
	PUSH	DS		; Save data segment.
	MOV	BX,DSKSEG	; DS is North Star controller segment.
	MOV	DS,BX
	PUSH	AX		; Store read/write flag.
	CBW			; Drive number is sixteen bits.
	MOV	BX,AX		; Put in BX.
	MOV	AX,DX		; Compute track & sector.
	MOV	DL,NSECT	; Ten sectors/track.
	DIV	AL,DL		; AL = track number, AH = sector number.
	MOV	CH,AH		; Sector number to CH.
	PUSH	CX		; Save sector number & number of sectors.
	MOV	DH,AL		; Put track number in DH.
	SEG	CS		; TRACKTAB is in the code segment.
	MOV	AH,[BX+TRACKTAB]	; Find out what the current track is.
	SEG	CS
	MOV	[BX+TRACKTAB],DH	; Update TRACKTAB.
	MOV	BP,CMMND+MOTOR+STPIN	; Assume step direction is in.
	MOV	CL,DH		; Put track number in CL.
	SUB	CL,AH		; Calculate how many steps required.
	JAE	DIRECTION	; Direction is correct if >= 0.
	DEC	BP		; Direction is out (STPOUT = STPIN-1).
	NEG	CL		; Make number of steps positive.
DIRECTION:
	IF	STEPD-1		; Multiply number of steps by two if step delay
	SAL	CL		;  is 40mS per step.
	ENDIF
	TEST	B,[CMMND+MOTOR+NOP],MO	; Turn motors on & check MO status.
	JZ	MOTORS		; If motors were off, wait for them to start.
	SEG	CS		; OLDDRIVE is in the code segment.
	CMP	BL,[OLDDRIVE]	; See if the correct drive is selected.
	JNZ	SELECT		; If wrong drive is selected, select right one.
	JP	SEEK		; Motors on, drive selected, go and step.
MOTORS:
	MOV	DL,MOTORD	; Wait for motors to come up to speed.
	CALL	WSECTOR
SELECT:
	CALL	ONESECT		; Wait for write gate to go off.
	MOV	AL,[BX+CMMND+MOTOR+DRVSEL]	; Select new drive.
	SEG	CS		; OLDDRIVE is in code segment.
	MOV	[OLDDRIVE],BL	; Update OLDDRIVE.
	MOV	DL,HEADD-1	; Full head load delay (-1 because waiting for
				;  the correct sector delays at least one more)
	MOV	AL,CL		; See if we've ever used the drive before.
	IF	STEPD-1		; Compute the actual number of steps if 40mS
	SAR	AL		;  step delay is used.
	ENDIF
	CMP	AL,NTRACK	; If the number of steps is >= NTRACK, we can't
	JAE	HEADDELAY	;  count on step time for head load delay.
	SUB	DL,CL		; Subtract stepping time.
	JB	SEEK		; Don't wait if we'll step long enough for the
				;  head to settle & the sector counter to sync.
HEADDELAY:
	CALL	WSECTOR
SEEK:
	IF	STEPD-1		; Convert back to the actual number of steps
	SAR	CL		;  rather than step time if the step time
	ENDIF			;  is 40mS per step.
	XOR	CH,CH		; CX = CL.
	JCXZ	SEEKCOMPLETE	; Jump if we're already there.
	SEG	DS		; BP normally uses stack segment.
	MOV	AL,[BP]		; Set the step direction.
	CALL	ONESECT		; Wait for the write gate to turn off.
;
; Step routine.  Step direction has already been given to the disk
; controller.  DH has destination track number.
; CX has number of sectors to step, >= 1.
; If track zero is ever reached, the head position is recalibrated using DH.
;
STEP:
	MOV	AL,[CMMND+MOTOR+NOP]	; Get `A' status.
	ROR	AL		; Track 0 bit to CF.
	JNC	STEPOK		; Recalibrate if track zero.
	MOV	CL,DH		; Track # to step count.
	JCXZ	SEEKCOMPLETE	; If destinination = 0, we're there.
	MOV	AL,[CMMND+MOTOR+STPIN]	; Set direction.
STEPOK:
	MOV	AL,[CMMND+MOTOR+STPON]
	AAM			; Waste time for > 10 uS.
	MOV	AL,[CMMND+MOTOR+STPOFF]
	MOV	DL,STEPD	; Step time (sectors).
	CALL	WSECTOR
	LOOP	STEP		; Loop till we get there.
SEEKCOMPLETE:
	POP	CX		; Restore sector number & number of sectors.
	MOV	BP,BX		; Put drive number in BP.
SECTORLOOP:
	MOV	DH,ERRLIM	; Soft error limit.
ERRORRETRY:
	DI			; Interrupts illegal till after read, write,
WAITSECTOR:			;  or error.
	CALL	ONESECT		; Wait for next sector to come by.
	MOV	AL,[CMMND+MOTOR+BSTAT+NOP]	; Get `B' status.
	AND	AL,0FH		; Mask to sector number.
	CMP	AL,CH
	JNE	WAITSECTOR	; Wait till the one we want comes by.
	POP	AX		; Get function.
	PUSH	AX		; Back on the stack for next time.
	AND	AH,AH		; AH = 1 -> read, AH = 0 -> write.
	JZ	WRITESECTOR	; Jump if write.
;
READSECTOR:
	MOV	SI,CMMND+MOTOR+RDBYTE+NOP
	PUSH	CX		; Save sector number and number of sectors.
	MOV	CX,352		; Time limit for sync byte.  352 passes through
				;  the loop @ 35 clocks/pass = 24 byte times.
RSYNCLP:
	TEST	B,[CMMND+MOTOR+NOP],BDY
	LOOPZ	RSYNCLP		; Test for sync byte. Loop till sync or timeout
	JNZ	READSECT	; Found sync byte, read data bytes.
	MOV	AL,8		; Error number 8, "Record Not Found".
	JP	ERROR
READSECT:
	MOV	CX,256		; Byte count.
	MOV	DL,CL		; CRC = 0.
READLOOP:
	AAD			; Waste time >= 7.5 uS.
	MOV	AL,[SI]		; Read a byte.
	STOB			; Store byte.
	XOR	DL,AL		; Compute CRC.
	ROL	DL
	LOOP	READLOOP	; Loop for 256 bytes.
	AAD			; Waste time >= 7.5 uS.
	MOV	AL,[SI]		; Get CRC from disk.
	CMP	AL,DL		; Same as computed?
	JE	NEXTSECTOR	; Jump if sucessful read.
	SUB	DI,256		; Back-up the index for retry.
	MOV	AL,4		; Error number 4, "CRC Error".
ERROR:
	EI			; Interrupts OK now.
	POP	CX		; Get sector number & number of sectors.
	DEC	DH		; Decrement error count.
	JNZ	ERRORRETRY	; Wait for the sector to come by again.
	POP	BX		; Pop junk off the stack.
	POP	DS		; Pop segment registers.
	POP	ES
	XOR	CH,CH		; CX is number of sectors left to read.
	STC			; Set CY flag to indicate error.
	RET	L		; Return.
;
WRITESECTOR:
	TEST	B,[CMMND+MOTOR+NOP],WP
	JZ	NOTPROT		; Jump if not protected.
	EI			; Interrupts OK now.
	POP	AX		; Pop junk off the stack.
	POP	DS
	POP	ES
	XOR	CH,CH		; CX = number of sectors left to write.
	MOV	AL,CH		; AL = 0 to indicate write protect.
	STC			; Set CY flag to indicate error.
	RET	L
NOTPROT:
	PUSH	CX		; Save sector number and number of sectors.
	MOV	AL,[CMMND+MOTOR+WRTSEC]
WWRT:
	TEST	B,[CMMND+MOTOR+NOP],WRT
	JZ	WWRT		; Loop till WRT bit goes high.
	MOV	CX,15		; Number of zeros to write.
	MOV	BX,WRTADR	; Address to write zeros.
WRTZERO:
	MOV	AL,[BX]		; Write a zero.
	AAD			; Waste time for >= 7.5 uS.
	LOOP	WRTZERO		; Write 15 of them.
	MOV	BL,0FBH		; Sync byte.
	MOV	AL,[BX]		; Write sync byte.
	AAD			; Waste time for >= 7.5 uS.
	MOV	CX,256		; Byte count.
	MOV	DL,CL		; CRC = 0.
WRTBYTE:
	SEG	ES		; Data is in extra segment.
	LODB			; Get write data.
	MOV	BL,AL		; Data to BL to write.
	MOV	AL,[BX]		; Write it.
	AAD			; Waste time for >= 7.5 uS.
	XOR	DL,BL		; Compute CRC.
	ROL	DL
	LOOP	WRTBYTE		; Write 256 bytes.
	MOV	BL,DL		; Write CRC byte.
	MOV	AL,[BX]
;
NEXTSECTOR:
	EI			; Interrupts OK now.
	POP	CX		; Get sector count.
	DEC	CL		; Decrement sector count.
	JZ	OKRETURN	; Return if done.
	INC	CH		; Increment sector number.
	CMP	CH,10		; Compare with number of sectors on track.
	JAE	NEEDSTEP
	JMP	SECTORLOOP	; Read another sector from same track.
NEEDSTEP:
	MOV	CH,0		; Reset sector number.
	CALL	ONESECT		; Wait for write gate to go off.
	MOV	AL,[CMMND+MOTOR+STPIN]
	MOV	AL,[CMMND+MOTOR+STPON]
	AAM			; Wait > 10 uS for step pulse width.
	MOV	AL,[CMMND+MOTOR+STPOFF]
	SEG	CS		; BP normally uses stack segment.
	INC	B,[BP+TRACKTAB]	; Increment the track table.
				; We don't have to wait for STEPD because
				;  waiting for the write gate to go off caused
				;  us to blow the sector and we have to wait
				;  a whole revolution anyway.
	JMP	SECTORLOOP	; Read a sector from the new track.
OKRETURN:
	POP	AX		; Get function, AH=0 -> write, AH=1 -> read.
	POP	DS		; Get original data & extra segments.
	POP	ES
	CLC			; No errors.
	RET	L
;
; Wait for sector routine.  ONESECT waits for the next sector.
; WSECTOR waits the number of sectors given by DL.
;
ONESECT:
	MOV	DL,1		; Wait for next sector.
WSECTOR:
	MOV	AL,[CMMND+MOTOR+RSETSF]
SECTLOOP:
	MOV	AL,[CMMND+MOTOR+NOP]
	TEST	AL,SF		; Check sector flag.
	JZ	SECTLOOP	; Loop till new sector.
	DEC	DL		; Decrement sector count.
	JNZ	WSECTOR		; Loop till zero.
	RET
;
DSKCHG:
	MOV	AH,0		; AH = 0 in case we don't know.
	SEG	CS
	CMP	AL,[OLDDRIVE]	; See if that's the last drive used.
	JNE	RETL		; Return if not.
	PUSH	DS		; See if the motors are still on.
	PUSH	BX
	MOV	BX,DSKSEG
	MOV	DS,BX
	TEST	B,[CMMND+NOP],MO	
	POP	BX
	POP	DS
	JZ	RETL		; Motors off, disk could be changed.
	MOV	AH,1		; If motors on, assume disk not changed.
RETL:
	RET	L
;
; Disk initialization tables.
;
INITTAB:
	DB	3		; Three drives.
	DW	DPT,DPT,DPT	; Address of disk parameter tables.
	DW	0		; Minimum buffer space
	DW	30		; Stack space
;
DPT:
	DW	256		; Sector size.
	DB	1		; One sector per allocation unit.
	DW	30		; Number of sectors allocated to system.
	DB	2		; Two allocation tables.
	DW	64		; Number of directory entries.
	DW	350		; Number of sectors on the disk.
;
; Storage locations for the disk drivers.
;
OLDDRIVE:
	DB	-1		; Which drive was used last.
TRACKTAB:
	DB	2*(NTRACK-1)+24	; Number of steps to restore the head
	DB	2*(NTRACK-1)+24	; if this drive has never been used.
	DB	2*(NTRACK-1)+24

DOSSEG:	EQU	($+15)/16+40H	; Compute segment to use for 86-DOS.
