/*
.nf
 *			  COPYRIGHT 1988
 *	    MASSACHUSETTS COMPUTER CORPORATION (MASSCOMP)
 *		       WESTFORD, MASSACHUSETTS
 *			ALL RIGHTS RESERVED.
 *
 *		      Author: Richard Carling
 *
 * THE INFORMATION IN THIS SOFTWARE IS SUBJECT TO CHANGE WITHOUT NOTICE AND
 * SHOULD NOT BE CONSTRUED AS A COMMITMENT BY MASSCOMP CORPORATION.
 * MASSCOMP MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR
 * ANY PURPOSE.  IT IS SUPPLIED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *
 * IF THE SOFTWARE IS MODIFIED IN A MANNER CREATING DERIVATIVE COPYRIGHT 
 * RIGHTS, APPROPRIATE LEGENDS MAY BE PLACED ON THE DERIVATIVE WORK IN 
 * ADDITION TO THAT SET FORTH ABOVE.
 *
 * Permission to use, copy, modify, and distribute this software and its

 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that the
 * copyright notice, and this permission notice appear in 
 * supporting documentation.
 */
 
#include <stdio.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

static Colormap colormap;
static int color_display = 0;
static long foreground, background;
static Display *display;
int fg_color = -1;		



struct DrawLine {
	unsigned short opcode;
	unsigned short length;
	double fx1, fx2, fy1, fy2;
	short x1;
	short y1;
	short x2;
	short y2;
	XColor xcolor;
} ;

ascii_loadline( fp, instruction, fw, fh )
FILE *fp;
short *instruction;
double fw, fh;
{
	char colorname[64];
	struct DrawLine *dl;
	int x1, y1, x2, y2;
	
	dl = (struct DrawLine *) instruction;
	dl->length = sizeof( struct DrawLine );
	
	/* this (too) simple parsing should be upgraded */
	
	fscanf( fp, "%d  %d  %d  %d  %s\n", &x1, &y1, &x2, &y2, colorname );

	XParseColor( display, colormap, colorname, &(dl->xcolor) );
	if (!XAllocColor( display, colormap, &(dl->xcolor) )) dl->xcolor.pixel = foreground;

	dl->x1 = x1;
	dl->y1 = y1;
	dl->x2 = x2;
	dl->y2 = y2;
	
	/* designed for color, if a B/W display, make all lines BLACK */

	if (!color_display) dl->xcolor.pixel = foreground;

	/* create scale independant information */
	
	if (fw < 2.0) fw = 2.0;
	if (fh < 2.0) fh = 2.0;

	dl->fx1 = (float)x1 / (fw - 1.);
	dl->fy1 = (float)y1 / (fh - 1.);
	dl->fx2 = (float)x2 / (fw - 1.);
	dl->fy2 = (float)y2 / (fh - 1.);
}


ascii_saveline( fp, instruction )
FILE *fp;
short *instruction;
{
	XColor xcolor;
	char colorname[64];
	struct DrawLine *dl;
	
	dl = (struct DrawLine *) instruction;
	sprintf( colorname, "#%2x%2x%2x", 
		dl->xcolor.red, dl->xcolor.green, dl->xcolor.blue );
	strip( colorname );
	
	fprintf( fp, "line %d  %d  %d  %d  %s\n", 
			dl->x1, dl->y1, dl->x2, dl->y2, colorname );
}

size_line()
{
	return sizeof( struct DrawLine );
}

add_line( instruction, x1, y1, x2, y2, color_index, fw, fh )
short *instruction;
int x1, y1, x2, y2, color_index;
double fw, fh;
{
	struct DrawLine *dl;
	
	dl = (struct DrawLine *) instruction;
	dl->length = sizeof( struct DrawLine );
	
	dl->x1 = x1;
	dl->y1 = y1;
	dl->x2 = x2;
	dl->y2 = y2;
	dl->xcolor.pixel = color_index;
	XQueryColor( display, colormap, &(dl->xcolor) );
	
	/* create scale independant information */
	
	/* make sure widget is at least 2 pixels x 2 pixels */
	if (fw < 2.0) fw = 2.0;
	if (fh < 2.0) fh = 2.0;

	dl->fx1 = (float)x1 / (fw - 1.);
	dl->fy1 = (float)y1 / (fh - 1.);
	dl->fx2 = (float)x2 / (fw - 1.);
	dl->fy2 = (float)y2 / (fh - 1.);
}

resize_line( instruction, fx, fy, fwidth, fheight ) short *instruction;
double fx, fy, fwidth, fheight;
{
	struct DrawLine *dl;

	dl = (struct DrawLine *) instruction;
	
	dl->x1 = (float) dl->fx1 * (fwidth - 1.);
	dl->y1 = (float) dl->fy1 * (fheight - 1.);
	dl->x2 = (float) dl->fx2 * (fwidth - 1.);
	dl->y2 = (float) dl->fy2 * (fheight - 1.);
}

draw_line( display, window, gc, instruction )
Display *display; Window window; GC gc;
short *instruction;
{

	struct DrawLine *dl;
	dl = (struct DrawLine *) instruction;
	if (fg_color != dl->xcolor.pixel) {
		  fg_color = dl->xcolor.pixel;
		  XSetForeground( display, gc, fg_color );
	}
	XDrawLine( display, window, gc, dl->x1, dl->y1, dl->x2, dl->y2 );
}

struct DrawRect {
	unsigned short opcode;
	unsigned short length;
	double fx, fy, fwidth, fheight;
	short x;
	short y;
	short width;
	short height;
	XColor xcolor;
} ;

ascii_loadrect( fp, instruction, fw, fh )
FILE *fp; short *instruction; 
double fw, fh;
{
	char colorname[64];
	struct DrawRect *dr;
	int x, y, width, height;
	
	dr = (struct DrawRect *) instruction;
	dr->length = sizeof( struct DrawRect );
	
	/* this (too) simple parsing will be upgraded in the next release */
	
	fscanf( fp, "%d  %d  %d  %d  %s\n", &x, &y, &width, &height, colorname );

	XParseColor( display, colormap, colorname, &(dr->xcolor) );
	if (!XAllocColor( display, colormap, &(dr->xcolor) )) dr->xcolor.pixel = foreground;

	dr->x = x;
	dr->y = y;
	dr->width = width;
	dr->height = height;


	if (!color_display) dr->xcolor.pixel = foreground;

	/* create scale independant information */
	
	if (fw < 2.0) fw = 2.0;
	if (fh < 2.0) fh = 2.0;

	dr->fx = (float)x / (fw - 1.);
	dr->fy = (float)y / (fh - 1.);
	dr->fwidth = (float)width / fw;
	dr->fheight = (float)height / fh;
}

ascii_saverect( fp, instruction )
FILE *fp; short *instruction; 
{
	char colorname[64];
	struct DrawRect *dr;

	dr = (struct DrawRect *) instruction;
	sprintf( colorname, "#%2x%2x%2x", 
		dr->xcolor.red, dr->xcolor.green, dr->xcolor.blue );
	strip( colorname );
	
	fprintf( fp, "rectangle %d  %d  %d  %d  %s\n", 
		dr->x, dr->y, dr->width, dr->height, colorname );
}


size_rect()
{
	return sizeof( struct DrawRect );
}

add_rect( instruction, x, y, width, height, color_index, fw, fh )
short *instruction;
unsigned int x, y, width, height, color_index;
double fw, fh;
{
	char colorname[64];
	struct DrawRect *dr;
	
	dr = (struct DrawRect *) instruction;
	dr->length = sizeof( struct DrawRect );
	
	dr->x = x;
	dr->y = y;
	dr->width = width;
	dr->height = height;
	dr->xcolor.pixel = color_index;
	XQueryColor( display, colormap, &(dr->xcolor) );
	
	/* create scale independant information */
	
	if (fw < 2.0) fw = 2.0;
	if (fh < 2.0) fh = 2.0;

	dr->fx = (float)x / (fw - 1.);
	dr->fy = (float)y / (fh - 1.);
	dr->fwidth = (float)width / fw;
	dr->fheight = (float)height / fh;
}


resize_rect( instruction, fx, fy, fwidth, fheight ) short *instruction;
double fx, fy, fwidth, fheight;
{
	struct DrawRect *dr;
	dr = (struct DrawRect *) instruction;
	
	dr->x =  dr->fx * fwidth;
	dr->y =  dr->fy * fheight;
	dr->width =  dr->fwidth * fwidth;
	dr->height = dr->fheight * fheight;
}



draw_rect( display, window, gc, instruction )
Display *display; Window window; GC gc; short *instruction;
{
	struct DrawRect *dr;
	
	dr = (struct DrawRect *) instruction;

	if (fg_color != dr->xcolor.pixel) {
		fg_color = dr->xcolor.pixel;
		XSetForeground( display, gc, fg_color );
	}
	XDrawRectangle( display, window, gc,
		dr->x, dr->y, dr->width, dr->height );
}





struct FillRect {
	unsigned short opcode;
	unsigned short length;
	double fx, fy, fwidth, fheight;
	short x;
	short y;
	short width;
	short height;
	XColor xcolor;
} ;


ascii_loadfrect( fp, instruction, fw, fh )
FILE *fp;
short *instruction;
double fw, fh;
{
	char colorname[64];
	struct FillRect *dfr;
	int x, y, width, height;
	
	dfr = (struct FillRect *) instruction;
	dfr->length = sizeof( struct FillRect );
	
	fscanf( fp, "%d  %d  %d  %d  %s\n", &x, &y, &width, &height, colorname );

	XParseColor( display, colormap, colorname, &(dfr->xcolor) );
	if (!XAllocColor( display, colormap, &(dfr->xcolor ))) dfr->xcolor.pixel = foreground;
	
	dfr->x = x;
	dfr->y = y;
	dfr->width = width;
	dfr->height = height;
	
	/* create scale independant information */
	
	if (fw < 2.0) fw = 2.0;
	if (fh < 2.0) fh = 2.0;

	dfr->fx = (float)x / (fw - 1.);
	dfr->fy = (float)y / (fh - 1.);
	dfr->fwidth = (float)width / fw;
	dfr->fheight = (float)height / fh;
}

ascii_savefrect( fp, instruction )
FILE *fp; short *instruction; 
{
	XColor xcolor;
	char colorname[64];
	struct FillRect *dfr;
	
	dfr = (struct FillRect *) instruction;

	sprintf( colorname, "#%2x%2x%2x", 
		dfr->xcolor.red, dfr->xcolor.green, dfr->xcolor.blue );
	strip( colorname );

	fprintf( fp, "filled_rectangle %d  %d  %d  %d  %s\n", 
		dfr->x, dfr->y, dfr->width, dfr->height, colorname );
}

size_frect()
{
	return sizeof( struct FillRect );
}

add_frect( instruction, x, y, width, height, color_index, fw, fh )
short *instruction;
unsigned int x, y, width, height, color_index;
double fw, fh;
{
	struct FillRect *dfr;
	
	dfr = (struct FillRect *) instruction;
	dfr->length = sizeof( struct FillRect );
	
	dfr->x = x;
	dfr->y = y;
	dfr->width = width;
	dfr->height = height;

	dfr->xcolor.pixel = color_index;
	XQueryColor( display, colormap, &(dfr->xcolor) );
	
	/* create scale independant information */
	
	if (fw < 2.0) fw = 2.0;
	if (fh < 2.0) fh = 2.0;

	dfr->fx = (float)x / (fw - 1.);
	dfr->fy = (float)y / (fh - 1.);
	dfr->fwidth = (float)width / fw;
	dfr->fheight = (float)height / fh;
}

resize_frect( instruction, fx, fy, fwidth, fheight )
short *instruction;
double fx, fy, fwidth, fheight;
{
	struct FillRect *dfr;
	dfr = (struct FillRect *) instruction;
	
	dfr->x =  dfr->fx * fwidth;
	dfr->y =  dfr->fy * fheight;
	dfr->width =  dfr->fwidth * fwidth;
	dfr->height = dfr->fheight * fheight;
	
}


draw_frect( display, window, gc, instruction )
Display *display; Window window; GC gc;
short *instruction;
{
	struct FillRect *dfr;

	dfr = (struct FillRect *) instruction;
	
	if (fg_color != dfr->xcolor.pixel) {
		fg_color = dfr->xcolor.pixel;
		XSetForeground( display, gc, fg_color );
	}
	if (color_display)
	    XFillRectangle( display, window, gc,
		dfr->x, dfr->y, dfr->width, dfr->height );
	else
           XDrawRectangle( display, window, gc,
		dfr->x, dfr->y, dfr->width, dfr->height );

}



#define NONE	01
#define CENTER	02
#define LEFT	04
#define RIGHT	010

struct TextInst {
	unsigned short opcode;
	unsigned short length;
	double fx, fy, fwidth, fheight;
	short justify;
	short ax;  /* actual text position */
	short ay;  /* actual text position */
	short x;
	short y;
	short width;
	short height;
	short string_width;
	short string_height;
	XColor xcolor;
/*	XtJustify  justify;   */
	XFontStruct *font;
	short strlength;
	char fontname[64];
	char string[64];
} ;

ascii_loadtext( fp, instruction, fw, fh )
FILE *fp; short *instruction;
double fw, fh;
{
	
	struct TextInst *dt;
	XFontStruct *fs;
	int x, y, width, height, color_index, strlength;
	char justify[32], colorname[64], string[64], fontname[64];
	
	dt = (struct TextInst *) instruction;
	dt->length = sizeof( struct TextInst );
	
	/* this (too) simple parsing will be upgraded in the next release */
	
	fscanf( fp, "%d  %d  %d  %d  %s  %s %s %[^\n] \n",
 		&x, &y, &width, &height, justify, colorname, fontname, string );
		
	XParseColor( display, colormap, colorname, &(dt->xcolor) );
	if (!XAllocColor( display, colormap, &(dt->xcolor) )) dt->xcolor.pixel = foreground;

	/* designed for color, sooo... make text black for B/W displays */

	if (!color_display) dt->xcolor.pixel = foreground;

	dt->x = x;
	dt->y = y;
	dt->width = width;
	dt->height = height;
	strcpy( dt->fontname, fontname );
	strcpy( dt->string, string );
	dt->strlength = strlen( string );

	dt->font = XLoadQueryFont( display, fontname );
	if (!dt->font) fprintf( stderr, "Font %s not found!\n", fontname );
	fs = dt->font;	
	
	dt->justify = 0;
	if (strcmp( justify, "none") == 0) dt->justify = NONE;
	if (strcmp( justify, "center") == 0) dt->justify = CENTER;
	if (strcmp( justify, "left") == 0) dt->justify = LEFT;
	if (strcmp( justify, "right") == 0) dt->justify = RIGHT;

	/* Calculate width and height of displayed text in pixels */
    
	dt->string_height = fs->max_bounds.ascent + fs->max_bounds.descent;
	dt->string_width = XTextWidth( fs, dt->string, (int) dt->strlength);

		
	dt->ax = x;
	dt->ay = y;

	if (dt->justify & CENTER) {
		
		/* XtJustifyCenter */
		dt->ax = (dt->width - dt->string_width) / 2;
		if (dt->ax < 0) dt->ax = 0;
		dt->ay = dt->y + (dt->height - dt->string_height) / 2
					+ fs->max_bounds.ascent;
	}
	if (dt->justify & LEFT) {
		
		/* XtJustifyLeft */
	
		dt->ax = (dt->width - dt->string_width) / 2;
		if (dt->ax < 0) dt->ax = 0;
		if (dt->ax > 4) dt->ax = 4;  /* pretty sleazy, eh */
		dt->ay = dt->y + (dt->height - dt->string_height) / 2
					+ fs->max_bounds.ascent;
	}
	if (dt->justify & RIGHT) {
		
		/* XtJustifyRight */
	
		dt->ax = dt->width - dt->string_width;
		if (dt->ax < 0) dt->ax = 0;
		dt->ay = dt->y + (dt->height - dt->string_height) / 2
					+ fs->max_bounds.ascent;
	}
	
	
	/* create scale independant information */
	
	if (fw < 2.0) fw = 2.0;
	if (fh < 2.0) fh = 2.0;

	dt->fx = (float)x / (fw - 1.);
	dt->fy = (float)y / (fh - 1.);
	dt->fwidth = (float)width / fw;
	dt->fheight = (float)height / fh;
}


ascii_savetext( fp, instruction )
FILE *fp; short *instruction;
{
	
	XColor xcolor;
	struct TextInst *dt;
	char justify[32], colorname[64], string[64], fontname[64];
	
	dt = (struct TextInst *) instruction;
	
	strcpy( justify, "none" );
	if (dt->justify & LEFT) strcpy( justify, "left" );
	if (dt->justify & RIGHT) strcpy( justify, "right" );
	if (dt->justify & CENTER) strcpy( justify, "center" );

	sprintf( colorname, "#%2x%2x%2x", 
		dt->xcolor.red, dt->xcolor.green, dt->xcolor.blue );
	strip( colorname );
	
	fprintf( fp, "text %d  %d  %d  %d  %s  %s %s %s\n",
 		dt->x, dt->y, dt->width, dt->height, 
			justify, colorname, dt->fontname, dt->string );
}



add_text( instruction, x, y, width, height, color_index, 
					justify, fontname, string, fw, fh )
short *instruction;
int x, y, width, height, color_index;
char *justify, *fontname, *string;
double fw, fh;
{
	
	XColor xcolor;
	struct TextInst *dt;
	XFontStruct *fs;
	int strlength;
	
	dt = (struct TextInst *) instruction;
	dt->length = sizeof( struct TextInst );
	
	dt->xcolor.pixel = color_index;
	XQueryColor( display, colormap, &(dt->xcolor) );

	dt->x = x;
	dt->y = y;
	dt->width = width;
	dt->height = height;

	strcpy( dt->fontname, fontname );
	strcpy( dt->string, string );
	dt->strlength = strlen( string );

	dt->font = XLoadQueryFont( display, fontname );
	if (!dt->font) fprintf( stderr, "Font %s not found!\n", fontname );
	fs = dt->font;	
	
	dt->justify = 0;
	if (strcmp( justify, "none") == 0) dt->justify = NONE;
	if (strcmp( justify, "center") == 0) dt->justify = CENTER;
	if (strcmp( justify, "left") == 0) dt->justify = LEFT;
	if (strcmp( justify, "right") == 0) dt->justify = RIGHT;

	/* Calculate width and height of displayed text in pixels */
    
	dt->string_height = fs->max_bounds.ascent + fs->max_bounds.descent;
	dt->string_width = XTextWidth( fs, dt->string, (int) dt->strlength);
		
	dt->ax = x;
	dt->ay = y;

	if (dt->justify & CENTER) {
		
		/* XtJustifyCenter */
		dt->ax = (dt->width - dt->string_width) / 2;
		if (dt->ax < 0) dt->ax = 0;
		dt->ay = dt->y + (dt->height - dt->string_height) / 2
					+ fs->max_bounds.ascent;
	}
	if (dt->justify & LEFT) {
		
		/* XtJustifyLeft */
	
		dt->ax = (dt->width - dt->string_width) / 2;
		if (dt->ax < 0) dt->ax = 0;
		if (dt->ax > 4) dt->ax = 4;  /* pretty sleazy, eh */
		dt->ay = dt->y + (dt->height - dt->string_height) / 2
					+ fs->max_bounds.ascent;
	}
	if (dt->justify & RIGHT) {
		
		/* XtJustifyRight */
	
		dt->ax = dt->width - dt->string_width;
		if (dt->ax < 0) dt->ax = 0;
		dt->ay = dt->y + (dt->height - dt->string_height) / 2
					+ fs->max_bounds.ascent;
	}
	
	
	/* create scale independant information */
	
	if (fw < 2.0) fw = 2.0;
	if (fh < 2.0) fh = 2.0;

	dt->fx = (float)x / (fw - 1.);
	dt->fy = (float)y / (fh - 1.);
	dt->fwidth = (float)width / fw;
	dt->fheight = (float)height / fh;
}



update_text_instruction( instruction, str )
short *instruction; char *str;
{
	struct TextInst *dt;

	dt = (struct TextInst *) instruction;

	dt->strlength = strlen( str );
	if (dt->strlength > 0) strcpy( dt->string, str );
	else dt->string[0] = 0;
}



size_text()
{
	return sizeof( struct TextInst );
}


resize_text( instruction, fx, fy, fwidth, fheight )
short *instruction;
double fx, fy, fwidth, fheight;
{
	struct TextInst *dt;
	XFontStruct *fs;

	dt = (struct TextInst *) instruction;
	
	dt->x =  dt->fx * fwidth;
	dt->y =  dt->fy * fheight;
	dt->width =  dt->fwidth * fwidth;
	dt->height = dt->fheight * fheight;

	dt->ax = dt->ax;
	dt->ay = dt->ay;

	fs = dt->font;	
	
	if (dt->justify & CENTER) {
		
		/* XtJustifyCenter */
		dt->ax = (dt->width - dt->string_width) / 2;
		if (dt->ax < 0) dt->ax = 0;
		dt->ay = dt->y + (dt->height - dt->string_height) / 2
					+ fs->max_bounds.ascent;
	}
	if (dt->justify & LEFT) {
		
		/* XtJustifyLeft */
	
		dt->ax = (dt->width - dt->string_width) / 2;
		if (dt->ax < 0) dt->ax = 0;
		if (dt->ax > 4) dt->ax = 4;  /* pretty sleazy, eh */
		dt->ay = dt->y + (dt->height - dt->string_height) / 2
					+ fs->max_bounds.ascent;
	}
	if (dt->justify & RIGHT) {
		
		/* XtJustifyRight */
	
		dt->ax = dt->width - dt->string_width;
		if (dt->ax < 0) dt->ax = 0;
		dt->ay = dt->y + (dt->height - dt->string_height) / 2
					+ fs->max_bounds.ascent;
	}
}


draw_text( display, window, gc, instruction )
Display *display; Window window; GC gc;
short *instruction;
{
	struct TextInst *dt;
	XGCValues values;

	dt = (struct TextInst *) instruction;
	fg_color = values.foreground = dt->xcolor.pixel;
	values.font = dt->font->fid;
	
	/* we take a hit here, aleays changing the GC to get the font 
	 * this could easily be cached, maybe next time.
	 */
	XChangeGC( display, gc, (unsigned) GCForeground | GCFont, &values );

	XDrawString( display, window, gc, dt->ax, dt->ay, dt->string, dt->strlength );
}


init_instructions( dpy ) Display *dpy;
{
	int screen;

	display = dpy; 

	/* get the default colormap and some key color values */
	
	screen = DefaultScreen( display );
	colormap = XDefaultColormap( display, screen );
	background = WhitePixel( display, screen );
	foreground = BlackPixel( display, screen );
	
	color_display = 0;
	if ( XDisplayCells( display, screen ) > 2) color_display = 1;

	/* register the display list event handlers */
	
	XtRegisterFtn( ascii_loadline, "ascii_loadline" );
	XtRegisterFtn( resize_line, "resize_line" );
	XtRegisterFtn( draw_line, "draw_line" );
	XtRegisterFtn( ascii_loadrect, "ascii_loadrect" );
	XtRegisterFtn( resize_rect, "resize_rect" );
	XtRegisterFtn( draw_rect, "draw_rect" );
	XtRegisterFtn( ascii_loadfrect, "ascii_loadfrect" );
	XtRegisterFtn( resize_frect, "resize_frect" );
	XtRegisterFtn( draw_frect, "draw_frect" );
	XtRegisterFtn( ascii_loadtext, "ascii_loadtext" );
	XtRegisterFtn( resize_text, "resize_text" );
	XtRegisterFtn( draw_text, "draw_text" );
	
	XtRegisterFtn( size_line,  "size_line" );
	XtRegisterFtn( size_rect,  "size_rect" );
	XtRegisterFtn( size_frect, "size_frect" );
	XtRegisterFtn( size_text,  "size_text" );
	
	XtRegisterFtn( add_line,  "add_line" );
	XtRegisterFtn( add_rect,  "add_rect" );
	XtRegisterFtn( add_frect, "add_frect" );
	XtRegisterFtn( add_text,  "add_text" );

	XtRegisterFtn( ascii_saveline,  "ascii_saveline" );
	XtRegisterFtn( ascii_saverect,  "ascii_saverect" );
	XtRegisterFtn( ascii_savefrect, "ascii_savefrect" );
	XtRegisterFtn( ascii_savetext,  "ascii_savetext" );
}

 
/* strip out all blanks!!!, for non-standard C compilers! */

strip( string) char *string;
{
  register char *p, *q;
  p = q = string;
  while (*q) { if (*q != ' ') *p++ = *q++; else q++; }
  *p = 0;
}

