/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

This code was derived from a file that contained the notice in
disclaimer.h.  Insofar as my modifications are concerned, I grant
the same permissions and make the same disclaimers.

Gene W. Dykes, Program of Computer Graphics, Cornell University

* * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "disclaimer.h"

/*
 * Lbl.c - Lbl widget
 */

#define XtStrlen(s)		((s) ? strlen(s) : 0)

#include <ctype.h>
#include <X11/IntrinsicP.h>
#include <X11/Xos.h>
#include <X11/StringDefs.h>
#include "LblP.h"

/*
 * Full class record constant
 */

/* Private Data */

/*
 * XtResource :
 *	name, class, type, size,
 *	offset, default_type, default_address
 *
 * These are the resources needed in addition to core resources
 */

static XtGravity	def_gravity = XtCenterGravity ;
static XtJustify	defJustify = XtJustifyCenter ;
static float		defLFactor = 1.0 ;
static int		defIHeight = 2 ;
static int		defIWidth = 2 ;
static int		def_one = 1 ;

#define offset(field) XtOffset(LblWidget, field)

static XtResource resources[] =
    {
     {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
	offset(lbl.foreground), XtRString, "Black"}
    ,{XtNfont,  XtCFont, XtRFontStruct, sizeof(XFontStruct *),
	offset(lbl.font),XtRString, "Fixed"}
    ,{XtNlabel,  XtCLabel, XtRString, sizeof(String),
	offset(lbl.lbl), XtRString, NULL}
    ,{XtNjustify, XtCJustify, XtRJustify, sizeof(XtJustify),
	offset(lbl.justify), XtRJustify, (caddr_t)&defJustify}
    ,{XtNinternalWidth, XtCWidth, XtRInt,  sizeof(Dimension),
	offset(lbl.internal_width), XtRInt, (caddr_t)&defIWidth}
    ,{XtNinternalHeight, XtCHeight, XtRInt, sizeof(Dimension),
	offset(lbl.internal_height), XtRInt, (caddr_t)&defIHeight}
    ,{XtNlineFactor, XtCFraction, XtRFloat, sizeof(float),
	offset(lbl.line_factor), XtRFloat, (caddr_t)&defLFactor}
    ,{XtNgravity, XtCGravity, XtRGravity, sizeof(XtGravity),
	offset(lbl.gravity), XtRGravity, (caddr_t)&def_gravity}
    } ;

static void Initialize() ;
static void Resize() ;
static void Redisplay() ;
static void Destroy() ;
static Boolean SetValues() ;
static XtGeometryResult QueryGeometry() ;
static void find_newlines () ;
static void SetTextWidthAndHeight () ;
static void GetnormalGC () ;

#define superclass (&widgetClassRec)

LblClassRec lblClassRec =
{
  {
  /* core_class fields */	
    /* superclass	  	*/	(WidgetClass) superclass,
    /* class_name	  	*/	"Lbl",
    /* widget_size	  	*/	sizeof(LblRec),
    /* class_initialize   	*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited       	*/	FALSE,
    /* initialize	  	*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize		  	*/	XtInheritRealize,
    /* actions		  	*/	NULL,
    /* num_actions	  	*/	0,
    /* resources	  	*/	resources,
    /* num_resources	  	*/	XtNumber(resources),
    /* xrm_class	  	*/	NULLQUARK,
    /* compress_motion	  	*/	TRUE,
    /* compress_exposure  	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest	  	*/	FALSE,
    /* destroy		  	*/	Destroy,
    /* resize		  	*/	Resize,
    /* expose		  	*/	Redisplay,
    /* set_values	  	*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus	 	*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private   	*/	NULL,
    /* tm_table		   	*/	NULL,
    /* query_geometry		*/	QueryGeometry,
  }
} ;

/* For use by clients... */
WidgetClass lblWidgetClass = (WidgetClass) &lblClassRec ;

/**
 ***
 **** Toolkit Procedures
 ***
 **/

/***** **** *** ** * Initialize * ** *** **** *****/

static void
Initialize (request, new)
    Widget request, new ;
{
LblWidget lw = (LblWidget) new ;

if (lw->lbl.lbl == NULL) 
    lw->lbl.lbl = lw->core.name ;
else
    {
    lw->lbl.lbl = strcpy( XtMalloc( strlen(lw->lbl.lbl) + 1 ),
			      lw->lbl.lbl ) ;
    }

GetnormalGC (lw) ;

lw->lbl.x = NULL ;
lw->lbl.y = NULL ;
lw->lbl.line_width = NULL ;
lw->lbl.len = NULL ;

find_newlines (lw->lbl.lbl, &lw->lbl.locations, &lw->lbl.n_lines) ;

SetTextWidthAndHeight (lw) ;

if (lw->core.width == 0)
    lw->core.width = lw->lbl.total_width + 2 * lw->lbl.internal_width ;
if (lw->core.height == 0)
    lw->core.height = lw->lbl.total_height + 2*lw->lbl.internal_height ;

(*XtClass(new)->core_class.resize) (new) ;

return ;
}

/***** **** *** ** * Redisplay * ** *** **** *****/

/*
 * Repaint the widget window
 */

static void
Redisplay (w, event, region)
    Widget w ;
    XEvent *event ;
    Region region ;
{
int i ;
LblWidget lw = (LblWidget) w ;

for (i=0;  i < lw->lbl.n_lines;  i++)
    {
    XDrawString
	(
	XtDisplay(w), XtWindow(w),
	lw->lbl.normal_GC,
	lw->lbl.x[i], lw->lbl.y[i],
	&lw->lbl.lbl[lw->lbl.locations[i]], (int) lw->lbl.len[i]
	) ;
    }
return ;
}

/***** **** *** ** * Resize * ** *** **** *****/

static void
Resize (w)
    Widget w ;
{
int i ;
LblWidget lw = (LblWidget) w ;

for (i=0;  i < lw->lbl.n_lines;  i++)
    {
    switch (lw->lbl.justify)
	{
	case XtJustifyLeft   :
	    lw->lbl.x[i] = 0 ;
	    break ;

	case XtJustifyRight  :
	    lw->lbl.x[i] = lw->lbl.total_width - lw->lbl.line_width[i] ;
	    break ;

	case XtJustifyCenter :
	    lw->lbl.x[i] = (lw->lbl.total_width - lw->lbl.line_width[i]) / 2 ;
	    break ;
	}
    switch (lw->lbl.gravity)
	{
	case XtWestGravity :
	case XtNorthWestGravity :
	case XtSouthWestGravity :
	    lw->lbl.x[i] += lw->lbl.internal_width ;
	    break ;
	case XtEastGravity :
	case XtNorthEastGravity :
	case XtSouthEastGravity :
	    lw->lbl.x[i] += ((int)lw->core.width - lw->lbl.internal_width -
			     lw->lbl.total_width) ;
	    break ;
	case XtNorthGravity :
	case XtCenterGravity :
	case XtSouthGravity :
	    lw->lbl.x[i] += ((int)lw->core.width - lw->lbl.total_width) / 2 ;
	    break ;
	}

    /* to protect against the dreaded unsigned int */
    if (lw->lbl.x[i] < 0)
	lw->lbl.x[i] = 0 ;

    lw->lbl.y[i] =
	lw->lbl.line_height * (i+1) +
	lw->lbl.line_height * (lw->lbl.line_factor - 1.0) * i -
	lw->lbl.font->max_bounds.descent ;

    switch (lw->lbl.gravity)
	{
	case XtNorthWestGravity :
	case XtNorthGravity :
	case XtNorthEastGravity :
	    lw->lbl.y[i] += lw->lbl.internal_height ;
	    break ;

	case XtWestGravity :
	case XtCenterGravity :
	case XtEastGravity :
	    lw->lbl.y[i] += ((int)lw->core.height - lw->lbl.total_height) / 2 ;
	    break ;

	case XtSouthWestGravity :
	case XtSouthGravity :
	case XtSouthEastGravity :
	    lw->lbl.y[i] += (int)lw->core.height - lw->lbl.total_height -
				lw->lbl.internal_height ;
	    break ;
	}
    }

return ;
}

/***** **** *** ** * SetValues * ** *** **** *****/

/*
 * Set specified arguments into widget
 */

/* ARGSUSED */
static Boolean
SetValues (current, request, new)
    Widget current, request, new ;
{
LblWidget curlw = (LblWidget) current ;
LblWidget reqlw = (LblWidget) request ;
LblWidget newlw = (LblWidget) new ;
Boolean need_to_redisplay = False ;
Boolean was_resized = False ;

if (newlw->lbl.lbl == NULL)
    {
    newlw->lbl.lbl = newlw->core.name ;
    }

if (curlw->lbl.lbl != newlw->lbl.lbl)
    {
    if (curlw->lbl.lbl != curlw->core.name)
	XtFree( (char *)curlw->lbl.lbl ) ;

    if (newlw->lbl.lbl != newlw->core.name)
	{
	newlw->lbl.lbl = strcpy(
		XtMalloc((unsigned) XtStrlen(newlw->lbl.lbl) + 1),
		newlw->lbl.lbl) ;
	}

    find_newlines (newlw->lbl.lbl, &newlw->lbl.locations, &newlw->lbl.n_lines) ;

    was_resized = True ;
    }

if (curlw->lbl.line_factor != newlw->lbl.line_factor)
    {
    union {int iii ; float xxx ;} uni ;
    XrmValue fromVal, toVal ;
    /*
     * This seems like a disgusting hack, but it's virtually impossible to
     * handle floats in resources...
     */
    uni.xxx = newlw->lbl.line_factor ;
    newlw->lbl.line_factor = *((float *) uni.iii) ;
    was_resized = True ;
    }

if (
   was_resized
   || curlw->lbl.font != newlw->lbl.font
   || curlw->lbl.internal_height != newlw->lbl.internal_height
   || curlw->lbl.internal_width != newlw->lbl.internal_width
   )
    {
    SetTextWidthAndHeight (newlw) ;
    (*XtClass(newlw)->core_class.resize) (new) ;
    need_to_redisplay = True ;
    }
else
if (
   curlw->lbl.justify != newlw->lbl.justify
   || curlw->lbl.gravity != newlw->lbl.gravity
   )
    {
    /* These things don't cause width and height to change */
    /* But resize does what we need for justification/gravity */
    (*XtClass(newlw)->core_class.resize) (new) ;
    need_to_redisplay = True ;
    }

if (was_resized)
    {
    newlw->core.width = newlw->lbl.total_width + 2*newlw->lbl.internal_width;
    newlw->core.height = newlw->lbl.total_height + 2*newlw->lbl.internal_height;
    }

if (curlw->core.border_width != newlw->core.border_width)
    {
    Mask changes_mask ;
    XWindowChanges changes ;

    changes_mask = CWBorderWidth ;
    changes.border_width = newlw->core.border_width ;
    if (XtIsRealized(current))
	{
	XConfigureWindow (XtDisplay(current), XtWindow(current),
				 changes_mask, &changes) ;
	}
    }

if (curlw->lbl.foreground != newlw->lbl.foreground
    || curlw->lbl.font->fid != newlw->lbl.font->fid)
    {
    XtDestroyGC (curlw->lbl.normal_GC) ;
    GetnormalGC (newlw) ;
    }

return need_to_redisplay ;
}

/***** **** *** ** * QueryGeometry * ** *** **** *****/

static XtGeometryResult
QueryGeometry (widget, requested, preferred)
    Widget widget ;
    XtWidgetGeometry *requested ;
    XtWidgetGeometry *preferred ;
{
/*
 * Examine bits in requested->request_mode
 * Evaluate preferred geometry of the widget
 * Store the result in preferred, setting bits cared about in request_mode
 *  (CWX, CWY,  CWWidth, CWHeight,  CWBorderWidth,  CWSibling,  CWStackMode)
 *
 * acceptable without modification				XtGeometryYes
 *
 * one field in requested != one field in preferred ||
 * one bit set in preferred that is not set in requested	XtGeometryAlmost
 *
 * if preferred == current					XtGeometryNo
 */

XtGeometryResult return_mode ;
LblWidget w = (LblWidget) widget ;
preferred->width = w->lbl.total_width + 2*w->lbl.internal_width ;
preferred->height = w->lbl.total_height + 2*w->lbl.internal_height ;
preferred->request_mode = (CWWidth | CWHeight) ;

if ((requested->request_mode & (CWWidth | CWHeight)) == 0)
    {
    /* parent isn't interested in anything we're interested in */
    return XtGeometryYes ;
    }

if (
    (
     ((requested->request_mode & CWWidth) != 0 &&
      preferred->width == requested->width)
		    ||
     ((requested->request_mode & CWWidth) == 0)
    )
		    &&
    (
     ((requested->request_mode & CWHeight) != 0 &&
      preferred->height == requested->height)
		    ||
     ((requested->request_mode & CWHeight) == 0)
    )
   )
    {
    /* Our values already identical to those the parent is interested in */
    return XtGeometryNo ;
    }

/*
 * That takes care of the simple cases, now we have to take a closer look...
 * I don't mind getting bigger than the smallest possible size.
 */

return_mode = XtGeometryYes ;

if (
    (requested->request_mode & CWHeight) &&
    (requested->height < preferred->height)
   )
    {
    return_mode = XtGeometryAlmost ;
    }
if (
    (requested->request_mode & CWWidth) &&
    (requested->width < preferred->width)
   )
    {
    return_mode = XtGeometryAlmost ;
    }

return return_mode ;
}

/***** **** *** ** * Destroy * ** *** **** *****/

static void
Destroy (widget)
    Widget widget ;
{
LblWidget lw = (LblWidget) widget ;
Cardinal  i ;

/*
 * free dynamically allocated data
 */

if (!XtIsRealized (widget))
    return ;

XtFree ((char *) lw->lbl.x) ;
XtFree ((char *) lw->lbl.y) ;
XtFree ((char *) lw->lbl.len) ;
XtFree ((char *) lw->lbl.line_width) ;
XtFree ((char *) lw->lbl.lbl) ;
XtFree ((char *) lw->lbl.locations) ;
XtDestroyGC (lw->lbl.normal_GC) ;

return ;
}

/**
 ***
 **** Private Procedures
 ***
 **/

/***** **** *** ** * SetTextWidthAndHeight * ** *** **** *****/

/*
 * Calculate width and height of displayed text in pixels
 */

static void
SetTextWidthAndHeight (lw)
    LblWidget lw ;
{
#define lwl	lw->lbl

int i ;
register XFontStruct	*fs = lwl.font ;

/*
 * allocate space for arrays of x, y, and len
 */

if (XtIsRealized ((Widget)lw))
    {
    if (lwl.x) XtFree ((char *) lwl.x) ;
    if (lwl.y) XtFree ((char *) lwl.y) ;
    if (lwl.len) XtFree ((char *) lwl.len) ;
    if (lwl.line_width) XtFree ((char *) lwl.line_width) ;
    }

lwl.x = (Position *) XtMalloc (lwl.n_lines * sizeof (Position)) ;
lwl.y = (Position *) XtMalloc (lwl.n_lines * sizeof (Position)) ;
lwl.len = (unsigned int *) XtMalloc (lwl.n_lines * sizeof (unsigned int)) ;
lwl.line_width = (int *) XtMalloc (lwl.n_lines * sizeof (int)) ;

/*
 * Now to find the width and height of the entire lbl
 * The height is (sum of line heights + sum of interline spacings)
 * The width is max (line widths)
 */

lwl.line_height = fs->max_bounds.ascent + fs->max_bounds.descent ;
lwl.total_height = lwl.line_height * lwl.n_lines +
		   lwl.line_height * (lwl.line_factor - 1.0) *
		   (lwl.n_lines - 1) ;

lwl.total_width = 0 ;
for (i=0;  i < lwl.n_lines;  i++)
    {
    lwl.len[i] = XtStrlen(&lwl.lbl[lwl.locations[i]]) ;
    lwl.line_width[i] = XTextWidth(fs, &lwl.lbl[lwl.locations[i]], lwl.len[i]) ;
    if (lwl.total_width < lwl.line_width[i])
	lwl.total_width = lwl.line_width[i] ;
    }

return ;
}

/***** **** *** ** * GetnormalGC * ** *** **** *****/

static void
GetnormalGC (lw)
    LblWidget lw ;
{
XGCValues	values ;

values.foreground	= lw->lbl.foreground ;
values.font		= lw->lbl.font->fid ;

lw->lbl.normal_GC = XtGetGC(
    (Widget)lw,
    (unsigned) GCForeground | GCFont,
    &values) ;
return ;
}

/***** **** *** ** * find_newlines * ** *** **** *****/

#define FIND_NEWLINES_MAX_LINES	100

static void
find_newlines (text, line_locations, n_lines)
    char *text ;
    unsigned int **line_locations ;
    unsigned int *n_lines ;
{
int string_length ;
int loc[FIND_NEWLINES_MAX_LINES] ;	/* local version of line_locations */
int n = 0 ;				/* local version of n_lines */
int i ;

loc[n++] = 0 ;				/* first line */

/*
 * check each character for a newline ('\n')
 */

string_length = strlen(text) ;
for (i=0;  i < string_length; i++)
    {
    if (text[i] == '\n')
	{
	if (n == FIND_NEWLINES_MAX_LINES)
	    {
	    char *my_text =
	    "Lbl: Too many newlines (%d) in a string passed to find_newlines\n";
	    char *etext = XtMalloc (strlen(my_text) + 6) ;
	    sprintf (etext, my_text, n) ;
	    XtWarning (etext) ;
	    }
	else
	    {
	    loc[n++] = i+1 ;		/* subsequent lines */
	    text[i] = '\0' ;
	    }
	}
    }

/* allocate space for the calling array */
*line_locations = (unsigned int *) XtMalloc (n * sizeof (unsigned int)) ;

/* Now copy the line locations to the calling array */
for (i=0;  i < n;  i++)
    {
    (*line_locations)[i] = loc[i] ;
    }

*n_lines = n ;
return ;
}


