/*
 * watch -- Simple program which watches for new mail (replaces obnoxious biff)
 * Ring bell and prints on status line (stdout if no status line) the
 * author of the mail and the subject ( "(No Subject)" if none).
 *
 * Author: Dan Heller island!argv@sun.com
 * Based on a great idea by Akkana <akkana@brain.ucsd.edu>
 *
 * Automatcially exits when user logs out.
 *
 * The program runs in background by default.  On startup, the process id of
 * the child is output to stdout.
 *
 * Options:
 *
 * -w for sunwindows (because you don't own your tty). This option need
 *    not be given if you are running suntools. If you are rlogged into another
 *    machine from a shelltool, then you will need to specify -w.
 *
 * -t displays the time whenever the status line is updated. If the timeout
 *    between updates is one minute, then the time displayed will be accurate.
 *
 * -T timeout   A timeout must be given -- this timeout is the number of
 *    seconds to wait between updates.  30 is the minimum timeout.
 *
 * -f filename  A filename must be specified. This file is read and its
 *    contents (one line) displayed on the status line (after the time, if
 *    specified).  The purpose for this flag is to allow the user to have
 *    his current working directory in the file.  Whenever the user changes
 *    directories, the current dir is put into the file, and the watch
 *    program is sent a SIGALRM and the status line is updated.  The alias
 *    and related commands to accomplish this:
 *
 *    % set watch = `watch`
 *    % alias cd 'cd \!*; pwd >! ~/file; kill -ALRM $watch'
 *
 *    The pid of the watch program is output and subsequently set in the
 *    "watch" shell variable.  You can send signals to the program by
 *    referencing the $watch variable as the process id of the program.
 *
 * -ts "string"
 * -fs "string"
 *    The strings are the escape sequences to go "to statusline" (-ts)
 *    and "from statusline" respectively.  This is provided for terminals
 *    whose termcap entries don't have this information even tho it exists
 *    or if you want to use an alternate status line (e.g. the top line of
 *    your terminal).  For example, the Wyse50's escape sequence to use the
 *    top statusline is ESC-F (^[F) -- to return would be just a carriage
 *    return (^M).  If -ts is given and -fs is missing, -fs defaults to ^M.
 *
 * -d Turns on debugging.  The program is not run in background.
 *
 * Bugs:
 *   If terminal has no status line capability, it prints to stdout which
 * can be annoying, but that's the user's fault for running it.
 *   The mail format requires the "From " line format to separate messages.
 * MMDF users can't use this program without hacking it.
 *
 * compile: cc -O -s watch.c -ltermlib -o watch
 */

#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <pwd.h>
#include <sys/time.h>
#include <ctype.h>
#include <strings.h>
#include <utmp.h>

#ifdef sun
#include <sundev/kbd.h>
#include <sundev/kbio.h>
extern char *sprintf();
#endif /* sun */

int debug, COLS = 80, iswindow, beeper, alrm();
int last_size, last_access, print_time, hs;

char *from(), *Time(), *print_dir, *username, *dir();
char spoolfile[80], bp[1024], ts[16], fs[16];
struct stat stbuf;

extern int errno;
extern char *sys_errlist[];

FILE *Tty;

char *usage = "usage: %s [-d] [-w] [-t] [-T timeout] [-f file] [-ts \"string\"] [-fs \"string\"]\n";

main(argc, argv)
char **argv;
{
    char *getenv(), *getname(), *ttyname(), *rindex(), *cmd, *tty;
    struct itimerval timer;
    int interval = 60, pid;

    if (cmd = rindex(*argv, '/'))
	cmd++;
    else
	cmd = *argv;

#ifdef sun
    if (getenv("WINDOW_PARENT"))
	iswindow = 1;
#endif /* sun */

    while (*++argv)
	if (!strcmp(*argv, "-d"))
	    debug = 1;
	else if (!strcmp(*argv, "-T")) {
	    if (!*++argv) {
		printf("%s: -T requires time argument of >= 30 seconds.\n",cmd);
		exit(1);
	    }
	    if ((interval = atoi(*argv)) < 30 && !debug)
		printf("Minimum time interval is %d seconds.\n", interval = 30);
	} else if (!strcmp(*argv, "-w"))
	    iswindow = 1;
	else if (!strcmp(*argv, "-t"))
	    print_time = 1;
	else if (!strcmp(*argv, "-f"))
	    if (!*++argv)
		printf("%s: -f requires filename argument.\n", cmd), exit(1);
	    else
		print_dir = *argv;
	else if (!strcmp(*argv, "-ts")) {
	    if (!*++argv) {
		printf("%s: -ts requires 'to statusline' sequence.\n", cmd);
		exit(1);
	    }
	    (void) strcpy(ts, *argv);
	    (void) strcpy(fs, "\012");
	} else if (!strcmp(*argv, "-fs")) {
	    if (!*++argv) {
		printf("%s: -ts requires 'from statusline' sequence.\n", cmd);
		exit(1);
	    }
	    (void) strcpy(fs, *argv);
	} else
	    fprintf(stderr, usage, cmd), exit(1);

    if (fs[0] && !ts[0])
	printf("%s: -fs requires the -ts option be specified as well.\n", cmd);

    username = getname();
    init_cadr(ts[0] != 0);
    beeper = beep();

    (void) sprintf (spoolfile, "/usr/spool/mail/%s", username);

    /* get the tty name */
    if (isatty(fileno(stderr)))
	tty = ttyname(fileno(stderr));
    else if (isatty(fileno(stdout)))
	tty = ttyname(fileno(stdout));
    else if (isatty(fileno(stdin)))
	tty = ttyname(fileno(stdin));
    else
	fprintf(stderr, "You don't have to redirect *everything*!\n"), exit(1);

    if (!debug && (pid = fork())) {	/* Parent goes into the background */
	printf("%d\n", pid);  /* print the pid of the child */
	fflush(stdout);
	exit(0);
    }
    (void) close(fileno(stdout));
    (void) close(fileno(stderr));
    sleep(1);	/* sleep to give parent a chance to print pid */

    if (!stat (spoolfile, &stbuf)) {
	last_size = stbuf.st_size;
	last_access = stbuf.st_atime;
    }

    timer.it_value.tv_sec = interval;
    timer.it_value.tv_usec = 0;
    timer.it_interval.tv_sec = interval;
    timer.it_interval.tv_usec = 0;

    (void) signal(SIGALRM, alrm);
    setitimer(ITIMER_REAL, &timer, 0);
    alrm();

    if (!(Tty = fopen(tty, "w")))
	exit(1);
    tty += 5; /* get rid of "/dev/" */
    alrm(); /* initialize */
    while (still_logged_in(tty))
	pause();
}

alrm()
{
    char buf[256], mail_msg[256];
    register char *dir_p = "", *time_p = "";
    char *fromp;
    static unread;
    int do_print = 0;

    /* if there's mail, set do_print to guarantee dir() to give the right
     * information. read mail into tmp buffer.
     */
    *mail_msg = 0;
    if (!stat(spoolfile, &stbuf)) {
	if (stbuf.st_size > last_size && *(fromp = from(last_size))) {
	    do_print = 1;
	    bell(&beeper);
	    (void) sprintf(mail_msg, "New mail %s", fromp);
	    unread = 2;
	} else if (stbuf.st_atime > last_access) {
	    last_access = stbuf.st_atime;
	    if (hs && unread)
		do_print = 1;
	    unread = 0;
	} else if (unread) {
	    if (unread > 1 && hs)
		unread = do_print = 1;
	    (void) strcpy(mail_msg, "(mail)");
	}
	last_size = stbuf.st_size;
    }

    if (print_time)
	time_p = Time(), do_print = 1;

    if (print_dir)
	dir_p = dir(&do_print);

    if (do_print) {
	(void) sprintf(buf, "%s%s%s %s%s", ts, time_p, dir_p, mail_msg, fs);
	fputs(buf, stdout), fflush(stdout);
    }
}

char *
dir(force)
int *force;
{
    FILE *fp;
    static char buf[128], buf2[128];
    char *p, *index();

    (void) strcpy(buf2, buf);
    if (!(fp = fopen(print_dir, "r")))
	return NULL;
    p = fgets(buf, sizeof buf, fp);
    fclose(fp);
    if (!p || !*force && !strcmp(buf, buf2))
	return NULL;
    *force = 1;
    if (p = index(buf, '\n'))
	*p = 0;
    return buf;
}

bell(ding)
register int *ding;
{
#ifdef KIOCCMD
    if (*ding != -1) {
  	int cmd = KBD_CMD_BELL;
        struct timeval timer;
        timer.tv_sec = 0;
        timer.tv_usec = 100000;
        if (ioctl(*ding, KIOCCMD, &cmd))
	    *ding = -1;
        select(32, 0,0,0, &timer);
        cmd = KBD_CMD_NOBELL;
        if (ioctl(*ding, KIOCCMD, &cmd))
	    *ding = -1;
    } else
#endif /* KIOCCMD */
	putchar(7), fflush(stdout);
}

/* return username: username doesn't have to be declared static cuz it
 * points to static data
 */
char *
getname()
{
    char *username, *getlogin();

    if (!(username = getlogin())) {
    	struct passwd *getpwuid(), *pwent;
    	pwent = getpwuid(getuid());
    	username = pwent->pw_name;
	endpwent(); /* close the passwd file */
    }
    return username;
}

/* Explanation of this silly routine:
 * return the fd of /dev/kbd if we're not rlogged in or from a pseudo-tty.
 * if return -1, bell() will just putchar(7), else it'll do an ioctl
 * to /dev/kbd to ring the bell on fd "kbd"
 */
beep()
{
    int kbd = -1;
#ifdef sun
    char *getenv(), *p = getenv("TERM");
    if (!iswindow && p && !strcmp(p, "sun")) {
	kbd = open("/dev/kbd", O_WRONLY, 0);
	if (debug)
	    fprintf(Tty, "%d\n", kbd);
    }
#endif /* sun */
    return kbd;
}

init_cadr(ts_known)
{
    char *getenv(), *tgetstr(), *termname;
    char *tmp[1];

    if (ts_known) {
	termname = getenv("TERM");
	COLS = 80;
	return;
    }

    if (!(termname = getenv("TERM"))) {
       puts("No terminal type!");
       goto no_type;
    }
    if (tgetent(bp, termname) <= 0) {
	perror(termname);
	COLS = 80;
    } else
	COLS = tgetnum("co");

    if (debug)
	fprintf(Tty, "Terminal type = \"%s\"\n", termname);
    if (hs = tgetflag("hs")) {
        tmp[0] = ts;
        tgetstr("ts", tmp);
        tmp[0] = fs;
        tgetstr("fs", tmp);
#ifdef sun
    } else if (!strncmp(termname, "sun", 3) || iswindow) {
	struct ttysize win;
	if (!ioctl(0, TIOCGSIZE, &win))
	    COLS = win.ts_cols;
	/* if iswindow is false, user is not running suntools */
	if (iswindow || !strncmp(ttyname(0), "/dev/ttyp", 9)) {
	    iswindow = hs = 1;
	    (void) strcpy(ts, "\033]l");
	    (void) strcpy(fs, "\033\\");
	}
#endif /* sun */
    } else if (!strncmp(termname, "wy", 2)) { /* bad hack for wyse's */
	(void) strcpy(ts, "\033f");
	(void) strcpy(fs, "\r");
	hs = 1;
    } else {
no_type:
	puts("No status line capability -- output to stdout");
	(void) strcpy(ts, " ");
	(void) strcpy(fs, "\r\n");
    }
    if (debug)
	bell(&beeper), fprintf(Tty, "%sThis is a test!%s", ts, fs);
}

char *
Time()
{
    static char time_buf[11];
    struct tm *T;
    long x;

    time(&x), T = localtime(&x);
    return sprintf(time_buf, "%d:%02d ",
	(T->tm_hour) ? ((T->tm_hour <= 12) ? T->tm_hour : T->tm_hour - 12) : 12,
	T->tm_min);
}

char *
from(offset)
register long offset;
{
    static char buf[256];
    char buf2[80];
    FILE *fp;
    register char *p;
    struct timeval times[2];
    int nomatch = 1;

    (void) strcpy(buf, "From \"unknown\"");
    if (!(fp = fopen(spoolfile, "r")) || fseek(fp, offset, 0)) {
	(void) strcpy(buf, sys_errlist[errno]);
	goto the_end;
    }
    while (fgets(buf, sizeof(buf), fp) && (nomatch = strncmp(buf, "From ", 5)))
	;
    if (nomatch) {
	buf[0] = 0;
	goto the_end;
    }

    if (p = index(buf+5, ' '))
	*p = 0;
    else
	goto the_end;
    p += strlen(strcpy(p, " -> "));

    while (fgets(buf2, sizeof(buf2), fp))
	if (*buf2 == '\n') {
	    (void) sprintf(p, " (No Subject)");
	    break;
	} else if (sscanf(buf2, "Subject: %[^\n]s", p))
	    break;
the_end:
    fclose(fp);
    times[0].tv_sec = stbuf.st_atime;
    times[0].tv_usec = 0;
    times[1].tv_sec = stbuf.st_mtime;
    times[1].tv_usec = 0;
    if (utimes(spoolfile, times))
	perror("utime");

    return buf;
}

/*
 * return 1 if user is still logged in this tty.  0 otherwise.
 * It could be that the user logs out and then back in on the
 * same tty before this gets called again (especially if you're
 * running in a window-based environment like suns or X windows).
 */
still_logged_in(tty)
char *tty;
{
    struct utmp buf;
    static int fd;
    static long pos;

    if (!fd && (fd = open("/etc/utmp", 0)) == -1)
	return 0;

    (void) lseek(fd, pos, 0);
    while (read(fd, (char *) &buf, sizeof buf) == sizeof buf)
	if (buf.ut_line[0] && !strcmp(buf.ut_line, tty))
	    break;
	else {
	    if (debug && buf.ut_line[0])
		fprintf(Tty, "%s <-> %s\n", buf.ut_line, tty);
	    pos += sizeof buf;
	}
    /* it's impossible to exit this loop without making a match */
    return buf.ut_name[0] != 0;
}
