/*
 * Author:  Dan Heller <island!argv@sun.com>
 * This code was stolen from Mailbox.c --the widget which supports "xbiff"
 * written by Jim Fulton which was apparently stolen from Clock.c, the widget
 * which supports "xclock."  Note, you are witnessing the big bang theory of
 * software development (everything is a subclass of universeWidgetClass).
 *
 * Major changes:
 * XtRemoveTimeOut() is called before calling XtAddTimeOut().  The old
 * mailbox would eventually timeout all the time rather than every 30 seconds
 * because the old timer was never removed.
 *
 * User can specify any icon he chooses for either the up flag or the down
 * flag.  Icons don't need to be the same size (defaults to flagup/flagdown).
 *
 * When new mail comes in, all new messages are scanned for author and subject
 * and are stored in the "mail" field of the mailwatch data structure.  This
 * data is passed as the call_data to a user supplied callback function.
 *
 * The mailbox flag goes up when there is new mail _and_ the user hasn't
 * read it yet.  As soon as the user updates the access time on the mailbox,
 * the flag goes down.  This removes the incredibly annoying habit xbiff 
 * had where you read some mail but not delete it from the mailbox and the
 * flag would remain up.
 *
 * Left button causes the widget to check for new mail.  Middle button
 * clears the flag and lets it think there is no new mail (reset).
 *
 * Destroy() will now destroy the flagup and flagdown pixmaps.
 *
 * BUGS:
 *    For Mailwatch to return any information to the callback routine(s),
 * it requires new mail be separated by the common usage of the "From "
 * string as the first 5 chars of a mail message.  This is how most mailers
 * deposit mail in /usr/spool/mail.  Sorry MMDF users.
 *
 * It is assumed that when the user starts up this widget, he knows what
 * his mail status is (how much he's got, etc).  Only new mail arrived
 * after the widget is created is accounted for.
 */
#include <X11/IntrinsicP.h>	/* for toolkit stuff */
#include <X11/Xos.h>
#include <X11/cursorfont.h>	/* for cursor constants */
#include <X11/StringDefs.h>	/* for useful atom names */
#include "MailwatchP.h"	/* for implementation mailbox stuff */
#include <stdio.h>	/* for printing error messages */
#include <sys/stat.h>	/* for stat() */
#include <pwd.h>	/* for getting username */
#include <X11/bitmaps/flagup>	/* for flag up (mail present) bits */
#include <X11/bitmaps/flagdown>	/* for flag down (mail not here) */
#include <errno.h>

static int pix_up_w, pix_dn_w, pix_up_h, pix_dn_h;
static struct stat stbuf;
static Pixmap ReadBitmapFile();
static void GetMailFile(), CloseDown();
static void check_mailbox(), redraw_mailbox();
static void Initialize(), Realize(), Destroy(), Redisplay();
static Boolean SetValues();

#define min(a,b) ((a) < (b) ? (a) : (b))
#define max(a,b) ((a) > (b) ? (a) : (b))

/* Initialization of defaults */
#define offset(field) XtOffset(MailwatchWidget,mailbox.field)
#define goffset(field) XtOffset(Widget,core.field)

static XtResource resources[] = {
    {XtNupdate, XtCInterval, XtRInt, sizeof(int),
	offset(update), XtRString, "30"},
    {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
	offset(foreground_pixel), XtRString, "black"},
    {XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
	goffset(background_pixel), XtRString, "white"},
    {XtNreverseVideo, XtCBoolean, XtRBoolean, sizeof(Boolean),
	offset(reverseVideo), XtRString, "FALSE"},
    {XtNfile, XtCFile, XtRString, sizeof(String),
	offset(filename), XtRString, NULL},
    {XtNdown_flag, XtCFile, XtRString, sizeof(String),
	offset(flagdown), XtRString, NULL},
    {XtNup_flag, XtCFile, XtRString, sizeof(String),
	offset(flagup), XtRString, NULL},
    {XtNmail, XtCString, XtRString, sizeof(String),
	offset(mail), XtRString, NULL},
    {XtNcallback, XtCCallback, XtRCallback, sizeof(caddr_t),
	offset(callback), XtRCallback, NULL},
};

#undef offset
#undef goffset

MailwatchClassRec mailwatchClassRec = {
    {	/* core fields */
	/* superclass		*/ &widgetClassRec,
	/* class_name		*/ "Mailwatch",
	/* widget_size		*/ sizeof(MailwatchRec),
	/* class_initialize	*/ NULL,
	/* class_part_initialize*/ NULL,
	/* class_inited 	*/ FALSE,
	/* initialize		*/ Initialize,
	/* initialize_hook	*/ NULL,
	/* realize		*/ Realize,
	/* actions		*/ NULL,
	/* num_actions		*/ 0,
	/* resources		*/ resources,
	/* resource_count	*/ XtNumber(resources),
	/* xrm_class		*/ NULL,
	/* compress_motion	*/ TRUE,
	/* compress_exposure	*/ TRUE,
	/* compress_enterleave	*/ TRUE,
	/* visible_interest	*/ FALSE,
	/* destroy		*/ Destroy,
	/* resize		*/ NULL,
	/* expose		*/ Redisplay,
	/* set_values		*/ SetValues,
	/* set_values_hook	*/ NULL,
	/* set_values_almost	*/ XtInheritSetValuesAlmost,
	/* get_values_hook	*/ NULL,
	/* accept_focus		*/ NULL,
	/* version		*/ XtVersion,
	/* callback_private	*/ NULL,
	/* tm_table		*/ NULL,
	/* query_geometry	*/ NULL,
    }
};

WidgetClass mailwatchWidgetClass = (WidgetClass) & mailwatchClassRec;

/*
 * private procedures
 */
static void
Initialize(request, new)
Widget request, new;
{
    MailwatchWidget w = (MailwatchWidget) new;
    register Display *dpy = XtDisplay(new);
    int depth = DefaultDepth(dpy, DefaultScreen(dpy)), wid, hgt;

    if (!w->mailbox.filename)
	GetMailFile(w);
    w->mailbox.mail = (String) XtMalloc(BUFSIZ);
    w->mailbox.mail[0] = 0;

    if (w->mailbox.reverseVideo) {
	Pixel tmp;

	tmp = w->mailbox.foreground_pixel;
	w->mailbox.foreground_pixel = w->core.background_pixel;
	w->core.background_pixel = tmp;
    }

    /*
     * build up the pixmaps that we'll put into the image
     */
    if (w->mailbox.flagup)
	w->mailbox.flagup_pixmap =
	    ReadBitmapFile(dpy, RootWindow(dpy, DefaultScreen(dpy)),
		w->mailbox.flagup, &pix_up_w, &pix_up_h);
    else
	w->mailbox.flagup_pixmap =
	    XCreatePixmapFromBitmapData(dpy, RootWindow(dpy,DefaultScreen(dpy)),
		flagup_bits,
		pix_up_w = flagup_width,
		pix_up_h = flagup_height,
		w->mailbox.foreground_pixel, w->core.background_pixel,
		depth);

    if (w->mailbox.flagdown)
	w->mailbox.flagdown_pixmap =
	    ReadBitmapFile(dpy, RootWindow(dpy, DefaultScreen(dpy)),
		w->mailbox.flagdown, &pix_dn_w, &pix_dn_h);
    else
	w->mailbox.flagdown_pixmap =
	    XCreatePixmapFromBitmapData(dpy, RootWindow(dpy,DefaultScreen(dpy)),
		flagdown_bits,
		pix_dn_w = flagdown_width,
		pix_dn_h = flagdown_height,
		w->mailbox.foreground_pixel, w->core.background_pixel,
		depth);

    /* the size of the icon should be the size of the larger icon. */
    if (w->core.width == 0)
	w->core.width = max(pix_up_w, pix_dn_w);
    if (w->core.height == 0)
	w->core.height = max(pix_up_h, pix_dn_h);
    if (stat(w->mailbox.filename, &stbuf) == 0) {
	w->mailbox.last_size = stbuf.st_size;
	w->mailbox.last_access = stbuf.st_atime;
    }
}

static void
HandleButtonPress(gw, closure, event)
Widget gw;
char *closure;
XEvent *event;
{
    MailwatchWidget w = (MailwatchWidget) gw;

    if (event->xbutton.button == 2) {
	w->mailbox.flag_up = FALSE; /* middle button resets mailbox */
	redraw_mailbox(w, NULL);
    } else {
	XtRemoveTimeOut(w->mailbox.interval_id);
	check_mailbox(w, TRUE);
    }
}

static void
clock_tic(client_data, id)
caddr_t client_data;
XtIntervalId *id;
{
    MailwatchWidget w = (MailwatchWidget) client_data;

    check_mailbox(w, FALSE);
}

static GC
GetNormalGC(w)
MailwatchWidget w;
{
    XtGCMask valuemask;
    XGCValues xgcv;

    valuemask = GCForeground | GCBackground | GCFunction | GCGraphicsExposures;
    xgcv.foreground = w->mailbox.foreground_pixel;
    xgcv.background = w->core.background_pixel;
    xgcv.function = GXcopy;
    xgcv.graphics_exposures = False;	/* this is Bool, not Boolean */
    w->mailbox.normal_GC = XtGetGC((Widget) w, valuemask, &xgcv);
}

static GC
GetInvertGC(w)
MailwatchWidget w;
{
    XtGCMask valuemask;
    XGCValues xgcv;

    valuemask = GCForeground | GCBackground | GCFunction | GCGraphicsExposures;
    xgcv.foreground = w->mailbox.foreground_pixel;
    xgcv.background = w->core.background_pixel;
    xgcv.function = GXcopyInverted;
    xgcv.graphics_exposures = False;	/* this is Bool, not Boolean */
    w->mailbox.invert_GC = XtGetGC((Widget) w, valuemask, &xgcv);
}

static Pixmap
ReadBitmapFile(dpy, root, file, width, height)
Display *dpy;
Window root;
char *file;
int *width, *height;
{
    Pixmap bitmap;
    int status, foo, bar;

    status = XReadBitmapFile(dpy, root, file, width, height, &bitmap,&foo,&bar);
    if (status == BitmapSuccess)
	return bitmap;
    if (status == BitmapOpenFailed)
	perror(file);
    else if (status == BitmapFileInvalid)
	fprintf(stderr, "%s: bad bitmap format file\n", file);
    exit(1);
}

static void
Realize(gw, valuemaskp, attr)
Widget gw;
XtValueMask *valuemaskp;
XSetWindowAttributes *attr;
{
    MailwatchWidget w = (MailwatchWidget) gw;
    register Display *dpy = XtDisplay(w);
    int depth = DefaultDepth(dpy, DefaultScreen(dpy));

    *valuemaskp |= (CWBitGravity | CWCursor);
    attr->bit_gravity = ForgetGravity;
    attr->cursor = XCreateFontCursor(dpy, XC_top_left_arrow);

    XtCreateWindow(gw, InputOutput, (Visual *) CopyFromParent,
	*valuemaskp, attr);

    GetNormalGC(w);
    GetInvertGC(w);

    XtAddEventHandler(gw, ButtonPressMask, FALSE, HandleButtonPress, NULL);
}

static void
Destroy(gw)
Widget gw;
{
    MailwatchWidget w = (MailwatchWidget) gw;

    XtFree(w->mailbox.mail);
    XtFree(w->mailbox.filename);
    XtRemoveTimeOut(w->mailbox.interval_id);
    XtDestroyGC(w->mailbox.normal_GC);
    XtDestroyGC(w->mailbox.invert_GC);
    XFreePixmap(XtDisplay(w), w->mailbox.flagup_pixmap);
    XFreePixmap(XtDisplay(w), w->mailbox.flagdown_pixmap);
}

static void
Redisplay(gw)
Widget gw;
{
    MailwatchWidget w = (MailwatchWidget) gw;

    XtRemoveTimeOut(w->mailbox.interval_id);
    check_mailbox(w, TRUE);
}

/*
 * Get the mail which is considered "new" and put it in the buffer
 * for the user to XtGet to query his mailbox.  This always overwrites
 * the previous info, so the user is responsible for querying this
 * buffer every N (update) seconds (to be in sync with the widget).
 *
 * Return TRUE if mail was gotten.
 */
static int
from(w)
MailwatchWidget w;
{
    char buf[256];
    FILE *fp;
    register char *p, *p2, *p3;
    struct timeval times[2];
    int cnt;

    if (!(p = w->mailbox.mail))
	return 1;
    *p = 0;
    if (!(fp = fopen(w->mailbox.filename, "r")) ||
	  fseek(fp, w->mailbox.last_size, 0L))
	goto the_end;
    while (fgets(buf, sizeof(buf), fp)) {
	if (strncmp(buf, "From ", 5))
	    continue;
	/* go to the first space after the author's address and  plug a null */
	for (p2 = buf+5; *p2 != ' '; p2++)
	    ;
	*p2 = 0;
	/* move forward past the Day, Month and Date -- go to the time */
	for (p2++, cnt = 0; cnt < 3; p2++)
	    if (*p2 == ' ') {
		cnt++;
		while (p2[1] == ' ')
		    p2++;
	    }
	/* set p3 to the start of the time and null terminate at first space */
	for (p3 = p2; *p2 != ' '; p2++)
	    ;
	*p2 = 0;
	(void) sprintf(p, "(%s) %s -> ", p3, buf+5);
	p += strlen(p);

	while (fgets(buf, sizeof(buf), fp))
	    if (*buf == '\n') {
		(void) sprintf(p, "(No Subject)");
		break;
	    } else if (sscanf(buf, "Subject: %[^\n]", p))
		break;
	p += strlen(p);
	*p++ = '\r', *p++ = '\n', *p = 0;
    }

the_end:
    if (fp)
	fclose(fp);
    times[0].tv_sec = stbuf.st_atime;
    times[0].tv_usec = 0;
    times[1].tv_sec = stbuf.st_mtime;
    times[1].tv_usec = 0;
    if (utimes(w->mailbox.filename, times))
	perror("utime");

    return w->mailbox.mail[0] != '\0';
}

static void
check_mailbox(w, force_redraw)
MailwatchWidget w;
Boolean force_redraw;
{
    char buf[256];
    int redraw = 0;

    if (stat(w->mailbox.filename, &stbuf) == -1) {
        if (w->mailbox.flag_up == TRUE)
            redraw = TRUE;
        w->mailbox.flag_up = FALSE;
        w->mailbox.last_size = 0;
    } else {
      if (stbuf.st_size > w->mailbox.last_size && from(w)) { 
	    redraw = TRUE;
	    w->mailbox.flag_up = TRUE;
	    XBell(XtDisplay(w), MAILBOX_VOLUME);
	    XtCallCallbacks(w, XtNcallback, w->mailbox.mail);
	    w->mailbox.last_access = stbuf.st_atime;
	} else if (stbuf.st_atime > w->mailbox.last_access) {
	    if (w->mailbox.flag_up == TRUE)
		redraw = TRUE;
	    w->mailbox.flag_up = FALSE;
	}
	w->mailbox.last_size = stbuf.st_size;
	/* else, no change -- leave it alone */
    }
    
    if (redraw || force_redraw)
	redraw_mailbox(w);

    /* reset timer */
    w->mailbox.interval_id = XtAddTimeOut(w->mailbox.update * 1000,
	clock_tic, (caddr_t) w);
}

/*
 * get user name for building mailbox
 */
static void
GetMailFile(w)
MailwatchWidget w;
{
    char *getlogin();
    char *username;
    int len;

    username = getlogin();
    if (!username) {
	struct passwd *pw = getpwuid(getuid());

	if (!pw) {
	    fprintf(stderr, "%s:  unable to find a username for you.\n",
		"Mailbox widget");
	    CloseDown(w, 1);
	}
	username = pw->pw_name;
    }
    if (!(w->mailbox.filename = (String) XtMalloc(strlen(MAILBOX_DIRECTORY)+1
	  + strlen(username)+1))) {
	fprintf(stderr, "can't allocate enough memory for widget.\n");
	exit(1);
    }
    sprintf(w->mailbox.filename, "%s/%s", MAILBOX_DIRECTORY, username);
}

static void
CloseDown(w, status)
MailwatchWidget w;
int status;
{
    Display *dpy = XtDisplay(w);

    XtDestroyWidget(w);
    XCloseDisplay(dpy);
    exit(status);
}

static Boolean
SetValues(gcurrent, grequest, gnew)
Widget gcurrent, grequest, gnew;
{
    MailwatchWidget current = (MailwatchWidget) gcurrent;
    MailwatchWidget new = (MailwatchWidget) gnew;
    Boolean redisplay = FALSE;

    if (current->mailbox.update != new->mailbox.update) {
	XtRemoveTimeOut(current->mailbox.interval_id);
	new->mailbox.interval_id = XtAddTimeOut(new->mailbox.update * 1000,
	    clock_tic,
	    (caddr_t) gnew);
    }
    if (current->mailbox.foreground_pixel != new->mailbox.foreground_pixel ||
	current->core.background_pixel != new->core.background_pixel) {
	XtDestroyGC(current->mailbox.normal_GC);
	XtDestroyGC(current->mailbox.invert_GC);
	GetNormalGC(new);
	GetInvertGC(new);
	redisplay = TRUE;
    }
    return (redisplay);
}

/*
 * drawing code
 */
static void
redraw_mailbox(mw)
MailwatchWidget mw;
{
    register Display *dpy = XtDisplay(mw);
    register Window win = XtWindow(mw);
    register int x, y, w, h;
    GC gc;
    Pixmap picture;
    Pixel back;

    if (mw->mailbox.flag_up) {	/* paint the "up" position */
	XFillRectangle(dpy, win, mw->mailbox.normal_GC,
	    0, 0, mw->core.width, mw->core.height);
	back = mw->mailbox.foreground_pixel;
	picture = mw->mailbox.flagup_pixmap;
	w = pix_up_w, h = pix_up_h;
	gc = mw->mailbox.invert_GC;
    } else {		/* paint the "down" position */
	XClearWindow(dpy, win);
	back = mw->core.background_pixel;
	picture = mw->mailbox.flagdown_pixmap;
	w = pix_dn_w, h = pix_dn_h;
	gc = mw->mailbox.normal_GC;
    }

    /* center the picture in the window */
    x = (mw->core.width - w) / 2;
    y = (mw->core.height - h) / 2;

    XCopyArea(dpy, picture, win, gc, 0, 0, w, h, x, y);
}
