
/*

	Copyright (c) 1986 	Chris Guthrie

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation.  No representations are made about the
suitability of this software for any purpose.  It is
provided "as is" without express or implied warranty.

*/

/*
 * X11 support and other enhancements added by Jeff Weinstein
 * (jeff@polyslo.calpoly.edu).  Please send all comments, bug
 * reports, fixes, suggestions regarding this version of XTrek
 * to me.  
 */

static char RCSID[] = "$Header: /blackbird/home/jeff/TAPE2/xtrek.new/RCS/daemon.c,v 3.1 88/09/20 00:43:47 jeff Exp $";

#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <stdio.h>
#ifndef ibm032
#include <sys/ipc.h>
#include <sys/shm.h>
#endif ibm032
#ifndef hpux
#include <sys/wait.h>
#endif hpux
#include <sys/ioctl.h>
#include <signal.h>
#include <setjmp.h>
#include <math.h>
#include "defs.h"
#include "struct.h"
#include "data.h"
#include "planets.h"

#ifdef hpux
#define bcopy(from, to, length)		memcpy((to), (from), (length))
#endif hpux

#define fuse(X) ((ticks % (X)) == 0)
/* Run the game */

long random();
long lseek();
static struct itimerval udt;
static int shmid;

static int debug = 0;
static int ticks = 0;
static int plfd;
static int remapgalaxy;
static int noshipclass;

jmp_buf env;

static int tcount[MAXTEAM + 1];

main(argc, argv)
int argc;
char **argv;
{
    int restart = 0;
    register int i;
    int	shmemKey = PKEY;
    struct memory	*sharedMemory;
#ifndef ibm032
    struct shmid_ds smbuf;
#endif ibm032
    int x = 0;
    int move();
    int reaper();
    int freemem();

    for (i=1; i<argc; ++i)
	if (argv[i][0] == '-')
	    switch(argv[i][1]) {
		case 'd':
			++debug;
			break;
		case 'S':
			++noshipclass;
			break;
		case 'r':
			++remapgalaxy;
			break;
		default:
			fprintf(stderr,"%s: unknown option %s\n", argv[0], argv[i]);
	    }

    srandom(getpid());
    if (!debug) {
	for (i = 0; i < NSIG; i++) {
	    (void) signal(i, freemem);
	}
	detach();
    }

    /* Kill any existing segments */
#ifndef ibm032
    if ((shmid = shmget(shmemKey, 0, 0)) >= 0) {
	fprintf(stderr, "Killing existing segment\n");
	shmctl(shmid, IPC_RMID, (struct shmid_ds *) 0);
    }

    shmid = shmget(shmemKey, sizeof(struct memory), IPC_CREAT | 0777);
    if (shmid < 0) {
	perror("can't open shared memory");
	exit (1);
    }
    /* Hose Ed's robots */
    shmctl(shmid, IPC_STAT, &smbuf);
    smbuf.shm_perm.uid = geteuid();
    smbuf.shm_perm.mode = 0700;
    shmctl(shmid, IPC_SET, &smbuf);

    sharedMemory = (struct memory *) shmat(shmid, 0, 0);
    if (sharedMemory == (struct memory *) -1) {
	perror("shm attach");
	exit (1);
    }
#else
    sharedMemory = (struct memory *)0xef100100;
    bzero( sharedMemory, sizeof(struct memory) );
    if ( (*(long *)0xef100000) == 0x4a454646 )
	{
	fprintf(stderr, "daemon already running.\n");
	exit(1);
	}
    (*(long *)0xef100000) = 0x4a454646;
#endif

    players = sharedMemory->players;
    torps = sharedMemory->torps;
    status = sharedMemory->status;
    planets = sharedMemory->planets;
    phasers = sharedMemory->phasers;
    mctl = sharedMemory->mctl;
    messages = sharedMemory->messages;

    for (i = 0; i < MAXPLAYER; i++)
	players[i].p_status = PFREE;

    plfd = open(PLFILE, O_RDWR, 0777);
    if (plfd < 0) {
	fprintf(stderr, "No planet file.  Restarting galaxy\n");
	restart++;
    }
    else {
	if (read(plfd, (char *) planets, sizeof(pdata)) != sizeof(pdata)) {
	    fprintf(stderr, "Planet file wrong size.  Restarting galaxy\n");
	    restart++;
	}
    }
    if (restart) {
	int ii;

	bcopy(pdata, planets, sizeof(pdata));

	for (ii = 0; ii < 10; ii++) for (i = ii; i < 40; i+=10) {
	    if (planets[i].pl_flags & PLHOME)
		planets[i].pl_flags |= (PLREPAIR|PLFUEL);
#ifdef REMAP
	    /* move non-home planets around */
	    else if (remapgalaxy) {
		int xoff,yoff,	/* offsets to keep teams' planets */
				/* in the appropriate quadrants   */
		    mindist;	/* distance to closest neighbor   */

#define BORDER (GWIDTH/20)
		switch (planets[i].pl_owner) {
		case FED:
		    xoff=BORDER;	yoff=GWIDTH/2;
		    break;
		case ROM:
		    xoff=BORDER;	yoff=BORDER;
		    break;
		case KLI:
		    xoff=GWIDTH/2;	yoff=BORDER;
		    break;
		case ORI:
		    xoff=GWIDTH/2;	yoff=GWIDTH/2;
		    break;
		}
		do {

#define TRIES 3
		    int newmd[TRIES],	/* distances to closest neighbors */
			newx[TRIES],	/* new x,y coordinates--          */
			newy[TRIES];	/*	we'll pick one later      */
		    int j,k;

#define RANDOM (random() % (GWIDTH/2 - BORDER))

		    for (j=0; j<TRIES; j++) {
			int kk;

			/* pick new x and y */
			newx[j]=xoff+RANDOM;
			newy[j]=yoff+RANDOM;

		        /* find closest planet */
		        newmd[j]=GWIDTH;
			for (kk = 0; kk < 10; kk++)
			for (k = kk; k < 40; k+=10) {
			    int dx, dy, dist;

			    /*
			     * don't worry about planets that
			     * haven't been positioned yet
			     */
			    if (k==i) goto done;

			    dx = newx[j] - planets[k].pl_x;
			    dy = newy[j] - planets[k].pl_y;
			    dist=(int)sqrt((double)(dx*dx + dy*dy));
			    if (dist < newmd[j]) newmd[j]=dist;
			}
			done: ;
		    }
#if TRIES == 2
		    /* pick greater newmd */
		    if (newmd[0] <= newmd[1]) {
			planets[i].pl_x=newx[1];
			planets[i].pl_y=newy[1];
			mindist	       =newmd[1];
		    }
		    else {
			planets[i].pl_x=newx[0];
			planets[i].pl_y=newy[0];
			mindist	       =newmd[0];
		    }
#endif TRIES == 2
#if TRIES == 3
		    /* pick greatest newmd */
#define greater(x,y,z) (((x) >= (y)) && ((x) >= (z)))
		    if (greater(newmd[0],newmd[1],newmd[2])) {
			planets[i].pl_x=newx[0];
			planets[i].pl_y=newy[0];
			mindist	       =newmd[0];
		    }
		    else if (greater(newmd[1],newmd[0],newmd[2])) {
			planets[i].pl_x=newx[1];
			planets[i].pl_y=newy[1];
			mindist	       =newmd[1];
		    }
		    else {
			planets[i].pl_x=newx[2];
			planets[i].pl_y=newy[2];
			mindist	       =newmd[2];
		    }
#endif TRIES == 3
		} while (mindist < 2*BORDER);
	    }
#endif REMAP
	    if (random() % 4 == 0)
		planets[i].pl_flags |= PLREPAIR;
	    if (random() % 2 == 0)
		planets[i].pl_flags |= PLFUEL;
	}
    }

    status->active = 0;
    if (noshipclass)
	status->configurable = NOT_CONFIGURABLE;

    (void) signal(SIGCLD, reaper);

    (void) signal(SIGALRM, move);
    udt.it_interval.tv_sec = 0;
    udt.it_interval.tv_usec = UPDATE;
    udt.it_value.tv_sec = 0;
    udt.it_value.tv_usec = UPDATE;
    (void) setitimer(ITIMER_REAL, &udt, (struct itimerval *) 0);

    (void) setjmp(env);

    while (1) {
	pause();
	if (debug) {
	    if (!(++x % 50))
		printf("Mark %d\n", x);
	}
    }
}

detach()
{
    int fd;

#ifdef hpux
    setpgrp();
#else hpux
    close(0);
    close(1);
    close(2);
    fd = open("/dev/tty", O_RDWR, 0);
    if (fd < 0)
	return;
    (void) ioctl(fd, TIOCNOTTY, (char *) NULL);
    (void) close(fd);
#endif hpux
}

/* These specify how often special actions will take place in
   UPDATE units (0.10 seconds, currently.)
*/

#define PLAYERFUSE	1
#define TORPFUSE	1
#define PHASERFUSE	1
#define TEAMFUSE	5
#define PLFIGHTFUSE	5
#define BEAMFUSE	10
#define SYNCFUSE	3000
#define PLANETFUSE	600

#define GHOSTTIME	(10 * 1000000 / UPDATE)	/* 10 secs */
#define OUTFITTIME	(2 * AUTOQUIT * 1000000 / UPDATE) /* 2 * AQ secs */

nplayers = 0;
dietime = -1;

move()
{
    if (++ticks == dietime)	/* no player for 1 minute. kill self */
	freemem();
    if (fuse(PLAYERFUSE)) {
	udplayers();
    }
    if (fuse(TORPFUSE)) {
	udtorps();
    }
    if (fuse(PHASERFUSE)) {
	udphaser();
    }
    if (fuse(TEAMFUSE)) {
	teamtimers();
    }
    if (fuse(PLFIGHTFUSE)) {
	plfight();
    }
    if (fuse(BEAMFUSE)) {
	beam();
    }
    if (fuse(SYNCFUSE)) {
	save_planets();
    }
    if (fuse(PLANETFUSE)) {
	udplanets();
    }
}

udplayers()
{
    register int i, k;
    register struct player *j;
    int maxspeed;
    int h, dx, dy;
    struct planet *l;

    nplayers = 0;
    tcount[FED] = tcount[ROM] = tcount[KLI] = tcount[ORI] = 0;
    for (i = status->active = 0, j = &players[i]; i < MAXPLAYER; i++, j++) {
	switch (j->p_status) {
	    case POUTFIT:
		if (++(j->p_ghostbuster) > OUTFITTIME)
		    j->p_status = PFREE;
		continue;
	    case PFREE:
		nplayers++;
		j->p_ghostbuster = 0;	/* stop from hosing new players */
		continue;
	    case PDEAD:
		if (--j->p_explode <= 0) { 	/* Ghost Buster */
		    j->p_status = PFREE;
		}
		continue;
	    case PEXPLODE:
		j->p_updates++;
		j->p_flags &= ~PFCLOAK;
		if (j->p_explode == (10/PLAYERFUSE))
		    blowup(j);		/* damage everyone else around */
		if (--j->p_explode <= 0) {
		    j->p_status = PDEAD;
		    j->p_explode = 600/PLAYERFUSE; /* set ghost buster */
		    for (h = 0, l = &planets[h]; h < MAXPLANETS; h++, l++) {
			dx = ABS(l->pl_x - j->p_x);
			dy = ABS(l->pl_y - j->p_y);
			if (dx < 4 * PFIREDIST && dy < 4 * PFIREDIST)
			    l->pl_flags |= PLREDRAW;
		    }
		}
		break;
	    case PALIVE:
		if (++(j->p_ghostbuster) > GHOSTTIME) {
		    j->p_status = PEXPLODE;
		    j->p_explode = 10/PLAYERFUSE;
		    ghostmess(j);
		    /* j->p_stats.st_losses++; */
		    j->p_whydead = KGHOST;
		    j->p_whodead = i;
		}
			
		status->active += (1<<i);
		tcount[j->p_team]++;
		j->p_updates++;

		/* cool weapons */
		j->p_wtemp -= j->p_ship.s_wcool;
		if (j->p_wtemp < 0)
		    j->p_wtemp = 0;
		if (j->p_flags & PFWEP) {
		    if (--j->p_wtime <= 0)
			j->p_flags &= ~PFWEP;
		}
		else if (j->p_wtemp > 1000) {
		    if (!(random() % 40)) {
			j->p_flags |= PFWEP;
			j->p_wtime = ((random() % 150) + 100) / PLAYERFUSE;
		    }
		}
		/* cool engine */
		j->p_etemp -= j->p_ship.s_ecool;
		if (j->p_etemp < 0)
		    j->p_etemp = 0;
		if (j->p_flags & PFENG) {
		    if (--j->p_etime <= 0)
			j->p_flags &= ~PFENG;
		}
		else if (j->p_etemp > 1000) {
		    if (!(random() % 40)) {
			j->p_flags |= PFENG;
			j->p_etime = ((random() % 150) + 100) / PLAYERFUSE;
			j->p_desspeed = 0;
		    }
		}

		/* Add fuel */
		if ((j->p_flags & PFORBIT) &&
		    (planets[j->p_planet].pl_flags & PLFUEL) &&
		    (!(planets[j->p_planet].pl_owner 
			& (j->p_swar | j->p_hostile)))) {
			    j->p_fuel += 8 * j->p_ship.s_recharge;
		}
		else
		    j->p_fuel += 2 * j->p_ship.s_recharge;

		if (j->p_fuel > j->p_ship.s_maxfuel)
		    j->p_fuel = j->p_ship.s_maxfuel;
		if (j->p_fuel < 0) {
		    j->p_desspeed = 0;
		    j->p_flags &= ~PFCLOAK;
		}

		/* repair shields */
		if (j->p_shield < j->p_ship.s_maxshields) {
		    if ((j->p_flags & PFREPAIR) && (j->p_speed == 0)) {
			j->p_subshield += j->p_ship.s_repair * 4;
			if ((j->p_flags & PFORBIT) &&
			    (planets[j->p_planet].pl_flags & PLREPAIR) &&
			    (!(planets[j->p_planet].pl_owner 
				& (j->p_swar | j->p_hostile)))) {
				    j->p_subshield += j->p_ship.s_repair * 4;
			}
		    }
		    else
			j->p_subshield += j->p_ship.s_repair * 2;
		    if (j->p_subshield / 1000) {
			j->p_shield += j->p_subshield / 1000;
			j->p_subshield %= 1000;
		    }
		    if (j->p_shield > j->p_ship.s_maxshields) {
			j->p_shield = j->p_ship.s_maxshields;
			j->p_subshield = 0;
		    }
		}

		/* repair damage */
		if (j->p_damage && !(j->p_flags & PFSHIELD)) {
		    if ((j->p_flags & PFREPAIR) && (j->p_speed == 0)) {
			j->p_subdamage += j->p_ship.s_repair * 2;
			if ((j->p_flags & PFORBIT) &&
			    (planets[j->p_planet].pl_flags & PLREPAIR) &&
			    (!(planets[j->p_planet].pl_owner 
				& (j->p_swar | j->p_hostile)))) {
				    j->p_subdamage += j->p_ship.s_repair * 2;
			}
		    }
		    else
			j->p_subdamage += j->p_ship.s_repair;
		    if (j->p_subdamage / 1000) {
			j->p_damage -= j->p_subdamage / 1000;
			j->p_subdamage %= 1000;
		    }
		    if (j->p_damage < 0) {
			j->p_damage = 0;
			j->p_subdamage = 0;
		    }
		}

		/* Charge for cloaking */
		if (j->p_flags & PFCLOAK) {
		    if (j->p_fuel < j->p_ship.s_cloakcost) {
			j->p_flags &= ~PFCLOAK;
		    }
		    else {
			j->p_fuel -= j->p_ship.s_cloakcost;
		    }
		}

		/* Move Player in orbit */
		if (j->p_flags & PFORBIT) {
		    j->p_dir += 2;
		    j->p_desdir = j->p_dir;
		    j->p_x = planets[j->p_planet].pl_x + ORBDIST
			* Cos[(unsigned char) (j->p_dir - (unsigned char) 64)];
		    j->p_y = planets[j->p_planet].pl_y + ORBDIST
			* Sin[(unsigned char) (j->p_dir - (unsigned char) 64)];
		}

		/* Move player through space */
		else {

		    if (j->p_dir != j->p_desdir)
			changedir(j);

		    /* Alter speed */
		    maxspeed = j->p_ship.s_maxspeed + 1;
		    maxspeed = maxspeed - ((j->p_damage * maxspeed) /
					   (j->p_ship.s_maxdamage));
		    if (maxspeed > j->p_ship.s_maxspeed)
			maxspeed = j->p_ship.s_maxspeed;

		    if (j->p_desspeed > maxspeed)
			j->p_desspeed = maxspeed;
		    if (j->p_flags & PFENG)
			j->p_desspeed = 0;

		    if (j->p_desspeed > j->p_speed) {
			j->p_subspeed += j->p_ship.s_accint;
		    }
		    if (j->p_desspeed < j->p_speed) {
			j->p_subspeed -= j->p_ship.s_decint;
		    }
		    if (j->p_subspeed / 1000) {
			j->p_speed += j->p_subspeed / 1000;
			j->p_subspeed /= 1000;
			if (j->p_speed < 0)
			    j->p_speed = 0;
			if (j->p_speed > j->p_ship.s_maxspeed)
			    j->p_speed = j->p_ship.s_maxspeed;
		    }
		    /* Charge for speed */

		    if (j->p_fuel < (j->p_ship.s_warpcost * j->p_speed)) {
			j->p_desspeed = 0;
		    }
		    else {
			j->p_fuel -= j->p_ship.s_warpcost * j->p_speed;
			j->p_etemp += j->p_speed;
		    }

		    j->p_x += (double) j->p_speed * Cos[j->p_dir] * WARP1;
		    j->p_y += (double) j->p_speed * Sin[j->p_dir] * WARP1;

		    /* Bounce off the side of the galaxy */
		    if (j->p_x < 0) {
			j->p_x = -j->p_x;
			if (j->p_dir == 192) 
			    j->p_dir = j->p_desdir = 64;
			else 
			    j->p_dir = j->p_desdir = 64 - (j->p_dir - 192);
		    }
		    else if (j->p_x > GWIDTH) {
			j->p_x = GWIDTH - (j->p_x - GWIDTH);
			if (j->p_dir == 64) 
			    j->p_dir = j->p_desdir = 192;
			else 
			    j->p_dir = j->p_desdir = 192 - (j->p_dir - 64);
		    }
		    if (j->p_y < 0) {
			j->p_y = -j->p_y;
			if (j->p_dir == 0) 
			    j->p_dir = j->p_desdir = 128;
			else 
			    j->p_dir = j->p_desdir = 128 - j->p_dir;
		    }
		    else if (j->p_y > GWIDTH) {
			j->p_y = GWIDTH - (j->p_y - GWIDTH);
			if (j->p_dir == 128) 
			    j->p_dir = j->p_desdir = 0;
			else 
			    j->p_dir = j->p_desdir = 0 - (j->p_dir - 128);
		    }
		}

		/* Set player's alert status */
#define YRANGE ((GWIDTH)/5)
#define RRANGE ((GWIDTH)/10)
		j->p_flags |= PFGREEN;
		j->p_flags &= ~(PFRED|PFYELLOW);
		for (k = 0; k < MAXPLAYER; k++) {
		    int dx, dy, dist;
		    if ((players[k].p_status != PALIVE) ||
			((!((j->p_swar | j->p_hostile) & players[k].p_team)) &&
			(!((players[k].p_swar | players[k].p_hostile) & 
			j->p_team)))) {
			    continue;
		    }
		    else if (j == &players[k]) {
			continue;
		    }
		    else {
			dx = j->p_x - players[k].p_x;
			dy = j->p_y - players[k].p_y;
			if (ABS(dx) > YRANGE || ABS(dy) > YRANGE)
			    continue;
			dist = dx * dx + dy * dy;
			if (dist <  RRANGE * RRANGE) {
			    j->p_flags |= PFRED;
			    j->p_flags &= ~(PFGREEN|PFYELLOW);
			}
			else if ((dist <  YRANGE * YRANGE) &&
			    (!(j->p_flags & PFRED))) {
			    j->p_flags |= PFYELLOW;
			    j->p_flags &= ~(PFGREEN|PFRED);
			}
		    }
		}
	    break;
	} /* end switch */
    }
    if (nplayers == MAXPLAYER) {
	if (dietime == -1)
	    dietime = ticks + 600 / PLAYERFUSE;
    }
    else {
	dietime = -1;
    }
}
changedir(sp)
struct player *sp;
{
    unsigned int ticks;

    if (sp->p_speed == 0) {
	sp->p_dir = sp->p_desdir;
	sp->p_subdir = 0;
    }
    else {
	sp->p_subdir += sp->p_ship.s_turns / (1 << sp->p_speed);
	ticks = sp->p_subdir / 1000;
	if (ticks) {
	    if (ticks > ABS(sp->p_dir - sp->p_desdir))
		sp->p_dir = sp->p_desdir;
	    else if ((unsigned char) (sp->p_dir - sp->p_desdir) > 127)
		sp->p_dir += ticks;
	    else 
		sp->p_dir -= ticks;
	    sp->p_subdir %= 1000;
	}
    }
}

udtorps()
{
    register int i;
    register struct torp *j;

    for (i = 0, j = &torps[i]; i < MAXPLAYER * MAXTORP; i++, j++) {
	switch (j->t_status) {
	    case TFREE:
		continue;
	    case TMOVE:
	    case TSTRAIGHT:
		j->t_x += (double) j->t_speed * Cos[j->t_dir] * WARP1;
		if (j->t_x < 0) {
		    j->t_x = 0;
		    explode(j);
		    break;
		}
		else if (j->t_x > GWIDTH) {
		    j->t_x = GWIDTH;
		    explode(j);
		    break;
		}
		j->t_y += (double) j->t_speed * Sin[j->t_dir] * WARP1;
		if (j->t_y < 0) {
		    j->t_y = 0;
		    explode(j);
		    break;
		}
		else if (j->t_y > GWIDTH) {
		    j->t_y = GWIDTH;
		    explode(j);
		    break;
		}

		/* Make sure that player torps wobble */
		if (j->t_status == TMOVE)
		    j->t_dir += (random() % 3) - 1;

		if (j->t_fuse-- <= 0) {
		    j->t_status = TFREE;
		    players[j->t_owner].p_ntorp--;
		}
		else if (near(j)) {
		    explode(j);
		}
		break;
	    case TDET:
		j->t_x += (double) j->t_speed * Cos[j->t_dir] * WARP1;
		if (j->t_x < 0)
		    j->t_x += GWIDTH;
		else if (j->t_x > GWIDTH)
		    j->t_x -= GWIDTH;
		j->t_y += (double) j->t_speed * Sin[j->t_dir] * WARP1;
		if (j->t_y < 0)
		    j->t_y += GWIDTH;
		else if (j->t_y > GWIDTH)
		    j->t_y -= GWIDTH;
		explode(j);
		break;
	    case TEXPLODE:
		if (j->t_fuse-- <= 0) {
		    j->t_status = TFREE;
		    players[j->t_owner].p_ntorp--;
		}
		break;
	    case TOFF:
		j->t_status = TFREE;
		players[j->t_owner].p_ntorp--;
		break;
	}
    }
}

/* See if there is someone close enough to explode for */
near(torp)
struct torp *torp;
{
    register int i;
    int dx, dy;
    register struct player *j;

    for (i = 0, j = &players[i]; i < MAXPLAYER; i++, j++) {
	if (!(j->p_status == PALIVE))
	    continue;
	if (torp->t_owner == j->p_no)
	    continue;
	/* This is tricky.  If you aren't at war with the team,
	   and that team is not at war with your team, continue.
	*/
	if ((!(torp->t_war & j->p_team)) &&
	    (!(torp->t_team & (j->p_swar | j->p_hostile))))
	        continue;
	dx = torp->t_x - j->p_x;
	dy = torp->t_y - j->p_y;
	if (ABS(dx) > EXPDIST || ABS(dy) > EXPDIST)
	    continue;
	if (dx * dx + dy * dy < EXPDIST * EXPDIST)
	    return 1;
    }
    return 0;
}

    

/* Do damage to all surrounding players */

explode(torp)
struct torp *torp;
{
    register int i;
    int dx, dy, dist;
    int damage;
    register struct player *j;

    for (i = 0, j = &players[i]; i < MAXPLAYER; i++, j++) {
	if (!(j->p_status == PALIVE))
	    continue;
	if (torp->t_owner == j->p_no)	/* This isn't realistic */
	    continue;
	dx = torp->t_x - j->p_x;
	dy = torp->t_y - j->p_y;
	if (ABS(dx) > DETDIST || ABS(dy) > DETDIST) /*XXX*/
	    continue;
	dist = dx * dx + dy * dy;
	if (dist > DETDIST * DETDIST)
	    continue;
	if (dist > EXPDIST * EXPDIST) {
	    damage = torp->t_damage * (DETDIST - sqrt((double) dist)) /
		(DETDIST - EXPDIST);
	}
	else {
	    damage = torp->t_damage;
	}
	if (damage > 0) {
	    /* First, check to see if torp owner has started a war */
	    if (players[torp->t_owner].p_hostile & j->p_team) {
		players[torp->t_owner].p_swar |= j->p_team;
	    }
	    /* Note that if a player is at peace with the victim, then
	       the torp has caused damage either accidently, or because
	       the victim was at war with, or hostile to, the player.
	       In either case, we don't consider the damage to be
	       an act of war */

	    if (j->p_flags & PFSHIELD) {
		j->p_shield -= damage;
		if (j->p_shield < 0) {
		    j->p_damage -= j->p_shield;
		    j->p_shield = 0;
		}
	    }
	    else {
		j->p_damage += damage;
	    }
	    if (j->p_damage >= j->p_ship.s_maxdamage) {
		j->p_status = PEXPLODE;
		j->p_explode = 10/PLAYERFUSE;
		if (!(j->p_flags & PFPRACTR)) {
		    players[torp->t_owner].p_kills += 1.0 + 
			j->p_armies * 0.1 + j->p_kills * 0.1;
		}
		killmess(j, &players[torp->t_owner]);
		j->p_stats.st_losses++;
		j->p_whydead = KTORP;
		j->p_whodead = torp->t_owner;
	    }
	}
    }
    torp->t_status = TEXPLODE;
    torp->t_fuse = 10/TORPFUSE;
}

udplanets()
{
    register int i;
    register struct planet *l;

    for (i = 0, l = &planets[i]; i < MAXPLANETS; i++, l++) {
	if (l->pl_couptime)	/* Decrement coup counter one minute */
	    l->pl_couptime--;
	if (l->pl_armies == 0)
	    continue;
	if ((random() % 3000) < l->pl_armies)
	    l->pl_armies -= (random() % l->pl_armies);
	if ((l->pl_armies < 4) && ((random() % 20) == 0)) {
	    l->pl_armies++;
	    continue;
	}
	if ((random() % 10) == 0)
	    l->pl_armies += (random() % 3) + 1;
	    
    }
}

udphaser()
{
    register int i;
    register struct phaser *j;
    register struct player *victim;

    for (i = 0, j = &phasers[i]; i < MAXPLAYER; i++, j++) {
	switch (j->ph_status) {
	    case PHFREE:
		continue;
	    case PHMISS:
		if (j->ph_fuse-- == 1)
		    j->ph_status = PHFREE;
		break;
	    case PHHIT:
		if (j->ph_fuse-- == 10) {
		    victim = &players[j->ph_target];

		    /* start a war, if necessary */
		    if (players[i].p_hostile & victim->p_team) {
			players[i].p_swar |= victim->p_team;
		    }

		    if (victim->p_flags & PFSHIELD) {
			victim->p_shield -= j->ph_damage;
			if (victim->p_shield < 0) {
			    victim->p_damage -= victim->p_shield;
			    victim->p_shield = 0;
			}
		    }
		    else {
			victim->p_damage += j->ph_damage;
		    }
		    if (victim->p_damage >= victim->p_ship.s_maxdamage) {
			victim->p_status = PEXPLODE;
			victim->p_explode = 10/PLAYERFUSE;
			if (!(victim->p_flags & PFPRACTR)) {
			    players[i].p_kills += 1.0 + 
				victim->p_armies * 0.1 +
				victim->p_kills * 0.1;
			}
			killmess(victim, &players[i]);
			victim->p_stats.st_losses++;
			victim->p_whydead = KPHASER;
			victim->p_whodead = i;
		    }
		}
		if (j->ph_fuse == 0)
		    j->ph_status = PHFREE;
		break;
	}
    }
}

int pl_warning[MAXPLANETS];	/* To keep planets shut up for awhile */
int tm_robots[MAXTEAM + 1];		/* To limit the number of robots */
int tm_coup[MAXTEAM + 1];		/* To allow a coup */

teamtimers()
{
    register int i;
    for (i = 0; i <= MAXTEAM; i++) {
	if (tm_robots[i] > 0)
	    tm_robots[i]--;
	if (tm_coup[i] > 0)
	    tm_coup[i]--;
    }
}

plfight()
{
    register int h, i;
    register struct player *j;
    register struct planet *l;
    int dx, dy;
    int damage;
    int dist;
    int rnd;
    char buf[80];
    char buf1[80];

    for (h = 0, l = &planets[h]; h < MAXPLANETS; h++, l++) {
	if (l->pl_flags & PLCOUP) {
	    l->pl_flags &= ~PLCOUP;
	    l->pl_owner = (l->pl_flags & ALLTEAM);
	    l->pl_armies = 4;
	}
	l->pl_flags &= ~PLREDRAW;
	if (pl_warning[h] > 0)
	    pl_warning[h]--;
    }
    for (i = 0, j = &players[i]; i < MAXPLAYER; i++, j++) {
	if (j->p_status == PALIVE) {

	    /* Have planets do damage to nearby players.  Note that
	       this is the most inefficient code in the game.  If
	       one needs cycles, simply change the code to have
	       planets only attack ships that are in orbit.   Thus
	       you get rid of the following for statement 
	    */
	    for (h = 0, l = &planets[h]; h < MAXPLANETS; h++, l++) {
		dx = ABS(l->pl_x - j->p_x);
		dy = ABS(l->pl_y - j->p_y);
		if (dx < 4 * PFIREDIST && dy < 4 * PFIREDIST)
		    l->pl_flags |= PLREDRAW;
		if (dx > PFIREDIST || dy > PFIREDIST)	/*XXX*/
		    continue;
		dist = (int) hypot((double) dx, (double) dy);
		if (dist > PFIREDIST)
		    continue;
		if ((j->p_swar | j->p_hostile) & l->pl_owner) {
		    if (l->pl_armies > 0)
			damage = l->pl_armies / 10 + 2;
		    else
			damage = 0;
		    if (damage > 0) {
			if (j->p_flags & PFSHIELD) {
			    j->p_shield -= damage;
			    if (j->p_shield < 0) {
				j->p_damage -= j->p_shield;
				j->p_shield = 0;
			    }
			}
			else {
			    j->p_damage += damage;
			}
			if (j->p_damage >= j->p_ship.s_maxdamage) {
			j->p_explode = 10/PLAYERFUSE;
			j->p_status = PEXPLODE;
			j->p_stats.st_losses++;
			(void) sprintf(buf, "%s (%c%x) killed by %s (%c)",
			    j->p_name,
			    teamlet[j->p_team],
			    j->p_no,
			    l->pl_name,
			    teamlet[l->pl_owner]);
			pmessage(buf, 0, MALL, "GOD->ALL");
			j->p_whydead = KPLANET;
			j->p_whodead = h;
			}
		    }
		}
	    }	/* End planet damage */

	    /* do bombing */
	    if ((!(j->p_flags & PFORBIT)) || (!(j->p_flags & PFBOMB)))
		continue;
	    l = &planets[j->p_planet];
	    if (j->p_team == l->pl_owner)
		continue;
	    if (!((j->p_swar | j->p_hostile) & l->pl_owner))
		continue;
	    if (l->pl_armies < 5)
		continue;

	    /* Warn owning team */
	    if (pl_warning[h] <= 0) {
		pl_warning[h] = 50/PLFIGHTFUSE; 
		(void) sprintf(buf, "We are being attacked by %s %c%x.",
			j->p_name,
			teamlet[j->p_team],
			j->p_no);
		(void) sprintf(buf1, "%-3s->%-3s",
		    l->pl_name, teamshort[l->pl_owner]);
		pmessage(buf, l->pl_owner, MTEAM, buf1);
	    }

	    /* start the war (if necessary) */
	    j->p_swar |= l->pl_owner;

	    rnd = random() % 100;
	    if (rnd < 50) {
		continue;
	    }
	    else if (rnd < 80) {
		l->pl_armies -= 1;
		j->p_kills += 0.02;
		j->p_stats.st_armsbomb++;
	    }
	    else if (rnd < 90) {
		l->pl_armies -= 2;
		j->p_kills += 0.04;
		j->p_stats.st_armsbomb += 2;
	    }
	    else  {
		l->pl_armies -= 3;
		j->p_kills += 0.06;
		j->p_stats.st_armsbomb += 3;
	    }

	    /* Send in a robot if there are no other defenders 
		and the planet is in the team's home space */

	    if ((tcount[l->pl_owner] == 0) && 
		(l->pl_flags & l->pl_owner) &&
		    tm_robots[l->pl_owner] == 0) {
			rescue(l->pl_owner, j->p_kills);
			tm_robots[l->pl_owner] = (1800 + 
			    (random() % 1800)) /
			    TEAMFUSE;
	    }
	}
    }
}

beam()
{
    register int i;
    register struct player *j;
    register struct planet *l;
    char buf[80];
    char buf1[80];

    for (i = 0, j = &players[i]; i < MAXPLAYER; i++, j++) {
	if ((j->p_status != PALIVE) || (!(j->p_flags & PFORBIT)))
	    continue;
	l = &planets[j->p_planet];
	if (j->p_flags & PFBEAMUP) {
	    if (l->pl_armies < 5)
		continue;
	    if (j->p_armies == j->p_ship.s_maxarmies)
		continue;
	    /* XXX */
	    if (j->p_armies == floor(j->p_kills * 2.0))
		continue;
	    if (j->p_team != l->pl_owner)
		continue;
	    if ((j->p_swar | j->p_hostile) & l->pl_owner)
		continue;
	    j->p_armies++;
	    l->pl_armies--;
	    continue;
	}
	if (j->p_flags & PFBEAMDOWN) {
	    if (j->p_armies == 0)
		continue;
	    if ((!((j->p_swar | j->p_hostile) & l->pl_owner))
		 && (j->p_team != l->pl_owner) && (l->pl_owner != NOBODY))
		continue;
	    if (j->p_team != l->pl_owner) {
		j->p_armies--;
		if (l->pl_armies) {
		    int oldowner = l->pl_owner;

		    /* start the war (if necessary) */
		    j->p_swar |= l->pl_owner;

		    l->pl_armies--;
		    j->p_kills += 0.02;
		    j->p_stats.st_armsbomb++;
		    if (l->pl_armies == 0) {
			/* Give planet to nobody */
			(void) sprintf(buf, "%s destroyed by %s (%c%x)",
			    l->pl_name,
			    j->p_name,
			    teamlet[j->p_team],
			    j->p_no);
			(void) sprintf(buf1, "%-3s->%-3s",
			    l->pl_name, teamshort[l->pl_owner]);
			pmessage(buf, l->pl_owner, MTEAM, buf1);
			l->pl_owner = NOBODY;
			checkgen(oldowner, j);
		    }
		}
		else { 	/* planet taken over */
		    l->pl_armies++;
		    j->p_stats.st_planets++;
		    j->p_kills += 0.25;
		    (void) sprintf(buf, "%s taken over by %s (%c%x)",
			l->pl_name,
			j->p_name,
			teamlet[j->p_team],
			j->p_no);
		    l->pl_owner = j->p_team;
		    l->pl_info = j->p_team;
		    checkwin(j);
		    /* now tell new team */
		    (void) sprintf(buf1, "%-3s->%-3s",
			l->pl_name, teamshort[l->pl_owner]);
		    pmessage(buf, l->pl_owner, MTEAM, buf1);
		}
	    }
	    else {
		j->p_armies--;
		l->pl_armies++;
	    }
	}

    }
}

blowup(sh)
struct player *sh;
{
    register int i;
    int dx, dy, dist;
    int damage;
    register struct player *j;

    for (i = 0, j = &players[i]; i < MAXPLAYER; i++, j++) {
	if (j->p_status != PALIVE)
	    continue;
	if (sh == j)
	    continue;
	/* the following keeps people from blowing up on others */
	me = sh;
	if ((me->p_whydead == KQUIT) && (friendlyPlayer(j)))
	    continue;
	dx = sh->p_x - j->p_x;
	dy = sh->p_y - j->p_y;
	if (ABS(dx) > DETDIST || ABS(dy) > DETDIST)
	    continue;
	dist = dx * dx + dy * dy;
	if (dist > DETDIST * DETDIST)
	    continue;
	if (dist > EXPDIST * EXPDIST) {
	    damage = 100 * (DETDIST - sqrt((double) dist)) /
		(DETDIST - EXPDIST);
	}
	else {
	    damage = 100;
	}
	if (damage > 0) {
	    if (j->p_flags & PFSHIELD) {
		j->p_shield -= damage;
		if (j->p_shield < 0) {
		    j->p_damage -= j->p_shield;
		    j->p_shield = 0;
		}
	    }
	    else {
		j->p_damage += damage;
	    }
	    if (j->p_damage >= j->p_ship.s_maxdamage) {
		j->p_status = PEXPLODE;
		j->p_explode = 10;
		if (!(j->p_flags & PFPRACTR)) {
		    sh->p_kills += 1.0 + 
			j->p_armies * 0.1 + j->p_kills * 0.1;
		}
		killmess(j, sh);
		j->p_stats.st_losses++;
		j->p_whydead = KSHIP;
		j->p_whodead = sh->p_no;
	    }
	}
    }
}

freemem()
{
    register int i;
    register struct player *j;

    /* Blow players out of the game */
    for (i = 0, j = &players[i]; i < MAXPLAYER; i++, j++) {
	j->p_status = PDEAD;
	j->p_whydead = KDAEMON;
	j->p_ntorp = 0;
    }
    save_planets();
    sleep(2);
#ifndef ibm032
    shmctl(shmid, IPC_RMID, (struct shmid_ds *) 0);
#else
    (*(long *)0xef100000) = 0;
#endif
    exit(0);
}

save_planets()
{
    if (plfd >= 0) {
	(void) lseek(plfd, (long) 0, 0);
	(void) write(plfd, (char *) planets, sizeof(pdata));
    }
}
/* This function checks to see if a team has been genocided --
   their last planet has been beamed down to zero.  It will set
   a timer on their home planet that will prevent them from
   couping for a random number of minutes.
*/
checkgen(loser, winner)
int loser;
struct player *winner;
{
    register int i;
    register struct planet *l;
    struct planet *homep;

    for (i = 0, l = &planets[i]; i < MAXPLANETS; i++, l++) {
	if (l->pl_owner == loser)
	    return;
	if (((l->pl_flags & ALLTEAM) == loser) && (l->pl_flags & PLHOME))
	    homep = l;		/* Keep track of his home planet for marking */
    }

    /* well, I guess they are losers.  Hose them now */
    winner->p_stats.st_genocides++;
    homep->pl_couptime = (18000 + (random() % 18000)) / PLANETFUSE;
}


/* This function is called when a planet has been taken over.
   It checks all the planets to see if the victory conditions
   are right.  If so, it blows everyone out of the game and
   resets the galaxy
*/
checkwin(winner)
struct player *winner;
{
    register int i, h;
    register struct planet *l;
    register struct player *j;
    int team[MAXTEAM + 1];

    for (i = 0; i < 4; i++)
	team[1<<i] = 0;
    
    for (i = 0, l = &planets[i]; i < MAXPLANETS; i++, l++)
	team[l->pl_owner]++;

    for (i = 0; i < 4; i++) {
	if (team[1<<i] >= VICTORY) {
	    /* We have a winning team */
	    for (h = 0, j = &players[0]; h < MAXPLAYER; h++, j++) {
		j->p_status = PDEAD;
		j->p_whydead = KWINNER;
		j->p_whodead = winner->p_no;
		j->p_ntorp = 0;
	    }
	    winner->p_stats.st_conqs++;
	    bcopy(pdata, planets, sizeof(pdata));
	    for (i = 0; i < 40; i++) {
		if (random() % 4 == 0)
		    planets[i].pl_flags |= PLREPAIR;
		if (random() % 2 == 0)
		    planets[i].pl_flags |= PLFUEL;
	    }
	    longjmp(env, 0);
	}
    }
}

pmessage(str, recip, group, addr)
char *str;
int recip;
int group;
char *addr;
{
    struct message *cur;

    if (++(mctl->mc_current) >= MAXMESSAGE)
	mctl->mc_current = 0;
    cur = &messages[mctl->mc_current];
    cur->m_no = mctl->mc_current;
    cur->m_flags = group;
    cur->m_time = ticks;
    cur->m_recpt = recip;
    (void) sprintf(cur->m_data, "%s %s", addr, str);
    cur->m_flags |= MVALID;
}

ghostmess(victim)
	struct player	*victim;
{
    char 		buf[80];
    static float	ghostkills = 0.0;

    ghostkills += 1.0 + victim->p_armies * 0.1 + victim->p_kills * 0.1;
    (void) sprintf(buf, "%s (%c%x) was kill %0.2f for the GhostBusters",
	victim->p_name, teamlet[victim->p_team], victim->p_no, ghostkills);
    pmessage(buf, 0, MALL, "GOD->ALL");
}

killmess(victim, killer)
struct player *victim, *killer;
{
    char buf[80];

    (void) sprintf(buf, "%s (%c%x) was kill %0.2f for %s (%c%x)",
	victim->p_name,
	teamlet[victim->p_team],
	victim->p_no,
	killer->p_kills,
	killer->p_name,
	teamlet[killer->p_team],
	killer->p_no);
    pmessage(buf, 0, MALL, "GOD->ALL");
}

/* Send in a robot to avenge the aggrieved team */
rescue(team, kills)
int team;
float kills;
{
    char *arg1;
    char *arg2 = "-lX";

    if (fork() == 0) {
	(void) close(0);
	(void) close(1);
	(void) close(2);
	(void) signal(SIGALRM, SIG_DFL);
	switch (team) {
	    case FED:
		arg1 = "-Tf";
		break;
	    case ROM:
		arg1 = "-Tr";
		break;
	    case KLI:
		arg1 = "-Tk";
		break;
	    case ORI:
		arg1 = "-To";
		break;
	}
	if (kills > 9.0)
	    kills = 9.0;
	(void) sprintf(arg2, "-l%d", (int) kills);
	execl(ROBOT, "robot", arg1, arg2, 0);
	fprintf(stderr,"daemon: couldn't execl %s\n",ROBOT);
	/* If we get here, we are hosed anyway */
    }
}

#include <sys/resource.h>

/* Don't fear the ... */

reaper(sig)
{
#ifdef hpux
    wait((int *) 0);
#else hpux
    while (wait3((union wait *) 0, WNOHANG, (struct rusage *) 0) > 0)
	;
#endif hpux
}

#ifdef hpux

srandom(foo)
int foo;
{
    rand(foo);
}

random()
{
    return(rand());
}

getrusage(foo, buf)
int foo;
struct rusage *buf;
{
    buf->ru_utime.tv_sec = 0;
    buf->ru_stime.tv_sec = 0;
}

#include <sys/signal.h>

int (*
signal(sig, funct))()
int sig;
int (*funct)();
{
    struct sigvec vec, oldvec;

    sigvector(sig, 0, &vec);
    vec.sv_handler = funct;
    sigvector(sig, &vec, (struct sigvec *) 0);
}

#endif hpux
