/*----------------------------------------------------------------------*
 | PROGRAM: xhanoi							|
 | AUTHOR:  Douglas Earl						|
 |	    Center for Information Technology Integration		|
 |	    University of Michigan					|
 |	    earl@citi.umich.edu						|
 |									|
 | DESCRIPTION: Interactive playing of the towers of hanoi game, or	|
 |    automatic solving of it.  Object: to move blocks from the		|
 |    first tower to the third tower.  Rules:				|
 |	1. can only move 1 block at a time				|
 |	2. can't put a larger block on top of a smaller block		|
 |    Quit with q or Q while in the window.				|
 |									|
 | COMPILE WITH: cc -o xhanoi -O xhanoi.c -lX				|
 |									|
 | BUGS: Resize events ignored, not able to specify -geom (due to	|
 |       hardcoded block locations).					|
 |									|
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/cursorfont.h>

#define MAX_BLOCKS 9
#define MAX_HEIGHT 30
#define MAX_WIDTH 100
#define DELTA_HEIGHT 5
#define DELTA_WIDTH 10
#define WINH 400
#define WINW 500
#define VOLUME 40

/* tower and yloc are 1-based arrays, i.e. the first piece of valid
   info is element 1, not 0,  This is done to be able to use num as a
   subscript into these arrays (e.g. tower[0][num[0]]--the block number
   of the highest block in tower 0).  When num is 0, we would get the 1st
   block in the tower if we used zero-based arrays, which is not what
   we want */
int tower[3][MAX_BLOCKS+1];	/* stack of block numbers for each tower */
int num[3];			/* number of blocks on each tower */
int xloc[3] = {50, 200, 350};
int yloc[MAX_BLOCKS+1] = {-1, 368, 336, 304, 272, 240, 208, 176, 144, 112};
char *str[MAX_BLOCKS] = {"Ann Arbor", "Michigan", "of", "Univ.", "CITI",
			 "", "", "", ""};
int blockid;			/* window id of last block drawn */
int wid, speed, top;
Display *disp;
GC gc;
XEvent  event;
char *help[] = {
"where options are:",
"    -d or -display host:disp	where to display",
"    -auto			xhanoi solves the puzzle automatically",
"    -s speed			with -auto: speed to move blocks (default 15)",
"    -n nBlocks			how many blocks in puzzle",
"    -r				reverse video (default is white on black)",
NULL};

/*--------------------------------------------------------------*
 |	Body of program						|
 *--------------------------------------------------------------*/

main(argc, argv)
    int argc;
    char **argv;

    {
    XSetWindowAttributes attr;
    XGCValues xgcv;
    Cursor cursor;
    Font font;
    char *display = NULL;
    int nblocks = 5;
    int autosolve = 0;
    int reverse = 0;
    int black, white;
    char **mess_ptr = help;


    speed = 15;
    while (*++argv) {
	if ((!strcmp(*argv, "-display")) || (!strcmp(*argv, "-d")))
	    display = *++argv;
	else if (!strcmp(*argv, "-s"))
	    speed = atoi(*++argv);
	else if (!strcmp(*argv, "-n")) {
	    nblocks = atoi(*++argv);
	    if (nblocks > MAX_BLOCKS) {
		fprintf(stderr, "Maximum number of blocks is %d\n",
			MAX_BLOCKS);
		exit(1);
		}
	    }
	else if (!strcmp(*argv, "-auto"))
	    autosolve = 1;
	else if (!strcmp(*argv, "-r"))
	    reverse = 1;
	else {
	    fprintf(stderr, "\nusage: xhanoi [-options]\n\n");
	    do
		fprintf(stderr, "%s\n", *mess_ptr);
		while (*++mess_ptr);
	    exit(1);
	    }
	}

    if (!(disp = XOpenDisplay(display))) {
	perror("Couldn't open display");
	exit(1);
	}

    if (reverse) {
	white = BlackPixel(disp, DefaultScreen(disp));
	black = WhitePixel(disp, DefaultScreen(disp));
        }
    else {
	black = BlackPixel(disp, DefaultScreen(disp));
	white = WhitePixel(disp, DefaultScreen(disp));
	}

    attr.background_pixel = black;
    attr.border_pixel = white;
    wid = XCreateWindow(disp, DefaultRootWindow(disp), 100, 100, WINW, WINH,
			2, 0, InputOutput, CopyFromParent,
			CWBackPixel | CWBorderPixel | CWOverrideRedirect,
			&attr);
    XStoreName(disp, wid, "xhanoi");
    XMapWindow(disp, wid);
    XSelectInput(disp, wid, KeyPressMask);

    font = XLoadFont(disp, "serifi10");
    xgcv.foreground = white;
    xgcv.background = black;
    xgcv.font = font;
    gc = XCreateGC(disp, wid, GCForeground | GCBackground | GCFont, &xgcv);

    cursor = XCreateFontCursor(disp, XC_man);
    XDefineCursor(disp, wid, cursor);

    num[0] = nblocks;  num[1] = 0;  num[2] = 0;
    DrawBlocks(nblocks, black, white, &top);
    if (autosolve) {
	sleep(1);
	while (1) {
	    Hanoi(nblocks, 0, 1, 2);
	    XSync(disp, 0);
	    sleep(2);
	    Hanoi(nblocks, 2, 1, 0);
	    XSync(disp, 0);
	    sleep(2);
	    }
	}
    else Interactive();
    }

/*--------------------------------------------------------------*
 |	Has a window become completely visible?			|
 *--------------------------------------------------------------*/

CheckVis(event)
    XEvent  event;

    {
    int vis_block, width, len;

    if (event.xvisibility.state == VisibilityUnobscured) {
	vis_block = blockid - event.xvisibility.window;
	width = MAX_WIDTH - (vis_block*DELTA_WIDTH);
	len = strlen(str[vis_block]);
	XDrawString(disp, event.xvisibility.window, gc,
		    width/2 - 4*len, MAX_HEIGHT/2+5,
		    str[vis_block], len);
	XSync(disp, 0);
	}
    }

/*--------------------------------------------------------------*
 |	Get user input for moves to make			|
 *--------------------------------------------------------------*/

Interactive()

    {
    Window root, child;
    int root_x, root_y, win_x, win_y, off_x, off_y, x, y;
    int curr_win, bnum;
    unsigned int mask;
    char ch;

    while (1) {
	XNextEvent(disp, &event);
	if (event.type == KeyPress) {
	    XLookupString(&event, &ch, 1, NULL, NULL);
	    if ((ch == 'q') || (ch == 'Q')) exit(0);
	    }
        else if (event.type == VisibilityNotify) CheckVis(event);
	else if (event.type == ButtonPress) {
	    curr_win = event.xbutton.window;
	    bnum = blockid - event.xbutton.window;
	    if ((bnum != tower[0][num[0]]) && (bnum != tower[1][num[1]]) &&
		(bnum != tower[2][num[2]])) {
		/* selected block not on top */
		XBell(disp, VOLUME);
		continue;
		}
	    XQueryPointer(disp, curr_win, &root, &child,
			&root_x, &root_y, &off_x, &off_y, &mask);
	    do {
		XQueryPointer(disp, wid, &root, &child, &root_x, &root_y,
			      &win_x, &win_y, &mask);
		/* add in offsets so cursor stays where it was clicked
		   in the block */
		win_x -= off_x;  win_y -= off_y;
		if ((win_x != x) || (win_y || y)) {
		    XMoveWindow(disp, curr_win, win_x, win_y);
		    x = win_x;  y = win_y;
		    }
		if (XPending(disp)) {
		    XNextEvent(disp, &event);
		    if (event.type == VisibilityNotify) CheckVis(event);
		    }
		} while (event.type != ButtonRelease);
	    UserMove(x, bnum);
	    }
	}
    }

/*--------------------------------------------------------------*
 |	Find user's destination tower and move it there		|
 *--------------------------------------------------------------*/

UserMove(x, bnum)
    int x, bnum;

    {
    int td, ts;  /* destination and source tower */

    td = (x <= 125 ? 0 : x <= 275 ? 1 : 2);
    ts = (bnum == tower[0][num[0]] ? 0 : bnum == tower[1][num[1]] ? 1 : 2);
    if (tower[td][num[td]] < bnum) {
	/* legal move -- td is empty or has a larger block */
	x = xloc[td]+(bnum*DELTA_WIDTH/2);
	num[td] += 1;
	XMoveWindow(disp, blockid - bnum, x, yloc[num[td]]);
	tower[td][num[td]] = tower[ts][num[ts]];
	num[ts] -= 1;
	}
    else {
	if (td != ts)
	    XBell(disp, VOLUME);
	x = xloc[ts]+(bnum*DELTA_WIDTH/2);
	XMoveWindow(disp, blockid - bnum, x, yloc[num[ts]]);
	}
    }

/*--------------------------------------------------------------*
 |	Solve the puzzle automatically				|
 *--------------------------------------------------------------*/

Hanoi(n, ts, tt, td)
    int n, ts, tt, td;  /* nblocks, source tower, temp tower, dest tower */

    {
    if (n==1) Move(ts, td);
    else
	{
	Hanoi(n-1, ts, td, tt);
	Move(ts, td);
	Hanoi(n-1, tt, ts, td);
	}
    }

/*--------------------------------------------------------------*
 |	Move block from tower ts to tower td			|
 *--------------------------------------------------------------*/

Move(ts, td)
    int ts, td;  /* tower source and tower destination */

    {
    int vel, dist, x, y, bnum;
    char ch;

    bnum = tower[ts][num[ts]];
    num[td] += 1;

    /* move the block above the tower */
    for (x = xloc[ts]+(bnum*DELTA_WIDTH/2), y = yloc[num[ts]]; y > top;
	 y -= speed) {
	    XMoveWindow(disp, blockid - bnum, x, y);
	    XSync(disp, 0);
	    }

    /* move it to the new tower */
    dist = abs(xloc[ts] - xloc[td]);
    if (xloc[ts] > xloc[td]) vel = -speed;  /* right to left */
    else vel = speed;   /* left to right */
    for (; dist > speed; x += vel, dist -= speed) {
	XMoveWindow(disp, blockid - bnum, x, y);
	XSync(disp, 0);
	}

    /* move it the final distance to the exact (x coord) location */
    x = xloc[td]+(bnum*DELTA_WIDTH/2);
    XMoveWindow(disp, blockid - bnum, x, y);

    /* move it back down */
    for (; y < yloc[num[td]]-speed; y += speed) {
	XMoveWindow(disp, blockid - bnum, x, y);
	XSync(disp, 0);
	}

    /* move it the final distance to the exact (y coord) location */
    XMoveWindow(disp, blockid - bnum, x, yloc[num[td]]);

    tower[td][num[td]] = tower[ts][num[ts]];
    num[ts] -= 1;

    /* check for quit or expose(s) */
    while (XPending(disp)) {
	XNextEvent(disp, &event);
	if (event.type == VisibilityNotify) CheckVis(event);
	else if (event.type == KeyPress) {
	    XLookupString(&event, &ch, 1, NULL, NULL);
	    if ((ch == 'q') || (ch == 'Q')) exit(0);
	    }
	}
    }

/*--------------------------------------------------------------*
 |	Initialize towers, draw the blocks			|
 *--------------------------------------------------------------*/

DrawBlocks(nblocks, black, white, top)
    int nblocks, black, white, *top;
    
    {
    int i, x, width, len, subwid, height;
    XSetWindowAttributes attr;

    /* allow 1-based subscripting of tower */
    tower[0][0] = -1;
    tower[1][0] = -1;
    tower[2][0] = -1;

    /* draw the towers */
    height = nblocks*MAX_HEIGHT + 25;
    for (i=0; i<=2; i++) {
	subwid = XCreateSimpleWindow(disp, wid,
	    i*150+95, WINH - height, 10, height, 1, white, black);
	XChangeWindowAttributes(disp, subwid, CWOverrideRedirect, &attr);
	XMapWindow(disp, subwid);
	}
    *top = WINH - (height + 10 + MAX_HEIGHT);

    /* draw the blocks */
    for (i = nblocks; i > 0; i--)
	{
	tower[0][i] = i-1;
	x = xloc[0]+((i-1)*DELTA_WIDTH/2);
	width = MAX_WIDTH - ((i-1)*DELTA_WIDTH);
        subwid = XCreateSimpleWindow(disp, wid, x, yloc[i],
	    width, MAX_HEIGHT, 1, white, black);
	XChangeWindowAttributes(disp, subwid, CWOverrideRedirect, &attr);
	XMapWindow(disp, subwid);

	XSelectInput(disp, subwid, ButtonPressMask | ButtonReleaseMask |
		     VisibilityChangeMask | KeyPressMask);
	len = strlen(str[i-1]);
	XDrawString(disp, subwid, gc, width/2 - 4*len, MAX_HEIGHT/2+5,
		    str[i-1], len);
	}
    blockid = subwid;
    XSync(disp, 0);
    }
