/* Dump or list fonts in a Macintosh resource file.
   This will run either on a Mac or on Unix (with a file
   containing the resource fork of a Mac file). */

char usage_string[]=
"usage: %s [-a] [-d dir] [-f] [-i id] [-n name]\n\
\t\t[-s size] [-t type] [-v] file\n";

/* Command line options */

int all;		/* Dump all resources of given type */
char *dirprefix;	/* directory prefix if non-NULL */
int force;		/* overwrite existing files */
int idmatch;		/* ID to match */
char *namatch;		/* name to match if non-NULL */
int namlen;		/* strlen(namatch) */
int sizematch;		/* size to match if non-zero */
char *type= "FONT";	/* resource type */
int verbose;		/* verbose output */
			/* implied by -a, -d, -f, -i, -n, -s */

#include <stdio.h>
#include <ctype.h>

#ifdef macintosh
#include "fcntl.h"
#define SYSV
#endif

#ifdef SYSV
#define index strchr
#endif

char *malloc();		/* libc interface */
char *index();

#define FALSE 0		/* Boolean constants */
#define TRUE 1

extern int optind;	/* getopt interface */
extern char *optarg;

char *progname;		/* Program name, for error messages */

short getshort();	/* Forward */
long getlong();
char *strdup();

/* Main program -- argument processing */

main(argc, argv)
	int argc;
	char **argv;
{
	progname= argv[0];
	
	for (;;) {
		int c= getopt(argc, argv, "ad:fi:n:s:t:xv");
		if (c == EOF)
			break;
		switch (c) {
		case 'a':
			all++;
			break;
		case 'd':
			dirprefix= optarg;
			break;
		case 'f':
			force++;
			break;
		case 'i':
			if (!isdigit(*optarg)) {
				fprintf(stderr, "nonumeric id\n");
				goto usage;
			}
			idmatch= atoi(optarg);
			break;
		case 'n':
			namatch= optarg;
			namlen= strlen(namatch);
			break;
		case 's':
			if (!isdigit(*optarg)) {
				fprintf(stderr, "nonumeric size\n");
				goto usage;
			}
			sizematch= atoi(optarg);
			break;
		case 't':
			if (strlen(type) != 4) {
				fprintf(stderr, "type must be 4 chars\n");
				goto usage;
			}
			type= optarg;
			break;
		case 'v':
			verbose++;
			break;
		default:
			goto usage;
		}
	}
	
	if (optind >= argc) {
 usage:
		fprintf(stderr, usage_string, progname);
		exit(2);
	}
	
	for (;optind < argc; optind++)
		process(argv[optind]);
	exit(0);
}

/* Process a resource file */

process(file)
	char *file;
{
	FILE *fp;
	char hdrbuf[256];
	long data_start;
	long map_start;
	long data_len;
	long map_len;
	char *map;
	char *typelist;
	int ntypes;
	char *namelist;
	int i;
	
#ifdef macintosh
	fp= fdopen(open(file, O_RSRC|O_RDONLY), "r");
#else
	fp= fopen(file, "r");
#endif
	if (fp == NULL)
		fatal("%s: can't open", file);
	
	if (fread(hdrbuf, 1, sizeof(hdrbuf), fp) != sizeof(hdrbuf))
		fatal("%s: can't read header", file);
	
	data_start= getlong(hdrbuf);
	map_start= getlong(hdrbuf+4);
	data_len= getlong(hdrbuf+8);
	map_len= getlong(hdrbuf+12);
	
	map= malloc((unsigned int)map_len);
	if (map == NULL)
		fatal("%s: can't allocate resource map (need 0x%lx bytes)",
			file, map_len);
	
	if (fseek(fp, map_start, 0) < 0)
		fatal("%s: can't seek to resource map", file);
	if (fread(map, 1, (int)map_len, fp) != map_len)
		fatal("%s: can't read resource map", file);
	
	typelist= map + getshort(map+24);
	ntypes= getshort(typelist) + 1;
	namelist= map + getshort(map+26);
	
	if (verbose)
		printf("Type    N offset\n");
	
	for (i= 0; i < ntypes; i++) {
		char *typentry= typelist + 2 + 8*i;
		int nres= getshort(typentry+4) + 1;
		int offset= getshort(typentry+6);
		if (verbose) {
			int j;
			for (j= 0; j < 4; j++)
				safeputc(typentry[j]);
			printf(" %4d %4d\n", nres, offset);
		}
		if (strncmp(typentry, type, 4) == 0)
			dumpfonts(fp, data_start,
				typelist + offset, nres, namelist);
	}
	
	fclose(fp);
}

char *fontname[256];

/* Dump font resources in detail */

dumpfonts(fp, data_start, rsrclist, nres, namelist)
	FILE *fp;
	long data_start;
	char *rsrclist;
	int nres;
	char *namelist;
{
	char namebuf[256];
	int i;
	
	if (verbose)
		printf("\t   ID nmof at dataof fID size name\n");
	
	for (i= 0; i < nres; i++) {
		char *name= namebuf;
		char *resentry= rsrclist + 12*i;
		int id= getshort(resentry);
		short name_offset= getshort(resentry+2);
		int attr= resentry[4] & 0xff;
		long data_offset= getlong(resentry+4) & 0xffffff;
		int font_id= id/128;
		int ptsize= id%128;
		
		if (verbose) {
			printf("\t%5d %4d %02x %6ld %3d %3d",
				id, name_offset, attr, data_offset,
				font_id, ptsize);
		}
		
		if (name_offset != -1) {
			strncpy(name, namelist + name_offset + 1,
				namelist[name_offset] & 0xff);
			name[namelist[name_offset] & 0xff]= '\0';
			if (verbose)
				printf("  \"%s\"", name);
			sanitize(name);
			fontname[font_id]= strdup(name);
		}
		else
			name= fontname[font_id];
		
		if (verbose)
			printf("\n");
		
		if (ptsize > 0 && selected(name, ptsize, id)) {
			if (name == NULL) {
				fprintf(stderr, "%s: no name for ID %d\n",
					progname, id);
				sprintf(namebuf, "%s%d", type, font_id);
				sanitize(namebuf);
				fontname[font_id]= strdup(namebuf);
				name= namebuf;
			}
			dumpfont(fp, name, ptsize,
				data_start + data_offset);
		}
	}
}

/* See whether the given resource should be dumped */

selected(name, ptsize, id)
	char *name;
	int ptsize;
	int id;
{
	if (all)
		return TRUE;
	if (sizematch != 0 && ptsize == sizematch)
		return TRUE;
	if (idmatch != 0 && id == idmatch)
		return TRUE;
	if (namatch != NULL && strncmp(name, namatch, namlen) == 0)
		return TRUE;
	return FALSE;
}

/* Sanitize a font name:
   Turn into lower case, replace funny chars by underscore */

#define FUNNY " /."

sanitize(p)
	char *p;
{
	for (; *p != '\0'; p++) {
		if (index(FUNNY, *p))
			*p= '_';
		else if (isupper(*p))
			*p= tolower(*p);
	}
}

/* Dump a font resource */

dumpfont(fp, name, ptsize, offset)
	FILE *fp;
	char *name;
	int ptsize;
	long offset;
{
	char file[256];
	FILE *ofp;
	static char buffer[10240];
	long length;
	int n;
	
	if (dirprefix == NULL)
		sprintf(file, "%s.%d", name, ptsize);
	else {
#ifdef macintosh
		sprintf(file, "%s%s.%d", dirprefix, name, ptsize);
#else
		sprintf(file, "%s/%s.%d", dirprefix, name, ptsize);
#endif
	}
	if (!force && access(file, 0) == 0) {
		fprintf(stderr, "%s: won't overwrite file %s\n",
			progname, file);
		return;
	}
	ofp= fopen(file, "w");
	if (ofp == NULL) {
		fprintf(stderr, "%s: can't create file %s\n",
			progname, file);
		return;
	}
	fseek(fp, offset, 0);
	if (fread(buffer, sizeof(long), 1, fp) != 1)
		fatal("can't read resource length");
	length= getlong(buffer);
	if (verbose)
		printf("Writing %ld bytes to file %s\n", length, file);
	while (length > 0) {
		n= sizeof buffer;
		if (n > length)
			n= length;
		if (fread(buffer, 1, n, fp) != n)
			fatal("can't read resource data");
		if (fwrite(buffer, 1, n, ofp) != n)
			fatal("can't write resource data to file %s", file);
		length -= n;
	}
	fclose(ofp);
}

/* Get a short in big-endian format from a buffer */

short
getshort(p)
	unsigned char *p;
{
	return (p[0] << 8) | p[1];
}

/* Get a long in big-endian format from a buffer */

long
getlong(p)
	unsigned char *p;
{
	return (((((p[0] << 8) | p[1]) << 8) | p[2]) << 8) | p[3];
}

/* Output a character if printable, else '?' */

safeputc(c)
	int c;
{
	if (c < ' ' || c >= 0x7f)
		c= '?';
	putchar(c);
}

/* Print fatal error message and exit */

fatal(fmt, a1, a2, a3, a4)
	char *fmt;
{
	fprintf(stderr, "%s: fatal: ", progname);
	fprintf(stderr, fmt, a1, a2, a3, a4);
	fprintf(stderr, "\n");
	exit(1);
}

/* Copy a string into malloc'ed storage */

char *
strdup(s)
	char *s;
{
	char *p= malloc(strlen(s) + 1);
	if (p != NULL)
		strcpy(p, s);
	return p;
}

#ifdef macintosh

/* Makeshift 'access', only works for the one call above */

access(file, flags)
	char *file;
	int flags;
{
	FILE *fp= fopen(file, "r");
	if (fp == NULL)
		return -1;
	fclose(fp);
	return 0;
}

/* You'll have to get the getopt source from unix */

#endif
