/* Convert Macintosh font to bdf format.
   Assumes the Mac font is on a file named "fontname.pointsize",
   in the format of a Mac FONT resource.
   
   Comments here and there document various aspects of the Mac
   font resource format, but for the complete truth I refer to
   Inside Macintosh, part I, chapter 7 (Font Manager), especially pp.
   I-227 and further (Format Of A Font).  See also part IV, chapter 5
   for additional information about Mac+ fonts.
   
   Public Domain by Guido van Rossum, February 1988.
   
   Based on a program by Craig Leres (leres@ucbarpa.berkeley.edu) to
   convert fonts in vfont format to bdf.
   
   DISCLAIMER: I have not really studied the bdf format.
   I do, however, believe that I have understood the Macintosh FONT
   resource format well. */

#include <stdio.h>
#include <strings.h>
#include <ctype.h>

long ftell();
char *malloc();

/* Lay-out of a Font Record header */

struct FontRec {
	unsigned short fontType;	/* PROPFONT or FIXEDFONT */
	short	firstChar,
		lastChar,
		widMax,
		kernMax,		/* Negative of max kern */
		nDescent,		/* negative of descent */
		fRectWidth,
		fRectHeight,
		owTLoc,			/* Offset to offset/width table [*] */
		ascent,
		descent,
		leading,
		rowWords;		/* Row width of bit image in words */
    
    /* Variable-length tables */
    
    /*	short	bitImage[rowWords][fRectHeight]; */
    /*	short	locTable[lastChar+3-firstChar]; */
    /*	short	owTable[lastchar+3-firstChar]; */
    
    /* [*]	owTLoc is the distance, IN WORDS, from itself to
    		the start of owTable */
};

/* Mac font types (magic words) */

#define PROPFONT	0x9000
#define FIXEDFONT	0xB000

char *progname;				/* Program name, for errors */

/* Fatal error message */

static void
punt(str)
char *str;
{
	fprintf(stderr, "%s: %s\n", progname, str);
	exit(1);
}

/* Determine byte order */

#define LIL_ENDIAN	1234		/* VAX style */
#define BIG_ENDIAN	4321		/* 68000 style */

int endian;

static void
endianism()
{
	union {
		short s;
		char c[2];
	} u;
	
	u.c[0]= 1;
	u.c[1]= 2;
	
	switch (u.s) {
	case 0x0201:
		endian= LIL_ENDIAN;
		break;
	case 0x0102:
		endian= BIG_ENDIAN;
		break;
	}
}

/* Macros to access the high and low byte of a short int */

#define HIBYTE(word) (((word)>>8) & 0xff)
#define LOBYTE(word) ((word)&0xff) 

/* Swap bytes from big-endian to this machine's type.
   The input data is assumed to be always in big-endian format.
   Byte order is determined dynamically so we need not know
   the machine's byte order at compile time */

static void
byteswap(array, nshorts)
	register unsigned short *array;
	register int nshorts;
{
	if (endian == 0)
		endianism(); /* Determine machine's endianism if necessary */
	if (endian == BIG_ENDIAN)
		return; /* No need to swap */
	for (; nshorts-- >= 0; array++)
		*array= HIBYTE(*array) | (LOBYTE(*array) << 8);
}

/* Mainline */

static void
mactobdf(macfile, macname, bdffile, bdfname)
	FILE *macfile;
	char *macname;
	FILE *bdffile;
	char *bdfname;
{
	struct FontRec hdr;	/* Font Record header */
	int fontsize;		/* point size of font */
	int nchars;		/* number of chars including 'missing char' */
	unsigned short *bits;	/* bit image */
	int nwords;		/* bit image size, in words */
	short *loctab;		/* location table */
	short *owtab;		/* offset/width table */
	int ndef;		/* number of characters really defined */
	int i;

	/* Read mac font header */
	if (fread((char*) &hdr, sizeof hdr, 1, macfile) != 1)
		punt("Error reading macfont header.");
	byteswap((short*) &hdr, sizeof hdr / sizeof(short));

	/* Check magic number.
	   The low two bits are masked off; newer versions of the
	   Font Manager use these to indicate the presence of optional
	   'width' and 'height' tables.  These are for fractional
	   character spacing, which I don't see how to use in bdf format
	   anyway. */
	if ((hdr.fontType & ~3) != PROPFONT &&
		(hdr.fontType & ~3) != FIXEDFONT) {
		fprintf(stderr, "Magic number: 0x%04x\n", hdr.fontType);
		punt("Bad magic number.");
	}

	/* Compute font size from mac file name, if possible */
	if (sscanf(macname, "%*[^.].%d", &fontsize) != 1) {
		fontsize= 12;
		fprintf(stderr, "%s: warning, using default point size %d\n",
			progname, fontsize);
	}
	
	nchars= (hdr.lastChar - hdr.firstChar + 1) + 1;
		/* One extra for "missing character image" */

	/* Start spewing stuff out */
	fprintf(bdffile, "STARTFONT 2.1\n");
	fprintf(bdffile, "FONT Apple Macintosh %s\n", macname);
	fprintf(bdffile, "SIZE %d 72 72\n", fontsize);
	fprintf(bdffile, "FONTBOUNDINGBOX %d %d %d %d\n",
	    hdr.fRectWidth, hdr.fRectHeight, hdr.kernMax, -hdr.descent);
	fprintf(bdffile, "STARTPROPERTIES 4\n");
	fprintf(bdffile, "FONT_ASCENT %d\n", hdr.ascent);
	fprintf(bdffile, "FONT_DESCENT %d\n", hdr.descent);
	fprintf(bdffile, "DEFAULT_CHAR %d\n", hdr.lastChar + 1);
	fprintf(bdffile, "COPYRIGHT \"Copyright Apple Computer.\"\n");
	fprintf(bdffile, "ENDPROPERTIES\n");
	
	/* Allocate memory for the various tables.
	   The location and offset/width tables have one entry more
	   than there are characters in the font,the latter being
	   a sentinel of some sort.  And remember that there is one
	   character in the font more than #{firstChar..lastChar},
	   since the missing character image is always present */
	
	nwords= hdr.rowWords * hdr.fRectHeight;
	bits= (unsigned short *) malloc(nwords * sizeof(short));
	loctab= (short *) malloc((nchars+1) * sizeof(short));
	owtab= (short *) malloc((nchars+1) * sizeof(short));
	if (bits == NULL || loctab == NULL || owtab == NULL)
		punt("Can't allocate memory for tables.");
	
	/* Read the tables.  They follow sequentially in the file */
	
	if (fread((char*)bits, sizeof(short), nwords, macfile) != nwords)
		punt("Can't read entire bit image.");
	if (fread((char*)loctab, sizeof(short), nchars+1, macfile) != nchars+1)
		punt("Can't read entire location table.");
	if (fread((char*)owtab, sizeof(short), nchars+1, macfile) != nchars+1)
		punt("Can't read entire offset/width table.");
	
	/* Note -- there may be excess data at the end of the file
	   (the optional width or height tables) */
	
	/* Byteswap the tables */
	byteswap(bits, nwords);
	byteswap(loctab, nchars+1);
	byteswap(owtab, nchars+1);
	
	/* Notes on the tables.
	
	   Table 'bits' contains a bitmap image of the entire font.
	   There are fRectHeight rows, each rowWords long.
	   The high bit of a word is leftmost in the image.
	   The characters are placed in this image in order of their
	   ASCII value.  The last image is that of the "missing
	   character"; every Mac font must have such an image
	   (traditionally a maximum-sized block).
	   
	   The location table (loctab) and offset/width table (owtab)
	   have one entry per character in the range firstChar..lastChar,
	   plus two extra entries: one for the "missing character" image
	   and a terminator.  They describe, respectively, where to
	   find the character in the bitmap and how to interpret it with
	   respect to the "character origin" (pen position on the base
	   line).
	   
	   The location table entry for a character contains the bit (!)
	   offset of the start of its image data in the font's bitmap.
	   The image data's width is computed by subtracting the start
	   from the start of the next character (hence the terminator).
	   
	   The offset/width table contains -1 for undefined characters;
	   for defined characters, the high byte contains the character
	   offset (distance between left of character image and
	   character origin), and the low byte contains the character
	   width (distance between the character origin and the origin
	   of the next character on the line). */
	
	/* Compute and print the number of defined characters */
	
	ndef= 0;
	for (i= 0; i < nchars; ++i) {
		if (owtab[i] != -1)
			++ndef;
	}
	fprintf(bdffile, "CHARS %d\n", ndef);
	
	/* Print the individual character definitions */
	
	for (i = 0; i < nchars; ++i) {
		int ascii;
		int width, height;
		int j;

		/* check to see if this character is defined */
		if (owtab[i] == -1)
			continue;
		
		/* ASCII value */
		ascii = i + hdr.firstChar;
		
		if (isprint(ascii) && !isspace(ascii))
			fprintf(bdffile, "STARTCHAR %c\n", ascii);
		else
			fprintf(bdffile, "STARTCHAR C%03o\n", ascii);
		fprintf(bdffile, "ENCODING %d\n", ascii);

		/* width in pixels */
		width = LOBYTE(owtab[i]);

		fprintf(bdffile, "SWIDTH %d %d\n", fontsize * width, 0);
		fprintf(bdffile, "DWIDTH %d %d\n", width, 0);

		/* height in pixels */
		height = hdr.fRectHeight;

		fprintf(bdffile, "BBX %d %d %d %d\n",
		    width, height, HIBYTE(owtab[i]), -hdr.descent);

		fprintf(bdffile, "BITMAP\n");
		
		for (j= 0; j < hdr.fRectHeight; ++j) {
			/* If you must make it faster,
			   expand putbits in-line! */
			putbits(bdffile, bits + j*hdr.rowWords,
				loctab[i], loctab[i+1] - loctab[i]);
		}
		
		fprintf(bdffile, "ENDCHAR\n");
	}
	
	fprintf(bdffile, "ENDFONT\n");

}

/* Get bit i of a scan line */

#define GETBIT(scanline, i) ((scanline[(i)/16] >> (15 - (i)%16)) & 1)

/* Extract a row of a character's image from a scan line and output it.
   The image starts at bit 'first' and is 'nbits' wide */

putbits(bdffile, scanline, first, nbits)
	FILE *bdffile;
	unsigned short *scanline;
	int first, nbits;
{
	register int i;
	register int byte= 0; /* Shift register */
	register int left= 8; /* Free slots in shift register */
	
	/* Avoid outputting blank lines by shifting a zero in */
	if (nbits == 0)
		--left;
	
	/* Shift bits in the shift register, emptying it every 8 bits.
	   Most of the program's execution time is spent in this loop */
	
	for (i= first; i < first+nbits; ++i) {
		byte= (byte<<1) | GETBIT(scanline, i);
#ifdef DEBUG
		/* Debug code to see whether I got it right.
		   If this comes out as pretty character images,
		   I have interpreted the table format correctly. */
		if (byte&1)
			putc('x', stderr);
		else
			putc(' ', stderr);
#endif
		if (--left == 0) {
			fprintf(bdffile, "%02x", byte);
			byte= 0;
			left= 8;
		}
	}
	
	/* Flush shift register (after shifting in 'left' zeros) */
	if (left < 8)
		fprintf(bdffile, "%02x", byte << left);
	
#ifdef DEBUG
	putc('\n', stderr);
#endif
	putc('\n', bdffile);
}

main(argc, argv)
	int argc;
	char **argv;
{
	char *macname, *bdfname;
	FILE *macfile, *bdffile;
	
	progname= argv[0];
	
	if (argc != 3) {
		fprintf(stderr, "usage: %s macname bdfname\n", progname);
		exit(2);
	}

	macname = argv[1];
	bdfname = argv[2];

	if ((macfile = fopen(macname, "r")) == NULL) {
		perror(macname);
		exit(1);
	}

	if ((bdffile = fopen(bdfname, "w")) == NULL) {
		perror(bdfname);
		exit(1);
	}

	mactobdf(macfile, macname, bdffile, bdfname);

	fclose(macfile);
	fclose(bdffile);

	exit(0);
}
