/* 
 * transform.c - several transform handling routines
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include "X.h"
#include "pubstr.h"
#include "pexarray.h"
#include "renderer.h"

#ifdef min
#undef min
#endif

#ifdef max
#undef max
#endif

void multm44();


/*****************************************************************
 * TAG( PexTransformPoint )
 * 
 * Transforms 3D modeling coordinates to clipping coordinates.
 * 
 * Inputs:
 * 	pt1 is untransformed coordinate
 * Outputs:
 * 	npt1 is transformed 4D coordinate
 * Assumptions:
 * 	...
 * Algorithm:
 * 	...
 */

void
PexTransformPoint(pRend, pt1, npt1)
    pexRendererPtr pRend;
    pexCoord3D *pt1;			/* Untransformed 3D point */
    pexCoord4D *npt1;			/* Transformed 4D point */
{
	pexCoord4D tmp1, tmp2;
	register pexPipelineContextPtr pc = pRend->pPC;
	int n = pc->viewIndex;
        register pexViewEntry *view = &(pRend->viewTable[n]);

	/* Transform through local & global */
	pt3xm44( pt1, pc->localTransform, &tmp1 );
	pt4xm44( &tmp1, pc->globalTransform, &tmp2 );

	/* Transform to VRC coordinates */
	pt4xm44( &tmp2, view->orientation, &tmp1 );

	/* Map view volume to NPC view clipping limits */
	pt4xm44( &tmp1, view->mapping, &tmp2 );

	/* Transform view clipping limits to -1->1, -1->1, 0->1 cube.
	 * This is the cube we actually clip on.
	 */
	pt4xm44( &tmp2, pRend->clipMatrix[n], npt1 );

}

/*****************************************************************
 * TAG( PexTransformPoints )
 * 
 * Transforms 3D modeling coordinates to clipping coordinates.
 * 
 * Inputs:
 * 	pRend:	Pointer to renderer to get transforms from.
 * 	pts:	Pointer to a pexArray1 describing the input points.
 * Outputs:
 * 	npts:	Pointer to a pexArray1 describing the output points.
 * Assumptions:
 * 	pts->n == npts->n.  pts->n is used as the number of points to
 * 	transform.
 * Algorithm:
 * 	Concatenate all the matrices together and map each point
 * 	through the combined matrix.
 */

void
PexTransformPoints(pRend, pts, npts)
    pexRendererPtr pRend;
    pexArray1 *pts;		/* Untransformed 3D points */
    pexArray1 *npts;		/* Transformed 4D point */
{
    pexPipelineContextPtr pc = pRend->pPC;
    int n = pc->viewIndex;
    pexViewEntry *view = &(pRend->viewTable[n]);
    pexMatrix ctm, mtmp;
    register int numPts = pts->n;
    register pexCoord3D * pt1;
    register pexCoord4D * npt1;

    /* Concatenate all matrices together */
    multm44( pc->localTransform, pc->globalTransform, mtmp );
    multm44( mtmp, view->orientation, ctm );
    multm44( ctm, view->mapping, mtmp );
    multm44( mtmp, pRend->clipMatrix[n], ctm );

    /* now loop and transform */
    pt1 = (pexCoord3D *)PexArray1Index(pts, 0);
    npt1 = (pexCoord4D *)PexArray1Index(npts, 0);
    while ( --numPts >= 0 )
    {
	pt3xm44( pt1, ctm, npt1 );
	pt1 = (pexCoord3D *)PexArray1Next(pts, pt1);
	npt1 = (pexCoord4D *)PexArray1Next(npts, npt1);
    }
}

/*****************************************************************
 * TAG( PexTransformNorm )
 * 
 * Transforms a normal vector from modeling coordinates to world coordinates.
 * 
 * Inputs:
 * 	nrm is untransformed normal vector
 * Outputs:
 * 	xnrm is transformed normal vector
 * Assumptions:
 * 	See NOTE.
 * Algorithm:
 * 	Pushes normal through upper 3x3 of concatenated local & global
 * 	transforms.
 * 	NOTE: This is not strictly correct.  Really should take
 * 	inverse transpose.  Only matters if differential scaling is present.
 */

void
PexTransformNorm(pRend, nrm, xnrm)
    pexRendererPtr pRend;
    pexVector3D *nrm;			/* Untransformed normal vector */
    pexVector3D *xnrm;			/* Transformed vector */
{
	pexCoord4D tmp1, tmp2;
	register pexPipelineContextPtr pc = pRend->pPC;
	int n = pc->viewIndex;
        register pexViewEntry *view = &(pRend->viewTable[n]);

	/* Transform through local & global */
	*(pexVector3D *)&tmp2 = *nrm;
	tmp2.w = 0;
	pt4xm44( &tmp2, pc->localTransform, &tmp1 );
	tmp1.w = 0;
	pt4xm44( &tmp1, pc->globalTransform, &tmp2 );

	*xnrm = *(pexVector3D *)&tmp2;
}

/*****************************************************************
 * TAG( PexTransformNorms )
 * 
 * Transforms Normal vectors from modeling to world coordinates.
 * 
 * Inputs:
 * 	pRend:	renderer to get transformation matrices from.
 * 	nrms is pointer to pexArray1 of untransformed normals.
 * Outputs:
 * 	xnrms is pointer to pexArray1 of transformed normals.
 * Assumptions:
 * 	nrms->n == xnrms->n.  nrms->n is used as number of normals.
 * 	See above.
 * Algorithm:
 * 	See above.
 */

void
PexTransformNorms(pRend, nrms, xnrms)
    pexRendererPtr pRend;
    pexArray1 *nrms;
    pexArray1 *xnrms;
{
    pexPipelineContextPtr pc = pRend->pPC;
    int n = pc->viewIndex;
    pexViewEntry *view = &(pRend->viewTable[n]);
    pexMatrix ctm;
    pexCoord4D tmp1, tmp2;
    register pexVector3D * nrm1, * xnrm1;
    register int numNrm = nrms->n;

    /* Concatenate all matrices together */
    multm44( pc->localTransform, pc->globalTransform, ctm );

    /* now loop and transform */
    nrm1 = (pexVector3D *)PexArray1Index(nrms, 0);
    xnrm1 = (pexVector3D *)PexArray1Index(xnrms, 0);
    while ( --numNrm >= 0 )
    {
	*(pexVector3D *)&tmp2 = *nrm1;
	tmp2.w = 0;
	pt4xm44( &tmp2, ctm, &tmp1 );
	*xnrm1 = *(pexVector3D *)&tmp1;
	nrm1 = (pexVector3D *)PexArray1Next(nrms, nrm1);
	xnrm1 = (pexVector3D *)PexArray1Next(xnrms, xnrm1);
    }
}


/*****************************************************************
 * TAG( PexTransformNpcToDev )
 * 
 * Transforms npc coordinates to device coordinates.
 * 
 * Inputs:
 * 	npcx, npcy;         Npc coordinates to transform.
 * Outputs:
 * 	*devx, *devy;       Transformed device coordinates
 * Assumptions:
 * 	Npcx and npcy coming into this routine are in clipping
 * 	coordinates -1->1, -1->1.  They must first be mapped to
 * 	the Npc subvolume.  Then mapped on to the X window.
 * Algorithm:
 * 	Dave sez: "This could be made much more efficient."
 */

PexTransformNpcToDev(pRend, npcx, npcy, devx, devy )
pexRendererPtr pRend;
float npcx, npcy;		/* Npc coordinate to transform */
INT16 *devx, *devy;		/* Transformed device coordinate */
{
	FLOAT llx, lly, urx, ury;
	FLOAT cx, cy, wx, wy;
	int n = pRend->pPC->viewIndex;
	register pexViewEntry *view = &(pRend -> viewTable[n]);

	/* Find the Npc-Subvolume limits */
	llx = view->clipLimits.minval.x;
	lly = view->clipLimits.minval.y;
	urx = view->clipLimits.maxval.x;
	ury = view->clipLimits.maxval.y;
	cx = (llx+urx) / 2.0;
	cy = (lly+ury) / 2.0;
	wx = (urx-llx);
	wy = (ury-lly);

	/* Map to the Npc-Subvolume */
	npcx = npcx * wx/2.0 + cx;
	npcy = npcy * wy/2.0 + cy;

	/* Map 0->1, 0->1 Npc Space to the X-window
	 * Here is DP's old version...
	 *     *devx =  (npcx-0.5) * Wscale + Woffset.x;
	 *     *devy = -(npcy-0.5) * Wscale + Woffset.y;
	 */
	*devx = (pRend -> viewport.maxval.x - pRend -> viewport.minval.x) *
	    npcx + pRend -> viewport.minval.x;
	*devy = (pRend -> viewport.maxval.y - pRend -> viewport.minval.y) *
	    (1.0 - npcy) + pRend -> viewport.minval.y;
}


/*****************************************************************
 * TAG( pt4xm44 )
 * 
 * Multiply a coordinate by a matrix.
 * 
 * Inputs:
 * 	pt;          Coordinate to transform.
 * 	m;           Matrix to multiply by.
 * Outputs:
 * 	npt;         Transformed point.
 * Assumptions:
 * 	...
 * Algorithm:
 * 	...
 */
	
pt4xm44(pt, m, npt)
    register pexCoord4D *pt;
    register pexMatrix m;
    register pexCoord4D *npt;
{
    npt->x = pt->x*m[0][0] + pt->y*m[1][0] + pt->z*m[2][0] + pt->w*m[3][0];
    npt->y = pt->x*m[0][1] + pt->y*m[1][1] + pt->z*m[2][1] + pt->w*m[3][1];
    npt->z = pt->x*m[0][2] + pt->y*m[1][2] + pt->z*m[2][2] + pt->w*m[3][2];
    npt->w = pt->x*m[0][3] + pt->y*m[1][3] + pt->z*m[2][3] + pt->w*m[3][3];
}


/*****************************************************************
 * TAG( pt3xm44 )
 * 
 * Multiply a coordinate by a matrix.
 * 
 * Inputs:
 * 	pt;          Coordinate to transform.
 * 	m;           Matrix to multiply by.
 * Outputs:
 * 	npt;         Transformed point.
 * Assumptions:
 * 	...
 * Algorithm:
 * 	...
 */

pt3xm44(pt, m, npt)
    register pexCoord3D *pt;
    register pexMatrix m;
    register pexCoord4D *npt;
{
    npt->x = pt->x*m[0][0] + pt->y*m[1][0] + pt->z*m[2][0] + m[3][0];
    npt->y = pt->x*m[0][1] + pt->y*m[1][1] + pt->z*m[2][1] + m[3][1];
    npt->z = pt->x*m[0][2] + pt->y*m[1][2] + pt->z*m[2][2] + m[3][2];
    npt->w = pt->x*m[0][3] + pt->y*m[1][3] + pt->z*m[2][3] + m[3][3];
}


/*****************************************************************
 * TAG( identm44 )
 * 
 * Set a mtrix to the identity.
 * 
 * Inputs:
 * 	 m;            Matrix to set to identity.
 * Outputs:
 * 	 m is identity.
 * Assumptions:
 * 	...
 * Algorithm:
 * 	...
 */

identm44( m )
    register pexMatrix m;
{
    int i,j;

    for(i=0; i<4; i++)
    {
        for(j=0; j<4; j++)
	    m[i][j] = 0.0;
	m[i][i] = 1.0;
    }
}


/* 
 * Vectm4(row,col) multiplies the row and col vectors, which are array[4]s 
 */
#define Vectm4(m1,row,m2,col) (m1[row][0]*m2[0][col])+\
                              (m1[row][1]*m2[1][col])+\
                              (m1[row][2]*m2[2][col])+\
                              (m1[row][3]*m2[3][col])
/*****************************************************************
 * TAG( multm44 )
 * 
 * multiply the first 4x4 matrix paramater by the second 4x4 matrix parameter.
 * 
 * Inputs:
 * 	m1, m2 - 4x4 matrices of type pexMatrix
 * Outputs:
 * 	returns the product of m1 and m2 in result
 * Assumptions:
 *	m1 and m2 should not be equal to result.
 * Algorithm:
 *	[None]
 */

void
multm44(m1, m2, result)
    register pexMatrix m1, m2, result;
{
    if ((m1 == result) || (m2 == result))
	printf("Hey!! you are doing math wrong!!! in (%s, %s)\n",
	       __LINE__, __FILE__);
    result[0][0] = Vectm4 (m1,0,m2,0);
    result[0][1] = Vectm4 (m1,0,m2,1);
    result[0][2] = Vectm4 (m1,0,m2,2);
    result[0][3] = Vectm4 (m1,0,m2,3);
    result[1][0] = Vectm4 (m1,1,m2,0);
    result[1][1] = Vectm4 (m1,1,m2,1);
    result[1][2] = Vectm4 (m1,1,m2,2);
    result[1][3] = Vectm4 (m1,1,m2,3);
    result[2][0] = Vectm4 (m1,2,m2,0);
    result[2][1] = Vectm4 (m1,2,m2,1);
    result[2][2] = Vectm4 (m1,2,m2,2);
    result[2][3] = Vectm4 (m1,2,m2,3);
    result[3][0] = Vectm4 (m1,3,m2,0);
    result[3][1] = Vectm4 (m1,3,m2,1);
    result[3][2] = Vectm4 (m1,3,m2,2);
    result[3][3] = Vectm4 (m1,3,m2,3);
}
