/* 
 * nurbs.c - NURBS rendering code.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
#include <stdio.h>
#ifdef TESTING
#include <math.h>
#endif
#include "misc.h"
#include "PEX.h"
#include "PEXproto.h"
#include "renderer.h"
#include "pc.h"
#include "nurb.h"
#include "polyline.h"
#include "quadmesh.h"
#include "alphamat.h"

#ifdef TESTING
#define Xalloc	malloc
#define Xfree	free
#endif

/*****************************************************************
 * TAG( mk_type2_kv )
 * 
 * Make a new knot vector for type 2 (constant between knots)
 * evaluation of a curve or surface.
 * Inputs:
 * 	ctlval:		The "control value".  If <= 0, evaluate only
 * 			at the knots.  If > 0, evaluate at the knots
 * 			and at the given number of points between the knots.
 * 	order:		Order of the spline.
 * 	tauvec:		The original knot vector.
 * 	ntau:		Size of tauvec.
 * Outputs:
 * 	tvec:		The new knot vector will be returned here.
 * 	nt:		Size of the new knot vector will be returned here.
 * 	jvec:		A "J vector" for passing to calc_alpha will be
 * 			returned here.
 * 	nj:		Size of jvec will be returned here.
 * Assumptions:
 * 	ntau >= 2*order.
 * 	order >= 2.
 * 	Knots in tauvec are in non-descending sequence.
 * Algorithm:
 * 	Replicate each interior knot in tauvec order-1 times.  If
 * 	ctlval > 0, insert ctlval new order-1-fold knots between each
 * 	pair of original interior knots.  jvec[*]+1 is the start of
 * 	each of the new multiple knots.
 */
mk_type2_kv( ctlval, order, tauvec, ntau, tvec, nt, jvec, nj )
int ctlval;
int order;
FLOAT tauvec[];
int ntau;
FLOAT **tvec;
int *nt;
int **jvec;
int *nj;
{
    FLOAT val;
    int i, j, k, m, mult, ji;

    if ( ctlval < 0 ) ctlval = 0;	/* < 0 is a pain */
    /*
     * How many knots in the new vector?  An upper bound is
     *   (ntau - 2 * order + 1 ) * ctlval * (order - 1) {newly added knots} +
     *   (ntau - 2 * order + 2) * (order - 1) {multiplied original knots} +
     *   2 * order - 2 {end knots}
     * It may be smaller than this if any of the original knots
     * already have multiplicity > 1, but we'll ignore this.
     */
    *nt = (ntau - 2 * order + 1 ) * (ctlval + 1) * (order - 1) +
	3 * order - 3;
    *tvec = (FLOAT *)malloc( *nt * sizeof(FLOAT) );

    if (!*tvec)
    {
	*jvec = NULL;
	*nt = *nj = 0;
	return;
    };
    
    /*
     * How many output points?  One for each original (interior) knot,
     * plus one for each new evaluation point.
     */
    *nj = 1 + (ntau - 2 * order + 1) * (ctlval + 1);
    *jvec = (int *)malloc( *nj * sizeof(int) );

    if (!*jvec)
    {
	free(*tvec);
	*tvec = NULL;
	*nt = *nj = 0;
	return;
    };

    /*
     * Make tvec.
     * Copy knots up to first interior knot.
     * Add knots to each existing knot to give it multiplicity order - 1.
     * Add multiple knots between existing knots if ctlval > 0.
     * jvec entries index the knot before each knot clump.
     */
    for ( mult = 1, m = 0, i = 0; i < order; i++ )
    {
	(*tvec)[i] = tauvec[i];
	if ( i != m && tauvec[i] == tauvec[m] )
	    mult++;
	else
	{
	    m = i;
	    mult = 1;
	}
    }

    for ( ji = 0, j = i; i < ntau; i++ )
    {
	/*
	 * As long as we are still on the same knot value in the
	 * original vector, continue to copy, and count multiplicity.
	 * When it changes, bring the multiplicity up to order-1, then
	 * create new knots.
	 */
	if ( tauvec[i] == tauvec[m] )
	    mult++;
	else
	{
	    if ( m <= ntau - order )
	    {
		for ( ; mult < order - 1; mult++ )
		    (*tvec)[j++] = tauvec[m];
		(*jvec)[ji++] = j - order;
	    }
	    if ( i <= ntau - order )
		for ( k = 0; k < ctlval; k++ )
		{
		    (*jvec)[ji++] = j - 1;
		    val = tauvec[m] + (tauvec[i] - tauvec[m]) *
			(k + 1) / (double)(ctlval + 1);
		    for ( mult = 0; mult < order - 1; mult++ )
			(*tvec)[j++] = val;
		}
	    m = i;
	    mult = 1;
	}
	(*tvec)[j++] = tauvec[i];
    }
    /* If the last knot had multiplicity of order, need to make a jvec entry */
    if ( mult >= order )
	(*jvec)[ji++] = j - mult - 1;

    /* Error check */
    if ( j > *nt )
	fprintf(stderr, "Generated %d knots, should have made %d\n", j, *nt);
    *nt = j;
    if ( ji > *nj )
	fprintf(stderr, "Generated %d points, should have made %d\n", ji, *nj);
    *nj = ji;
}


/*****************************************************************
 * TAG( alpha_mul )
 * 
 * Multiply a control mesh by an alpha matrix, giving a new control mesh.
 * Inputs:
 * 	a_mat:		Alpha matrix to multiply by.
 * 	o_mesh:		Original control mesh.  The row direction
 * 			(indexed by colstride) is the "active"
 * 			direction.  In other words, all the values in
 * 			a given column are treated identically.
 * Outputs:
 * 	n_mesh:		New control mesh.
 * Assumptions:
 * 	o_mesh->nrow == n_mesh->nrow.  o_mesh->nrow is used.
 * 	n_mesh->ncol >= a_mat->width.  a_mat->width is used.
 * 	o_mesh->ncol >= MAX(a_col->colstart + a_col->length - 1) for
 * 		a_col ranging over all columns of a_mat.
 * Algorithm:
 * 	Each column of the alpha matrix is used to compute a single
 * 	new column as a combination of the appropriate original columns.
 * 	Thus, the columns of n_mesh correspond to the columns of
 * 	a_mat, and the columns of o_mesh correspond to the rows of
 * 	a_mat.  As a matrix formula: n_mesh[i,*] = o_mesh[i,*] * a_mat
 *
 * Note:
 * 	This routine should be called once to refine a curve, and once
 * 	per row or column of the control mesh to refine a surface.
 */
alpha_mul( a_mat, o_mesh, n_mesh )
    alpha_mat *a_mat;
    pexArray2 *o_mesh;
    pexArray2 *n_mesh;
{
    int i, j, k;
    alpha_col * a_col;
    char * n_col, * n_row, * o_col1, * o_col, * o_row;
    FLOAT sum;

    /* Step through columns of alpha matrix */
    for ( i = a_mat->width, a_col = alpha_mat_col(a_mat, 0),
	  n_col = PexArray2Index(n_mesh, 0, 0);
	  i > 0;
	  i--, a_col = alpha_mat_next(a_mat, a_col),
	  n_col = PexArray2NextCol(n_mesh, n_col))
    {
	/* Get 1st column pointer in o_mesh */
	o_col1 = PexArray2Index(o_mesh, 0, a_col->colstart);
	/* Step through rows */
	for ( j = 0, o_row = o_col1, n_row = n_col;
	      j < o_mesh->nrow;
	      j++, o_row = PexArray2NextRow(o_mesh, o_row),
	      n_row = PexArray2NextRow(n_mesh, n_row))
	{
	    /*
	     * Step through rows of alpha matrix & columns of original
	     * mesh.
	     */
	    for ( k = 0, sum = 0.0, o_col = o_row;
		  k < a_col->length;
		  k++, o_col = PexArray2NextCol(o_mesh, o_col) )
		sum += *(FLOAT *)o_col * a_col->val[k];
	    *(FLOAT *)n_row = sum;
	}
    }
}




/*****************************************************************
 * TAG( newalpha_mat )
 * 
 * Creates a new alpha_mat data structure of a given size.
 * Inputs:
 * 	order:		Spline order.
 * 	width:		Number of columns in the resulting matrix.
 * Outputs:
 * 	Returns a newly created alpha matrix.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
alpha_mat *
newalpha_mat( order, width )
int order;
int width;
{
    alpha_mat * retval;
    int stride = sizeof(alpha_col) + (order - 1) * sizeof(FLOAT);

    retval = (alpha_mat *)malloc( sizeof(alpha_mat) + width * stride );
    if (retval)
    {
	retval->order = order;
	retval->width = width;
	retval->colStride = stride;
	retval->colOffset = sizeof(alpha_mat);
    }
    return retval;
}



/*****************************************************************
 * TAG( calc_alpha )
 * 
 * Calculate the alpha matrix to transform the tau knot vector into
 * the t knot vector.
 * Inputs:
 * 	order:		B-spline order
 * 	tauvec:		Original knot vector.
 * 	ntau:		Size of tauvec.
 * 	tvec:		New (refined) knot vector.
 * 	nt:		Size of tvec.
 * 	jvec:		If non-NULL, list of alpha matrix colum
 * 			indices.  Only the columns of the alpha matrix
 * 			corresponding to these indices will be
 * 			calculated.  Thus, column i of the calculated
 * 			alpha matrix will correspond to column jvec[i]
 * 			in the true alpha matrix.
 * 	nj:		Size of jvec.  If 0, or if jvec is NULL, jvec
 * 			will be ignored.
 * Outputs:
 * 	Builds and returns an alpha matrix.
 * Assumptions:
 * 	As necessary.  See Oslo Algorithm papers.
 * Algorithm:
 * 	From "Making the Oslo Algorithm More Efficient", Lyche &
 * 	Moerken, SIAM J. Numer. Anal., Vol. 23, No. 3, June 1986.
 */
static alpha_mat *
calc_alpha( order, tauvec, ntau, tvec, nt, jvec, nj )
    int order;
    FLOAT tauvec[];
    int ntau;
    FLOAT tvec[];
    int nt;
    int jvec[];
    int nj;
{
    alpha_mat * a_mat;
    register alpha_col * a_col;
    int i, j, ji, u, up, v, p, ih, il, iu, rt, lt, rtau0, ltaun;
    FLOAT tj, b, b1, d1, d2, *xi, *ah;
    Bool alloc_jvec = FALSE;

    if ( jvec == NULL || nj == 0 )
    {
	jvec = (int *)malloc( (nj = nt - order) * sizeof(int) );
	if (!jvec) return NULL;
	alloc_jvec = TRUE;
	for ( ji = 0; ji < nj; ji++ )
	    jvec[ji] = ji;
    }

    a_mat = newalpha_mat( order, nj );
    if (!a_mat)
	goto CrashAndBurn1;

    xi = (FLOAT *)malloc( (order - 1) * sizeof(FLOAT) );
    if (!xi)
	goto CrashAndBurn2;

    ah = (FLOAT *)malloc( order * sizeof(FLOAT) );
    if (!ah)
	goto CrashAndBurn3;
    

    /* Compute left and right multiplicities of tau endpoints */
    for ( j = 0; j < ntau; j++ )
	if ( tauvec[j] == tauvec[0] )
	    rtau0 = j + 1;
	else
	    break;
    for ( j = 1; j < ntau; j++ )
	if ( tauvec[ntau - j] == tauvec[ntau - 1] )
	    ltaun = j + 1;
	else
	    break;

    /*
     * For each knot in the new vector by Algorithm 1 from Lyche and
     * Moerken (p 671)
     */
    for ( ji = 0; ji < nj; ji++ )
    {
	j = jvec[ji];
	/* Get left and right multiplicities */
	for ( p = 0; p < nt - j; p++ )
	    if ( tvec[j + p] == tvec[j] )
		rt = p + 1;
	    else
		break;
	for ( p = 0; p < j + order; p++ )
	    if ( tvec[j + order - p] == tvec[j + order] )
		lt = p + 1;
	    else
		break;
	/* Get pointer to alpha matrix column */
	a_col = alpha_mat_col( a_mat, ji );

	/* Check for bounds constraints */
	if ( tvec[j] < tauvec[0] ||
	     tvec[j] == tauvec[0] && rt > rtau0 ||
	     tvec[j + order] > tauvec[ntau - 1] ||
	     tvec[j + order] == tauvec[ntau - 1] && lt > ltaun )
	{
	    /* All elements in this column are 0 */
	    a_col->length = 0;
	}
	else
	{
	    for ( p = 0; p < order; p++ )
		ah[p] = 0;

	    /* Compute u (starting offset) */
	    for ( u = 0; u < ntau; u++ )
		if ( tauvec[u] <= tvec[j] && tvec[j] < tauvec[u + 1] )
		    break;
	    /* Steps 1, 2 */
	    for ( i = j + 1, up = u;
		  tvec[i] == tauvec[up] && i < j + order;
		  i++, up-- )
		;
	    /* Step 3 */
	    ih = up + 1;
	    v = 0;
	    /* Step 4 */
	    for ( p = 1; p < order; p++ )
		if ( tvec[j+p] == tauvec[ih] )
		    ih++;
		else
		    xi[v++] = tvec[j + p];
	    /* Step 5 */
	    ah[order-1] = 1;
	    /* Step 6 */
	    for ( p = 1; p <= v; p++ )
	    {
		/* 6.1 */
		b1 = 0;
		tj = xi[p - 1];
		/* 6.2 */
		if ( p > up )
		    b1 = ((tj - tauvec[0]) * ah[order - up - 1]) /
			(tauvec[p + order - v - 1] - tauvec[0]);
		/* 6.3 */
		il = max( 1, up - p + 1 );
		iu = min( up, ntau - order - 1 + v - p );
		/* 6.4 */
		for ( i = il; i <= iu; i++ )
		{
		    /* 6.4.1 */
		    d1 = tj - tauvec[i];
		    d2 = tauvec[i + p + order - v - 1] - tj;
		    /* 6.4.2 */
		    b = ah[i + order - up - 1] / (d1 + d2);
		    /* 6.4.3 */
		    ah[i + order - up - 2] = d2 * b + b1;
		    /* 6.4.4 */
		    b1 = d1 * b;
		}
		/* 6.5 */
		ah[iu + order - up - 1] = b1;
		/* 6.6 */
		if ( iu < up )
		    ah[iu + order - up - 1] = b1 +
			(tauvec[ntau - 1] - tj) * ah[iu + order - up] /
			    (tauvec[ntau - 1] - tauvec[iu + 1]);
	    }
#if 0
	    printf( "j=%d, up=%d, v=%d, il=%d, iu=%d, xi=", j, up, v, il, iu );
	    for ( p = 0; p < v; p++ )
		printf( "%g ", xi[p] );
	    printf( ", ah =" );
	    for ( p = 0; p < order; p++ )
		printf( "%g ", ah[p] );
	    printf( "\n" );
#endif
	    /*
	     * Copy into alpha matrix */
	    il = max( up - v, 0 );
	    iu = min( up, ntau - order - 1 );
	    a_col->colstart = il;
	    a_col->length = iu + 1 - a_col->colstart;
	    for ( i = 0; i < a_col->length; i++ )
		a_col->val[i] = ah[i + il + order - up - 1];
	}
    }

    free( ah );
    free( xi );
    if ( alloc_jvec )
	free( jvec );
    return a_mat;   /* Normal exit */

 CrashAndBurn3:
    free( xi );
 CrashAndBurn2:
    free(a_mat);
 CrashAndBurn1:
    if ( alloc_jvec )
	free( jvec );
    return NULL;
}


pr_alpha_mat( a_mat )
alpha_mat * a_mat;
{
    int row, col;
    int done = 0, active;
    alpha_col * a_col;

    printf( "\t" );
    for ( col = 0; col < a_mat->width; col++ )
	printf( "%4d\t", col );
    printf( "\n" );
    for ( row = 0; !done; row++ )
    {
	active = -1;
	for ( col = 0; col < a_mat->width; col++ )
	{
	    a_col = alpha_mat_col( a_mat, col );
	    if ( row >= a_col->colstart &&
		 row < a_col->colstart + a_col->length )
	    {
		if ( active == -1 )
		    printf( "%4d\t", row );
		if ( col - active > 1 )
		    printf(
			"%*.*s", col - active - 1, col - active - 1,
			"\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t" );
		active = col;
		printf( "%7.5f ", a_col->val[row - a_col->colstart] );
	    }
	}
	if ( active >= 0 )
	    printf( "\n" );
	else
	    done = 1;
    }
}

/*****************************************************************
 * TAG( PexNurbCurveToPolyline )
 * 
 * Convert a NURB Curve to a Polyline.
 * Inputs:
 * 	pRend:	renderer pointer, used to get approximation parameters.
 * 	crv:	The curve.
 * Outputs:
 * 	Returns a newly created polyline that approximates the curve.
 * Assumptions:
 * 	For now, assumes that the approximation method is #2 (constant
 * 	between knots).
 * 	Curve must be well-formed (good knot vector, etc.)
 * 	Knot vector in Curve data structure is stored as an array of
 * 	FLOATs.
 * Algorithm:
 * 	Call mk_type2_kv to get a refinement knot vector,
 * 	then call calc_alpha to get the alpha matrix,
 * 	allocate a new polyline structure,
 * 	then call alpha_mul to evaluate the curve into the polyline.
 */
pexPolylinePtr
PexNurbCurveToPolyline( pRend, crv )
pexRendererPtr pRend;
pexNurbCurvePtr crv;
{
    FLOAT 	* tvec, w;
    pexCoord4D	* pts;
    int		* jvec, nj, nt, i;
    alpha_mat	* a_mat;
    pexPolylinePtr	p_poly;
    pexArray2	orig, new;

    mk_type2_kv( (int)pRend->pPC->curveApprox.tolerance, crv->curveOrder,
		 PexNurbCurveKnot(crv, 0), crv->numKnots,
		 &tvec, &nt, &jvec, &nj );
    if (!tvec || !jvec )
	return NULL;

    a_mat = calc_alpha( crv->curveOrder,
			PexNurbCurveKnot(crv, 0), crv->numKnots,
			tvec, nt, jvec, nj );
    if (!a_mat)
	goto die1;
    
    p_poly = NewPexPolyline(nj);
    
    if (!p_poly)
	goto die2;
    
    /* There should be a routine to do this stuff */
    p_poly->numVertices = nj;
    p_poly->vertexStride = sizeof(pexCoord3D);
    p_poly->vertexOffset = sizeof(pexPolylineRec);

    orig.body = (char *)PexNurbCurvePoint(crv, 0);
    orig.ncol = crv->numPoints;
    orig.nrow = (crv->coordType == Rational) ? 4 : 3;
    orig.colStride = crv->pointStride;
    orig.rowStride = sizeof(FLOAT);

    if ( crv->coordType != Rational )
    {
	new.body = (char *)PexPolylineVertex(p_poly, 0);
	new.ncol = nj;
	new.nrow = 3;
	new.colStride = p_poly->vertexStride;
	new.rowStride = sizeof(FLOAT);
	alpha_mul(a_mat, &orig, &new);
    }
    else
    {
	pts = (pexCoord4D *)malloc(nj * sizeof(pexCoord4D));

	if (!pts)
	    goto die3;
	
	new.body = (char *)pts;
	new.ncol = nj;
	new.nrow = 4;
	new.colStride = sizeof(pexCoord4D);
	new.rowStride = sizeof(FLOAT);
	alpha_mul(a_mat, &orig, &new);
	for ( i = 0; i < nj; i++ )
	{
	    if ( (w = pts[i].w) != 1.0 && w != 0.0 )
	    {
		pts[i].x /= w;
		pts[i].y /= w;
		pts[i].z /= w;
	    }
	    *PexPolylineVertex(p_poly, i) = *(pexCoord3D *)&pts[i];
	}
	free(pts);
    }

    free(a_mat);
    free(tvec);
    free(jvec);

    return p_poly;

 die3:
    free(p_poly);
 die2:
    free(a_mat);
 die1:
    free(tvec);
    free(jvec);
    return NULL;
}

/*
 * Functions for computing normal approximations. Used by
 * PexNurbSurfaceToQuadmesh.
 */
#define VectorLengthSquared(v)	((v).x*(v).x + (v).y*(v).y + (v).z*(v).z)
#define VectorCrossProduct( v1, v2, v3 )\
    (v3).x = ((v1).y * (v2).z) - ((v1).z * (v2).y);\
    (v3).y = ((v1).z * (v2).x) - ((v1).x * (v2).z);\
    (v3).z = ((v1).x * (v2).y) - ((v1).y * (v2).x)

#define PointMinus( p1, p0, v ) \
    (v).x = (p1).x - (p0).x; \
    (v).y = (p1).y - (p0).y; \
    (v).z = (p1).z - (p0).z

static CornerNormal(p_quad, i0, j0, i1, j1, i2, j2, id, jd)
    pexQuadMeshPtr p_quad;
    int i0, j0, i1, j1, i2, j2, id, jd;
{
    pexCoord3D pt0, pt1, pt2;
    pexVector3D v0, v1, norm;

    pt0 = *PexQuadMeshVertex(p_quad, i0, j0);
    pt1 = *PexQuadMeshVertex(p_quad, i1, j1);
    pt2 = *PexQuadMeshVertex(p_quad, i2, j2);
    PointMinus(pt1, pt0, v0);
    PointMinus(pt2, pt0, v1);

    VectorCrossProduct(v0, v1, norm);

    /* Test for degeneracy */
    if (VectorLengthSquared(norm) < 1e-6)
    {
	/* Do it again with the diagonal */
	pt2 = *PexQuadMeshVertex(p_quad, id, jd);
	if (VectorLengthSquared(v0) < 1e-6)
	{
	    PointMinus(pt2, pt0, v0);
	}
	else
	{
	    PointMinus(pt2, pt0, v1);
	}
	VectorCrossProduct(v0, v1, norm);
    }
    *PexQuadMeshVertexNorm(p_quad, i0, j0) = norm;
}

static UEdgeNormal(p_quad, i0, j0, incr)
    pexQuadMeshPtr p_quad;
    int i0, j0, incr;
{
    pexCoord3D ept, lEpt, rEpt, ipt;
    pexVector3D v0, v1, norm;

    ept = *PexQuadMeshVertex(p_quad, i0, j0);	/* the edge point */
    lEpt = *PexQuadMeshVertex(p_quad, i0, j0 - 1);	/* Point to the left */
    rEpt = *PexQuadMeshVertex(p_quad, i0, j0 + 1);	/* to the right */
    ipt = *PexQuadMeshVertex(p_quad, i0 + incr, j0);	/* Inner point */

    PointMinus(ipt, ept, v0);
    PointMinus(rEpt, lEpt, v1);

    VectorCrossProduct(v0, v1, norm);

    /* Degenerate edge? */
    if (VectorLengthSquared(norm) < 1e-6)
    {
	/*
	 * Assume diagonals from this edge point do not give
	 * degenerate result.
	 */
	lEpt = *PexQuadMeshVertex(p_quad, i0 + incr, j0 - 1);
	rEpt = *PexQuadMeshVertex(p_quad, i0 + incr, j0 + 1);

	PointMinus(rEpt, ept, v0);
	PointMinus(lEpt, ept, v1);

	VectorCrossProduct(v0, v1, norm);
    }
    if ( incr < 0 )
    {
	norm.x = -norm.x;
	norm.y = -norm.y;
	norm.z = -norm.z;
    }

    *PexQuadMeshVertexNorm(p_quad, i0, j0) = norm;
}


static VEdgeNormal(p_quad, i0, j0, incr)
    pexQuadMeshPtr p_quad;
    int i0, j0, incr;
{
    pexCoord3D ept, uEpt, dEpt, ipt;
    pexVector3D v0, v1, norm;

    ept = *PexQuadMeshVertex(p_quad, i0, j0);	/* the edge point */
    uEpt = *PexQuadMeshVertex(p_quad, i0 - 1, j0);	/* Point above (up) */
    dEpt = *PexQuadMeshVertex(p_quad, i0 + 1, j0);	/* below (down) */
    ipt = *PexQuadMeshVertex(p_quad, i0, j0 + incr);	/* Inner point */

    PointMinus(dEpt, uEpt, v0);
    PointMinus(ipt, ept, v1);

    VectorCrossProduct(v0, v1, norm);

    /* Degenerate edge? */
    if (VectorLengthSquared(norm) < 1e-6)
    {
	/*
	 * Assume diagonals from this edge point do not give
	 * degenerate result.
	 */
	uEpt = *PexQuadMeshVertex(p_quad, i0 - 1, j0 + incr);
	dEpt = *PexQuadMeshVertex(p_quad, i0 + 1, j0 + incr);

	PointMinus(uEpt, ept, v0);
	PointMinus(dEpt, ept, v1);

	VectorCrossProduct(v0, v1, norm);
    }
    if ( incr < 0 )
    {
	norm.x = -norm.x;
	norm.y = -norm.y;
	norm.z = -norm.z;
    }

    *PexQuadMeshVertexNorm(p_quad, i0, j0) = norm;
}


static InteriorNormal(p_quad, i0, j0)
    pexQuadMeshPtr p_quad;
    int i0, j0;
{
    pexCoord3D upt, dpt, lpt, rpt;
    pexVector3D v0, v1, norm;

    upt = *PexQuadMeshVertex(p_quad, i0 - 1, j0);	/* up */
    dpt = *PexQuadMeshVertex(p_quad, i0 + 1, j0);	/* down */
    lpt = *PexQuadMeshVertex(p_quad, i0, j0 - 1);	/* left */
    rpt = *PexQuadMeshVertex(p_quad, i0, j0 + 1);	/* right */

    PointMinus(dpt, upt, v0);
    PointMinus(rpt, lpt, v1);
    VectorCrossProduct(v0, v1, norm);

    *PexQuadMeshVertexNorm(p_quad, i0, j0) = norm;
}

/*****************************************************************
 * TAG( PexNurbSurfaceToQuadmesh )
 * 
 * Convert a NURB Curve to a Polyline.
 * Inputs:
 * 	pRend:	renderer pointer, used to get approximation parameters.
 * 	srf:	The surface.
 * 	doNorms:TRUE if vertex normals should be created.
 * Outputs:
 * 	Returns a newly created quadmesh that approximates the surface.
 * Assumptions:
 * 	For now, assumes that the approximation method is #2 (constant
 * 	between knots).
 * 	Surface must be well-formed (good knot vectors, etc.)
 * 	Knot vectors in surface data structure are stored as arrays of
 * 	FLOATs.
 * 	All degeneracies occur only at corners or along edges.
 * Algorithm:
 * 	Call mk_type2_kv to get a refinement knot vectors,
 * 	then call calc_alpha to get the alpha matrices,
 * 	allocate a new quadmesh structure,
 * 	then call alpha_mul (twice) to evaluate the surface into the quadmesh.
 *
 * 	If vertex normals are requested, compute an approximation as follows:
 * 	At a corner, use the cross product of the two edges
 * 	incident on the corner.  If this is 0, use the diagonal of the
 * 	facet together with one of the edges.
 *
 * 	Along the edge of the mesh, take the cross product of the
 * 	interior incident edge with the vector between the two
 * 	adjacent edge points.
 *
 * 	In the interior, take the cross product of the vectors that
 * 	are the differences of the adjacent points in the row and
 * 	column directions (respectively).
 */
pexQuadMeshPtr
PexNurbSurfaceToQuadmesh( pRend, srf, doNorms )
pexRendererPtr pRend;
pexNurbSurfacePtr srf;
Bool doNorms;
{
    FLOAT 	* t1vec, * t2vec, w;
    FLOAT	* ptmesh;
    pexCoord4D	* pts;
    int		* j1vec, * j2vec, nj1, nj2, nt1, nt2, i, j;
    alpha_mat	* a_mat;
    pexQuadMeshPtr	p_quad;
    pexArray2	orig, new, mesh;

    mk_type2_kv( (int)pRend->pPC->surfaceApprox.sTolerance,
   		 srf->uOrder,
		 PexNurbSurfaceUKnot(srf, 0), srf->numUknots,
		 &t1vec, &nt1, &j1vec, &nj1 );
    if (!t1vec || !j1vec )
	return NULL;

    mk_type2_kv( (int)pRend->pPC->surfaceApprox.tTolerance,
		 srf->vOrder,
		 PexNurbSurfaceVKnot(srf, 0), srf->numVknots,
		 &t2vec, &nt2, &j2vec, &nj2 );
    if (!t2vec || !j2vec)
    {
	goto BadAlloc1;
    }
    
    /* Make storage for the quadmesh */
    /* Note: should calculate vertex normals.  Do it later. */
    p_quad = NewPexQuadMesh(nj2, nj1, NonRational, 0, doNorms ? GANormal : 0);

    if (!p_quad)
	goto BadAlloc2;

    InitPexQuadMesh(p_quad, nj2, nj1, NonRational, 0, 0,
		    doNorms ? GANormal : 0);

    /* Intermediate storage */
    ptmesh = (FLOAT *)malloc( srf->numVPoints * nj1 * sizeof(FLOAT) *
			      ((srf->coordType == Rational) ? 4 : 3) );

    if (!ptmesh)
	goto BadAlloc3;

    mesh.body = (char *)ptmesh;
    mesh.nrow = srf->numVPoints;
    mesh.ncol = nj1;
    mesh.colStride = ((srf->coordType == Rational) ? 4 : 3) * sizeof(FLOAT);
    mesh.rowStride = mesh.ncol * mesh.colStride;

    /* First refine the rows (U direction) */
    a_mat = calc_alpha( srf->uOrder,
			PexNurbSurfaceUKnot(srf, 0), srf->numUknots,
			t1vec, nt1, j1vec, nj1 );
    if (!a_mat)
	goto BadAlloc4;
    
    for ( i = 0; i < mesh.nrow; i++ )
    {
	/* Describe this row as a 2-D array of FLOATs */
	orig.body = (char *)PexNurbSurfacePoint(srf, i, 0);
	orig.ncol = srf->numUPoints;
	orig.nrow = (srf->coordType == Rational) ? 4 : 3;
	orig.colStride = srf->pointNStride;
	orig.rowStride = sizeof(FLOAT);
	
	new.body = PexArray2Index(&mesh, i, 0);
	new.ncol = mesh.ncol;
	new.nrow = orig.nrow;
	new.colStride = orig.colStride;
	new.rowStride = sizeof(FLOAT);
	alpha_mul(a_mat, &orig, &new);
    }
    free(a_mat);
    
    /* Refine the columns (V direction) */
    a_mat = calc_alpha( srf->vOrder,
		        PexNurbSurfaceVKnot(srf, 0), srf->numVknots,
		        t2vec, nt2, j2vec, nj2 );
    if (!a_mat)
	goto BadAlloc4;
    
    /* Temp storage if rational */
    if ( srf->coordType == Rational )
    {
	pts = (pexCoord4D *)malloc(nj2 * sizeof(pexCoord4D));

	if (!pts)
	    goto BadAlloc5;
    }
    
    for ( i = 0; i < mesh.ncol; i++ )
    {
	orig.body = PexArray2Index(&mesh, 0, i);
	orig.ncol = mesh.nrow;	/* Same as numVPoints */
	orig.nrow = (srf->coordType == Rational) ? 4 : 3;
	orig.colStride = mesh.rowStride;
	orig.rowStride = sizeof(FLOAT);
	if ( srf->coordType != Rational )
	{
	    new.body = (char *)PexQuadMeshVertex(p_quad, 0, i);
	    new.ncol = nj2;
	    new.nrow = 3;
	    new.colStride = p_quad->vertexMStride;
	    new.rowStride = sizeof(FLOAT);
	    alpha_mul(a_mat, &orig, &new);
	}
	else
	{
	    new.body = (char *)pts;
	    new.ncol = nj2;
	    new.nrow = 4;
	    new.colStride = sizeof(pexCoord4D);
	    new.rowStride = sizeof(FLOAT);
	    alpha_mul(a_mat, &orig, &new);
	    for ( j = 0; j < nj2; j++ )
	    {
		if ( (w = pts[j].w) != 1.0 && w != 0.0 )
		{
		    pts[j].x /= w;
		    pts[j].y /= w;
		    pts[j].z /= w;
		}
		*PexQuadMeshVertex(p_quad, j, i) = *(pexCoord3D *)&pts[j];
	    }
	}
    }
    if ( srf->coordType == Rational )
	free(pts);

    free(a_mat);
    free(t1vec);
    free(j1vec);
    free(t2vec);
    free(j2vec);
    free(ptmesh);

    /* If vertex normals were requested, calculate them */
    if ( doNorms )
    {
	/* Do the corners */
	CornerNormal(p_quad,
		     0, 0,	/* Corner point */
		     1, 0,	/* First adjacent point */
		     0, 1,	/* Second adjacent point */
		     1, 1);	/* Diagonal */
	CornerNormal(p_quad, 0, nj1-1, 0, nj1-2, 1, nj1-1, 1, nj1-2);
	CornerNormal(p_quad, nj2-1, 0, nj2-1, 1, nj2-2, 0, nj2-2, 1);
	CornerNormal(p_quad, nj2-1, nj1-1, nj2-2, nj1-1, nj2-1, nj1-2,
		     nj2-2, nj1-2);

	/* Do the edge points */
	for ( i = 1; i < nj1 - 1; i++ )
	{
	    UEdgeNormal(p_quad,
			0, i,	/* Edge point*/
			1);	/* Multiplier & row increment */
	    UEdgeNormal(p_quad, nj2-1, i, -1);
	}
	for ( i = 1; i < nj2 - 1; i++ )
	{
	    VEdgeNormal(p_quad,
			i, 0,	/* Edge point */
			1);	/* Multiplier & row increment */
	    VEdgeNormal(p_quad, i, nj1-1, -1);
	}

	/* Interior points */
	for ( i = 1; i < nj2 - 1; i++ )
	    for ( j = 1; j < nj1 - 1; j++ )
		InteriorNormal(p_quad, i, j);
    }

    return p_quad;     /* Normal exit */

 BadAlloc5:            /* We need error exits at various depths */
		       /* in the procedure */
    free(a_mat);
 BadAlloc4:
    free(ptmesh);
 BadAlloc3:
    free(p_quad);
 BadAlloc2:
    free(j1vec);
    free(j2vec);
 BadAlloc1:
    free(t1vec);
    free(t2vec);
    return NULL;

}



/*****************************************************************
 * TAG( mipexNurbCurve3D )
 * 
 * 'renderer' for 3D NURB Curves.
 * Inputs:
 * 	pRend:	The renderer.
 * 	pCrv:	The curve.
 * Outputs:
 * 	Renders the curve.
 * Assumptions:
 *	[None]
 * Algorithm:
 * 	Computes a polyline that approximates the curve, then draws it
 * 	using the polyline3D routine from the renderer.
 */
mipexNurbCurve3D(pRend, pCrv)
pexRendererPtr pRend;
pexNurbCurvePtr pCrv;
{
    pexPolylinePtr pPoly, PexNurbCurveToPolyline();

    pPoly = PexNurbCurveToPolyline(pRend, pCrv);
    if (pPoly)
    {
	(*pRend->Polyline3D)(pRend, pPoly);
	FreePexPolyline(pPoly);
	return Success;
    }
    else return BadAlloc;
}


/*****************************************************************
 * TAG( mipexNurbSurface3DFlat )
 * 
 * 'Draw' a NURB surface with no or flat (faceted) shading.
 * Inputs:
 * 	pRend:	The renderer
 * 	pSrf:	The surface
 * Outputs:
 * 	Draws the surface.
 * Assumptions:
 *	[None]
 * Algorithm:
 * 	Converts the surface to a quadmesh and draws it via the renderer.
 */
mipexNurbSurface3DFlat(pRend, pSrf)
pexRendererPtr pRend;
pexNurbSurfacePtr pSrf;
{
    pexQuadMeshPtr pQuad, PexNurbSurfaceToQuadmesh();

    pQuad = PexNurbSurfaceToQuadmesh(pRend, pSrf, FALSE);
    if (pQuad)
    {
	(*pRend->QuadMesh3D)(pRend, pQuad);
	FreePexQuadMesh(pQuad);
	return Success;
    }
    else return BadAlloc;
}


/*****************************************************************
 * TAG( mipexNurbSurface3DSmooth )
 * 
 * 'Draw' a NURB surface with smooth shading.
 * Inputs:
 * 	pRend:	The renderer
 * 	pSrf:	The surface
 * Outputs:
 * 	Draws the surface.
 * Assumptions:
 *	[None]
 * Algorithm:
 * 	Converts the surface to a quadmesh and draws it via the renderer.
 */
mipexNurbSurface3DSmooth(pRend, pSrf)
pexRendererPtr pRend;
pexNurbSurfacePtr pSrf;
{
    pexQuadMeshPtr pQuad, PexNurbSurfaceToQuadmesh();

    pQuad = PexNurbSurfaceToQuadmesh(pRend, pSrf, TRUE);
    if (pQuad)
    {
	(*pRend->QuadMesh3D)(pRend, pQuad);
	FreePexQuadMesh(pQuad);
	return Success;
    }
    else return BadAlloc;
}


#ifdef TESTING
main( argc, argv )
char ** argv;
{
    int i, ntau, nt, nx, nj, order;
    FLOAT tauvec[40], xvec[2][40], yvec[2][40];
#if 0
    FLOAT tvec[40];
    int jvec[40];
#else
    FLOAT *tvec;
    int *jvec;
    int ninterp;
#endif
    pexArray2 orig, new;
    alpha_mat * a_mat;

#if 0
    for ( i = 2, ntau = 0; i < argc; i++ )
#else
    for ( i = 3, ntau = 0; i < argc; i++ )
#endif
    {
	if ( *argv[i] == ',' )
	    break;
	tauvec[ntau++] = atof( argv[i] );
    }
#if 0
    for ( nt = 0, i++; i < argc; i++ )
    {
	if ( *argv[i] == ',' )
	    break;
	tvec[nt++] = atof( argv[i] );
    }
#endif
    for ( nx = 0, i++; i < argc; i++, nx++ )
    {
	if ( *argv[i] == ',' )
	    break;
	xvec[0][nx] = atof( argv[i] );
	xvec[1][nx] = nx;
    }
#if 0
    for ( nj = 0, i++; i < argc; i++, nj++ )
    {
	if ( *argv[i] == ',' )
	    break;
	jvec[nj] = atoi( argv[i] );
    }
#endif

    if ( ntau == 0 || nt == 0 || nx == 0 )
    {
#if 0
	fprintf( stderr,
	    "Usage: nurbs order tauvalues  ... , tvalues ... , xvalues \n" );
#else
	fprintf( stderr,
	    "Usage: nurbs order ninterp tauvalues ... , xvalues \n" );
#endif
	exit( 1 );
    }
	
    order = atoi( argv[1] );
#if 0
#else
    ninterp = atoi( argv[2] );
    mk_type2_kv( ninterp, order, tauvec, ntau, &tvec, &nt, &jvec, &nj );
#endif
    printf( "tau =\t" );
    for ( i = 0; i < ntau; i++ )
	printf( "%g\t", tauvec[i] );
    printf( "\n" );

    printf( "t =\t" );
    for ( i = 0; i < nt; i++ )
	printf( "%g\t", tvec[i] );
    printf( "\n" );

    if ( nj > 0 )
    {
	printf( "j =\t" );
	for ( i = 0; i < nj; i++ )
	    printf( "%d\t", jvec[i] );
	printf( "\n" );
    }

    a_mat = calc_alpha( order, tauvec, ntau, tvec, nt, jvec, nj );
    if ( nj == 0 )
	nj = nt - order;

    printf( "Resulting alpha matrix = \n" );
    pr_alpha_mat( a_mat );

    orig.body = (char *)xvec;
    orig.ncol = nx;
    orig.nrow = 2;
    orig.colStride = sizeof(FLOAT);
    orig.rowStride = 40 * orig.colStride;

    new.body = (char *)yvec;
    new.ncol = nj;
    new.nrow = 2;
    new.colStride = sizeof(FLOAT);
    new.rowStride = 40 * new.colStride;

    alpha_mul( a_mat, &orig, &new );

    printf( "Original control polygon:\n" );
    for ( i = 0; i < nx; i++ )
	printf( "%7.5f ", xvec[0][i] );
    printf( "\n" );
    for ( i = 0; i < nx; i++ )
	printf( "%7.5f ", xvec[1][i] );
    printf( "\n" );

    printf( "Refined control polygon:\n" );
    for ( i = 0; i < nj; i++ )
	printf( "%7.5f ", yvec[0][i] );
    printf( "\n" );
    for ( i = 0; i < nj; i++ )
	printf( "%7.5f ", yvec[1][i] );
    printf( "\n" );
}
#endif
