/* 
 * micolor.c - Routines to initialize and keep a colorTable for screen 0
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
#include <stdio.h>
#include "X.h"
#include "misc.h"
#include "colormap.h"
#include "colormapst.h"
#include "scrnintstr.h"
#include "resource.h"
#include "dixstruct.h"
#include "gcstruct.h"
#include "extnsionst.h"
#include "renderer.h"

#define EXT
#include "colortable.h"

#define PIXMAP_SIZE 8
static unsigned char PexMonochromePixmapData[NUM_PIXMAPS][PIXMAP_SIZE] =
{
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x77, 0xff, 0xff, 0xff, 0x77, 0xff, 0xff, 0xff,
    0x77, 0xff, 0xdd, 0xff, 0x77, 0xff, 0xdd, 0xff,
    0x55, 0xff, 0xdd, 0xff, 0x55, 0xff, 0xdd, 0xff,
    0x55, 0xff, 0x55, 0xff, 0x55, 0xff, 0x55, 0xff,
    0x55, 0xbb, 0x55, 0xff, 0x55, 0xbb, 0x55, 0xff,
    0x55, 0xbb, 0x55, 0xee, 0x55, 0xbb, 0x55, 0xee,
    0x55, 0xaa, 0x55, 0xee, 0x55, 0xaa, 0x55, 0xee,
    0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
    0x11, 0xaa, 0x55, 0xaa, 0x11, 0xaa, 0x55, 0xaa,
    0x11, 0xaa, 0x44, 0xaa, 0x11, 0xaa, 0x44, 0xaa,
    0x00, 0xaa, 0x44, 0xaa, 0x00, 0xaa, 0x44, 0xaa,
    0x00, 0xaa, 0x00, 0xaa, 0x00, 0xaa, 0x00, 0xaa,
    0x00, 0x22, 0x00, 0xaa, 0x00, 0x22, 0x00, 0xaa,
    0x00, 0x22, 0x00, 0x88, 0x00, 0x22, 0x00, 0x88,
    0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x88,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
};

static CARD16 PexColorValues[MAX_PEX_COLOR][3] =
{
    0,     0,     0,		/* black */
    65535, 65535, 65535,	/* white */
    65535, 0,     0,		/* red */
    0,     65535, 0,		/* green */
    0,     0,     65535,	/* blue */
    65535, 0,     65535,	/* magenta */
    0,     65535, 65535,	/* cyan */
    65535, 65535, 0,		/* yellow */
};

/*
 * fill shades with a 'grey' scale from black to color towards white
 * From 0 to TRUE_COLOR_SHADE is a scale from black to the color.  From
 * TRUE_COLOR_SHADE to NUM_SHADES-1 adds white to the color.
 */

static void 
ShadeColor(color, shades)
    register CARD16 *color;
    register xColorItem *shades;
    
{
    CARD16 red, green, blue;
    register CARD16 i;

    if (color[0] == 0xffff && color[1] == 0xffff && color[2] == 0xffff)
    {
	i = 1;

	shades[0].red = red = color[0] / (NUM_SHADES - 1);
	shades[0].green = green = color[1] / (NUM_SHADES - 1);
	shades[0].blue = blue = color[2] / (NUM_SHADES - 1);
    }
    else
    {
	shades[0].red = red = color[0] / (TRUE_COLOR_SHADE);
	shades[0].green = green = color[1] / (TRUE_COLOR_SHADE);
	shades[0].blue = blue = color[2] / (TRUE_COLOR_SHADE);
    
	for (i = 1; i < TRUE_COLOR_SHADE; i++)
	{ 
	    shades[i].red = red + shades[i-1].red;
	    shades[i].green = green + shades[i-1].green;
	    shades[i].blue = blue + shades[i-1].blue;
	}
	
	/* Shade towards white */
	red = (0xffff - color[0]) / (NUM_SHADES - TRUE_COLOR_SHADE);
	green = (0xffff - color[1]) / (NUM_SHADES - TRUE_COLOR_SHADE);
	blue = (0xffff - color[2]) / (NUM_SHADES - TRUE_COLOR_SHADE);
    }
    
    for (; i < NUM_SHADES - 1; i++)
    {
	shades[i].red = red + shades[i-1].red;
	shades[i].green = green + shades[i-1].green;
	shades[i].blue = blue + shades[i-1].blue;
    }
    
};

static void 
ShadeMonochrome(color, shades, black, white)
    register CARD16 *color;
    register xColorItem *shades;
    Pixel black, white;
{
    register CARD16 i;

    
    for(i = 0; i < NUM_SHADES - 1; i++)
	if ( i < (NUM_SHADES - 1) / 2 && i < TRUE_COLOR_SHADE - 1 )
	    shades[i].pixel = black;
	else
	    shades[i].pixel = white;

    for(i = 0; i < NUM_SHADES - 1; i++)
	shades[i].pixel = (Pixel)
	    PexMonochromePixmaps[(int)((i * NUM_PIXMAPS)/NUM_SHADES)];
};


/*****************************************************************
 * TAG( PexInitColormap )
 * 
 * 
 * Inputs:
 *	[None]
 * Outputs:
 * 	True if colors allocated ok.
 * Assumptions:
 * 	We assume that the screen that we a re interested in is screen 0.
 * 	Namely, that if we dereference the X global structure screenInfo for
 * 	screen 0's default colormap, and then load that colormap with our
 * 	colors, that the clients and the dd server code will be able to use
 * 	those colors. 
 * Algorithm:
 *	[None]
 */
int
PexInitColormap()
{
    Pixel indices[MAX_PEX_COLOR*NUM_SHADES];
    ScreenPtr pScreen = &(screenInfo.screen[0]);
    ColormapPtr cmap = (ColormapPtr)
	LookupID(pScreen->defColormap,RT_COLORMAP,RC_CORE);
    int i;

    int ok, monochrome;
    
    monochrome = (pScreen->rootDepth == 1);

    if (!monochrome)
	ok = AllocColorCells(serverClient->index, cmap,
			     (MAX_PEX_COLOR-1)*(NUM_SHADES-1)+1, 0, TRUE,
			     indices, 0);
    else ok = Success;
    
    if (ok != Success)
    {
	fprintf(stderr, "PEX could not allocate %d colors.\n",
		MAX_PEX_COLOR*NUM_SHADES);
	return (ok);        /* not ok!!! */
    }

    /*
    for (i=0;i<MAX_PEX_COLOR*NUM_SHADES;i++)
    {
	if (!(i & ~0x10))
	    printf("\n %2x:", i);
	printf(" %d,", indices[i]);
    }
    */

    bzero(PexColorTable, sizeof(Pixel)*MAX_PEX_COLOR*(NUM_SHADES-1));

    if (monochrome)
    {
	extern GCPtr CreateScratchGC();
	GCPtr pGC = CreateScratchGC(pScreen, pScreen->rootDepth);
	int scanwidth[PIXMAP_SIZE];
	DDXPointRec scanpoints[PIXMAP_SIZE];
	unsigned int scandata[PIXMAP_SIZE];
	
	for(i = 0; i < PIXMAP_SIZE; i++)
	{
	    scanwidth[i] = PIXMAP_SIZE;
	    scanpoints[i].x = 0;
	    scanpoints[i].y = i;
	}

	for (i = 0; i < NUM_PIXMAPS; i++)
	{
	    int j;
	    
	    for(j = 0; j < PIXMAP_SIZE; j++)
		scandata[j] = PexMonochromePixmapData[i][j] << 24;
	    
	    PexMonochromePixmaps[i] = (PixmapPtr)(pScreen->CreatePixmap)
		(pScreen, PIXMAP_SIZE, PIXMAP_SIZE, pScreen->rootDepth);
	    
	    if (!PexMonochromePixmaps[i])
		fprintf(stderr,
			"Pex could not allocate 8x8 pixmap for shading\n");
	    else
	    {
		pGC->alu = GXcopy;
		pGC->bgPixel = pScreen->whitePixel;
		pGC->fgPixel = pScreen->blackPixel;
		pGC->stateChanges |= GCFunction | GCForeground | GCBackground;
		ValidateGC(PexMonochromePixmaps[i], pGC);
		
		(*GetGCValue(pGC, SetSpans))(PexMonochromePixmaps[i], pGC,
					     scandata, scanpoints, scanwidth,
					     PIXMAP_SIZE, TRUE);
	    }
	}
	for(i=0;i<MAX_PEX_COLOR;i++)
	    ShadeMonochrome(PexColorValues[i],
			    &(PexColorTable[i*(NUM_SHADES-1)]),
			    pScreen->blackPixel, pScreen->whitePixel);
	
	FreeScratchGC(pGC);
    }
    else
    {
	for (i=0; i<NUM_SHADES-1; i++)
	{
	    PexColorTable[i].flags = DoRed | DoGreen | DoBlue;
	    PexColorTable[i].pixel = indices[0];
	}
	for (; i<MAX_PEX_COLOR*(NUM_SHADES-1); i++) {
	    PexColorTable[i].flags = DoRed | DoGreen | DoBlue;
	    PexColorTable[i].pixel = indices[i - NUM_SHADES + 2];
	}
	
	for(i=1;i<MAX_PEX_COLOR;i++)
	    ShadeColor(PexColorValues[i], &(PexColorTable[i*(NUM_SHADES-1)]));
	
	StoreColors(cmap, (MAX_PEX_COLOR-1)*(NUM_SHADES-1) + 1,
		    &PexColorTable[NUM_SHADES - 2]);
    }
    
/*
    for (i=0; i<(MAX_PEX_COLOR)*(NUM_SHADES-1); i++)
    {
	if (!(i & 0x3))
	    printf("\n %3d:", i);
	printf(" %3d (%4x %4x %4x),", PexColorTable[i].pixel,
	       PexColorTable[i].red, PexColorTable[i].green,
	       PexColorTable[i].blue);
    }
*/

    return ok;
}


