/* 
 * pexarray.h - Declare generic vector and array types.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#ifndef PEX_ARRAY_H
#define PEX_ARRAY_H
#include "Xmd.h"        /* for CARD32 */

/*****************************************************************
 * TAG( pexArray1 pexArray2 )
 *
 * Generic 1-D and 2-D arrays with strides.
 * 
 */

typedef struct _pexArray1 {
    char      * body;		/* Pointer to the body of the array */
    int		n;		/* Number of elements in the array */
    CARD32	stride;		/* Stride (in sizeof(char) units) */
} pexArray1;

typedef struct _pexArray2 {
    char      * body;
    int		nrow, ncol;	/* Size of the array */
    CARD32	colStride;	/* Stride from one column to the next */
    CARD32	rowStride;	/* Stride from one row to the next */
} pexArray2;

/*****************************************************************
 * TAG( PexArray1Index PexArray1Next )
 *
 * Get the pointer to a particular array element (as a char *), or to
 * the next member from the given one.
 * Inputs:
 * 	pv:	Pointer to the pexArray1.
 * 	idx:	Index into the array.
 * 	pelt:	Pointer to the current element.
 * Outputs:
 * 	Returns pointer to desired element.
 */

#define	PexArray1Index(pv, idx)	((pv)->body + (idx) * (pv)->stride)
#define PexArray1Next(pv, pelt)	((char *)(pelt) + (pv)->stride)

/*****************************************************************
 * TAG( PexArray2Index PexArray2NextCol PexArray2NextRow )
 *
 * Get the pointer to a particular array element, or to the adjacent
 * element in the next row or column from the given one.
 * Inputs:
 * 	pa:	Pointer to the pexArray2.
 * 	row:	Row index into the array.
 * 	col:	Column index into the array.
 * 	pelt:	Pointer to the current element (for next).
 * Outputs:
 * 	Returns a pointer to the desired element.
 */

#define	PexArray2Index(pa, row, col) \
    ((pa)->body + (row) * (pa)->rowStride + (col) * (pa)->colStride)
#define PexArray2NextCol(pa, pelt) ((char *)(pelt) + (pa)->colStride)
#define PexArray2NextRow(pa, pelt) ((char *)(pelt) + (pa)->rowStride)

/*****************************************************************
 * TAG( Pex1DData Pex2DData )
 *
 * Data access macros for the variable sized data structures.  These
 * macros are used VERY frequently, so they do not check for input
 * error.
 * 
 * Inputs:
 * 	pObj - pointer to a variable sized object
 * 	n - Index into the vector, or column index for the array
 * 	m - Row index into the array.
 * 	data - The data type requested.  (must match the structure element
 * 	     name.)
 * 	     
 * Outputs:
 * 	pointer to the data field desired as a char *.
 * 	
 * Assumptions:
 * 	Assumptions are that n (and m) are bounded correctly,
 * 	and that the field 'data' is present in the object.
 * 	
 * Algorithm:
 * 	simple pointer arithmetic.
 */

#if defined(__STDC__) && !defined(__HIGHC__)
#define Pex1DData(pObj, n, data)\
    ((char *)(pObj)+\
     ((n) * ((pObj)->data##Stride))+\
     ((pObj)->data##Offset))
#else
#define Pex1DData(pObj, n, data)\
    ((char *)(pObj)+\
     ((n) * ((pObj)->data/**/Stride))+\
     ((pObj)->data/**/Offset))
#endif

#if defined(__STDC__) && !defined(__HIGHC__)
#define Pex2DData(pObj,m,n,data)\
   ((char *)(pObj)+\
     ((m) * ((pObj)->data##MStride))+\
     ((n) * ((pObj)->data##NStride))+\
     ((pObj)->data##Offset))

#else /* Ansi C does token concat with ## */
#define Pex2DData(pObj,m,n,data)\
    ((char *)(pObj)+\
     ((m) * ((pObj)->data/**/MStride))+\
     ((n) * ((pObj)->data/**/NStride))+\
     ((pObj)->data/**/Offset))
#endif

/*****************************************************************
 * TAG( Pex1DDataNext Pex2DDataNextN Pex2DDataNextM )
 *
 * Incremental data access macros for the variable sized data
 * structures.  These macros are used VERY frequently, so they do not
 * check for input error.
 * 
 * Inputs:
 * 	pObj - pointer to a variable sized object
 * 	pElt - pointer to the current element.
 * 	data - The data type requested.  (must match the structure element
 * 	     name.)
 * 	     
 * Outputs:
 * 	pointer (as a char *) to the next element of the desired data field.
 * 	
 * Assumptions:
 * 	The field 'data' is present in the object.
 * 	
 * Algorithm:
 * 	simple pointer arithmetic.
 */

#if defined(__STDC__) && !defined(__HIGHC__)
#define Pex1DDataNext(pObj, pElt, data)\
    ((char *)(pElt) + (pObj)->data##Stride)
#else
#define Pex1DDataNext(pObj, pElt, data)\
    ((char *)(pElt) + (pObj)->data/**/Stride)
#endif

#if defined(__STDC__) && !defined(__HIGHC__)
#define Pex2DDataNextN(pObj, pElt, data)\
   ((char *)(pElt) + (pObj)->data##NStride)
#define Pex2DDataNextM(pObj, pElt, data)\
   ((char *)(pElt) + (pObj)->data##MStride)
#else
#define Pex2DDataNextN(pObj, pElt, data)\
   ((char *)(pElt) + (pObj)->data/**/NStride)
#define Pex2DDataNextM(pObj, pElt, data)\
   ((char *)(pElt) + (pObj)->data/**/MStride)
#endif

#endif /* PEX_ARRAY_H */

