/* 
 * PEXNurb.h - Header file for non-uniform rational B-spline curves & surfaces
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
#ifndef PEX_NURB_H
#define PEX_NURB_H

#include "PEXproto.h"
#include "pexarray.h"

typedef struct _pexNurbCurve
{
    pexElementInfo	head;
    CARD16              curveOrder;
    pexCoordType        coordType;
    FLOAT               tMin;
    FLOAT               tMax;
    CARD32              numKnots;
    CARD32              numPoints;
    CARD32              pointStride;
    CARD32              pointOffset;
    CARD32              knotStride;
    CARD32              knotOffset;
    /* Point list and knot list follow here */
} *pexNurbCurvePtr, pexNurbCurveRec;



/*****************************************************************
 * TAG( PexNurbCurveKnot PexNurbCurvePoint )
 * 
 * Data access macros for the Nurb Curve.
 * This macro is used VERY frequently, so it is not expected to
 * check for input error.
 * 
 * Inputs:
 * 	pNC - pointer to a Polyline
 * 	N - Indices into the point array
 * 	data - The data type requested.  (must match the structure element
 * 	     name.
 * 	     
 * Outputs:
 * 	pointer to the data field desired.
 * 	
 * Assumptions:
 * 	Assumptions are that N is bounded correctly,
 * 	and that the field 'data' is present in the object.
 * 	(The bitmasks determine this.)
 * 	
 * Algorithm:
 * 	simple pointer arithmetic.
 */
#define PexNurbCurveKnot(pNC, n)\
   (FLOAT *)	 Pex1DData(pNC, n, knot)
#define PexNurbCurvePoint(pNC, n)\
   (pexCoord4D*) Pex1DData(pNC, n, point)

#if defined(__STDC__)
extern pexNurbCurvePtr	UnpackPexNurbCurve(pexNonUniformBSplineCurve *);
extern pexNurbCurvePtr sUnpackPexNurbCurve(pexNonUniformBSplineCurve *);
extern pexNurbCurvePtr fUnpackPexNurbCurve(pexNonUniformBSplineCurve *);
extern pexNurbCurvePtr sfUnpackPexNurbCurve(pexNonUniformBSplineCurve *);

extern pexNurbCurvePtr	NewPexNurbCurve(CARD32 numPoints, CARD16 order
					pexCoordType coordType);
extern pexNurbCurvePtr	CopyPexNurbCurve(pexNurbCurvePtr);
extern void		FreePexNurbCurve(pexNurbCurvePtr);
extern Bool		ErrorCheckPexNurbCurve(pexNurbCurvePtr);
#else
extern pexNurbCurvePtr	UnpackPexNurbCurve();
extern pexNurbCurvePtr sUnpackPexNurbCurve();  /* s means swap */
extern pexNurbCurvePtr fUnpackPexNurbCurve();  /* f means float convert */
extern pexNurbCurvePtr sfUnpackPexNurbCurve(); 

extern pexNurbCurvePtr	NewPexNurbCurve();
extern pexNurbCurvePtr	CopyPexNurbCurve();
extern void		FreePexNurbCurve();
extern Bool		ErrorCheckPexNurbCurve();
#endif /* if defined... */

/********************************************************************
 * NURB Surface
 */

typedef struct _pexNurbSurface
{
    pexElementInfo	head;
    pexCoordType        coordType;
    CARD16              uOrder;
    CARD16              vOrder;
    CARD16		pad;
    FLOAT               uMin;
    FLOAT               uMax;
    FLOAT               vMin;
    FLOAT               vMax;
    CARD32              numUknots;
    CARD32              numVknots;
    CARD32              numUPoints;
    CARD32              numVPoints;
    /* CARD32              numCurveLists; */

    /* M->U, N->V (use M & N so that generic data access macros work) */
    CARD32              pointMStride; /* A 2D list of Control Points */
    CARD32              pointNStride; /* V is the dominant Stride (larger) */
    CARD32              pointOffset;

    CARD32              uKnotStride;    /* Two lists of Knot values */
    CARD32              uKnotOffset;
    CARD32              vKnotStride;
    CARD32              vKnotOffset;

/*  CARD32              listStride;  no trimming curves, this won't work anyway
    CARD32              listOffset;*/
    /* Point list and knot list follow here */
} *pexNurbSurfacePtr, pexNurbSurfaceRec;


/*****************************************************************
 * TAG( PexNurbSurfacePoint PexNurbSurfaceUKnot )
 * TAG( PexNurbSurfaceVKnot )
 *
 * Data access macros for the Nurb Surface.
 * These macro is used VERY frequently, so it is not expected to
 * check for input error.
 * 
 * Inputs:
 * 	pNC - pointer to a Polyline
 * 	N - Indices into the knot arrays
 * 	data - The data type requested.  (must match the structure element
 * 	     name.
 * 	     
 * Outputs:
 * 	pointer to the data field desired (e.g. FLOAT * ).
 * 	
 * Assumptions:
 * 	Assumptions are that N is bounded correctly,
 * 	and that the field 'data' is present in the object.
 * 	(The bitmasks determine this.)
 * 	
 * Algorithm:
 * 	simple pointer arithmetic.
 */

#define PexNurbSurfacePoint(pNS, v, u)\
     (pexCoord4D *)Pex2DData(pNS, v, u, point)

#define PexNurbSurfaceUKnot(o, n)\
    (FLOAT *)Pex1DData(o, n, uKnot)
#define PexNurbSurfaceVKnot(o, n)\
    (FLOAT *)Pex1DData(o, n, vKnot)

/*
 * external function definitions.
 */

#if defined(__STDC__)
extern pexNurbSurfacePtr UnpackPexNurbSurface(pexNonUniformBSplineSurface *);
extern pexNurbSurfacePtr sUnpackPexNurbSurface(pexNonUniformBSplineSurface *);
extern pexNurbSurfacePtr fUnpackPexNurbSurface(pexNonUniformBSplineSurface *);
extern pexNurbSurfacePtr sfUnpackPexNurbSurface(pexNonUniformBSplineSurface *);

extern pexNurbSurfacePtr NewPexNurbSurface(CARD32 numUPoints,
					   CARD32 numVPoints,
					   CARD16 uOrder, CARD16 vOrder,
					   pexCoordType coordType);
extern pexNurbSurfacePtr CopyPexNurbSurface(pexNurbSurfacePtr);
extern void		 FreePexNurbSurface(pexNurbSurfacePtr);
extern Bool		 ErrorCheckPexNurbSurface(pexNurbSurfacePtr);

#else /* NOT __STDC__ */

extern pexNurbSurfacePtr UnpackPexNurbSurface();
extern pexNurbSurfacePtr sUnpackPexNurbSurface();  /* s means swap */
extern pexNurbSurfacePtr fUnpackPexNurbSurface();  /* f means float convert */
extern pexNurbSurfacePtr sfUnpackPexNurbSurface(); 

extern pexNurbSurfacePtr NewPexNurbSurface();
extern pexNurbSurfacePtr CopyPexNurbSurface();
extern void		 FreePexNurbSurface();
extern Bool		 ErrorCheckPexNurbSurface();
#endif /* if defined... */
#endif PEX_NURB_H
