/* 
 * PUtil.c - Utilities for dealing with matricies.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <math.h>
#include "phigs.h"
#include "phigserr.h"
#include "pexDICE.h"

/*
 ******************************************************************************
 *
 *	Function:	ptranslate3(trans_vector, error_ind, matrix)
 *
 *		A 3D homogeneous transformation matrix to perform the specified
 *		3D translation is returned in matrix.
 *
 *	On Entry:	
 *
 *	On Exit:
 *
 *	Notes:
 *
 *
 ******************************************************************************
 */
void
ptranslate3(trans_vector, error_ind, matrix)

    Pvector3 *trans_vector;	/* translation vector	*/
    Pint *error_ind;		/* OUT error indicator	*/
    Pmatrix3 matrix;		/* OUT transformation matrix	*/
{
    *error_ind = PNO_ERROR;
    
    /* Create a traslation matrix */
    pident3(matrix);
    matrix[3][0] = trans_vector->x;
    matrix[3][1] = trans_vector->y;
    matrix[3][2] = trans_vector->z;

}

/*
 ******************************************************************************
 *
 *	Function:	ptranslate(trans_vector, error_ind, matrix)
 *
 *		A 2D homogeneous transformation matrix to perform the specified
 *		2D translation is returned in matrix.
 *
 *	On Entry:	
 *
 *	On Exit:	
 *
 *	Notes:
 *		
 *
 ******************************************************************************
 */
void
ptranslate(trans_vector, error_ind, matrix)

    Pvector	*trans_vector;	/* translation vector	*/
    Pint *error_ind;		/* OUT error indicator	*/
    Pmatrix	matrix;			/* OUT transformation matrix	*/
{
    *error_ind = PNO_ERROR;
    
    pident3(matrix);
    matrix[2][0] = trans_vector->x;
    matrix[2][1] = trans_vector->y;
    
}

/*
 ******************************************************************************
 *
 *	Function:	pscale3(scale_vector, error_ind, matrix)
 *
 *		A 3D homogeneous transformation matrix to perform the specified
 *		3D scaling is returned in matrix.
 *
 *	On Entry:	
 *
 *	On Exit:	
 *
 *	Notes:
 *		
 *
 ******************************************************************************
 */
void
pscale3(scale_vector, error_ind, matrix)

    Pvector3 *scale_vector;	/* scale factor vector	*/
    Pint *error_ind;		/* OUT error indicator	*/
    Pmatrix3 matrix;		/* OUT transformation matrix	*/
{
    *error_ind = PNO_ERROR;
    
    /* Create a scaling matrix */
    pident3(matrix);
    matrix[0][0] = scale_vector->x;
    matrix[1][1] = scale_vector->y;
    matrix[2][2] = scale_vector->z;
    
}

/*
 ******************************************************************************
 *
 *	Function:	pscale(scale_vector, error_ind, matrix)
 *
 *		A 2D homogeneous transformation matrix to perform the specified
 *		2D scaling is returned in matrix.
 *
 *	On Entry:	
 *
 *	On Exit:	
 *
 *	Notes:
 *		
 *
 ******************************************************************************
 */
void
pscale(scale_vector, error_ind, matrix)

    Pvector	*scale_vector;	/* scale factor vector	*/
    Pint *error_ind;		/* OUT error indicator	*/
    Pmatrix	matrix;			/* OUT transformation matrix	*/
{
    *error_ind = PNO_ERROR;
    
    /* Create the scaling matrix */
    pident3(matrix);
    matrix[0][0] = scale_vector->x;
    matrix[1][1] = scale_vector->y;
    
}

/*
 ******************************************************************************
 *
 *	Function:	protatex(angle, error_ind, matrix)
 *
 *		A 3D homogeneous transformation matrix to perform the specified
 *		rotation around the X axis is returned in matrix.
 *
 *	On Entry:	Pfloat	angle;		rotation angle
 *
 *	On Exit:
 *				Pint	*error_ind;		OUT error indicator
 *				Pmatrix3	matrix;		OUT transformation matrix
 *
 *	Notes:
 *	
 *
 ******************************************************************************
 */
void
protatex(angle, error_ind, matrix)

    Pfloat		angle;		/* rotation angle	*/
    Pint		*error_ind;	/* OUT error indicator	*/
    Pmatrix3	matrix;		/* OUT transformation matrix	*/
{    
    Pfloat c, s;	/* Sin and cosine of angle */
     
    *error_ind = PNO_ERROR;
    
    /* Create the X-rotation matrix */
    pident3(matrix);
    c = cos(angle);
    s = sin(angle);
    matrix[1][1] = c;
    matrix[2][2] = c;
    matrix[1][2] = s;
    matrix[2][1] = -s;
    
}

/*
 ******************************************************************************
 *
 *	Function:	protatey(angle, error_ind, matrix)
 *
 *		A 3D homogeneous transformation matrix to perform the specified
 *		rotation around the Y axis is returned in matrix.
 *
 *	On Entry:
 *				Pfloat	angle;		rotation angle
 *
 *	On Exit:
 *				Pint	*error_ind;		OUT error indicator
 *				Pmatrix3	matrix;		OUT transformation matrix
 *
 *	Notes:
 *
 *
 ******************************************************************************
 */
void
protatey(angle, error_ind, matrix)

    Pfloat		angle;		/* rotation angle	*/
    Pint		*error_ind;	/* OUT error indicator	*/
    Pmatrix3	matrix;		/* OUT transformation matrix	*/
{
    Pfloat c, s;	/* Sin and cosine of angle */
    
    *error_ind = PNO_ERROR;
    
    /* Create the Y-rotation matrix */
    pident3(matrix);
    c = cos(angle);
    s = sin(angle);
    matrix[0][0] = c;
    matrix[2][2] = c;
    matrix[2][0] = s;
    matrix[0][2] = -s;
    
}

/*
 ******************************************************************************
 *
 *	Function: protatez(angle, error_ind, matrix)
 *
 *		A 3D homogeneous transformation matrix to perform the specified
 *		rotation around the Z axis is returned in matrix.
 *
 *	On Entry:
 *				Pfloat	angle;		rotation angle
 *
 *	On Exit:
 *				Pint	*error_ind;		OUT error indicator
 *				Pmatrix3	matrix;		OUT transformation matrix
 *
 *	Notes:
 *
 *
 ******************************************************************************
 */
void
protatez(angle, error_ind, matrix)

    Pfloat		angle;		/* rotation angle	*/
    Pint		*error_ind;	/* OUT error indicator	*/
    Pmatrix3	matrix;		/* OUT transformation matrix	*/
{
    Pfloat c, s;	/* Sin and cosine of angle */
    
    *error_ind = PNO_ERROR;
    
    /* Create the Z-rotation matrix */
    pident3(matrix);
    c = cos(angle);
    s = sin(angle);
    matrix[0][0] = c;
    matrix[1][1] = c;
    matrix[0][1] = s;
    matrix[1][0] = -s;
    
}

/*
 ******************************************************************************
 *
 *	Function: protate(angle, error_ind, matrix)
 *
 *		A 2D homogeneous transformation matrix to perform the specified
 *		2D rotation is returned in matrix.
 *
 *	On Entry:
 *				Pfloat	angle;		rotation angle
 *
 *	On Exit:
 *				Pint	*error_ind;		OUT error indicator
 *				Pmatrix	matrix;		OUT transformation matrix
 *
 *	Notes:
 *
 *
 ******************************************************************************
 */
void
protate(angle, error_ind, matrix)

    Pfloat	angle;		/* rotation angle	*/
    Pint	*error_ind;	/* OUT error indicator	*/
    Pmatrix	matrix;		/* OUT transformation matrix	*/
{ 

    Pfloat c, s;	/* Sin and cosine of angle */
    
    *error_ind = PNO_ERROR;
    
    /* Create the 2D rotation matrix */
    pident3(matrix);
    c = cos(angle);
    s = sin(angle);
    matrix[0][0] = c;
    matrix[1][1] = c;
    matrix[0][1] = s;
    matrix[1][0] = -s;
    
}

/*
 ******************************************************************************
 *
 *	Function: pcomposematrix3(a, b, error_ind, result)
 *
 *		The 3D homogeneous transformation matrix to perform the 
 *		transformation [a] * [b] is returned in result.
 *       
 *	On Entry:
 *				Pmatrix3	a;				matrix a
 *				Pmatrix3	b;				matrix b
 *
 *	On Exit:
 *				Pint	*error_ind;		error indicator
 *				Pmatrix3	result;		result matrix
 *
 *	Notes:
 *
 *
 ******************************************************************************
 */
void
pcomposematrix3(a, b, error_ind, result)

   Pmatrix3	a;		/* matrix a	*/
    Pmatrix3	b;		/* matrix b	*/
    Pint	*error_ind;	/* OUT error indicator	*/
    Pmatrix3	result;	/* OUT result matrix	*/
{
    int i, j;		/* Loop counters */
    
    *error_ind = PNO_ERROR;
    
    for (i=0; i<4; i++)
	for (j=0; j<4; j++)
	    result[i][j]=a[i][0]*b[0][j]+a[i][1]*b[1][j]+a[i][2]*b[2][j]+
		a[i][3]*b[3][j];
    
}

/*
 ******************************************************************************
 *
 *	Function: pcomposematrix(a, b, error_ind, result)
 *
 *		The 2D homogeneous transformation matrix to perform the 
 *		transformation [a] * [b] is returned in result.
 *       
 *	On Entry:
 *				Pmatrix	a;				matrix a
 *				Pmatrix	b;				matrix b
 *
 *	On Exit:
 *				Pint	*error_ind;		error indicator
 *				Pmatrix	result;		result matrix
 *
 *	Notes:
 *
 *
 ******************************************************************************
 */
void
pcomposematrix(a, b, error_ind, result)

    Pmatrix	a;			/* matrix a	*/
    Pmatrix	b;			/* matrix b	*/
    Pint	*error_ind;	/* OUT error indicator	*/
    Pmatrix	result;		/* OUT result matrix	*/
{
    Pint i, j;
    
    *error_ind = PNO_ERROR;
    
    for (i=0; i<2; i++)
	for (j=0; j<2; j++)
	    result[i][j]=a[i][0]*b[0][j]+a[i][1]*b[1][j]+a[i][2]*b[2][j];
    
}

/*
 ******************************************************************************
 *
 *	Function: ptranvc3(vc, matrix, error_ind, result)
 *
 *		The result of applying transformation matrix to vc is retuned
 *		as result.
 *       
 *	On Entry:
 *				Ppoint3	*vc;			vector
 *				Pmatrix3	matrix;		transformation matrix
 *
 *	On Exit:
 *				Pint	*error_ind;		error indicator
 *				Ppoint3	*result;		transformed vector
 *
 *	Notes:
 *
 *
 ******************************************************************************
 */
void
ptranvc3(vc, matrix, error_ind, result)

    Ppoint3	*vc;		/* vector	*/
    Pmatrix3	matrix;	/* transformation matrix	*/
    Pint	*error_ind;	/* OUT error indicator	*/
    Ppoint3	*result;	/* OUT transformed vector	*/
{
    Pfloat w;
   
    *error_ind = PNO_ERROR;
    
    result->x = 
	vc->x*matrix[0][0]+vc->y*matrix[1][0]+vc->z*matrix[2][0];
    result->y = 
	vc->x*matrix[0][1]+vc->y*matrix[1][1]+vc->z*matrix[2][1];
    result->z = 
	vc->x*matrix[0][2]+vc->y*matrix[1][2]+vc->z*matrix[2][2];
    
}

/*
 ******************************************************************************
 *	Function:  ptranpt(pt, matrix, error_ind, result)
 *       
 *		The result of applying transformation matrix to pt is retuned
 *		as result.
 *       
 *	On Entry: 
 *				Ppoint	*pt;			point
 *				Pmatrix	matrix;		transformation matrix
 *	On Exit:  
 *				Pint	*error_ind;		error indicator
 *				Ppoint	*result;		transformed point
 *	Notes: none
 *
 *
 ******************************************************************************
 */
void
ptranpt(pt, matrix, error_ind, result)

    Ppoint	*pt;		/* point	*/
    Pmatrix	matrix;		/* transformation matrix	*/
    Pint	*error_ind;	/* OUT error indicator	*/
    Ppoint	*result;	/* OUT transformed point	*/
{
    Pfloat w;
    
    *error_ind = PNO_ERROR;
    
    result->x = 
	pt->x*matrix[0][0]+pt->y*matrix[1][0]+matrix[2][0];
    result->y = 
	pt->x*matrix[0][1]+pt->y*matrix[1][1]+matrix[2][1];
    w = 
	pt->x*matrix[0][2]+pt->y*matrix[1][2]+matrix[2][2];
    
    if ((w<0.00001)&&(w> -.00001)) { 
    }
    else {
	result->x = result->x/w; 
	result->y = result->y/w; 
    }
    
}

/*
 ******************************************************************************
 *
 *	Function: ptranvec3(pt, matrix, error_ind, result)
 *
 *		The result of applying transformation matrix to pt is retuned
 *		as result.
 *       
 *	On Entry:
 *				Ppoint3	*pt;			point
 *				Pmatrix3	matrix;		transformation matrix
 *
 *	On Exit:
 *				Pint	*error_ind;		error indicator
 *				Ppoint3	*result;		transformed point
 *
 *	Notes:
 *
 *
 ******************************************************************************
 */
void
ptranvec3(pt, matrix, error_ind, result)

    Ppoint3	*pt;		/* point	*/
    Pmatrix3	matrix;		/* transformation matrix	*/
    Pint	*error_ind;	/* OUT error indicator	*/
    Ppoint3	*result;	/* OUT transformed point	*/
{
    Pfloat w;
    
    *error_ind = PNO_ERROR;
    
    result->x = 
	pt->x*matrix[0][0]+pt->y*matrix[1][0]+pt->z*matrix[2][0]+matrix[3][0];
    result->y = 
	pt->x*matrix[0][1]+pt->y*matrix[1][1]+pt->z*matrix[2][1]+matrix[3][1];
    result->z = 
	pt->x*matrix[0][2]+pt->y*matrix[1][2]+pt->z*matrix[2][2]+matrix[3][2];
    w = 
	pt->x*matrix[0][3]+pt->y*matrix[1][3]+pt->z*matrix[2][3]+matrix[3][3];
    
    if ((w<0.00001)&&(w> -.00001)) { 
    }
    else {
	result->x = result->x/w; 
	result->y = result->y/w; 
	result->z = result->z/w;
    }
}

/*
 ******************************************************************************
 *	Function:  pbuildtran3(pt,shift,x_angle,y_angle,z_angle,
 *												scale,error_ind,matrix)
 *
 *		A 3D homogeneous transformation matrix to perform the specified 
 *		transformation is returned in matrix.  The order of transformation 
 *		is: scale, rotate (both relative to the specified fixed point), 
 *		and shift.
 *       
 *	On Entry: 
 *				Ppoint3	*pt;			fixed point
 *				Pvector3	*shift;		shift vector
 *				Pfloat	x_angle;		rotation angle X	(radians)
 *				Pfloat	y_angle;		rotation angle Y	(radians)
 *				Pfloat	z_angle;		rotation angle Z	(radians)
 *				Pvector3	*scale;		scale vector
 *	On Exit:  
 *				Pint	*error_ind;		error indicator
 *				Pmatrix3	matrix;		transformation matrix
 *	Notes: none
 *
 ******************************************************************************
 */
void
pbuildtran3(pt, shift, x_angle, y_angle, z_angle, scale, error_ind, matrix)

    Ppoint3	*pt;		/* fixed point	*/
    Pvector3	*shift;	/* shift vector	*/
    float	x_angle;	/* rotation angle X	*/
    Pfloat	y_angle;	/* rotation angle Y	*/
    Pfloat	z_angle;	/* rotation angle Z	*/
    Pvector3	*scale;	/* scale vector	*/
    Pint	*error_ind;	/* OUT error indicator	*/
    Pmatrix3	matrix;	/* OUT transformation matrix	*/
{
    Ppoint3 pttmp;
    Pmatrix3 matrixtmp;
    
    *error_ind = PNO_ERROR;
    
    /* Translate to -pt */
    pttmp.x = -pt->x;
    pttmp.y = -pt->y;
    pttmp.z = -pt->z;
    ptranslate3(&pttmp, error_ind, matrix);
    
    /* Scale by scale */
    pscale3(scale, error_ind, matrixtmp);
    pcomposematrix3(matrix, matrixtmp, error_ind, matrix);
    
    /* Rotate by x_angle */
    protatex(x_angle, error_ind, matrixtmp);
    pcomposematrix3(matrix, matrixtmp, error_ind, matrix);
    
    /* Rotate by y_angle */
    protatey(y_angle, error_ind, matrixtmp);
    pcomposematrix3(matrix, matrixtmp, error_ind, matrix);
    
    /* Rotate by z_angle */
    protatez(z_angle, error_ind, matrixtmp);
    pcomposematrix3(matrix, matrixtmp, error_ind, matrix);
    
    /* Translate back by pt */
    ptranslate3(pt, error_ind, matrixtmp);
    pcomposematrix3(matrix, matrixtmp, error_ind, matrix);
    
    /* Translate by shift */
    ptranslate3(shift, error_ind, matrixtmp);
    pcomposematrix3(matrix, matrixtmp, error_ind, matrix);
    
    
}

/*
 ******************************************************************************
 *	Function:  pbuildtran(pt, shift, angle, scale, error_ind, matrix)
 *       
 *		A homogeneous transformation matrix to perform the specified 
 *		transformation is returned in matrix.  The order of transformation 
 *		is: scale, rotate (both relative to the specified fixed point), 
 *		and shift.
 *       
 *	On Entry: 
 *				Ppoint	*pt;			fixed point
 *				Pvector	*shift;		shift vector
 *				Pfloat	angle;		rotation angle	(radians)
 *				Pvector	*scale;		scale vector
 *	On Exit:  
 *				Pint	*error_ind;		error indicator
 *				Pmatrix	matrix;		transformation matrix
 *	Notes: none
 *
 ******************************************************************************
 */
void
pbuildtran(pt, shift, angle, scale, error_ind, matrix)

    Ppoint	*pt;		/* fixed point	*/
    Pvector	*shift;		/* shift vector	*/
    Pfloat	angle;		/* rotation angle	*/
    Pvector	*scale;		/* scale vector	*/
    Pint	*error_ind;	/* OUT error indicator	*/
    Pmatrix	matrix;		/* OUT transformation matrix	*/
{
    Ppoint pttmp;
    Pmatrix matrixtmp;
    
    *error_ind = PNO_ERROR;
    
    /* Translate to -pt */
    pttmp.x = -pt->x;
    pttmp.y = -pt->y;
    ptranslate(&pttmp, error_ind, matrix);
    
    /* Scale by scale */
    pscale(scale, error_ind, matrixtmp);
    pcomposematrix(matrix, matrixtmp, error_ind, matrix);
    
    /* Rotate by angle */
    protate(angle, error_ind, matrixtmp);
    pcomposematrix(matrix, matrixtmp, error_ind, matrix);
    
    /* Translate back by pt */
    ptranslate(pt, error_ind, matrixtmp);
    pcomposematrix(matrix, matrixtmp, error_ind, matrix);
    
    /* Translate by shift */
    ptranslate(shift, error_ind, matrixtmp);
    pcomposematrix(matrix, matrixtmp, error_ind, matrix);
    

}

/*
 ******************************************************************************
 *	Function:  pcomposetran3(matrix, pt, shift, x_angle, 
 *									y_angle, z_angle, scale, error_ind, result)
 *
 *		A 3D homogeneous transformation matrix is returned in result which
 *		is the composition of the specified matrix with the matrix defined by
 *		the fixed point, shift, rotate, and scale parameters.  The order of 
 *		transformation is: scale, rotate (both relative to the specified 
 *		fixed point), and shift.
 *       
 *	On Entry: 
 *				Pmatrix3	matrix;		transformation matrix
 *				Ppoint3	*pt;			fixed point
 *				Pvector3	*shift;		shift vector
 *				Pfloat	x_angle;		rotation angle X	(radians)
 *				Pfloat	y_angle;		rotation angle Y	(radians)
 *				Pfloat	z_angle;		rotation angle Z
 *				Pvector3	*scale;		scale vector
 *	On Exit:  
 *				Pint	*error_ind;		error indicator
 *				Pmatrix3	result;		transformation matrix
 *	Notes: none
 *
 ******************************************************************************
 */
void
pcomposetran3(matrix, pt, shift, x_angle, y_angle, z_angle, scale, error_ind, result)

    Pmatrix3	matrix;	/* transformation matrix	*/
    Ppoint3	*pt;		/* fixed point	*/
    Pvector3	*shift;	/* shift vector	*/
    Pfloat	x_angle;	/* rotation angle X	*/
    Pfloat	y_angle;	/* rotation angle Y	*/
    Pfloat	z_angle;	/* rotation angle Z	*/
    Pvector3	*scale;	/* scale vector	*/
    Pint	*error_ind;	/* OUT error indicator	*/
    Pmatrix3	result;	/* OUT transformation matrix	*/
{
    
    pbuildtran3(pt, shift, x_angle, y_angle, z_angle, scale, error_ind, result);
    pcomposematrix3( matrix, result, error_ind, result );
    
}

/*
 ******************************************************************************
 *	Function:  pcomposetran(matrix, pt, shift, angle, scale, 
 *													error_ind, result)
 *       
 *	On Entry: 
 *				Pmatrix	matrix;		transformation matrix
 *				Ppoint	*pt;			fixed point
 *				Pvector	*shift;		shift vector
 *				Pfloat	angle;		rotation angle
 *				Pvector	*scale;		scale vector
 *	On Exit:  
 *				Pint	*error_ind;		error indicator
 *				Pmatrix	result;		transformation matrix
 *	Notes: none
 *
 ******************************************************************************
 */
void
pcomposetran(matrix, pt, shift, angle, scale, error_ind, result)

    Pmatrix	matrix;		/* transformation matrix	*/
    Ppoint	*pt;		/* fixed point	*/
    Pvector	*shift;		/* shift vector	*/
    Pfloat	angle;		/* rotation angle	*/
    Pvector	*scale;		/* scale vector	*/
    Pint	*error_ind;	/* OUT error indicator	*/
    Pmatrix	result;		/* OUT transformation matrix	*/
{
    
    pbuildtran(pt, shift, angle, scale, error_ind, result);
    pcomposematrix( matrix, result, error_ind, result );

}


/*
 ******************************************************************************
 *	Function:  pident3( matrix )
 *
 *	Return a 4x4 identity matrix.
 *       
 *	On Entry: 
 *	On Exit:  
 *				Pmatrix3	result;		identity matrix
 *	Notes: none
 *
 ******************************************************************************
 */
pident3(matrix)

    Pmatrix3 matrix;
{
    int i,j;
    
    for(i=0; i<4; i++) {
	for(j=0; j<4; j++)
	    matrix[i][j] = 0.0;
	matrix[i][i] = 1.0;
    }
}
 

/************************************************************************
*
* FUNCTION: atan3( opp, adj )
*
* If adj == 0.0, returns 0.0
* Else, returns atan2 of num, denom.
*
* Input:        opp     Opposite side of triangle
*               adj     Adjacent side of triangle
*
* Output:       none
* Returns:      Angle of the triangle
*
* Warnings:     none
*
************************************************************************/
double atan3(opp, adj)

    double opp, adj;
{
    float temp;
    
    if( fabs( adj ) < 0.000001 && fabs( opp ) < 0.000001 )
	return(0.0);
    else
	return(atan2(opp, adj));
}

