/* 
 * PNurb.c - Routines to send Nurb packets.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include "pexDICE.h"

PexNurbCurve (pexi,
	      curveOrder,
	      tMin, tMax,
	      coordType,
	      numKnots,
	      knots,
	      numPoints,
	      points)
    
    register pexC *pexi;
    CARD16 curveOrder, coordType;
    float tMin, tMax;
    CARD32 numKnots, numPoints;
    float *knots, *points;
{
    register Display *dpy = pexi->phigsDisplay;
    register pexRenderOutputCommandsReq *req;
    CARD32 totalLen, pointSize;
    int one_shot = 0;

    pexNonUniformBSplineCurve oc;

    if( ( req = pexi->curRenderOut ) == 0 )
    {
	PexRenderOutputCommands (pexi);
	one_shot++;
	req = pexi->curRenderOut;
    }

    pointSize = ((coordType == Rational) ?
		 sizeof(pexCoord4D) :
		 sizeof(pexCoord3D));

    totalLen = ((sizeof(pexNonUniformBSplineCurve)) +   /* space for struct */
		(pointSize * numPoints) +     /* ...for vertices */
		(sizeof(float) * numKnots));

    oc.head.elementType = OCNurbCurve;
    oc.head.length = totalLen>>2;

    oc.curveOrder = curveOrder;
    oc.coordType = coordType;
    oc.tMin = tMin;
    oc.tMax = tMax;
    oc.numPoints = numPoints;
    oc.numKnots = numKnots;

    pexPackData (pexi, (char *)(&oc), sizeof(pexNonUniformBSplineCurve));
    pexPackData (pexi, knots, sizeof(float)*numKnots);
    pexPackData (pexi, points, pointSize*numPoints);

    req->numCommands++;
    req->length += (totalLen>>2);

    if (one_shot)
	PexEndRendComm(pexi);
};



PexNurbSurface (pexi,
		uOrder, vOrder,
		uMin, uMax,
		vMin, vMax,
		coordType,
		numUknots,
		numVknots,
		numUpoints,
		numVpoints,
		uKnots,
		vKnots,
		points)
    
    register pexC *pexi;
    CARD16 uOrder, vOrder, coordType;
    float uMin, uMax;
    float vMin, vMax;
    CARD32 numUknots, numUpoints;
    CARD32 numVknots, numVpoints;
    float *uKnots, *vKnots, *points;
{
    register Display *dpy = pexi->phigsDisplay;
    register pexRenderOutputCommandsReq *req;
    CARD32 totalLen, pointSize, numPoints;
    int one_shot = 0;

    pexNonUniformBSplineSurface oc;

    if( ( req = pexi->curRenderOut ) == 0 )
    {
	PexRenderOutputCommands (pexi);
	one_shot++;
	req = pexi->curRenderOut;
    }

    numPoints = numUpoints * numVpoints;
    pointSize = ((coordType == Rational) ?
		 sizeof(pexCoord4D) :
		 sizeof(pexCoord3D));

    totalLen = ((sizeof(pexNonUniformBSplineSurface)) +   /* space for struct */
		(pointSize * numPoints) +     /* ...for vertices */
		(sizeof(float) * numUknots) +
		(sizeof(float) * numVknots));

    oc.head.elementType = OCNurbSurface;
    oc.head.length = totalLen>>2;

    oc.type = coordType;
    oc.uOrder = uOrder;
    oc.vOrder = vOrder;
    oc.pad = 0;
    oc.uMin = uMin;
    oc.uMax = uMax;
    oc.vMin = vMin;
    oc.vMax = vMax;
    oc.numUknots = numUknots;
    oc.numVknots = numVknots;
    oc.numPoints = numPoints;
    oc.numLists = 0;
    
    pexPackData (pexi, (char *)(&oc), sizeof(pexNonUniformBSplineSurface));
    pexPackData (pexi, uKnots, sizeof(float)*numUknots);
    pexPackData (pexi, vKnots, sizeof(float)*numVknots);
    pexPackData (pexi, points, pointSize*numPoints);

    req->numCommands++;
    req->length += (totalLen>>2);

    if (one_shot)
	PexEndRendComm(pexi);
};
