/* xdvorak installs a Dvorak keyboard map on an X server. */

/* Install a Dvorak keyboard map with the command:

		xdvorak

With no arguments, the program toggles between a Dvorak and a qwerty
keyboard map.  Any arguments force the installation of a qwerty
keyboard map.  Execute "xdvorak -qwerty" when you log out. */

/* John D. Ramsdell -- July 1988. */

static char copyright[] = "Copyright 1988 by The MITRE Corporation.";
/* Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies.  The
MITRE Corporation makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.  */

/* Compile with the command: cc -O -o xdvorak xdvorak.c -lX */

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

int is_qwerty(dpy)		/* Root window is checked for a */
     Display *dpy;		/* property that gives the keycode */
{				/* of the key marked D. */
  KeyCode current_key_marked_D;
  Window root;
  Atom key, type, actual_type_return;
  int actual_format_return;
  unsigned long nitems_return;
  unsigned long bytes_after_return;
  unsigned char *prop_return;

  current_key_marked_D = XKeysymToKeycode(dpy, XK_D);
  key = XInternAtom(dpy, "KEYCODE_OF_KEY_MARKED_D", False);
  type = XInternAtom(dpy, "KEYCODE", False);
  root = XDefaultRootWindow(dpy);

  XGetWindowProperty(dpy, root, key, 0L, 1L, False, type,
		     &actual_type_return, &actual_format_return,
		     &nitems_return, &bytes_after_return,
		     &prop_return);

  if (actual_type_return != type) { /* Property not there. */
    XChangeProperty(dpy, root, key, type, 8*sizeof(KeyCode),
		    PropModeReplace,
		    (unsigned char *)(&current_key_marked_D), 1);
    XFlush(dpy);		/* Record keycode with root window. */
    return True;
  } else
    return *((KeyCode *) prop_return) == current_key_marked_D;
}

#define NCHAR 256
KeySym qwerty[NCHAR], dvorak[NCHAR]; /* These will contain the permutations. */
int old_keysym_to_map_index[NCHAR];
Display *dpy;

/* translate from marked key to dvorak value. */
#define translate(Q, D) { KeySym q = Q; KeySym d = D; \
if (q < NCHAR) dvorak[q] = d; if (d < NCHAR) qwerty[d] = q; }

/* The following is a fairly machine independent description of
minimal changes required to make a Dvorak keyboard.  You might want to
tailor this for your machine.  */

make_permutation()
{
  int i;
  for (i = 0; i < NCHAR; i++) {
    dvorak[i] = i;		/* Initialize with the  */
    qwerty[i] = i;		/* identify permutation. */
  }

  translate (XK_minus, 	XK_bracketleft);
  /* Next row. */
  translate (XK_Q,	XK_quoteright);
  translate (XK_W,	XK_comma);
  translate (XK_E,	XK_period);
  translate (XK_R,	XK_P);
  translate (XK_T,	XK_Y);
  translate (XK_Y,	XK_F);
  translate (XK_U,	XK_G);
  translate (XK_I,	XK_C);
  translate (XK_O,	XK_R);
  translate (XK_P,	XK_L);
  translate (XK_bracketleft,	XK_slash);
  /* Next row. */
  translate (XK_S,	XK_O);
  translate (XK_D,	XK_E);
  translate (XK_F,	XK_U);
  translate (XK_G,	XK_I);
  translate (XK_H,	XK_D);
  translate (XK_J,	XK_H);
  translate (XK_K,	XK_T);
  translate (XK_L,	XK_N);
  translate (XK_semicolon,	XK_S);
  translate (XK_quoteright,	XK_minus);
  /* Next row. */
  translate (XK_Z,	XK_semicolon);
  translate (XK_X,	XK_Q);
  translate (XK_C,	XK_J);
  translate (XK_V,	XK_K);
  translate (XK_B,	XK_X);
  translate (XK_N,	XK_B);
  translate (XK_comma,	XK_W);
  translate (XK_period,	XK_V);
  translate (XK_slash,	XK_Z);
}

int permute_keyboard_map(force_qwerty)
     int force_qwerty;
{
  KeySym *new_map, *old_map, *permutation;
  int min_keycode = dpy->min_keycode;
  int max_keycode = dpy->max_keycode;
  int keysyms_per_keycode;
  int am_qwerty = is_qwerty(dpy);

  if (force_qwerty) {
    if (am_qwerty) return 0;	/* No change needed. */
    permutation = qwerty;
  } else permutation = am_qwerty ? dvorak : qwerty; /* Else toggle. */
  make_permutation();
  old_map =			/* old_map will not be modified. */
    XGetKeyboardMapping(dpy,
			min_keycode,
			max_keycode - min_keycode + 1,
			&keysyms_per_keycode);
  new_map =			/* new_map gets the new map. */
    XGetKeyboardMapping(dpy,
			min_keycode,
			max_keycode - min_keycode + 1,
			&keysyms_per_keycode);
#if defined DEBUG
  printf("max, min, per= %d, %d, %d\n",
	 max_keycode, min_keycode, keysyms_per_keycode);
#endif
  { int i, k;		/* Construct inverse map. */
    for (i = min_keycode, k = 0; i <= max_keycode; i++) {
      if (old_map[k] < NCHAR && old_map[k] != NoSymbol)
	old_keysym_to_map_index[old_map[k]] = k;
      k += keysyms_per_keycode;
    }
  }

  { int i, j, k, l;		/* permute new_map. */
    for (i = min_keycode, k = 0; i <= max_keycode; i++)
      if (old_map[k] >= NCHAR || old_map[k] == NoSymbol
	  || old_map[k] == permutation[old_map[k]])
	k += keysyms_per_keycode;
      else {
	l = old_keysym_to_map_index[permutation[old_map[k]]];
	if (k == l) k += keysyms_per_keycode;
	else
	  for (j = 0; j < keysyms_per_keycode; j++, k++, l++) {
	    new_map[k] = old_map[l];
#if defined DEBUG
	    printf("Mapping %s->%s \t(%d->%d)\n",
		   XKeysymToString(old_map[k]),
		   XKeysymToString(new_map[k]),
		   l, k);
#endif
	  }
      }
  }
	    
  XChangeKeyboardMapping(dpy,	/* Install new map. */
			 min_keycode,
			 keysyms_per_keycode,
			 new_map,
			 max_keycode - min_keycode + 1);

  XFree((char *) old_map);	/* Release storage. */
  XFree((char *) new_map);
  XFlush(dpy);			/* Force out changes. */
  printf("%s keyboard installed.\n",
	 permutation == dvorak ? "Dvorak" : "Qwerty");
  return 0;			/* Permutation succeeded. */
}

main(argc, argv)
     int argc;
     char **argv;
{ argv;				/* Not used. */

  if (!(dpy = XOpenDisplay(NULL))) {
    perror("Cannot open display\n");
    exit(1);
  }

  /* Force installation of qwerty when any extra arguments are given. */
  exit (permute_keyboard_map(argc > 1));
}
