/*
 * 
 * $Copyright
 * Copyright 1993, 1994 , 1995 Intel Corporation
 * INTEL CONFIDENTIAL
 * The technical data and computer software contained herein are subject
 * to the copyright notices; trademarks; and use and disclosure
 * restrictions identified in the file located in /etc/copyright on
 * this system.
 * Copyright$
 * 
 */
 
/*
 * Copyright (c) 1991, Locus Computing Corporation
 * All rights reserved
 */
/* 
 * HISTORY
 * $Log: bvp_vpops.c,v $
 * Revision 1.12  1994/11/18  20:51:03  mtm
 * Copyright additions/changes
 *
 * Revision 1.11  1994/04/28  19:20:33  chrisp
 * Changes for introduction of tncgen and support of i386 builds.
 * Generated prototype header files changed for consistency with tnc/.
 *
 *  Reviewer: dleslie, cfj
 *  Risk: M
 *  Benefit or PTS #: 9188
 *  Testing: Builds and tested on i386 platform.
 *  Module(s):
 *      Modified Files:
 *      	bvp_init.c bvp_vpops.c bvp_vpsops.c
 *      Added Files:
 *      	gen_dispatch_table.sh gen_op_macros.sh gen_prototypes.sh
 *      	gen_table_struct.sh tncgen vp.ops vprocgen.mk vps.ops
 *      Removed Files:
 *      	maketables.sh
 *
 * Revision 1.10  1994/03/14  02:09:51  slk
 * Checkpoint Restart Code Drop
 *  Reviewer: Stefan Tritscher
 *  Risk: Medium
 *  Benefit or PTS #: Enhancement
 *  Testing: Locus VSTNC, EATS TCP-IP, Individual Checkpoint/Restart tests.
 *  Module(s):
 *
 * Revision 1.9  1993/10/11  13:15:40  nandy
 * Added uid and ruid to get_attr().
 *
 * Revision 1.8  1993/07/19  23:00:38  robboy
 * Integrate OSF/Locus Lite server changes
 *
 * Revision 1.7  1993/07/14  18:47:21  cfj
 * OSF/1 AD 1.0.4 code drop from Locus.
 *
 * Revision 1.1.1.6  1993/07/01  21:10:49  cfj
 * Adding new code from vendor
 *
 * Revision 1.6  1993/05/06  19:11:02  stefan
 * ad103+tnc merged with Intel code.
 *
 * Revision 1.5  1993/04/03  03:13:21  brad
 * Merge of PFS branch (tagged PFS_End) into CVS trunk (tagged
 * Main_Before_PFS_Merge).  The result is tagged PFS_Merge_Into_Main_April_2.
 *
 * Revision 1.4  1992/12/11  03:07:47  cfj
 * Merged 12-1-92 bug drop from Locus.
 *
 * Revision 1.1.2.2.2.1  1992/12/16  06:06:18  brad
 * Merged trunk (as of the Main_After_Locus_12_1_92_Bugdrop_OK tag)
 * into the PFS branch.
 *
 * Revision 1.3  1992/11/30  22:58:34  dleslie
 * Copy of NX branch back into main trunk
 *
 * Revision 1.1.2.2  1992/11/06  20:34:55  dleslie
 * Merged bug drop from Locus November 3, 1992, with NX development
 *
 * Revision 1.1.2.1  1992/11/05  23:47:09  dleslie
 * Local changes for NX through noon, November 5, 1992.
 *
 * Revision 1.1.1.4  1993/05/03  17:55:23  cfj
 * Initial 1.0.3 code drop
 *
 * Revision 3.39  93/07/12  16:53:10  yazz
 * Correct RCS comment.
 *
 * Revision 3.38  93/06/22  18:11:44  yazz
 * [ Bug #0251 ] Remove calls to the now-defunct routine pproc_clear_sleep_-
 *      wait(). (roman)
 * [ Bug #0287 ] Make exiting proc awaken its parent, not itself. (roman)
 * Also remove spurious, unreachable return statement.
 *
 * 
 * Revision 3.37  93/06/16  15:31:49  klh
 * 	Revision 2.18  93/06/04  12:22:18  rabii
 * 		Added node argument to bvpop_fork, and pproc_fork (rabii)
 * 
 * Revision 3.36  93/04/22  16:43:17  roman
 * [Bug 222] Waiting for pgid has problems.
 *
 * Revision 3.35  93/03/15  15:39:15  roman
 * [Bug 0136] Do not add orphaned child to foster parent list of init -    
 *         the child is already on the parent-child-sibling list of init.
 * 
 * Revision 3.34  92/11/10  12:51:50  chrisp
 * Replace forward reference for bvpop_resign_pgrp().
 * 
 * Revision 3.33  92/10/28  16:03:19  roman
 * Include procedure prototypes now generated by maketables.sh
 * 	for better type-checking.
 * 
 * Revision 3.32  92/10/16  11:33:00  chrisp
 * In bvpop_get_task_port(), return ESRCH and MACH_PORT_NULL if the target
 * 	process is zombied (durriya@osf.org).
 * 
 * Revision 3.31  92/10/09  10:03:05  roman
 * Add procedure prototypes for clean i860 compilation.
 * Change format of procedure compilation to be consistant.
 * 
 * Revision 3.30  92/10/08  17:20:52  chrisp
 * Names of operations tablees changed for consistency.
 * bvpop_wait() altered to use new routines pproc_assert_sleep() and
 * 	pproc_clear_sleep_wait() to prevent signal delivery
 * 	while searching child for status.
 * 
 * Revision 3.29  92/10/01  10:38:19  roman
 * Add logic for foster parent and foster child handling; i.e. logic for
 * 	having a foster parent inherit resource usage of processes
 * 	that are children of init due to the fact that their parent
 * 	process exited.
 * 
 * Revision 3.28  92/09/29  08:34:36  roman
 * Change name of set_state vproc op to report_state.
 * Add terminal_sigpgrp vproc op and get rid of VPROC_TERMINAL_ACCESS
 * 	processing the the sigpgrp vproc op.
 * get rid of killall vproc op - replaced with virtual process system
 * 	operations (vpsop).
 * Change reference to pvproc fields to use the new (better) names.
 * Change invocations of vproc locking to use the new macros.
 * 
 * Revision 3.27  92/09/22  11:02:49  chrisp
 * [Bug #59] VPOP_SIGPGRP() should return ESUCCESS if at least one process
 * 	has been signaled and should return any error received from
 * 	pproc_signal() [e.g. EPERM] if no process is signaled.
 * 
 * Revision 3.26  92/09/21  14:15:41  chrisp
 * pproc_wakeup() required only when signaling parent of stopped or exited
 * child - move call from VPOP_SIGPROC() to VPOP_SET_STATE().
 * 
 * Revision 3.25  92/08/25  14:54:27  chrisp
 * Fix for OSF bug #368: ^Z job control defective. Correct orphan process group
 * counting logic.
 * 
 * Revision 3.24  92/07/07  15:20:47  roman
 * Fix minor typo.
 * Fix setting nice value of pgrp in same manner as setting nice value of
 * 	process.
 * 
 * Revision 3.23  92/07/06  09:07:49  chrisp
 * In bvpop_proc_nice(), for VPROC_SET case when calling pproc_set_nice,
 * 	privelege does not depend on the flag parameter.
 * 
 * Revision 3.22  92/06/17  09:18:55  roman
 * [Bug 0026] New vproc op VPOP_GET_TASK_PORT() added to get task_by_pid() 
 *	system call working.
 * 
 * Revision 3.21  92/06/05  15:46:31  roman
 * Get rid of syscall_on_master() assertions. These require including
 * 	<sys/user.h>, which is not aesthetically proper for this
 * 	file.
 * 
 * Revision 3.20  92/05/29  10:41:45  chrisp
 * In bvpop_sigproc(), test for attempt to signal a zombie process and return
 * immediately if so.
 * 
 * Revision 3.19  92/05/22  09:56:56  chrisp
 * bvpop_pgrp_nice() always returning ESCRCH - "=" -> "==".
 * 
 * Revision 3.18  92/04/14  14:31:10  roman
 * Simplify the check for setting a process that is already a pgrp leader
 * 	into its own pgrp. This check falls out of the previous check
 * 	for setting a process into a pgrp it was already a member of.
 * 
 * Revision 3.17  92/04/14  10:51:40  roman
 * Fix for "vproc not in session list" panic: treat as a no-op setting a process
 * 	into the group of which it is already a member.
 * 
 * Revision 3.16  92/03/19  15:51:20  chrisp
 * Major change in the way orphan pgrp tracking ia achieved. This brings the
 * non-TNC code into line with TNC.
 * 
 * Revision 3.15  92/03/12  15:51:33  roman
 * vproc_hash_list becomes vproc_list_lock. Missing processing for
 * 	VPROC_CHECK_CTTY option added.
 * 
 * Revision 3.14  92/03/03  16:32:13  chrisp
 * VPOP_CTTY_GETATTR() and VPOP_SET_CTTY() now take an additional parameter, the
 * session vproc, which may howver be null.
 * 
 * Revision 3.13  92/02/14  09:05:24  roman
 * Correct spelling mistake in comment.
 * 
 * Revision 3.12  92/01/30  14:20:54  chrisp
 * Correct bvpop_wait in the case of stopped (traced) child.
 * Add bvpop_ptrace.
 * 
 * Revision 3.11  92/01/28  16:34:04  roman
 * Include assert.h and parallel.h.
 * 
 * Revision 3.10  92/01/03  11:16:55  chrisp
 * Corrected setpgid() checks for group vproc not subject vproc and not
 * already a pgrp leader.
 * 
 * Revision 3.9  91/12/20  16:57:42  chrisp
 * Multiple changes to keep pace with changes in the TNC variant: in particular,
 * orphan pgrp handling revamped.
 * 
 * Revision 3.8  91/12/19  11:15:33  chrisp
 * Correct comment text (ctty_getattr).
 * 
 * Revision 3.7  91/12/04  08:10:24  chrisp
 * Correct some instances of "if (!x&y)" needing to be "if (!(x&y))".
 * Correct bvpop_exit in the case of session leader exit with no ctty set.
 * 
 * Revision 3.6  91/11/26  12:00:16  chrisp
 * vpop_sigproc corrected to return errors from pproc_ to caller.
 * vpop_setpgid now returns EPERM if session leader attempts to set pgid.
 * 
 * Revision 3.5  91/11/15  09:57:50  chrisp
 * Correct permission checks for VPOP_SET_CTTY().
 * 
 * Revision 3.4  91/11/14  15:00:45  chrisp
 * Radical change for controlling terminal handling.
 * Signaling operations revised.
 * 
 * Revision 3.3  91/11/01  17:27:29  roman
 * Add new VPOP_FREE vproc op. This is now responsible for doing the
 * vproc_port_allocate, which is no longer done when reaping the zombie.
 * 
 * Revision 3.2  91/10/31  09:42:59  roman
 * Get rid of VPROC_HOLD and VPROC_RELEASE that were done only for
 * TNC functionality.
 * 
 * Revision 3.1  91/10/30  12:54:46  roman
 * Change some comments for VPROC_HOLD and VPROC_RELEASE macros. Change
 * the pattern of VPROC_HOLD and VPROC_RELEASE for session leaders (a
 * session leader exists for every vproc in the session). Fix up the
 * bvpop_fork() logic to be more clean.
 * 
 * Revision 3.0  91/10/25  10:34:51  roman
 * Initials submission of this file, which has vproc operations which
 * do not support distributed TNC functionality.
 * 
 */
#include <machine/reg.h>
#include <sys/unix_defs.h>
#include <sys/vproc.h>
#include <vproc/bpvproc.h>
#include <sys/proc.h>
#include <sys/tty.h>
#include <sys/vnode.h>
#include <sys/resource.h>
#include <sys/signal.h>
#include <sys/wait.h>
#include <sys/errno.h>
#include <sys/ptrace.h>
#include <uxkern/import_mach.h>
#if	MACH
#include <kern/assert.h>
#include <kern/parallel.h>
#endif


/* External declarations */
extern struct vproc *vproc_alloc();

/* Declarations of forward references */
int bvpop_resign_pgrp(struct vproc *);
int bvp_adjust_job_control_count(struct vproc *, int);
int bvp_remove_foster_child(struct vproc *, struct vproc *);
int bvp_orphan_pgrp_check(struct vproc *, pid_t, pid_t);
pid_t bvp_pidgen();

/* Include procedure prototypes generated from header files */
#include <vproc/bvproc_protos_gen.h>

/*
 * NAME: bvpop_fork
 */
int
bvpop_fork(
	struct vproc *pp, 		/* parent vproc of process */
	pid_t *pid,			/* process ID generated */
	node_t node,			/* node number for task */
	caddr_t new_state,		/* thread info */
	unsigned int new_state_count)	/* server thread info */
{
	register struct vproc *vc;
	register struct pvproc *pvc;
	register struct pvproc *pvp;
	struct vproc *w;
	register error;
	struct proc *procp;
	int hashidx;
	extern struct vproc_ops bvproc_ops_table;

	pvp = PVP(pp);

	/* allocate a vproc and zeroize its pvproc */
	if ((vc = vproc_alloc()) == NULL)
		return(ENOMEM);

	*pid = bvp_pidgen();

	/* do the physical fork */
	if ((error = pproc_fork(pvp->pvp_pproc, 
				*pid, 
				&procp,
				node,
				(thread_state_t) new_state,
				new_state_count)) != 0 ) {
		vproc_dealloc(vc);
		return(error);
	}

	pvc = PVP(vc);
	bzero((caddr_t) pvc, sizeof(struct pvproc));

	/* put the vproc on the vproc hash chain */
	VPROC_LIST_LOCK();
	vc->vp_pid = *pid;
	w = vproc_hash[hashidx = VPROCPIDHASH(*pid)];
	vc->vp_hashbwd = NULL;
	vc->vp_hashfwd = w;
	vproc_hash[hashidx] = vc;
	if (w)
		w->vp_hashbwd = vc;
	VPROC_HOLD(vc, "dvpop_fork(active)");
	VPROC_LIST_UNLOCK();

	vc->vp_ops = &bvproc_ops_table;

	/*
	 * Allocate a Mach port for the vproc.
	 */
	(void) vproc_port_allocate(vc);

	/*
	 * point the pvproc at the physical process
	 */
	pvc->pvp_pproc = procp;

	/* sort out the immediate family relationships */
	pvc->pvp_childl = pvp->pvp_head_childl; /* put child on parent's */
	pvp->pvp_head_childl = vc;	/* child list */
	pvc->pvp_head_childl = NULL;	/* child has no children */
	pvc->pvp_ppid = pp->vp_pid;	/* set child's parent id */
	pvc->pvp_foster_ppid = pp->vp_pid; /* set child's foster parent id */
	pvc->pvp_pgid = pvp->pvp_pgid;	/* child's pgrp is parent's */
	pvc->pvp_sid = pvp->pvp_sid;	/* child's session is parent's */

	/* propagate the parent's susceptability to controlling terminal */
	pvc->pvp_flag |= pvp->pvp_flag&PV_SCTTY;

	if (pvc->pvp_pgid != 0) {
		register struct vproc *g;
		/* the child joins its parent's group */
		g = VPROCPTR(pvc->pvp_pgid);
		pvc->pvp_pgrpl = PVP(g)->pvp_head_pgrpl;
		PVP(g)->pvp_head_pgrpl = vc;
		VPROC_HOLD(g, "bvpop_fork(pgrp leader)");
	}

	if (pvc->pvp_sid != 0)
		VPROC_HOLD(VPROCPTR(pvc->pvp_sid),"bvpop_fork(session leader)");

	/* initialize the process structure with values from the pvproc */
	pproc_set_attr(pvc->pvp_pproc, 
		       vc, 
		       &vc->vp_pid, 
		       &pvc->pvp_ppid, 
		       &pvc->pvp_pgid, 
		       &pvc->pvp_sid,
		       0);

	return(ESUCCESS);
}


/*
 * NAME: bvp_pidgen
 *
 * FUNCTION: Generate a process ID
 */
pid_t
bvp_pidgen()
{
	static int mpid = 1;

	/* start the pid at 2 since 0 and 1 are used by the system */
	mpid++;
	if ( mpid > 30000 ) 
		mpid = 2;

	/* verify the pid is not being used */
	while ( VPROCPTR(mpid) != NULL ) {
		mpid++;
		if (mpid >= 30000)
			mpid = 2;
	}

	return(mpid);
}


/*
 * NAME: bvpop_exit
 */
int
bvpop_exit(
	struct vproc *v,	/* vproc of process exiting system */
	int rv)			/* exit code */
{
	register struct vproc *vc;
	register struct pvproc *pvp = PVP(v);
	register struct vproc *vfp = VPROCPTR(pvp->pvp_foster_ppid);
	pid_t t_pgid;

	/* Session leaders exit involves signaling foreground pgrp */
	if (pvp->pvp_flag&PV_SESSIONLEADER) {
		if ((VPOP_CTTY_GETATTR(v, v, &t_pgid, 0, 0, 0) == ESUCCESS) &&
		    (t_pgid != 0)) {
			register struct vproc *vfg = LOCATE_VPROC_PID(t_pgid);
			if (vfg) {
				(void) VPOP_SIGPGRP(vfg,
						    SIGHUP,
						    0, 
						    NULL,
						    VPROC_HAS_PRIV |
							VPROC_NOSIGSELF);
				VPROC_RELEASE(vfg, "bvpop_exit");
			}
			/* Release the controlling terminal */
			(void) VPOP_SET_CTTY(v, v, 0, VPROC_SET);
		}
	}
			
	/*
	 * The process exiting may orphan its own pgrp. While it is also
	 * possible that children of this process may have their pgrps
	 * orphaned, this is taken care of in the following loop.
	 */
	if (pvp->pvp_pgid != 0 &&
			pvp->pvp_sid == PVP(VPROCPTR(pvp->pvp_ppid))->pvp_sid &&
			pvp->pvp_pgid != PVP(VPROCPTR(pvp->pvp_ppid))->pvp_pgid)
		(void) bvp_adjust_job_control_count(VPROCPTR(pvp->pvp_pgid),
						    PV_SUBTRACT | PV_EXIT);
	
	/* Reassign children, N.B. pvp_head_childl becomes next sibling */
	while ((vc = pvp->pvp_head_childl) != NULL) {
		register struct vproc *ip = VPROCPTR(1);
		register struct pvproc *pvc = PVP(vc);

		/* Remove child from parent's list */
		pvp->pvp_head_childl = pvc->pvp_childl;

		/* Reassign the child may orphan its pgrp */
		if (pvc->pvp_pgid != 0 &&
				pvc->pvp_sid == pvp->pvp_sid && 
				pvc->pvp_pgid != pvp->pvp_pgid)
			(void) bvp_adjust_job_control_count(
					VPROCPTR(pvc->pvp_pgid),
					PV_SUBTRACT | PV_EXIT);

		/* Send child to the init institution */
		pvc->pvp_ppid = 1;
		pproc_set_attr(pvc->pvp_pproc,0,0,&pvc->pvp_ppid,0,0,0);
		pvc->pvp_childl = PVP(ip)->pvp_head_childl;
		PVP(ip)->pvp_head_childl = vc;

		/* Assign child to foster parent, unless the foster parent
		 * is init (there is no point in having the foster parent
		 * be init, since init is already the parent).
		 */
		if (vfp->vp_pid != 1) {
			pvc->pvp_foster_ppid = vfp->vp_pid;
			pvc->pvp_foster_childl
					= PVP(vfp)->pvp_head_foster_childl;
			PVP(vfp)->pvp_head_foster_childl = vc;
		} else {
			pvc->pvp_foster_ppid = 1;
		}

		/* If proc is a zombie, signal INIT to cleanup */
		if (pvc->pvp_flag & PV_SZOMB)
			(void) VPOP_SIGPROC(ip, SIGCHLD, 0, VPROC_HAS_PRIV);
		else
			/* kill the child if it's being traced */
			(void) VPOP_SIGPROC(vc, SIGKILL, 0,
					    VPROC_HAS_PRIV | VPROC_IFTRACED);

		/* 
		 * In TNC, clearing pending terminal access signals for
		 * potentially remote descendants isn't so easy and it
		 * certainly isn't atomic. Hence, the call to spgrp() and
		 * described as:
		 * 	Protect this process from future
		 * 	tty signals, clear TSTP/TTIN/TTOU if pending
		 * isn't attempted.
		 */
		   
	}

	/* Reassign foster children to new foster parent */
	while ((vc = pvp->pvp_head_foster_childl) != NULL) {

		/* Remove child from parent's list */
		pvp->pvp_head_foster_childl = PVP(vc)->pvp_foster_childl;

		/* Assign child to foster parent, unless the foster parent
		 * is init (there is no point in having the foster parent
		 * be init, since init is already the parent).
		 */
		if (vfp->vp_pid != 1) {
			PVP(vc)->pvp_foster_ppid = vfp->vp_pid;
			PVP(vc)->pvp_foster_childl = 
					PVP(vfp)->pvp_head_foster_childl;
			PVP(vfp)->pvp_head_foster_childl = vc;
		} else {
			ASSERT(PVP(vc)->pvp_ppid == 1);
			PVP(vc)->pvp_foster_ppid = 1;
		}
	}

	/* Now let's get physical */
	pproc_exit(pvp->pvp_pproc, rv);
	
	/* Mark it zombie */
	(void) VPOP_REPORT_STATE(v, VPROC_ZOMBIE);

	return(ESUCCESS);
}


/*
 * NAME:	bvp_adjust_job_control_count
 *
 * FUNCTION:	Update pgrp job count for process group.
 *
 * RETURNS:	EPERM, if the specified vproc is not a process group leader.
 * 		ESUCCESS on completion.
 */
int
bvp_adjust_job_control_count(
	struct vproc *g,	/* vproc of pgrp leader */
	int  cause)		/* PV_ADD, PV_REMOVE, or PV_EXIT */
{
	register struct pvproc *pvp = PVP(g);

	/*
	 * Whether a process group is not orphaned is determined by there
	 * being at least one member with a parent in the same session
	 * but in a different process group. The count of such process
	 * group members is the jobc field. It is incremented as each
	 * such process joins the group, and decremented as each such member
	 * leaves the group. If such a member (or its parent) exits, the
	 * count is decremented and if it reaches 0 the group becomes
	 * orphaned and the group is signaled.
	 */
	if ((cause & PV_ADD) != 0)
		pvp->pvp_jobc++;
	else
		pvp->pvp_jobc--;

	/*
	 * Here if the group has been orphaned by an exiting process.
	 * Signal the group with SIGHUP/SIGCONT if any member is stopped.
	 */
	if ((cause & PV_EXIT) != 0 && pvp->pvp_jobc == 0) {
		register struct vproc *q;
		for (q = pvp->pvp_head_pgrpl; q != NULL; q = PVP(q)->pvp_pgrpl){
			/*
			 * post SIGHUP conditionally on the process being 
			 * stopped
			 */
			if (VPOP_SIGPROC(q, 
					 SIGHUP, 
					 0, 
					 VPROC_HAS_PRIV|VPROC_IFSTOPPED) 
							== ESUCCESS) {
				register struct vproc *r;

				/*
				 * here if a stopped member exists
				 * send it a SIGCONT
				 */
				(void) VPOP_SIGPROC(q, 
						    SIGCONT, 
						    0, 
						    VPROC_HAS_PRIV);

				/*
				 * and scan again through all other members 
				 * sending * them the signal pair
				 */
				for (r = pvp->pvp_head_pgrpl; 
						r; 
						r = PVP(r)->pvp_pgrpl) {
					/* skip the first stopped process */
					if (r == q)
						continue;
					(void) VPOP_SIGPROC(r, 
							    SIGHUP,
							    0,
							    VPROC_HAS_PRIV);
					(void) VPOP_SIGPROC(r, 
							    SIGCONT,
							    0,
							    VPROC_HAS_PRIV);
				}
				break;
			}
		}
	}
	return(ESUCCESS);
}


/*
 * NAME: bvpop_free
 */
int
bvpop_free(struct vproc *v)
{
	(void) vproc_port_deallocate(v, TRUE);
	return(ESUCCESS);
}


/*
 * NAME: bvpop_wait
 */
int
bvpop_wait(
	struct vproc *vp,		/* parent vproc */
	pid_t pid,			/* pid value, WAIT_ANY, 0, -pgrp id */
	int options,			/* options to vary system call */
	int *wstat,			/* child termination status */
	struct rusage_dev *ru_loc,	/* ptr to child resource usage area */
	pid_t *ret_val)			/* awaited pid */
{
	register struct pvproc *pvp = PVP(vp);
	register struct vproc *vc;
	register struct vproc *vo;
	register struct pvproc *pvc;
	int wait_satisfied;
	register f;
	int error = ESUCCESS;

	if (pid == 0)
		pid = -pvp->pvp_pgid;

loop:
	f = 0;
	(void) pproc_assert_sleep_wait(pvp->pvp_pproc);
	for (vo = NULL, vc = pvp->pvp_head_childl; 
				vc != NULL; 
				vo = vc, vc = PVP(vc)->pvp_childl) {
		if (pid != WAIT_ANY && vc->vp_pid != pid &&
		    PVP(vc)->pvp_pgid != -pid)
			continue;
#if SEC_BASE
		/*
		 * If we are init (pid 1) don't count unkillable processes.
		 * This is to avoid waiting forever for audit and policy
	   	 * daemons to teminate as a result of a signal.
	    	 */
		/* TNC can't easily do this */
#endif
		f++;

		pvc = PVP(vc);

		if (pvc->pvp_flag & PV_SZOMB || pvc->pvp_flag & PV_SSTOP) {

			wait_satisfied = pproc_reap(pvc->pvp_pproc, 
						    options, 
						    wstat, 
						    ru_loc) == ESUCCESS;

			if ((pvc->pvp_flag & PV_SZOMB) &&
					(!(options & VPROC_WNOWAIT))) {
				if (vo == NULL)
				    pvp->pvp_head_childl = pvc->pvp_childl;
				else
				    PVP(vo)->pvp_childl = pvc->pvp_childl;

				/*
				 * Accumulate child's stats in parent 
				 * or in foster parent.
				 */
				if (pvc->pvp_ppid == pvc->pvp_foster_ppid)
				    pproc_add_rusage(pvp->pvp_pproc, 
						     ru_loc);
				else {
				    struct vproc *vfp;
				    vfp = VPROCPTR(pvc->pvp_foster_ppid);
				    pproc_add_rusage(PVP(vfp)->pvp_pproc, 
						     ru_loc);
				    bvp_remove_foster_child(vc, vfp);
				}

				/* Remember child's ID */
				*ret_val = vc->vp_pid;

				/* Null out the proc pointer */ 
				pvc->pvp_pproc = NULL;

				/* Resign from process group */
				bvpop_resign_pgrp(vc);

				/* Not using session leader any more */
				if (pvc->pvp_sid != 0)
					VPROC_RELEASE(VPROCPTR(pvc->pvp_sid), 
						"bvpop_wait(session leader)");

				/* free the vproc */
				VPROC_RELEASE(vc, "bvpop_wait(active)");

				break;

			}
		
			/* stop & child is traced by parent or WUNTRACED*/
			if (pvc->pvp_flag & PV_SSTOP && wait_satisfied) {
				*ret_val = vc->vp_pid;
				break;
			}
		}
	}
	if (vc != NULL)
		error = ESUCCESS;
	else if (f == 0)
		error = ECHILD;
	else if (options & VPROC_WNOHANG) {
		*ret_val = 0;
		error = ESUCCESS;
	}
	else if (!(error = pproc_sleep_waiting(pvp->pvp_pproc))) {
		goto loop;
	}
	return (error);
}


/*
 * NAME: bvp_remove_foster_child
 */
int
bvp_remove_foster_child(
	struct vproc *vc,		/* vproc of foster child process */
	struct vproc *vfp)		/* vproc of foster parent process */
{
	struct pvproc	*pvc = PVP(vc);
	struct pvproc	*pvfp = PVP(vfp);
	struct vproc	*v;

	if (pvfp->pvp_head_foster_childl == vc)
		pvfp->pvp_head_foster_childl = pvc->pvp_foster_childl;
	else {
		for (v = pvfp->pvp_head_foster_childl;
				PVP(v)->pvp_foster_childl != vc;
				v = PVP(v)->pvp_foster_childl)
			if (v == NULL)
				panic("bvp_remove_foster_child");
		PVP(v)->pvp_foster_childl = pvc->pvp_foster_childl;
	}
}


/*
 * NAME: bvpop_proc_nice
 */
int
bvpop_proc_nice(
	struct vproc *v,		/* vproc of target process */
	int *nice,			/* nice to set or returned value */
	int flag)			/* set or get */
{
	uid_t euid = 0, ruid = 0;
	int has_priv = 0;

	/* find caller's privilege if we're setting nice */
	if (flag == VPROC_SET)
		(void) pproc_get_attr(0,0,0,0,0,&has_priv,&euid,&ruid,0,0);

	switch (flag&(VPROC_SET|VPROC_GET)) {
	case VPROC_GET:
		return(pproc_get_nice(PVP(v)->pvp_pproc, nice));
	case VPROC_SET:	
		return(pproc_set_nice(PVP(v)->pvp_pproc,
				      euid, 
				      ruid, 
				      has_priv,
				      *nice));
	default:
		return(EINVAL);
	}
}

/*
 * NAME: bvpop_pgrp_nice
 *
 * FUNCTION:  Read/write process nice value for process group.
 */
int
bvpop_pgrp_nice(
	struct vproc *g,		/* vproc of target process group */
	int *nice,			/* nice to set or returned value */
	int flag)			/* set or get */
{
	uid_t euid = 0, ruid = 0;
	int has_priv = 0;
	register struct vproc *v;
	register low = PRIO_MAX + 1;
	register found = 0, error = 0;
	int member_nice;

	/* find caller's privilege if we're setting nice */
	if (flag == VPROC_SET)
		(void) pproc_get_attr(0,0,0,0,0,&has_priv,&euid,&ruid,0,0);
	switch (flag&(VPROC_SET|VPROC_GET)) {
	case VPROC_GET:
		/* visit all group members to compute lowest nice value */
		for (v = PVP(g)->pvp_head_pgrpl; 
				v != NULL; 
				v = PVP(v)->pvp_pgrpl) {
			(void)pproc_get_nice(PVP(v)->pvp_pproc, &member_nice);
			if ( member_nice < low )
				low = member_nice;
		}
		if (low == PRIO_MAX + 1)
			return(ESRCH);
		*nice = low;
		return(ESUCCESS);

	case VPROC_SET:	
		/* visit all group members setting nice */
		for (v = PVP(g)->pvp_head_pgrpl; 
				v != NULL; 
				v = PVP(v)->pvp_pgrpl) {
			error = pproc_set_nice(PVP(v)->pvp_pproc,
					       euid, 
					       ruid, 
					       has_priv ? VPROC_HAS_PRIV : 0,
					       *nice);
			found++;
		}
		if (found == 0)
			return(ESRCH);
		return(error);
	default:
		return(EINVAL);
	}
}


/*
 * NAME: bvpop_sigproc
 *
 * FUNCTION: Sends signal to process.
 */
int
bvpop_sigproc(
	struct vproc *vto,	/* pgrp leader vproc to be signaled */
	int signo,		/* signal to be sent */
	int arg,		/* argument (for SIGMIGRATE) */
	int flags)		/* modifies posting behavior */
{
	register int error = 0;
	uid_t uid;
	uid_t ruid;
	pid_t pid, sid;
	boolean_t has_priv = TRUE;

	/*
	 * Simply don't bother to signal a zombie process
	 */
	if (PVP(vto)->pvp_flag & PV_SZOMB)
		return(ESUCCESS);

	if (!(flags&VPROC_HAS_PRIV))
		pproc_get_attr(0,&pid,0,0,&sid,&has_priv,&uid,&ruid,0,0);

	/*
	 * If delivery is conditional on stopped state,
	 * return error if process not stopped.
	 */
	if (flags&VPROC_IFSTOPPED && !(PVP(vto)->pvp_flag&PV_SSTOP))
		return(ESRCH);

	/*
	 * If delivery is conditional on there being a controlling terminal
	 * set but there isn't, simply return.
	 */
	if (flags&VPROC_CHECK_CTTY && !(PVP(vto)->pvp_flag&PV_SCTTY))
		return(ESUCCESS);

	error = pproc_psignal(PVP(vto)->pvp_pproc,
			      signo,
			      arg,
			      !(flags&VPROC_CANT_BLOCK),
			      flags&VPROC_CHILD_STOP,
			      flags&VPROC_IFTRACED,
			      has_priv,
			      uid,
			      ruid,
			      sid);

	return(error);
}


/*
 * NAME: bvpop_sigpgrp
 *
 * FUNCTION: Sends signal to process group.
 */
int
bvpop_sigpgrp(
	struct vproc *gto,	/* pgrp leader vproc to be signaled */
	int signo,		/* signal to be sent */
	int arg,		/* argument (for SIGMIGRATE) */
	int *nproc,		/* number of processes signaled */
	int flags)		/* modifies posting behavior */
{ 
	register int error = ESRCH;
	register int nsuccesses = 0;
	uid_t uid;
	uid_t ruid;
	pid_t pid, sid;
	boolean_t has_priv = TRUE;
	register struct vproc *w;

	if (!(PVP(gto)->pvp_flag&PV_PGRPLEADER))
		return(ESRCH);

	if (!(flags&VPROC_HAS_PRIV))
		pproc_get_attr(0,&pid,0,0,&sid,&has_priv,&uid,&ruid,0,0);

	/* check for terminal access signals which shouldn't be delivered */
	if (signo == SIGTTIN || signo == SIGTTOU || signo == SIGTSTP)
		if (PVP(gto)->pvp_jobc == 0)
			return(ENOTTY);

	for (w = PVP(gto)->pvp_head_pgrpl; 
			w != NULL; 
			w = PVP(w)->pvp_pgrpl) {
		if (flags&VPROC_NOSIGSELF  && w->vp_pid == pid ||
		    flags&VPROC_CHECK_CTTY && !(PVP(w)->pvp_flag&PV_SCTTY))
			continue;
		error = pproc_psignal(PVP(w)->pvp_pproc,
				      signo,
				      arg,
			              !(flags&VPROC_CANT_BLOCK),
				      0,
				      0,
			              has_priv,
			              uid,
			              ruid,
			              sid);
		if (error == ESUCCESS)
			nsuccesses++;
	}
	
	if (nproc)
		*nproc = nsuccesses;
	return(nsuccesses ? ESUCCESS : error);
}


/*
 * NAME: bvpop_report_state
 *
 * FUNCTION: Alter the state of a proc.
 */
int
bvpop_report_state(
	struct vproc *v,		/* vproc of calling process */
	int state)			/* zombie or stop or unstop */
{
	int error = ESUCCESS;
	register struct pvproc *pv = PVP(v);
	register struct vproc  *p  = VPROCPTR(pv->pvp_ppid);

	/* action depends on new state */
	switch (state) {
	case VPROC_ZOMBIE:
		pv->pvp_flag |= PV_SZOMB;
		/* take the opportunity here to signal the parent */
		error = VPOP_SIGPROC(p, SIGCHLD, 0, VPROC_HAS_PRIV);
		(void) pproc_wakeup(PVP(p)->pvp_pproc);
		break;
	case VPROC_STOP:
		pv->pvp_flag |= PV_SSTOP;
		/* signal the parent */
		error = VPOP_SIGPROC(p, SIGCHLD, 0, VPROC_HAS_PRIV |
						      VPROC_CHILD_STOP);
		(void) pproc_wakeup(PVP(p)->pvp_pproc);
		break;	
	case VPROC_UNSTOP:
		pv->pvp_flag &= ~PV_SSTOP;
		break;
	}
	return(error);
}


/*
 * NAME: bvpop_setsid
 *
 * FUNCTION: Create a new session.
 *
 */
int
bvpop_setsid(
	struct vproc *v)	/* vproc of process becoming session leader */
{
	register struct pvproc *pvp;
	boolean_t false = FALSE;

	pvp = PVP(v);

	if (pvp->pvp_pgid == v->vp_pid ||
	    pvp->pvp_flag & PV_PGRPLEADER)
		return(EPERM);

	/* This may orphan old pgrp or pgrps of children */
	ASSERT(pvp->pvp_jobc == 0);
	bvp_orphan_pgrp_check(v, v->vp_pid, v->vp_pid);

	/* resign from current process group */
	if (pvp->pvp_pgid != 0)
		bvpop_resign_pgrp(v);

	/* don't need the current session leader any more */
	if (pvp->pvp_sid != 0)
		VPROC_RELEASE(VPROCPTR(pvp->pvp_sid),
			      "bvpop_setsid(session leader)");

	/*
	 * Add process to its own pgrp list.
	 */
	ASSERT(pvp->pvp_head_pgrpl == NULL);
	pvp->pvp_head_pgrpl = v;
	pvp->pvp_pgrpl = NULL;
	VPROC_HOLD(v, "bvpop_setsid(pgrp leader)");

	/*
	 * Add pgrp to session list - except the session leader isn't
	 * so we just bump reference counts.
	 */ 
	ASSERT(pvp->pvp_sessionl == NULL);
	VPROC_HOLD(v, "bvpop_setsid(session leader)");

	/* set pvproc accordingly */
	pvp->pvp_sid  = pvp->pvp_pgid = v->vp_pid;
	pvp->pvp_flag &= ~PV_SCTTY;
	pvp->pvp_flag |= (PV_PGRPLEADER | PV_SESSIONLEADER);

	/* keep physical proc updated */
	pproc_set_attr(pvp->pvp_pproc,0,0,0,&v->vp_pid,&v->vp_pid,&false);

	return(ESUCCESS);
}


/*
 * NAME: bvpop_resign_pgrp
 *
 * FUNCTION: Resign a process from a process group.
 */
int
bvpop_resign_pgrp(
	struct vproc *v)	/* vproc of process resigning from pgrp */
{
	register struct vproc *g;

	if (PVP(v)->pvp_pgid == 0)
		return(ESUCCESS);	/* can't resign from an empty pgrp */

	g = VPROCPTR(PVP(v)->pvp_pgid);
	if (PVP(g)->pvp_head_pgrpl == v)
		/* v is the first in the list */
		PVP(g)->pvp_head_pgrpl = PVP(v)->pvp_pgrpl;
	else {
		register struct vproc *w;
		/* walk through the remaining list looking for v */
		for (w = PVP(g)->pvp_head_pgrpl; PVP(w)->pvp_pgrpl != v; 
				w = PVP(w)->pvp_pgrpl)
			if (w == NULL)
				panic("bvpop_resign_pgrp: not in pgrp list");
		PVP(w)->pvp_pgrpl = PVP(v)->pvp_pgrpl;
	}

	/* is anyone still here ? */
	if (PVP(g)->pvp_head_pgrpl == NULL) {
		if (PVP(g)->pvp_sid != 0) {
			register struct vproc *s;
			/* no more process group - resign from session too */
			s = VPROCPTR(PVP(g)->pvp_sid);
			if (s != g) {
				register struct vproc *w;
				/* walk the session list chain looking for g */
				for (w = s; PVP(w)->pvp_sessionl != g;
						w = PVP(w)->pvp_sessionl) 
					if (w == NULL)
						panic("bvpop_resign_pgrp: not in session list");
				PVP(w)->pvp_sessionl = PVP(g)->pvp_sessionl;
			}
		}
		PVP(g)->pvp_flag &= ~PV_PGRPLEADER;
	}

	VPROC_RELEASE(g, "bvpop_resign_pgrp(pgrp leader)");

	/*
	 * In case the parent of the process resigning is in a wait(),
	 * wake it up (it may be waiting for this exact pgrp).
	 */
	(void) pproc_wakeup(PVP(VPROCPTR(PVP(v)->pvp_ppid))->pvp_pproc);

	return(ESUCCESS);
}


/*
 * NAME: bvpop_setpgid
 *
 * FUNCTION: Change the process group id of a process.
 *
 */
int
bvpop_setpgid(
	struct vproc *v,	/* vproc of target process */
	struct vproc *g,	/* vproc of process group to be assigned */
	pid_t pid,		/* pid of calling process */ 
	pid_t sid)		/* session id of calling process */ 
{
	register struct vproc	*vs;	/* session leader vproc */
	struct pvproc		*pvp = PVP(v);
	int			group_jobc;
	
	/*
	 * First some checks ...
	 */

	/*
	 * The process having its group set must be the caller
	 * or a child of the caller. If a child, it must not have exec'ed
	 */
	if (v->vp_pid != pid) {
		if (pvp->pvp_ppid != pid)
			return(ESRCH);
		else {
			int has_execd;
			(void) pproc_get_attr(pvp->pvp_pproc,
					      0,0,0,0,0,0,0,0,&has_execd);
			if (has_execd)
				return(EACCES);
		}
	}
	/*
	 * The process having its group set cannot be a session leader.
	 */
	if (pvp->pvp_flag & PV_SESSIONLEADER)
		return(EPERM);
	/*
	 * If the process to be set is a child of the caller,
	 * the child must be in the same session as the caller.
	 */
	if (pvp->pvp_ppid == pid && pvp->pvp_sid != sid)
		return(EPERM);
	/*
	 * The process group must be the process itself
	 * or a group in the same session as the caller.
	 */
	if (g->vp_pid != v->vp_pid)
		if ((!(PVP(g)->pvp_flag&PV_PGRPLEADER)) ||
		    PVP(g)->pvp_sid != sid)
			return(EPERM);

	/* Determine the group leadership status of pgrp being assigned into */
	if ((PVP(g)->pvp_flag & PV_PGRPLEADER) == 0)
		group_jobc = VPROC_NOT_PGRPLEADER;
	else
		group_jobc = (PVP(g)->pvp_jobc != 0) ? VPROC_JOB_CONTROL
						     : VPROC_ORPHANED;

	/*
	 * To set a process into a group of which it is already a member
	 * is short-circuited to avoid fruitless process group manipulations.
	 * In addition, the permature removal of the pgrp vproc from its
	 * session list in the case when the process leader no longer exists
	 * but has a single member, would be erroneous.
	 *
	 * Shells frequently indulge in behavior of performing a series
	 * of setpgid(pid,pid) calls for a process (POSIX recommends
	 * doing it twice - once from the parent and once from the child,
	 * but shells have been spotted doing this four times per process).
	 * As this operation can cause a pgrp to be torn down
	 * (when resigning from the pgrp) and then set-up again, it would
	 * be nice to avoid the extra work where possible.
	 */
	if (g->vp_pid != pvp->pvp_pgid) {

		/*
		 * This operation may orphan the old pgrp or unorphan the new 
		 * pgrp or it may either orphan or unorphan the pgrps of 
		 * children of this process.
		 */
		if (g->vp_pid != pvp->pvp_pgid) {
			ASSERT(v->vp_pid != g->vp_pid || (pvp->pvp_flag & PV_PGRPLEADER) != 0 || pvp->pvp_jobc == 0);
			bvp_orphan_pgrp_check(v, g->vp_pid, pvp->pvp_sid);
		}

		/* resign from current group */
		if (pvp->pvp_pgid != 0)
			bvpop_resign_pgrp(v);

		pvp->pvp_pgid = g->vp_pid;	

		/* Create a pgrp if necessary */
		if (group_jobc == VPROC_NOT_PGRPLEADER) {
			ASSERT(v->vp_pid == g->vp_pid);
			if (pvp->pvp_sid) {
				/* add new pgrp to session list */
				vs = VPROCPTR(pvp->pvp_sid);
				PVP(g)->pvp_sessionl = PVP(vs)->pvp_sessionl;
				PVP(vs)->pvp_sessionl = g;
			}

			/*
			 * Add process to its own pgrp list
			 */
			ASSERT(pvp->pvp_head_pgrpl == NULL);
			pvp->pvp_head_pgrpl = v;
			pvp->pvp_pgrpl = NULL;
			VPROC_HOLD(v, "bvpop_setpgid(pgrp leader)");
			pvp->pvp_flag |=  PV_PGRPLEADER;
		} else {

			/*
			 * Add process to pgrp list
			 */
			pvp->pvp_pgrpl = PVP(g)->pvp_head_pgrpl;
			PVP(g)->pvp_head_pgrpl = v;
			VPROC_HOLD(g, "bvpop_setpgid(pgrp leader)");
		}

		/* keep physical proc updated */
		pproc_set_attr(pvp->pvp_pproc,
			       0,0,0,
			       &pvp->pvp_pgid,
			       &pvp->pvp_sid,
			       0);
	}
	
	return(ESUCCESS);
}


/*
 * NAME:	bvp_orphan_pgrp_check
 *
 * FUNCTION:	Re-assess elibility to job control due to change of pgrp.
 *
 * RETURNS:	None
 */
int
bvp_orphan_pgrp_check(
	struct vproc *v,	/* vproc of process to changing or exiting */
	pid_t new_pgid,		/* pgrp we are changing to */
	pid_t new_sid)		/* session we are changing to */
{
	register struct pvproc *pvp = PVP(v);
	register struct pvproc *pvp_pp = PVP(VPROCPTR(pvp->pvp_ppid));
	register struct vproc *vc;

	/*
	 * Subtract from current pgrp's jobc if we were contributing to this 
	 * pgrp being unorphaned
	 */
	if (pvp->pvp_pgid != 0
			&& pvp_pp->pvp_sid == pvp->pvp_sid
			&& pvp_pp->pvp_pgid != pvp->pvp_pgid)
		(void) bvp_adjust_job_control_count(VPROCPTR(pvp->pvp_pgid),
						    PV_SUBTRACT);
	/*
	 * Add to new pgrp's jobc if we are going to contribute to it
	 * being unorphaned.
	 */
	if (new_pgid != 0
			&& pvp_pp->pvp_sid == new_sid
			&& pvp_pp->pvp_pgid != new_pgid)
		(void) bvp_adjust_job_control_count(VPROCPTR(new_pgid),
						    PV_ADD);

	/*
	 * Re-assess the orphan status of the pgrp's of the children. This
	 * should be an infrequent operation in practice, since a parent
	 * seldom changes process groups or sessions.
	 */
	for (vc = pvp->pvp_head_childl; vc; vc = PVP(vc)->pvp_childl) {
		register struct pvproc *pvc = PVP(vc);
		if (pvc->pvp_pgid != 0) {
			boolean_t is_contributor_before =
				pvp->pvp_sid == pvc->pvp_sid 
					&& pvp->pvp_pgid != pvc->pvp_pgid;
			boolean_t is_contributor_after =
				new_sid == pvc->pvp_sid
					&& new_pgid != pvc->pvp_pgid;
			if (is_contributor_before && !is_contributor_after)
				(void) bvp_adjust_job_control_count(
						VPROCPTR(pvc->pvp_pgid),
						PV_SUBTRACT);
			else if (!is_contributor_before && is_contributor_after)
				(void) bvp_adjust_job_control_count(
						VPROCPTR(pvc->pvp_pgid), 
						PV_ADD);
		}
	}
}


/*
 * NAME: bvpop_get_attr
 *
 * FUNCTION: Get relationship attributes for process.
 */
int
bvpop_get_attr(
	struct vproc *v,	/* vproc of calling process */
	pid_t *ppid,		/* returned id of parent */
	pid_t *pgid,		/* returned process group id*/
	pid_t *sid,		/* returned session id */
	uid_t *uid,		/* returned user id */
	uid_t *ruid,		/* returned real user id */
	int *job_control,	/* returns if process subject to job control */
	boolean_t *has_sctty)	/* returns TRUE if has session tty control */
{
	struct pvproc *pvp;

	pvp = PVP(v);

	if (ppid)
		*ppid = pvp->pvp_ppid;
	if (pgid)
		*pgid = pvp->pvp_pgid;
	if (sid)
		*sid = pvp->pvp_sid;
	if(uid)
		*uid = pvp->pvp_pproc->p_rcred->cr_uid;
	if(ruid)
		*ruid = pvp->pvp_pproc->p_ruid;
	if (job_control) {
		if (pvp->pvp_flag&PV_PGRPLEADER)
			*job_control = pvp->pvp_jobc ? VPROC_JOB_CONTROL
						     : VPROC_ORPHANED;
		else
			*job_control = VPROC_NOT_PGRPLEADER;
	}

	if (has_sctty)
		*has_sctty = pvp->pvp_flag&PV_SCTTY;

	return(ESUCCESS);

}


/*
 * NAME: bvpop_setpinit
 *
 * FUNCTION: Converts process to a child of init.
 *
 */
int
bvpop_setpinit(
	struct vproc *v)	/* vproc in question */
{
	register struct vproc *pp;
	register struct vproc *ip;

	pp = VPROCPTR(PVP(v)->pvp_ppid);

	/* remove process from current parent */
	if (PVP(pp)->pvp_head_childl == v)
		PVP(pp)->pvp_head_childl = PVP(v)->pvp_childl;
	else {
		register struct vproc *w;
		for (w = PVP(pp)->pvp_head_childl;
		     PVP(w)->pvp_childl != v;
		     w = PVP(w)->pvp_childl)
			;
		PVP(w)->pvp_childl = PVP(v)->pvp_childl;
	}

	/* let init adopt it */
	ip = VPROCPTR(1);
	PVP(v)->pvp_ppid = 1;
	PVP(v)->pvp_foster_ppid = 1;
	pproc_set_attr(PVP(v)->pvp_pproc, 0, 0, &PVP(v)->pvp_ppid, 0, 0, 0);
	PVP(v)->pvp_childl = PVP(ip)->pvp_head_childl;
	PVP(ip)->pvp_head_childl = v;

	/* if proc is a zombie, signal INIT to cleanup */
	if (PVP(v)->pvp_flag & PV_SZOMB)
		(void) VPOP_SIGPROC(ip, SIGCHLD, 0, VPROC_HAS_PRIV);

	return(ESUCCESS);
}


/*
 * NAME: bvpop_ctty_getattr
 *
 * FUNCTION: Get (a pointer to) the tty struct of the controlling terminal
 *		for the caller's vproc.
 */
int
bvpop_ctty_getattr(
	struct vproc *v,		/* vproc of queried process */
	struct vproc *s,		/* vproc of session leader */
	pid_t *t_pgid,			/* returned foreground pgrp */
	dev_t *dev,			/* returned controlling device */
	node_t *node,			/* returned controlling node, ignored */
	struct tty **ttyp)		/* returned ptr to controlling tty */
{
	register int error;
	register struct vproc *vs = s;
	struct tty *tp;

	if (!(PVP(v)->pvp_flag&PV_SCTTY))
		return(ENOTTY);

	if (vs == NULL)
		if ((vs=VPROCPTR(PVP(v)->pvp_sid)) == NULL)
			return(ESRCH);

	tp = PVP(vs)->pvp_cttyp;
	if (ttyp)
		*ttyp = tp;
	if (dev)
		*dev = tp? tp->t_dev : 0;
	if (t_pgid)
		*t_pgid = tp? tp->t_pgid : 0;
	return(ESUCCESS);
}


/*
 * NAME: bvpop_set_ctty
 *
 * FUNCTION: Set (a pointer to) the tty struct of the controlling terminal
 *		for the caller's vproc.
 */
int
bvpop_set_ctty(
	struct vproc *v,	/* vproc of calling process */
	struct vproc *s,	/* vproc of session leader */
	struct tty *ttyp,	/* pointer to controlling tty (or NULL) */
	int flags)		/* set or get */
{
	boolean_t true = TRUE;
	boolean_t false = FALSE;

	if (flags&VPROC_SET) {
		/* permitted only for session leader */
		if (!(PVP(v)->pvp_flag&PV_SESSIONLEADER))
			return(EPERM);

		/* check for special case of releasing controlling tty */
		if (ttyp == NULL) {
			pproc_release_cttyv(PVP(v)->pvp_cttyp);
		} else {
			PVP(v)->pvp_flag |= PV_SCTTY;
			pproc_set_attr(PVP(v)->pvp_pproc,0,0,0,0,0,&true);
		}
		PVP(v)->pvp_cttyp = ttyp;
	}
	else if (flags&VPROC_CLEAR) {
		/* not permitted for session leader */
		if (PVP(v)->pvp_flag&PV_SESSIONLEADER)
			return(EPERM);

		PVP(v)->pvp_flag &= ~PV_SCTTY;
		pproc_set_attr(PVP(v)->pvp_pproc,0,0,0,0,0,&false);
	}
		
	return(ESUCCESS);
}


/*
 * NAME: bvpop_ptrace
 *
 * FUNCTION: Process tracing functions for a child process.
 */
bvpop_ptrace(
	struct vproc *vd,	/* vproc of calling (debugger) process */
	struct vproc *ve,	/* vproc of traced process */
	int req,		/* ptrace request */
	int *addr,		/* address within process */
	int data,		/* data to be written */
	int *retval)		/* returned data if read from address */
{
	register struct vproc *vp;

	/*
	 * Check that the traced process is indeed a child of the debugger.
	 * We don't do this check if the caller is requesting to be traced.
	 */
	if (req == VPROC_PT_TRACE_ME) {
		vp = vd;
	} else {
		vp = PVP(vd)->pvp_head_childl;
		for (; vp != ve; vp = PVP(vp)->pvp_childl)
			if (vp == NULL)
				return(ESRCH);
	}

	/*
	 * We translate the request code from "VPROC space"
	 * to "PROC space" - even though this is an identity operation.
	 */
	switch (req) {
	case VPROC_PT_TRACE_ME:	/* enable tracing for caller */
		req = PT_TRACE_ME; break;
	case VPROC_PT_READ_I:	/* read word in child's I space */
		req = PT_READ_I; break;
	case VPROC_PT_READ_D:	/* read word in child's D space */
		req = PT_READ_D; break;
	case VPROC_PT_READ_U:	/* read word in child's user structure */
		req = PT_READ_U; break;
	case VPROC_PT_WRITE_I:	/* write word in child's I space */
		req = PT_WRITE_I; break;
	case VPROC_PT_WRITE_D:	/* write word in child's D space */
		req = PT_WRITE_D; break;
	case VPROC_PT_WRITE_U:	/* write word in child's user structure */
		req = PT_WRITE_U; break;
	case VPROC_PT_CONTINUE:	/* continue the child */
		req = PT_CONTINUE; break;
	case VPROC_PT_KILL:	/* kill the child process */
		req = PT_KILL; break;
	case VPROC_PT_STEP:	/* single step the child */
		req = PT_STEP; break;
	default:
		return(EINVAL);
	}

	/*
	 * Call the physical level to perform the trace request.
	 */
	return(pproc_ptrace(PVP(vp)->pvp_pproc, req, addr, data, retval));
}


/*
 * NAME:	bvpop_get_task_port
 *
 * FUNCTION:	Get Mach task port corresponding to physical process
 *
 * RETURNS:	ESRCH, if vproc is not active
 *		EACCES, if calling process does not have permission
 *		ESUCCESS, if successful
 */
int
bvpop_get_task_port(
	struct vproc *v,	/* vproc of target process */
	mach_port_t *task_port)	/* task port for physical process */
{
	uid_t		euid;
	int		has_priv;
	struct pvproc	*pvp = PVP(v);

	if (pvp->pvp_flag & PV_SZOMB) {
		*task_port = MACH_PORT_NULL;
		return(ESRCH);
	}

	(void) pproc_get_attr(0,0,0,0,0,&has_priv,&euid,0,0,0);

	return(pproc_get_task_port(pvp->pvp_pproc, 
				   euid, 
				   has_priv, 
				   task_port));
}


/*
 * NAME: bvpop_terminal_sigpgrp
 *
 * FUNCTION: Sends terminal access control signal to process group.
 */
int
bvpop_terminal_sigpgrp(
	struct vproc *v,	/* process whose pgrp is to be signalled */
	int signo)		/* signal to be sent */
{ 
	boolean_t is_masked, is_ignored;

	/*
	 * This is a terminal access control signal.
	 * The vpop is directed at the caller's own vproc.
	 * We check the caller's masks before re-directing the operation
	 * to the caller's pgrp.
	 */
	pproc_chksigmask(PVP(v)->pvp_pproc, signo,
				&is_masked, &is_ignored);
	if (is_masked || is_ignored)
		return(ENOTTY);

	return(bvpop_sigpgrp(VPROCPTR(PVP(v)->pvp_pgid),
			     signo,
			     0,
			     NULL,
			     VPROC_HAS_PRIV | VPROC_CHECK_CTTY));
}


