/*
 * 
 * $Copyright
 * Copyright 1993, 1994 , 1995 Intel Corporation
 * INTEL CONFIDENTIAL
 * The technical data and computer software contained herein are subject
 * to the copyright notices; trademarks; and use and disclosure
 * restrictions identified in the file located in /etc/copyright on
 * this system.
 * Copyright$
 * 
 */
 
/*
 * @OSF_COPYRIGHT@
 */
/*
 * HISTORY
 * $Log: tcp_input.c,v $
 * Revision 1.6  1995/03/18  01:18:28  hobbes
 *  Reviewer: John Litvin
 *  Risk: Low
 *  Benefit or PTS #: 12435 -- part of tcp protocol stack fail to lock stats
 * 			    structure before updating
 *  Testing: TCP/IP EATS
 *  Module(s): tcp_input.c
 *
 * Revision 1.5  1994/11/18  20:35:34  mtm
 * Copyright additions/changes
 *
 * Revision 1.4  1993/10/08  01:23:12  cfj
 * Add forward references to tcp_xmit_timer() and use ANSI prototypes
 * in the function definition to satisfy the PGI 4.5 compilers.
 *
 * Revision 1.3  1993/05/06  20:27:03  brad
 * ad103+tnc merged with Intel code.
 *
 * Revision 1.1.1.1  1993/05/03  17:34:12  cfj
 * Initial 1.0.3 code drop
 *
 * Revision 1.2  1992/11/30  22:29:35  dleslie
 * Copy of NX branch back into main trunk
 *
 * Revision 1.1.2.1  1992/11/05  23:27:49  dleslie
 * Local changes for NX through noon, November 5, 1992.
 *
 * Revision 4.1  1992/11/04  00:21:49  cfj
 * Bump major revision number.
 *
 * Revision 2.3  1992/10/22  23:18:17  hobbes
 * Added the RFC_1323 extensions.
 *
 * Revision 2.2  1991/08/31  13:44:24  rabii
 * 	Initial V2.0 Checkin
 *
 * Revision 3.1  91/07/31  15:36:49  sp
 * Upgrade to 1.0.2
 * 
 * Revision 1.13.4.2  91/03/15  17:49:33  tmt
 * 	Add Ansi declaration for tcp_mss().
 * 	[91/03/13  19:12:13  tmt]
 * 
 * Revision 1.13  90/10/31  14:01:49  devrcs
 * 	Add missing semi-colon for NETSTAT_LOCK().
 * 	[90/10/23  20:01:14  brezak]
 * 
 * Revision 1.12  90/10/07  14:35:45  devrcs
 * 	Added EndLog Marker.
 * 	[90/09/28  11:16:19  gm]
 * 
 * Revision 1.11  90/09/13  11:48:15  devrcs
 * 	Fix maxseg buffering efficiency/roundoff problem.
 * 	Keep max_rcvd hiwater in header prediction case.
 * 	Fix uninitialized "inp" in tcp_input.
 * 	[90/08/22  16:21:28  tmt]
 * 
 * Revision 1.10  90/08/09  13:26:30  devrcs
 * 	Finish Reno update - REASS_MBUF, NTOHS.
 * 	[90/07/25  11:10:36  tmt]
 * 
 * 	Do last pcb cache with help from in_pcblookup.
 * 	Add Reno header prediction.
 * 	[90/07/24  14:14:08  tmt]
 * 
 * Revision 1.9  90/07/27  09:01:55  devrcs
 * 	Lock so_rcv in tcp_mss, as appropriate.
 * 	[90/07/23  11:22:46  tmt]
 * 
 * 	Update to BSD Reno release.
 * 	Remove or rearrange route locks - route.c handles.
 * 	[90/07/19  17:37:59  tmt]
 * 
 * Revision 1.8  90/06/22  20:39:22  devrcs
 * 	Parallelization repairs. Socket locks, then inpcb locks.
 * 	Take inpcbhead locks directly. Do refcounting right.
 * 	Handle urgent data properly if !tcp_compat_42.
 * 	[90/06/07  16:13:09  tmt]
 * 
 * Revision 1.7  90/04/27  19:19:04  devrcs
 * 	Change code to handle new _NO_BITFIELDS tcp header.
 * 	[90/04/20  13:02:38  tmt]
 * 
 * Revision 1.6  90/04/14  00:33:27  devrcs
 * 	Add void's, delete a function decl.
 * 	[90/04/09  16:34:19  tmt]
 * 
 * Revision 1.4  90/01/18  08:47:43  gm
 * 	Modify tcp_pulloutofband to take mbuf argument. Did a dtom() which was
 * 	broken by earlier performance mod.
 * 	[90/01/10  10:39:20  tmt]
 * 
 * 	Apply the idea from one of Van Jabobson's speedups to tcp frag
 * 	reassembly, i.e. do pullup only if queuing a fragment. It may
 * 	be worth fixing it completely in future (struct mbuf *next, *prev).
 * 	Rearrange locks slightly.
 * 	Do looser maxseg sanity checking, to allow tiny buffers if requested
 * 	but to find negative and (some) byteswapped bogons.
 * 	Do cluster rounding in tcp_mss only if route is to an interface that
 * 	does trailers, since that's why we might want it. Makes bigger segsize
 * 	on some interfaces (e.g. ethernet if MCLBYTES==1024).
 * 	[90/01/08  16:12:10  tmt]
 * 
 * 	OSF/1 "one" snapshot revision.
 * 	[90/01/02  12:00:00  tmt]
 * 
 * 	- Base is BSD 4.4 (Alpha) networking.
 * 	- Encore multiprocessing merged in with some structural
 * 	  modifications to support flexible configuration.
 * 	- Glue for compiling and running in MACH or Unix 4.4 environments,
 * 	  lock testing under Unix, thread or software interrupt netisr's,
 * 	  locking and/or spl synchronization, single or multiple CPUs.
 * 	[89/12/20  12:00:00  tmt]
 * 
 * Revision 1.3  90/01/03  12:41:56  gm
 * 	Fixes for first snapshot.
 * 	[90/01/03  09:39:11  gm]
 * 
 * Revision 1.2  89/12/26  10:14:58  gm
 * 	New networking code from BSD.
 * 	[89/12/16            tmt]
 * 
 * $EndLog$
 */
/*
 * Copyright (C) 1988,1989 Encore Computer Corporation.  All Rights Reserved
 *
 * Property of Encore Computer Corporation.
 * This software is made available solely pursuant to the terms of
 * a software license agreement which governs its use. Unauthorized
 * duplication, distribution or sale are strictly prohibited.
 *
 */
/*
 * Copyright (c) 1982, 1986, 1988, 1990 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted provided
 * that: (1) source distributions retain this entire copyright notice and
 * comment, and (2) distributions including binaries display the following
 * acknowledgement:  ``This product includes software developed by the
 * University of California, Berkeley and its contributors'' in the
 * documentation or other materials provided with the distribution and in
 * all advertising materials mentioning features or use of this software.
 * Neither the name of the University nor the names of its contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *	Base:	tcp_input.c	7.21 (Berkeley) 4/8/89
 *	Merged:	tcp_input.c	7.25 (Berkeley) 6/30/90
 */

#include "net/net_globals.h"
#include "machine/endian.h"

#include "sys/param.h"
#include "sys/time.h"
#include "sys/errno.h"

#include "sys/mbuf.h"
#include "sys/socket.h"
#include "sys/socketvar.h"
#include "sys/protosw.h"

#include "net/if.h"
#include "net/route.h"

#include "netinet/in.h"
#include "netinet/in_systm.h"
#include "netinet/ip.h"
#include "netinet/in_pcb.h"
#include "netinet/ip_var.h"
#include "netinet/tcp.h"
#include "netinet/tcpip.h"
#include "netinet/tcp_fsm.h"
#include "netinet/tcp_seq.h"
#include "netinet/tcp_timer.h"
#include "netinet/tcp_var.h"
#include "netinet/tcp_debug.h"

#include "net/net_malloc.h"

LOCK_ASSERTL_DECL

int	tcprexmtthresh = 3;

#if	RFC_1323
extern u_long sb_max;
#endif

/*
 *  Forward references
 */
#if __STDC__ == 1

#ifdef RFC_1323
void tcp_xmit_timer(struct tcpcb *, short);
#else
void tcp_xmit_timer(struct tcpcb *);
#endif /* RFC_1323 */

#else
void tcp_xmit_timer();
#endif /* __STDC__ */


/*
 * Insert segment ti into reassembly queue of tcp with
 * control block tp.  Return TH_FIN if reassembly now includes
 * a segment with FIN.  The macro form does the common case inline
 * (segment is the next to be received on an established connection,
 * and the queue is empty), avoiding linkage into and removal
 * from the queue and repetition of various conversions.
 * Set DELACK for segments received in order, but ack immediately
 * when segments are out of order (so fast retransmit can work).
 */
#define	TCP_REASS(tp, ti, m, so, flags) { \
	if ((ti)->ti_seq == (tp)->rcv_nxt && \
	    (tp)->seg_next == (struct tcpiphdr *)(tp) && \
	    (tp)->t_state == TCPS_ESTABLISHED) { \
		(tp)->t_flags |= TF_DELACK; \
		(tp)->rcv_nxt += (ti)->ti_len; \
		flags = (ti)->ti_flags & TH_FIN; \
		NETSTAT_LOCK(&tcpstat.tcps_lock); \
		tcpstat.tcps_rcvpack++;\
		tcpstat.tcps_rcvbyte += (ti)->ti_len;\
		NETSTAT_UNLOCK(&tcpstat.tcps_lock); \
		SOCKBUF_LOCK(&(so)->so_rcv); \
		sbappend(&(so)->so_rcv, (m)); \
		SOCKBUF_UNLOCK(&(so)->so_rcv); \
		sorwakeup(so); \
	} else { \
		(flags) = tcp_reass((tp), (ti), (m)); \
		tp->t_flags |= TF_ACKNOW; \
	} \
}

tcp_reass(tp, ti, m)
	register struct tcpcb *tp;
	register struct tcpiphdr *ti;
	register struct mbuf *m;
{
	register struct tcpiphdr *q;
	struct socket *so = tp->t_inpcb->inp_socket;
	int flags;

	/*
	 * Call with ti==0 after become established to
	 * force pre-ESTABLISHED data up to user socket.
	 */
	if (ti == 0)
		goto present;

	/*
	 * Find a segment which begins after this one does.
	 */
	for (q = tp->seg_next; q != (struct tcpiphdr *)tp;
	    q = (struct tcpiphdr *)q->ti_next)
		if (SEQ_GT(q->ti_seq, ti->ti_seq))
			break;

	/*
	 * If there is a preceding segment, it may provide some of
	 * our data already.  If so, drop the data from the incoming
	 * segment.  If it provides all of our data, drop us.
	 */
	if ((struct tcpiphdr *)q->ti_prev != (struct tcpiphdr *)tp) {
		register int i;
		q = (struct tcpiphdr *)q->ti_prev;
		/* conversion to int (in i) handles seq wraparound */
		i = q->ti_seq + q->ti_len - ti->ti_seq;
		if (i > 0) {
			if (i >= ti->ti_len) {
				NETSTAT_LOCK(&tcpstat.tcps_lock);
				tcpstat.tcps_rcvduppack++;
				tcpstat.tcps_rcvdupbyte += ti->ti_len;
				NETSTAT_UNLOCK(&tcpstat.tcps_lock);
				m_freem(m);
				return (0);
			}
			m_adj(m, i);
			ti->ti_len -= i;
			ti->ti_seq += i;
		}
		q = (struct tcpiphdr *)(q->ti_next);
	}
	NETSTAT_LOCK(&tcpstat.tcps_lock);
	tcpstat.tcps_rcvoopack++;
	tcpstat.tcps_rcvoobyte += ti->ti_len;
	NETSTAT_UNLOCK(&tcpstat.tcps_lock);

	REASS_MBUF(ti) = m;		/* XXX */

	/*
	 * While we overlap succeeding segments trim them or,
	 * if they are completely covered, dequeue them.
	 */
	while (q != (struct tcpiphdr *)tp) {
		register int i = (ti->ti_seq + ti->ti_len) - q->ti_seq;
		if (i <= 0)
			break;
		if (i < q->ti_len) {
			q->ti_seq += i;
			q->ti_len -= i;
			m_adj(REASS_MBUF(q), i);
			break;
		}
		q = (struct tcpiphdr *)q->ti_next;
		m = REASS_MBUF((struct tcpiphdr *)q->ti_prev);
		remque(q->ti_prev);
		m_freem(m);
	}

	/*
	 * Stick new segment in its place.
	 */
	insque(ti, q->ti_prev);

present:
	/*
	 * Present data to user, advancing rcv_nxt through
	 * completed sequence space.
	 */
	if (TCPS_HAVERCVDSYN(tp->t_state) == 0)
		return (0);
	ti = tp->seg_next;
	if (ti == (struct tcpiphdr *)tp || ti->ti_seq != tp->rcv_nxt)
		return (0);
	if (tp->t_state == TCPS_SYN_RECEIVED && ti->ti_len)
		return (0);
	do {
		tp->rcv_nxt += ti->ti_len;
		flags = ti->ti_flags & TH_FIN;
		remque(ti);
		m = REASS_MBUF(ti);
		ti = (struct tcpiphdr *)ti->ti_next;
		if (so->so_state & SS_CANTRCVMORE)
			m_freem(m);
		else {
			SOCKBUF_LOCK(&so->so_rcv);
			sbappend(&so->so_rcv, m);
			SOCKBUF_UNLOCK(&so->so_rcv);
		}
	} while (ti != (struct tcpiphdr *)tp && ti->ti_seq == tp->rcv_nxt);
	sorwakeup(so);
	return (flags);
}

/*
 * TCP input routine, follows pages 65-76 of the
 * protocol specification dated September, 1981 very closely.
 */
void
tcp_input(m, iphlen)
	register struct mbuf *m;
	int iphlen;
{
	register struct tcpiphdr *ti;
	register struct inpcb *inp = 0;
	struct mbuf *om = 0;
	int len, tlen, off;
	register struct tcpcb *tp = 0;
	register int tiflags;
	struct socket *so = 0;
	int todrop, acked, ourfinisacked, needoutput = 0;
	short ostate;
	struct in_addr laddr;
	struct tcpiphdr tcp_saveti;
	int dropsocket = 0;
	int iss = 0;
#if	RFC_1323
	u_long tiwin, ts_val, ts_ecr;
	int ts_present = 0;
#endif

	NETSTAT_LOCK(&tcpstat.tcps_lock);
	tcpstat.tcps_rcvtotal++;
	NETSTAT_UNLOCK(&tcpstat.tcps_lock);
	/*
	 * Get IP and TCP header together in first mbuf.
	 * Note: IP leaves IP header in first mbuf.
	 */
	ti = mtod(m, struct tcpiphdr *);
	if (iphlen > sizeof (struct ip))
		ip_stripoptions(m, (struct mbuf *)0);
	if (m->m_len < sizeof (struct tcpiphdr)) {
		if ((m = m_pullup(m, sizeof (struct tcpiphdr))) == 0) {
			NETSTAT_LOCK(&tcpstat.tcps_lock);
			tcpstat.tcps_rcvshort++;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
			return;
		}
		ti = mtod(m, struct tcpiphdr *);
	}

	/*
	 * Checksum extended TCP header and data.
	 */
	tlen = ((struct ip *)ti)->ip_len;
	len = sizeof (struct ip) + tlen;
	ti->ti_next = ti->ti_prev = 0;
	ti->ti_x1 = 0;
	ti->ti_len = (u_short)tlen;
	ti->ti_len = htons((u_short)ti->ti_len);
	if (ti->ti_sum = in_cksum(m, len)) {
#if	INETPRINTFS
		if (inetprintfs)
			printf("tcp sum: src 0x%x\n", ntohl(ti->ti_src));
#endif
		NETSTAT_LOCK(&tcpstat.tcps_lock);
		tcpstat.tcps_rcvbadsum++;
		NETSTAT_UNLOCK(&tcpstat.tcps_lock);
		goto drop;
	}

	/*
	 * Check that TCP offset makes sense,
	 * pull out TCP options and adjust length.		XXX
	 */
	off = (ti->ti_xoff & 0xf0) >> 2;
	if (off < sizeof (struct tcphdr) || off > tlen) {
#if	INETPRINTFS
		if (inetprintfs)
			printf("tcp off: src 0x%x off %d\n",
							ntohl(ti->ti_src), off);
#endif
		NETSTAT_LOCK(&tcpstat.tcps_lock);
		tcpstat.tcps_rcvbadoff++;
		NETSTAT_UNLOCK(&tcpstat.tcps_lock);
		goto drop;
	}
	tlen -= off;
	ti->ti_len = tlen;
	if (off > sizeof (struct tcphdr)) {
#if	RFC_1323
 		u_long *opl;		/* u_long pointer into options. */
#endif
		if (m->m_len < sizeof(struct ip) + off) {
			if ((m = m_pullup(m, sizeof (struct ip) + off)) == 0) {
				NETSTAT_LOCK(&tcpstat.tcps_lock);
				tcpstat.tcps_rcvshort++;
				NETSTAT_UNLOCK(&tcpstat.tcps_lock);
				return;
			}
			ti = mtod(m, struct tcpiphdr *);
		}
#if	RFC_1323
		/* Do quick retrieval of time-stamp options. ("options
		 * prediction?")  If time-stamp is only option and it's
		 * formatted as recommended in RFC 1323, we quickly get the
		 * values now and not bother allocating an mbuf, calling
		 * tcp_dooptions(), etc.
		 */
		if ( off == sizeof(struct tcphdr)+12 &&
			!(ti->ti_flags & TH_SYN) && *(opl=mtod(m, u_long *)
			+ sizeof(struct tcpiphdr)/sizeof(u_long) ) ==
			htonl(  TCPOPT_NOP<<24 | TCPOPT_NOP<<16 |
				TCPOPT_TIMESTAMP<<8 | 10 ) ) {
			
			ts_present = 1;
			ts_val = ntohl( opl[1] );
			ts_ecr = ntohl( opl[2] );
		}
		else {
			/* get options, but they'll be removed LATER */
			om = m_get(M_DONTWAIT, MT_DATA);
			if (om == 0)
				goto drop;
			om->m_len = off - sizeof (struct tcphdr);
			{ caddr_t op = mtod(m, caddr_t) +
				sizeof (struct tcpiphdr);
			  bcopy(op, mtod(om, caddr_t), (unsigned)om->m_len);
			}
		}
#else
		om = m_get(M_DONTWAIT, MT_DATA);
		if (om == 0)
			goto drop;
		om->m_len = off - sizeof (struct tcphdr);
		{ caddr_t op = mtod(m, caddr_t) + sizeof (struct tcpiphdr);
		  bcopy(op, mtod(om, caddr_t), (unsigned)om->m_len);
		  m->m_len -= om->m_len;
		  m->m_pkthdr.len -= om->m_len;
		  bcopy(op+om->m_len, op,
		   (unsigned)(m->m_len-sizeof (struct tcpiphdr)));
		}
#endif
	}
	tiflags = ti->ti_flags;

	/*
	 * Convert TCP protocol specific fields to host format.
	 */
	NTOHL(ti->ti_seq);
	NTOHL(ti->ti_ack);
	NTOHS(ti->ti_win);
	NTOHS(ti->ti_urp);

	/*
	 * Locate pcb for segment.
	 *
	 * BSD "Reno" caches the previous pcb looked up here. This is
	 * problematic with parallelization and thread safety, due to
	 * the synchronization required, and taking/releasing refcounts,
	 * so we push it down to in_pcblookup (where it belongs?).
	 */
findpcb:
	inp = in_pcblookup
		(&tcb, ti->ti_src, ti->ti_sport, ti->ti_dst, ti->ti_dport,
		INPLOOKUP_WILDCARD|INPLOOKUP_USECACHE);

	/*
	 * If the state is CLOSED (i.e., TCB does not exist) then
	 * all data in the incoming segment is discarded.
	 * If the TCB exists but is in CLOSED state, it is embryonic,
	 * but should either do a listen or a connect soon.
	 */
	if (inp == 0)
		goto dropwithreset;
	so = inp->inp_socket;		/* Follow lock order here! */
	SOCKET_LOCK(so);
	INPCB_LOCK(inp);
	tp = intotcpcb(inp);
	if (tp == 0) {
		INPCB_UNLOCK(inp);
		goto dropwithreset;
	}
	if (tp->t_state == TCPS_CLOSED)
		goto drop;

#if RFC_1323
	/* Unscale the window into a 32-bit value.
	 */
	if ((tiflags & TH_SYN) == 0)
		tiwin = ti->ti_win << tp->snd_scale;
	else
		tiwin = ti->ti_win;
#endif

	if (so->so_options & (SO_DEBUG|SO_ACCEPTCONN)) {
		if (so->so_options & SO_DEBUG) {
			ostate = tp->t_state;
			tcp_saveti = *ti;
		}
		if (so->so_options & SO_ACCEPTCONN) {
			INPCB_UNLOCK(inp);	/* Unlock for new create */
			INPCBRC_UNREF(inp);
			tp = 0; inp = 0;
			so = sonewconn(so, 0);
			if (so == 0)
				goto drop;
			/*
			 * This is ugly, but ....
			 *
			 * Mark socket as temporary until we're
			 * committed to keeping it.  The code at
			 * ``drop'' and ``dropwithreset'' check the
			 * flag dropsocket to see if the temporary
			 * socket created here should be discarded.
			 * We mark the socket as discardable until
			 * we're committed to it below in TCPS_LISTEN.
			 */
			dropsocket++;
			inp = (struct inpcb *)so->so_pcb;
			INPCB_LOCK(inp);
			INHEAD_WRITE_LOCK(&tcb);
			INPCBRC_REF(inp);	/* re-reference (total 2) */
			inp->inp_laddr = ti->ti_dst;
			inp->inp_lport = ti->ti_dport;
			INHEAD_WRITE_UNLOCK(&tcb);
			inp->inp_options = ip_srcroute();
			tp = intotcpcb(inp);
			tp->t_state = TCPS_LISTEN;

#if RFC_1323
			/* Compute proper scaling value from buffer space
			 */
			while ( tp->request_r_scale < TCP_MAX_WINSHIFT &&
			   TCP_MAXWIN<<tp->request_r_scale<so->so_rcv.sb_hiwat)
				tp->request_r_scale++;
#endif
		}
	}

	/*
	 * Segment received on connection.
	 * Reset idle time and keep-alive timer.
	 */
	tp->t_idle = 0;
	tp->t_timer[TCPT_KEEP] = tcp_keepidle;

	/*
	 * Process options if not in LISTEN state,
	 * else do it below (after getting remote address).
	 */
	if (om && tp->t_state != TCPS_LISTEN) {
#if	RFC_1323
		tcp_dooptions(tp, om, ti, &ts_present, &ts_val, &ts_ecr );
#else
		tcp_dooptions(tp, om, ti);
#endif
		om = 0;
	}

	/* 
	 * Header prediction: check for the two common cases
	 * of a uni-directional data xfer.  If the packet has
	 * no control flags, is in-sequence, the window didn't
	 * change and we're not retransmitting, it's a
	 * candidate.  If the length is zero and the ack moved
	 * forward, we're the sender side of the xfer.  Just
	 * free the data acked & wake any higher level process
	 * that was blocked waiting for space.  If the length
	 * is non-zero and the ack didn't move, we're the
	 * receiver side.  If we're getting packets in-order
	 * (the reassembly queue is empty), add the data to
	 * the socket buffer and note that we need a delayed ack.
	 */
#if	RFC_1323
	if (tp->t_state == TCPS_ESTABLISHED &&
	    (tiflags & (TH_SYN|TH_FIN|TH_RST|TH_URG|TH_ACK)) == TH_ACK &&
	    ( ! ts_present || (int) ( ts_val - tp->ts_recent ) >= 0 ) &&
	    ti->ti_seq == tp->rcv_nxt &&
	    tiwin && tiwin == tp->snd_wnd &&
	    tp->snd_nxt == tp->snd_max) {

		/* If last ACK falls within this segments sequence number,
		 *  record the time-stamp.
		 */
		if (  ts_present && SEQ_LEQ( ti->ti_seq, tp->last_ack_sent ) &&
		   SEQ_LT( tp->last_ack_sent, ti->ti_seq+ti->ti_len) ) {
			tp->ts_recent_age = 0;
			tp->ts_recent = ts_val;
		}
#else
	if (tp->t_state == TCPS_ESTABLISHED &&
	    (tiflags & (TH_SYN|TH_FIN|TH_RST|TH_URG|TH_ACK)) == TH_ACK &&
	    ti->ti_seq == tp->rcv_nxt &&
	    ti->ti_win && ti->ti_win == tp->snd_wnd &&
	    tp->snd_nxt == tp->snd_max) {
#endif

		if (ti->ti_len == 0) {
			if (SEQ_GT(ti->ti_ack, tp->snd_una) &&
			    SEQ_LEQ(ti->ti_ack, tp->snd_max) &&
			    tp->snd_cwnd >= tp->snd_wnd) {
				/*
				 * this is a pure ack for outstanding data.
				 */
				/*++tcppredack;*/
#if	RFC_1323
				if (ts_present)
					tcp_xmit_timer(tp, tcp_now-ts_ecr+1);
				else if (tp->t_rtt &&
					SEQ_GT(ti->ti_ack,tp->t_rtseq))
					tcp_xmit_timer(tp, tp->t_rtt);
#else
				if (tp->t_rtt && SEQ_GT(ti->ti_ack,tp->t_rtseq))
					tcp_xmit_timer(tp);
#endif
				acked = ti->ti_ack - tp->snd_una;
				NETSTAT_LOCK(&tcpstat.tcps_lock);
				tcpstat.tcps_rcvackpack++;
				tcpstat.tcps_rcvackbyte += acked;
				NETSTAT_UNLOCK(&tcpstat.tcps_lock);
				tp->snd_una = ti->ti_ack;
				m_freem(m);

				/*
				 * If all outstanding data are acked, stop
				 * retransmit timer, otherwise restart timer
				 * using current (possibly backed-off) value.
				 * If process is waiting for space,
				 * wakeup/selwakeup/signal.  If data
				 * are ready to send, let tcp_output
				 * decide between more output or persist.
				 */
				if (tp->snd_una == tp->snd_max)
					tp->t_timer[TCPT_REXMT] = 0;
				else if (tp->t_timer[TCPT_PERSIST] == 0)
					tp->t_timer[TCPT_REXMT] = tp->t_rxtcur;

				SOCKBUF_LOCK(&so->so_snd);
				sbdrop(&so->so_snd, acked);
				needoutput = (so->so_snd.sb_cc != 0);
				if (so->so_snd.sb_flags & SB_NOTIFY) {
					SOCKBUF_UNLOCK(&so->so_snd);
					sowwakeup(so);
				} else
					SOCKBUF_UNLOCK(&so->so_snd);
				goto done;
			}
		} else if (ti->ti_ack == tp->snd_una &&
		    tp->seg_next == (struct tcpiphdr *)tp &&
		    ti->ti_len <= sbspace(&so->so_rcv)) { /* XXX lock it */
			/*
			 * this is a pure, in-sequence data packet
			 * with nothing on the reassembly queue and
			 * we have enough buffer space to take it.
			 */
			/*++tcppreddat;*/
			tp->rcv_nxt += ti->ti_len;
			NETSTAT_LOCK(&tcpstat.tcps_lock);
			tcpstat.tcps_rcvpack++;
			tcpstat.tcps_rcvbyte += ti->ti_len;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
#if RFC_1323
			/*
			 * Drop TCP, IP headers and TCP options.  Then add data
			 * to socket buffer
			 */
			m->m_data +=
			   sizeof(struct tcpiphdr)+off-sizeof(struct tcphdr);
			m->m_len -=
			   sizeof(struct tcpiphdr)+off-sizeof(struct tcphdr);
			m->m_pkthdr.len -=
			   sizeof(struct tcpiphdr)+off-sizeof(struct tcphdr);
#else
			/*
			 * Drop TCP and IP headers then add data
			 * to socket buffer
			 */
			m->m_data += sizeof(struct tcpiphdr);
			m->m_len -= sizeof(struct tcpiphdr);
			m->m_pkthdr.len -= sizeof(struct tcpiphdr);
#endif
			SOCKBUF_LOCK(&so->so_rcv);
			sbappend(&so->so_rcv, m);
			SOCKBUF_UNLOCK(&so->so_rcv);
			sorwakeup(so);
			tp->t_flags |= TF_DELACK;
			len = so->so_rcv.sb_hiwat - (tp->rcv_adv - tp->rcv_nxt);
			if (len > tp->max_rcvd)
				tp->max_rcvd = len;
			goto done;
		}
	}

#if	RFC_1323
	/*
	 * Drop TCP, IP headers and TCP options.
	 */
	m->m_data += sizeof(struct tcpiphdr)+off-sizeof(struct tcphdr);
	m->m_len -= sizeof(struct tcpiphdr)+off-sizeof(struct tcphdr);
	m->m_pkthdr.len -= sizeof(struct tcpiphdr)+off-sizeof(struct tcphdr);
#else
	/*
	 * Drop TCP and IP headers; TCP options were dropped above.
	 */
	m->m_data += sizeof(struct tcpiphdr);
	m->m_len -= sizeof(struct tcpiphdr);
	m->m_pkthdr.len -= sizeof(struct tcpiphdr);
#endif

	/*
	 * Calculate amount of space in receive window,
	 * and then do TCP input processing.
	 * Receive window is amount of space in rcv queue,
	 * but not less than advertised window.
	 */
	{ int win;

	SOCKBUF_LOCK(&so->so_rcv);
	win = sbspace(&so->so_rcv);
	SOCKBUF_UNLOCK(&so->so_rcv);
	if (win < 0)
		win = 0;
	tp->rcv_wnd = max(win, (int)(tp->rcv_adv - tp->rcv_nxt));
	}

	switch (tp->t_state) {

	/*
	 * If the state is LISTEN then ignore segment if it contains an RST.
	 * If the segment contains an ACK then it is bad and send a RST.
	 * If it does not contain a SYN then it is not interesting; drop it.
	 * Don't bother responding if the destination was a broadcast.
	 * Otherwise initialize tp->rcv_nxt, and tp->irs, select an initial
	 * tp->iss, and send a segment:
	 *     <SEQ=ISS><ACK=RCV_NXT><CTL=SYN,ACK>
	 * Also initialize tp->snd_nxt to tp->iss+1 and tp->snd_una to tp->iss.
	 * Fill in remote peer address fields if not previously specified.
	 * Enter SYN_RECEIVED state, and process any other fields of this
	 * segment in this state.
	 */
	case TCPS_LISTEN: {
		struct mbuf *am;
		register struct sockaddr_in *sin;

		if (tiflags & TH_RST)
			goto drop;
		if (tiflags & TH_ACK)
			goto dropwithreset;
		if ((tiflags & TH_SYN) == 0)
			goto drop;
		if (m_broadcast(m))
			goto drop;
		am = m_get(M_DONTWAIT, MT_SONAME);	/* XXX */
		if (am == NULL)
			goto drop;
		am->m_len = sizeof (struct sockaddr_in);
		sin = mtod(am, struct sockaddr_in *);
		sin->sin_family = AF_INET;
		sin->sin_len = sizeof(*sin);
		sin->sin_addr = ti->ti_src;
		sin->sin_port = ti->ti_sport;
		INHEAD_WRITE_LOCK(&tcb);
		laddr = inp->inp_laddr;
		if (inp->inp_laddr.s_addr == INADDR_ANY)
			inp->inp_laddr = ti->ti_dst;
		if (in_pcbconnect(inp, am)) {
			inp->inp_laddr = laddr;
			INHEAD_WRITE_UNLOCK(&tcb);
			(void) m_free(am);
			goto drop;
		}
		INHEAD_WRITE_UNLOCK(&tcb);
		(void) m_free(am);
		dropsocket = 0;		/* committed to socket */
		(void) tcp_template(tp);
		if (om) {
#if RFC_1323
			tcp_dooptions(tp, om, ti, &ts_present, &ts_val,
				&ts_ecr );
#else
			tcp_dooptions(tp, om, ti);
#endif
			om = 0;
		}
		TCPMISC_LOCK();
		if (iss)
			tp->iss = iss;
		else
			tp->iss = tcp_iss;
		tcp_iss += TCP_ISSINCR/2;
		TCPMISC_UNLOCK();
		tp->irs = ti->ti_seq;
		tcp_sendseqinit(tp);
		tcp_rcvseqinit(tp);
		tp->t_flags |= TF_ACKNOW;
		tp->t_state = TCPS_SYN_RECEIVED;
		tp->t_timer[TCPT_KEEP] = TCPTV_KEEP_INIT;
		NETSTAT_LOCK(&tcpstat.tcps_lock);
		tcpstat.tcps_accepts++;
		NETSTAT_UNLOCK(&tcpstat.tcps_lock);
		goto trimthenstep6;
		}

	/*
	 * If the state is SYN_SENT:
	 *	if seg contains an ACK, but not for our SYN, drop the input.
	 *	if seg contains a RST, then drop the connection.
	 *	if seg does not contain SYN, then drop it.
	 * Otherwise this is an acceptable SYN segment
	 *	initialize tp->rcv_nxt and tp->irs
	 *	if seg contains ack then advance tp->snd_una
	 *	if SYN has been acked change to ESTABLISHED else SYN_RCVD state
	 *	arrange for segment to be acked (eventually)
	 *	continue processing rest of data/controls, beginning with URG
	 */
	case TCPS_SYN_SENT:
		if ((tiflags & TH_ACK) &&
		    (SEQ_LEQ(ti->ti_ack, tp->iss) ||
		     SEQ_GT(ti->ti_ack, tp->snd_max)))
			goto dropwithreset;
		if (tiflags & TH_RST) {
			if (tiflags & TH_ACK)
				tp = tcp_drop(tp, ECONNREFUSED);
			goto drop;
		}
		if ((tiflags & TH_SYN) == 0)
			goto drop;
		if (tiflags & TH_ACK) {
			tp->snd_una = ti->ti_ack;
			if (SEQ_LT(tp->snd_nxt, tp->snd_una))
				tp->snd_nxt = tp->snd_una;
		}
		tp->t_timer[TCPT_REXMT] = 0;
		tp->irs = ti->ti_seq;
		tcp_rcvseqinit(tp);
		tp->t_flags |= TF_ACKNOW;
		if (tiflags & TH_ACK && SEQ_GT(tp->snd_una, tp->iss)) {
			NETSTAT_LOCK(&tcpstat.tcps_lock);
			tcpstat.tcps_connects++;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
			soisconnected(so);
			tp->t_state = TCPS_ESTABLISHED;
#if	RFC_1323
			/* Do window scaling on this connection? */
			if ( (tp->t_flags & (TF_RCVD_SCALE|TF_REQ_SCALE)) ==
				(TF_RCVD_SCALE|TF_REQ_SCALE) ) {
				tp->snd_scale = tp->requested_s_scale;
				tp->rcv_scale = tp->request_r_scale;
			}
#endif
			(void) tcp_reass(tp, (struct tcpiphdr *)0,
				(struct mbuf *)0);
			/*
			 * if we didn't have to retransmit the SYN,
			 * use its rtt as our initial srtt & rtt var.
			 */
			if (tp->t_rtt)
#if	RFC_1323
				tcp_xmit_timer( tp, tp->t_rtt );
#else
				tcp_xmit_timer(tp);
#endif
		} else
			tp->t_state = TCPS_SYN_RECEIVED;

trimthenstep6:
		/*
		 * Advance ti->ti_seq to correspond to first data byte.
		 * If data, trim to stay within window,
		 * dropping FIN if necessary.
		 */
		ti->ti_seq++;
		if (ti->ti_len > tp->rcv_wnd) {
			todrop = ti->ti_len - tp->rcv_wnd;
			m_adj(m, -todrop);
			ti->ti_len = tp->rcv_wnd;
			tiflags &= ~TH_FIN;
			NETSTAT_LOCK(&tcpstat.tcps_lock);
			tcpstat.tcps_rcvpackafterwin++;
			tcpstat.tcps_rcvbyteafterwin += todrop;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
		}
		tp->snd_wl1 = ti->ti_seq - 1;
		tp->rcv_up = ti->ti_seq;
		goto step6;
	}

	/*
	 * States other than LISTEN or SYN_SENT.
	 * First check time-stamp, if present. Only done if RFC_1323 is TRUE.
	 * Then check that at least some bytes of segment are within 
	 * receive window.  If segment begins before rcv_nxt,
	 * drop leading data (and SYN); if nothing left, just ack.
	 */

#if RFC_1323
	/* PAWS:  If we have an time-stamp reply on this segment and it's
	 * less than ts_recent, drop it.
	 */
	if ( ts_present && ! (tiflags & TH_RST) &&
	    (int)( ts_val - tp->ts_recent ) < 0 ) {

		/* Check to see if ts_recent is over 24 days old.
		 */
		if ( tp->ts_recent_age > (24 * 24 * 60 * 60 * PR_SLOWHZ) )

			/* Invalidate ts_recent.  If this segment updates
			 * ts_recent, the age will be reset later and ts_recent
			 * will get a valid value.  If it does not, setting
			 * ts_recent to zero will at least satisfy the
			 * requirement that zero be placed in the time-stamp
			 * echo reply when ts_recent isn't valid.  The
			 * age isn't reset until we get a valid ts_recent
			 * because we don't want out-of-order segments to be
			 * dropped when ts_recent is old.
			 */
			tp->ts_recent = 0;
		else {
			NETSTAT_LOCK(&tcpstat.tcps_lock);
			tcpstat.tcps_rcvduppack++;
			tcpstat.tcps_rcvdupbyte += ti->ti_len;
			tcpstat.tcps_pawsdrop++;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
			goto dropafterack;
		}
	}
#endif /* RFC_1323 */

	todrop = tp->rcv_nxt - ti->ti_seq;
	if (todrop > 0) {
		if (tiflags & TH_SYN) {
			tiflags &= ~TH_SYN;
			ti->ti_seq++;
			if (ti->ti_urp > 1) 
				ti->ti_urp--;
			else
				tiflags &= ~TH_URG;
			todrop--;
		}
		if (todrop > ti->ti_len ||
		    todrop == ti->ti_len && (tiflags&TH_FIN) == 0) {
			NETSTAT_LOCK(&tcpstat.tcps_lock);
			tcpstat.tcps_rcvduppack++;
			tcpstat.tcps_rcvdupbyte += ti->ti_len;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
			/*
			 * If segment is just one to the left of the window,
			 * check two special cases:
			 * 1. Don't toss RST in response to 4.2-style keepalive.
			 * 2. If the only thing to drop is a FIN, we can drop
			 *    it, but check the ACK or we will get into FIN
			 *    wars if our FINs crossed (both CLOSING).
			 * In either case, send ACK to resynchronize,
			 * but keep on processing for RST or ACK.
			 */
			if ((tiflags & TH_FIN && todrop == ti->ti_len + 1) ||
			    (tcp_compat_42 &&
			     (tiflags&TH_RST && ti->ti_seq == tp->rcv_nxt-1))) {
				todrop = ti->ti_len;
				tiflags &= ~TH_FIN;
				tp->t_flags |= TF_ACKNOW;
			} else
				goto dropafterack;
		} else {
			NETSTAT_LOCK(&tcpstat.tcps_lock);
			tcpstat.tcps_rcvpartduppack++;
			tcpstat.tcps_rcvpartdupbyte += todrop;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
		}
		m_adj(m, todrop);
		ti->ti_seq += todrop;
		ti->ti_len -= todrop;
		if (ti->ti_urp > todrop)
			ti->ti_urp -= todrop;
		else {
			tiflags &= ~TH_URG;
			ti->ti_urp = 0;
		}
	}

	/*
	 * If new data are received on a connection after the
	 * user processes are gone, then RST the other end.
	 */
	if ((so->so_state & SS_NOFDREF) &&
	    tp->t_state > TCPS_CLOSE_WAIT && ti->ti_len) {
		tp = tcp_close(tp);
		NETSTAT_LOCK(&tcpstat.tcps_lock);
		tcpstat.tcps_rcvafterclose++;
		NETSTAT_UNLOCK(&tcpstat.tcps_lock);
		goto dropwithreset;
	}

	/*
	 * If segment ends after window, drop trailing data
	 * (and PUSH and FIN); if nothing left, just ACK.
	 */
	todrop = (ti->ti_seq+ti->ti_len) - (tp->rcv_nxt+tp->rcv_wnd);
	if (todrop > 0) {
		NETSTAT_LOCK(&tcpstat.tcps_lock);
		tcpstat.tcps_rcvpackafterwin++;
		if (todrop >= ti->ti_len) {
			tcpstat.tcps_rcvbyteafterwin += ti->ti_len;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
			/*
			 * If a new connection request is received
			 * while in TIME_WAIT, drop the old connection
			 * and start over if the sequence numbers
			 * are above the previous ones.
			 */
			if (tiflags & TH_SYN &&
			    tp->t_state == TCPS_TIME_WAIT &&
			    SEQ_GT(ti->ti_seq, tp->rcv_nxt)) {
				iss = tp->rcv_nxt + TCP_ISSINCR;
				tp = tcp_close(tp);
				INPCBRC_UNREF(inp);
				SOCKET_UNLOCK(so);
				so = 0;
				goto findpcb;
			}
			/*
			 * If window is closed can only take segments at
			 * window edge, and have to drop data and PUSH from
			 * incoming segments.  Continue processing, but
			 * remember to ack.  Otherwise, drop segment
			 * and ack.
			 */
			if (tp->rcv_wnd == 0 && ti->ti_seq == tp->rcv_nxt) {
				tp->t_flags |= TF_ACKNOW;
				NETSTAT_LOCK(&tcpstat.tcps_lock);
				tcpstat.tcps_rcvwinprobe++;
				NETSTAT_UNLOCK(&tcpstat.tcps_lock);
			} else
				goto dropafterack;
		} else {
			tcpstat.tcps_rcvbyteafterwin += todrop;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
		}
		m_adj(m, -todrop);
		ti->ti_len -= todrop;
		tiflags &= ~(TH_PUSH|TH_FIN);
	}

#if RFC_1323
	/* If last ACK falls within this segments sequence numbers,
	 * record its time-stamp.
	 */
	if ( ts_val &&  SEQ_LEQ( ti->ti_seq, tp->last_ack_sent ) &&
	   SEQ_LT( tp->last_ack_sent, ti->ti_seq+ti->ti_len) ) {
		tp->ts_recent_age = 0;
		tp->ts_recent = ts_val;
	}
#endif

	/*
	 * If the RST bit is set examine the state:
	 *    SYN_RECEIVED STATE:
	 *	If passive open, return to LISTEN state.
	 *	If active open, inform user that connection was refused.
	 *    ESTABLISHED, FIN_WAIT_1, FIN_WAIT2, CLOSE_WAIT STATES:
	 *	Inform user that connection was reset, and close tcb.
	 *    CLOSING, LAST_ACK, TIME_WAIT STATES
	 *	Close the tcb.
	 */
	if (tiflags&TH_RST) switch (tp->t_state) {

	case TCPS_SYN_RECEIVED:
		so->so_error = ECONNREFUSED;
		goto close;

	case TCPS_ESTABLISHED:
	case TCPS_FIN_WAIT_1:
	case TCPS_FIN_WAIT_2:
	case TCPS_CLOSE_WAIT:
		so->so_error = ECONNRESET;
	close:
		tp->t_state = TCPS_CLOSED;
		NETSTAT_LOCK(&tcpstat.tcps_lock);
		tcpstat.tcps_drops++;
		NETSTAT_UNLOCK(&tcpstat.tcps_lock);
		tp = tcp_close(tp);
		goto drop;

	case TCPS_CLOSING:
	case TCPS_LAST_ACK:
	case TCPS_TIME_WAIT:
		tp = tcp_close(tp);
		goto drop;
	}

	/*
	 * If a SYN is in the window, then this is an
	 * error and we send an RST and drop the connection.
	 */
	if (tiflags & TH_SYN) {
		tp = tcp_drop(tp, ECONNRESET);
		goto dropwithreset;
	}

	/*
	 * If the ACK bit is off we drop the segment and return.
	 */
	if ((tiflags & TH_ACK) == 0)
		goto drop;
	
	/*
	 * Ack processing.
	 */
	switch (tp->t_state) {

	/*
	 * In SYN_RECEIVED state if the ack ACKs our SYN then enter
	 * ESTABLISHED state and continue processing, otherwise
	 * send an RST.
	 */
	case TCPS_SYN_RECEIVED:
		if (SEQ_GT(tp->snd_una, ti->ti_ack) ||
		    SEQ_GT(ti->ti_ack, tp->snd_max))
			goto dropwithreset;
		NETSTAT_LOCK(&tcpstat.tcps_lock);
		tcpstat.tcps_connects++;
		NETSTAT_UNLOCK(&tcpstat.tcps_lock);
		soisconnected(so);
		tp->t_state = TCPS_ESTABLISHED;
#if RFC_1323
		/* Do window scaling? */
		if ( (tp->t_flags & (TF_RCVD_SCALE|TF_REQ_SCALE)) ==
			(TF_RCVD_SCALE|TF_REQ_SCALE) ) {
			tp->snd_scale = tp->requested_s_scale;
			tp->rcv_scale = tp->request_r_scale;
		}
#endif
		(void) tcp_reass(tp, (struct tcpiphdr *)0, (struct mbuf *)0);
		tp->snd_wl1 = ti->ti_seq - 1;
		/* fall into ... */

	/*
	 * In ESTABLISHED state: drop duplicate ACKs; ACK out of range
	 * ACKs.  If the ack is in the range
	 *	tp->snd_una < ti->ti_ack <= tp->snd_max
	 * then advance tp->snd_una to ti->ti_ack and drop
	 * data from the retransmission queue.  If this ACK reflects
	 * more up to date window information we update our window information.
	 */
	case TCPS_ESTABLISHED:
	case TCPS_FIN_WAIT_1:
	case TCPS_FIN_WAIT_2:
	case TCPS_CLOSE_WAIT:
	case TCPS_CLOSING:
	case TCPS_LAST_ACK:
	case TCPS_TIME_WAIT:

		if (SEQ_LEQ(ti->ti_ack, tp->snd_una)) {
#if RFC_1323
			if (ti->ti_len == 0 && tiwin == tp->snd_wnd) {
#else
			if (ti->ti_len == 0 && ti->ti_win == tp->snd_wnd) {
#endif
				NETSTAT_LOCK(&tcpstat.tcps_lock);
				tcpstat.tcps_rcvdupack++;
				NETSTAT_UNLOCK(&tcpstat.tcps_lock);
				/*
				 * If we have outstanding data (other than
				 * a window probe), this is a completely
				 * duplicate ack (ie, window info didn't
				 * change), the ack is the biggest we've
				 * seen and we've seen exactly our rexmt
				 * threshhold of them, assume a packet
				 * has been dropped and retransmit it.
				 * Kludge snd_nxt & the congestion
				 * window so we send only this one
				 * packet.
				 *
				 * We know we're losing at the current
				 * window size so do congestion avoidance
				 * (set ssthresh to half the current window
				 * and pull our congestion window back to
				 * the new ssthresh).
				 *
				 * Dup acks mean that packets have left the
				 * network (they're now cached at the receiver) 
				 * so bump cwnd by the amount in the receiver
				 * to keep a constant cwnd packets in the
				 * network.
				 */
				if (tp->t_timer[TCPT_REXMT] == 0 ||
				    ti->ti_ack != tp->snd_una)
					tp->t_dupacks = 0;
				else if (++tp->t_dupacks == tcprexmtthresh) {
					tcp_seq onxt = tp->snd_nxt;
					u_int win =
					    min(tp->snd_wnd, tp->snd_cwnd) / 2 /
						tp->t_maxseg;

					if (win < 2)
						win = 2;
					tp->snd_ssthresh = win * tp->t_maxseg;
					tp->t_timer[TCPT_REXMT] = 0;
					tp->t_rtt = 0;
					tp->snd_nxt = ti->ti_ack;
					tp->snd_cwnd = tp->t_maxseg;
					(void) tcp_output(tp);
					tp->snd_cwnd = tp->snd_ssthresh +
					       tp->t_maxseg * tp->t_dupacks;
					if (SEQ_GT(onxt, tp->snd_nxt))
						tp->snd_nxt = onxt;
					goto drop;
				} else if (tp->t_dupacks > tcprexmtthresh) {
					tp->snd_cwnd += tp->t_maxseg;
					(void) tcp_output(tp);
					goto drop;
				}
			} else
				tp->t_dupacks = 0;
			break;
		}
		/*
		 * If the congestion window was inflated to account
		 * for the other side's cached packets, retract it.
		 */
		if (tp->t_dupacks > tcprexmtthresh &&
		    tp->snd_cwnd > tp->snd_ssthresh)
			tp->snd_cwnd = tp->snd_ssthresh;
		tp->t_dupacks = 0;
		if (SEQ_GT(ti->ti_ack, tp->snd_max)) {
			NETSTAT_LOCK(&tcpstat.tcps_lock);
			tcpstat.tcps_rcvacktoomuch++;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
			goto dropafterack;
		}
		acked = ti->ti_ack - tp->snd_una;
		NETSTAT_LOCK(&tcpstat.tcps_lock);
		tcpstat.tcps_rcvackpack++;
		tcpstat.tcps_rcvackbyte += acked;
		NETSTAT_UNLOCK(&tcpstat.tcps_lock);

#if	RFC_1323
		/*
		 * If we have a time-stamp reply, update smoothed
		 * round trip time.  If no time-stamp is present but
		 * transmit timer is running and timed sequence
		 * number was acked, update smoothed round trip time.
		 * Since we now have an rtt measurement, cancel the
		 * timer backoff (cf., Phil Karn's retransmit alg.).
		 * Recompute the initial retransmit timer.
		 */
		if (ts_present)
			tcp_xmit_timer(tp, tcp_now-ts_ecr+1 );
		else if (tp->t_rtt && SEQ_GT(ti->ti_ack, tp->t_rtseq))
			tcp_xmit_timer(tp,tp->t_rtt);
#else
		/*
		 * If transmit timer is running and timed sequence
		 * number was acked, update smoothed round trip time.
		 * Since we now have an rtt measurement, cancel the
		 * timer backoff (cf., Phil Karn's retransmit alg.).
		 * Recompute the initial retransmit timer.
		 */
		if (tp->t_rtt && SEQ_GT(ti->ti_ack, tp->t_rtseq))
			tcp_xmit_timer(tp);
#endif

		/*
		 * If all outstanding data is acked, stop retransmit
		 * timer and remember to restart (more output or persist).
		 * If there is more data to be acked, restart retransmit
		 * timer, using current (possibly backed-off) value.
		 */
		if (ti->ti_ack == tp->snd_max) {
			tp->t_timer[TCPT_REXMT] = 0;
			needoutput = 1;
		} else if (tp->t_timer[TCPT_PERSIST] == 0)
			tp->t_timer[TCPT_REXMT] = tp->t_rxtcur;
		/*
		 * When new data is acked, open the congestion window.
		 * If the window gives us less than ssthresh packets
		 * in flight, open exponentially (maxseg per packet).
		 * Otherwise open linearly: maxseg per window
		 * (maxseg^2 / cwnd per packet), plus a constant
		 * fraction of a packet (maxseg/8) to help larger windows
		 * open quickly enough.
		 */
		{
		register u_int cw = tp->snd_cwnd;
		register u_int incr = tp->t_maxseg;

		if (cw > tp->snd_ssthresh)
			incr = incr * incr / cw + incr / 8;
#if RFC_1323
		tp->snd_cwnd = min(cw + incr, TCP_MAXWIN<<tp->snd_scale);
#else
		tp->snd_cwnd = min(cw + incr, TCP_MAXWIN);
#endif
		}
		SOCKBUF_LOCK(&so->so_snd);
		if (acked > so->so_snd.sb_cc) {
			tp->snd_wnd -= so->so_snd.sb_cc;
			sbdrop(&so->so_snd, (int)so->so_snd.sb_cc);
			ourfinisacked = 1;
		} else {
			sbdrop(&so->so_snd, acked);
			tp->snd_wnd -= acked;
			ourfinisacked = 0;
		}
		if (so->so_snd.sb_flags & SB_NOTIFY) {
			SOCKBUF_UNLOCK(&so->so_snd);
			sowwakeup(so);
		} else
			SOCKBUF_UNLOCK(&so->so_snd);
		tp->snd_una = ti->ti_ack;
		if (SEQ_LT(tp->snd_nxt, tp->snd_una))
			tp->snd_nxt = tp->snd_una;

		switch (tp->t_state) {

		/*
		 * In FIN_WAIT_1 STATE in addition to the processing
		 * for the ESTABLISHED state if our FIN is now acknowledged
		 * then enter FIN_WAIT_2.
		 */
		case TCPS_FIN_WAIT_1:
			if (ourfinisacked) {
				/*
				 * If we can't receive any more
				 * data, then closing user can proceed.
				 * Starting the timer is contrary to the
				 * specification, but if we don't get a FIN
				 * we'll hang forever.
				 */
				if (so->so_state & SS_CANTRCVMORE) {
					soisdisconnected(so);
					tp->t_timer[TCPT_2MSL] = tcp_maxidle;
				}
				tp->t_state = TCPS_FIN_WAIT_2;
			}
			break;

	 	/*
		 * In CLOSING STATE in addition to the processing for
		 * the ESTABLISHED state if the ACK acknowledges our FIN
		 * then enter the TIME-WAIT state, otherwise ignore
		 * the segment.
		 */
		case TCPS_CLOSING:
			if (ourfinisacked) {
				tp->t_state = TCPS_TIME_WAIT;
				tcp_canceltimers(tp);
				tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;
				soisdisconnected(so);
			}
			break;

		/*
		 * In LAST_ACK, we may still be waiting for data to drain
		 * and/or to be acked, as well as for the ack of our FIN.
		 * If our FIN is now acknowledged, delete the TCB,
		 * enter the closed state and return.
		 */
		case TCPS_LAST_ACK:
			if (ourfinisacked) {
				tp = tcp_close(tp);
				goto drop;
			}
			break;

		/*
		 * In TIME_WAIT state the only thing that should arrive
		 * is a retransmission of the remote FIN.  Acknowledge
		 * it and restart the finack timer.
		 */
		case TCPS_TIME_WAIT:
			tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;
			goto dropafterack;
		}
	}

step6:
	/*
	 * Update window information.
	 * Don't look at window if no ACK: TAC's send garbage on first SYN.
	 */
	if ((tiflags & TH_ACK) &&
	    (SEQ_LT(tp->snd_wl1, ti->ti_seq) || tp->snd_wl1 == ti->ti_seq &&
	    (SEQ_LT(tp->snd_wl2, ti->ti_ack) ||
#if	RFC_1323
	     tp->snd_wl2 == ti->ti_ack && tiwin > tp->snd_wnd))) {
#else
	     tp->snd_wl2 == ti->ti_ack && ti->ti_win > tp->snd_wnd))) {
#endif
		/* keep track of pure window updates */
		if (ti->ti_len == 0 &&
#if	RFC_1323
		    tp->snd_wl2 == ti->ti_ack && tiwin > tp->snd_wnd) {
#else
		    tp->snd_wl2 == ti->ti_ack && ti->ti_win > tp->snd_wnd) {
#endif
			NETSTAT_LOCK(&tcpstat.tcps_lock);
			tcpstat.tcps_rcvwinupd++;
			NETSTAT_UNLOCK(&tcpstat.tcps_lock);
		}
#if	RFC_1323
		tp->snd_wnd = tiwin;
#else
		tp->snd_wnd = ti->ti_win;
#endif
		tp->snd_wl1 = ti->ti_seq;
		tp->snd_wl2 = ti->ti_ack;
		if (tp->snd_wnd > tp->max_sndwnd)
			tp->max_sndwnd = tp->snd_wnd;
		needoutput = 1;
	}

	/*
	 * Process segments with URG.
	 */
	if ((tiflags & TH_URG) && ti->ti_urp &&
	    TCPS_HAVERCVDFIN(tp->t_state) == 0) {
		SOCKBUF_LOCK(&so->so_rcv);
		/*
		 * This is a kludge, but if we receive and accept
		 * random urgent pointers, we'll crash in
		 * soreceive.  It's hard to imagine someone
		 * actually wanting to send this much urgent data.
		 */
		if (ti->ti_urp + so->so_rcv.sb_cc > sb_max) {
			SOCKBUF_UNLOCK(&so->so_rcv);
			ti->ti_urp = 0;			/* XXX */
			tiflags &= ~TH_URG;		/* XXX */
			goto dodata;			/* XXX */
		}
		/*
		 * If this segment advances the known urgent pointer,
		 * then mark the data stream.  This should not happen
		 * in CLOSE_WAIT, CLOSING, LAST_ACK or TIME_WAIT STATES since
		 * a FIN has been received from the remote side. 
		 * In these states we ignore the URG.
		 *
		 * According to RFC961 (Assigned Protocols),
		 * the urgent pointer points to the last octet
		 * of urgent data.  We continue, however,
		 * to consider it to indicate the first octet
		 * of data past the urgent section as the original 
		 * spec states (in one of two places).
		 * ... UNLESS !tcp_compat_42
		 */
		if (SEQ_GT(ti->ti_seq+ti->ti_urp-(!tcp_compat_42),tp->rcv_up)) {
			tp->rcv_up = ti->ti_seq + ti->ti_urp;
			so->so_oobmark = so->so_rcv.sb_cc +
			    (tp->rcv_up - tp->rcv_nxt) - 1;
			if (so->so_oobmark == 0)
				so->so_state |= SS_RCVATMARK;
			sohasoutofband(so);
			tp->t_oobflags &= ~(TCPOOB_HAVEDATA | TCPOOB_HADDATA);
		}
		/*
		 * Remove out of band data so doesn't get presented to user.
		 * This can happen independent of advancing the URG pointer,
		 * but if two URG's are pending at once, some out-of-band
		 * data may creep in... ick.
		 */
		if (ti->ti_urp <= ti->ti_len &&
		    (so->so_options & SO_OOBINLINE) == 0)
			tcp_pulloutofband(so, ti, m);
		SOCKBUF_UNLOCK(&so->so_rcv);
	} else
		/*
		 * If no out of band data is expected,
		 * pull receive urgent pointer along
		 * with the receive window.
		 */
		if (SEQ_GT(tp->rcv_nxt, tp->rcv_up))
			tp->rcv_up = tp->rcv_nxt;
dodata:							/* XXX */

	/*
	 * Process the segment text, merging it into the TCP sequencing queue,
	 * and arranging for acknowledgment of receipt if necessary.
	 * This process logically involves adjusting tp->rcv_wnd as data
	 * is presented to the user (this happens in tcp_usrreq.c,
	 * case PRU_RCVD).  If a FIN has already been received on this
	 * connection then we just ignore the text.
	 */
	if ((ti->ti_len || (tiflags&TH_FIN)) &&
	    TCPS_HAVERCVDFIN(tp->t_state) == 0) {
		TCP_REASS(tp, ti, m, so, tiflags);
		/*
		 * Note the amount of data that peer has sent into
		 * our window, in order to estimate the sender's
		 * buffer size.
		 */
		len = so->so_rcv.sb_hiwat - (tp->rcv_adv - tp->rcv_nxt);
		if (len > tp->max_rcvd)
			tp->max_rcvd = len;
	} else {
		m_freem(m);
		tiflags &= ~TH_FIN;
	}

	/*
	 * If FIN is received ACK the FIN and let the user know
	 * that the connection is closing.
	 */
	if (tiflags & TH_FIN) {
		if (TCPS_HAVERCVDFIN(tp->t_state) == 0) {
			socantrcvmore(so);
			tp->t_flags |= TF_ACKNOW;
			tp->rcv_nxt++;
		}
		switch (tp->t_state) {

	 	/*
		 * In SYN_RECEIVED and ESTABLISHED STATES
		 * enter the CLOSE_WAIT state.
		 */
		case TCPS_SYN_RECEIVED:
		case TCPS_ESTABLISHED:
			tp->t_state = TCPS_CLOSE_WAIT;
			break;

	 	/*
		 * If still in FIN_WAIT_1 STATE FIN has not been acked so
		 * enter the CLOSING state.
		 */
		case TCPS_FIN_WAIT_1:
			tp->t_state = TCPS_CLOSING;
			break;

	 	/*
		 * In FIN_WAIT_2 state enter the TIME_WAIT state,
		 * starting the time-wait timer, turning off the other 
		 * standard timers.
		 */
		case TCPS_FIN_WAIT_2:
			tp->t_state = TCPS_TIME_WAIT;
			tcp_canceltimers(tp);
			tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;
			soisdisconnected(so);
			break;

		/*
		 * In TIME_WAIT state restart the 2 MSL time_wait timer.
		 */
		case TCPS_TIME_WAIT:
			tp->t_timer[TCPT_2MSL] = 2 * TCPTV_MSL;
			break;
		}
	}
	if (so->so_options & SO_DEBUG)
		tcp_trace(TA_INPUT, ostate, tp, &tcp_saveti, 0);

	/*
	 * Return any desired output.
	 */
done:
	if (needoutput || (tp->t_flags & TF_ACKNOW))
		(void) tcp_output(tp);
	INPCB_UNLOCK(inp);
	INPCBRC_UNREF(inp);
	SOCKET_UNLOCK(so);
	return;

dropafterack:
	/*
	 * Generate an ACK dropping incoming segment if it occupies
	 * sequence space, where the ACK reflects our state.
	 */
	if (tiflags & TH_RST)
		goto drop;
	m_freem(m);
	tp->t_flags |= TF_ACKNOW;
	(void) tcp_output(tp);
	INPCB_UNLOCK(inp);
	INPCBRC_UNREF(inp);
	SOCKET_UNLOCK(so);
	return;

dropwithreset:
	if (om) {
		(void) m_free(om);
		om = 0;
	}
	/*
	 * Generate a RST, dropping incoming segment.
	 * Make ACK acceptable to originator of segment.
	 * Don't bother to respond if destination was broadcast.
	 */
	if ((tiflags & TH_RST) || m_broadcast(m))
		goto drop;
	if (tiflags & TH_ACK)
		tcp_respond(tp, ti, m, (tcp_seq)0, ti->ti_ack, TH_RST);
	else {
		if (tiflags & TH_SYN)
			ti->ti_len++;
		tcp_respond(tp, ti, m, ti->ti_seq+ti->ti_len, (tcp_seq)0,
		    TH_RST|TH_ACK);
	}
	if (inp) {
		if (tp) INPCB_UNLOCK(inp);
		INPCBRC_UNREF(inp);
	}
	if (so)
		SOCKET_UNLOCK(so);
	/* destroy temporarily created socket */
	if (dropsocket)
		(void) soabort(so);
	return;

drop:
	if (om)
		(void) m_free(om);
	/*
	 * Drop space held by incoming segment and return.
	 */
	if (tp && (tp->t_inpcb->inp_socket->so_options & SO_DEBUG))
		tcp_trace(TA_DROP, ostate, tp, &tcp_saveti, 0);
	m_freem(m);
	if (inp) {
		if (tp) INPCB_UNLOCK(inp);
		INPCBRC_UNREF(inp);
	}
	if (so)
		SOCKET_UNLOCK(so);
	/* destroy temporarily created socket */
	if (dropsocket)
		(void) soabort(so);
	return;
}

void
#if	RFC_1323
tcp_dooptions(tp, om, ti, ts_present, ts_val, ts_ecr)
#else
tcp_dooptions(tp, om, ti)
#endif
	struct tcpcb *tp;
	struct mbuf *om;
	struct tcpiphdr *ti;
#if	RFC_1323
	int *ts_present;
	u_long *ts_val, *ts_ecr;
#endif
{
	register u_char *cp;
	u_short mss;
	int opt, optlen, cnt;
#if	RFC_1323
	extern int tcp_dont_tsecho;
#endif

	cp = mtod(om, u_char *);
	cnt = om->m_len;
	for (; cnt > 0; cnt -= optlen, cp += optlen) {
		opt = cp[0];
		if (opt == TCPOPT_EOL)
			break;
		if (opt == TCPOPT_NOP)
			optlen = 1;
		else {
			optlen = cp[1];
			if (optlen <= 0)
				break;
		}
		switch (opt) {

		default:
			continue;

		case TCPOPT_MAXSEG:
			if (optlen != 4)
				continue;
			if (!(ti->ti_flags & TH_SYN))
				continue;
			bcopy((char *) cp + 2, (char *) &mss, sizeof(mss));
			NTOHS(mss);
			(void) tcp_mss(tp, mss);	/* sets t_maxseg */
			break;

#if	RFC_1323
		case TCPOPT_WINDOW:
			if (optlen != 3)
				continue;
			if (!(ti->ti_flags & TH_SYN))
				continue;
			tp->t_flags |= TF_RCVD_SCALE;
			tp->requested_s_scale = MIN( cp[2], TCP_MAX_WINSHIFT );
			break;

		case TCPOPT_TIMESTAMP:
			if (optlen != 10)
				continue;
			
			/* Put value into ts_val in host byte order.
			 * Could be unaligned.
			 */
			*ts_present = 1;
			*ts_val = cp[2]<<24 | cp[3]<<16 | cp[4]<<8 | cp[5];
			*ts_ecr = cp[6]<<24 | cp[7]<<16 | cp[8]<<8 | cp[9];

			/* A time-stamp received in a SYN makes
			 * it ok to send time-stamp requests and replies.
			 */
			if ( ti->ti_flags & TH_SYN ) {
				tp->t_flags |= TF_RCVD_TSTMP;
				tp->ts_recent = *ts_val;
				tp->ts_recent_age = 0;
			}
			break;
		
		case TCPOPT_SACK_PERMITTED:
			if (optlen != 2)
				continue;
			if (!(ti->ti_flags & TH_SYN))
				continue;
			tp->t_flags |= TF_SACK_PERMIT;
			break;
#endif /* RFC_1323 */
		}
	}
	(void) m_free(om);
}

/*
 * Pull out of band byte out of a segment so
 * it doesn't appear in the user's data queue.
 * It is still reflected in the segment length for
 * sequencing purposes.
 */
void
tcp_pulloutofband(so, ti, m)
	struct socket *so;
	struct tcpiphdr *ti;
	register struct mbuf *m;
{
	int cnt = ti->ti_urp - 1;
	
	while (cnt >= 0) {
		if (m->m_len > cnt) {
			char *cp = mtod(m, caddr_t) + cnt;
			struct tcpcb *tp = sototcpcb(so);

			tp->t_iobc = *cp;
			tp->t_oobflags |= TCPOOB_HAVEDATA;
			bcopy(cp+1, cp, (unsigned)(m->m_len - cnt - 1));
			m->m_len--;
			return;
		}
		cnt -= m->m_len;
		m = m->m_next;
		if (m == 0)
			break;
	}
	panic("tcp_pulloutofband");
}

/*
 * Collect new round-trip time estimate
 * and update averages and current timeout.
 */
#if __STDC__ == 1

#if	RFC_1323
void tcp_xmit_timer(struct tcpcb *tp, short rtt)
#else
void tcp_xmit_timer(struct tcpcb *tp)
#endif /* RFC_1323 */

#else

void
#if	RFC_1323
tcp_xmit_timer(tp, rtt)
#else
tcp_xmit_timer(tp)
#endif
	register struct tcpcb *tp;
#if	RFC_1323
	short	rtt;
#endif /* RFC_1323 */

#endif /* __STDC__ */
{
	register short delta;

	NETSTAT_LOCK(&tcpstat.tcps_lock);
	tcpstat.tcps_rttupdated++;
	NETSTAT_UNLOCK(&tcpstat.tcps_lock);
	if (tp->t_srtt != 0) {
		/*
		 * srtt is stored as fixed point with 3 bits after the
		 * binary point (i.e., scaled by 8).  The following magic
		 * is equivalent to the smoothing algorithm in rfc793 with
		 * an alpha of .875 (srtt = rtt/8 + srtt*7/8 in fixed
		 * point).  Adjust t_rtt (rtt for RFC_1323) to origin 0.
		 */
#if	RFC_1323
		delta = rtt - 1 - (tp->t_srtt >> TCP_RTT_SHIFT);
#else
		delta = tp->t_rtt - 1 - (tp->t_srtt >> TCP_RTT_SHIFT);
#endif
		if ((tp->t_srtt += delta) <= 0)
			tp->t_srtt = 1;
		/*
		 * We accumulate a smoothed rtt variance (actually, a
		 * smoothed mean difference), then set the retransmit
		 * timer to smoothed rtt + 4 times the smoothed variance.
		 * rttvar is stored as fixed point with 2 bits after the
		 * binary point (scaled by 4).  The following is
		 * equivalent to rfc793 smoothing with an alpha of .75
		 * (rttvar = rttvar*3/4 + |delta| / 4).  This replaces
		 * rfc793's wired-in beta.
		 */
		if (delta < 0)
			delta = -delta;
		delta -= (tp->t_rttvar >> TCP_RTTVAR_SHIFT);
		if ((tp->t_rttvar += delta) <= 0)
			tp->t_rttvar = 1;
	} else {
		/* 
		 * No rtt measurement yet - use the unsmoothed rtt.
		 * Set the variance to half the rtt (so our first
		 * retransmit happens at 2*rtt)
		 */
#if	RFC_1323
		tp->t_srtt = rtt << TCP_RTT_SHIFT;
		tp->t_rttvar = rtt << (TCP_RTTVAR_SHIFT - 1);
#else
		tp->t_srtt = tp->t_rtt << TCP_RTT_SHIFT;
		tp->t_rttvar = tp->t_rtt << (TCP_RTTVAR_SHIFT - 1);
#endif
	}
	tp->t_rtt = 0;
	tp->t_rxtshift = 0;

	/*
	 * the retransmit should happen at rtt + 4 * rttvar.
	 * Because of the way we do the smoothing, srtt and rttvar
	 * will each average +1/2 tick of bias.  When we compute
	 * the retransmit timer, we want 1/2 tick of rounding and
	 * 1 extra tick because of +-1/2 tick uncertainty in the
	 * firing of the timer.  The bias will give us exactly the
	 * 1.5 tick we need.  But, because the bias is
	 * statistical, we have to test that we don't drop below
	 * the minimum feasible timer (which is 2 ticks).
	 */
	TCPT_RANGESET(tp->t_rxtcur, TCP_REXMTVAL(tp),
	    tp->t_rttmin, TCPTV_REXMTMAX);
	
	/*
	 * We received an ack for a packet that wasn't retransmitted;
	 * it is probably safe to discard any error indications we've
	 * received recently.  This isn't quite right, but close enough
	 * for now (a route might have failed after we sent a segment,
	 * and the return path might not be symmetrical).
	 */
	tp->t_softerror = 0;
}

/*
 * Determine a reasonable value for maxseg size.
 * If the route is known, check route for mtu.
 * If none, use an mss that can be handled on the outgoing
 * interface without forcing IP to fragment; if using trailers and bigger than
 * an mbuf cluster (MCLBYTES), round down to nearest multiple of MCLBYTES
 * to utilize large mbufs.  If no route is found, route has no mtu,
 * or the destination isn't local, use a default, hopefully conservative
 * size (usually 512 or the default IP max size, but no more than the mtu
 * of the interface), as we can't discover anything about intervening
 * gateways or networks.  We also initialize the congestion/slow start
 * window to be a single segment if the destination isn't local.
 * While looking at the routing entry, we also initialize other path-dependent
 * parameters from pre-set or cached values in the routing entry.
 */
#if	RFC_1323
/*
 * We must also take into account that time-stamp options may be placed in
 * each segment.   -skibo
 */
#endif

#ifndef _NO_PROTO
tcp_mss(
	register struct tcpcb *tp,
	u_short offer)
#else
tcp_mss(tp, offer)
	register struct tcpcb *tp;
	u_short offer;
#endif
{
	struct route *ro;
	register struct rtentry *rt;
	struct ifnet *ifp;
	register int rtt, mss;
	u_long bufsize;
	struct inpcb *inp;
	struct socket *so;
	ROUTE_LOCK_DECL()
	extern int tcp_mssdflt, tcp_rttdflt;
#if	RFC_1323
	int room_for_options = 0;

	if ( tp->t_flags & TF_REQ_TSTMP )
		room_for_options += 12;	/* leave room for time-stamping */
#endif

	inp = tp->t_inpcb;
	ro = &inp->inp_route;

	ROUTE_WRITE_LOCK();
	if ((rt = ro->ro_rt) == (struct rtentry *)0) {
		/* No route yet, so try to acquire one */
		if (inp->inp_faddr.s_addr != INADDR_ANY) {
			ro->ro_dst.sa_family = AF_INET;
			ro->ro_dst.sa_len = sizeof(ro->ro_dst);
			((struct sockaddr_in *) &ro->ro_dst)->sin_addr =
				inp->inp_faddr;
			rtalloc(ro);
		}
		if ((rt = ro->ro_rt) == (struct rtentry *)0) {
			ROUTE_WRITE_UNLOCK();
#if	RFC_1323
			return (tcp_mssdflt - room_for_options);
#else
			return (tcp_mssdflt);
#endif
		}
	}
	ifp = rt->rt_ifp;
	ROUTE_WRITE_UNLOCK();
	so = inp->inp_socket;

#ifdef RTV_MTU	/* if route characteristics exist ... */
	/*
	 * While we're here, check if there's an initial rtt
	 * or rttvar.  Convert from the route-table units
	 * to scaled multiples of the slow timeout timer.
	 */
	if (tp->t_srtt == 0 && (rtt = rt->rt_rmx.rmx_rtt)) {
		if (rt->rt_rmx.rmx_locks & RTV_MTU)
			tp->t_rttmin = rtt / (RTM_RTTUNIT / PR_SLOWHZ);
		tp->t_srtt = rtt / (RTM_RTTUNIT / (PR_SLOWHZ * TCP_RTT_SCALE));
		if (rt->rt_rmx.rmx_rttvar)
			tp->t_rttvar = rt->rt_rmx.rmx_rttvar /
			    (RTM_RTTUNIT / (PR_SLOWHZ * TCP_RTTVAR_SCALE));
		else
			/* default variation is +- 1 rtt */
			tp->t_rttvar =
			    tp->t_srtt * TCP_RTTVAR_SCALE / TCP_RTT_SCALE;
		TCPT_RANGESET(tp->t_rxtcur,
		    ((tp->t_srtt >> 2) + tp->t_rttvar) >> 1,
		    tp->t_rttmin, TCPTV_REXMTMAX);
	}
	/*
	 * if there's an mtu associated with the route, use it
	 */
	if (rt->rt_rmx.rmx_mtu)
#if RFC_1323
		mss = rt->rt_rmx.rmx_mtu - sizeof(struct tcpiphdr)
			- room_for_options;
#else
		mss = rt->rt_rmx.rmx_mtu - sizeof(struct tcpiphdr);
#endif /* RFC_1323 */
	else
#endif /* RTV_MTU */
	{
#if	RFC_1323
		mss = ifp->if_mtu - sizeof(struct tcpiphdr) - room_for_options;
#else
		mss = ifp->if_mtu - sizeof(struct tcpiphdr);
#endif /* RFC_1323 */
		if (mss > MCLBYTES && (ifp->if_flags & IFF_NOTRAILERS) == 0) {
			/* only want cluster rounding if trailers */
#if	(MCLBYTES & (MCLBYTES - 1)) == 0
			mss &= ~(MCLBYTES-1);
#else
			mss = mss / MCLBYTES * MCLBYTES;
#endif
		}
		if (!in_localaddr(inp->inp_faddr))
#if RFC_1323
			mss = min(mss, tcp_mssdflt-room_for_options);
#else
			mss = min(mss, tcp_mssdflt);
#endif /* RFC_1323 */
	}
	/*
	 * The current mss, t_maxseg, is initialized to the default value.
	 * If we compute a smaller value, reduce the current mss.
	 * If we compute a larger value, return it for use in sending
	 * a max seg size option, but don't store it for use
	 * unless we received an offer at least that large from peer.
	 * However, do not accept offers under 32 bytes.
	 */
	if (offer)
		mss = min(mss, offer);
	mss = max(mss, 32);		/* sanity */
	if (mss < tp->t_maxseg || offer != 0) {
		/*
		 * If there's a pipesize, change the socket buffer
		 * to that size.  Make the socket buffers an integral
		 * number of mss units; if the mss is larger than
		 * the socket buffer, decrease the mss.
		 */
		LOCK_ASSERT("tcp_mss so", SOCKET_ISLOCKED(so));
		SOCKBUF_LOCK(&so->so_snd);
#ifdef RTV_SPIPE
		if ((bufsize = rt->rt_rmx.rmx_sendpipe) == 0)
#endif
			bufsize = so->so_snd.sb_hiwat;
		/*
		 * Want mss <= 2 * bufsize so buffering may behave optimally
		 * and delayed ack algorithms work as intended, among others.
		 * Try to maintain this even if bufsize % mss != 0.
		 * (e.g. bufsize == 4096, mss ~= 1460 as for ethernet)
		 * Adequate for now... evaluate. (tmt)
		 */
		if (bufsize < mss * 2)
			mss = max(32, bufsize / 2);
		else {
			bufsize = (bufsize / mss) * mss;
			/* Very rough roundoff error detector/adjuster */
			if (bufsize <= mss * 2) bufsize *= 2;
			while (sbreserve(&so->so_snd, bufsize) == 0)
				if ((bufsize -= mss) < mss)
					break;
		}
		SOCKBUF_UNLOCK(&so->so_snd);
		tp->t_maxseg = mss;

		SOCKBUF_LOCK(&so->so_rcv);
#ifdef RTV_RPIPE
		if ((bufsize = rt->rt_rmx.rmx_recvpipe) == 0)
#endif
			bufsize = so->so_rcv.sb_hiwat;
		if (bufsize > mss * 2) {
			bufsize = (bufsize / mss) * mss;
			if (bufsize <= mss * 2) bufsize *= 2;
			while (sbreserve(&so->so_rcv, bufsize) == 0)
				if ((bufsize -= mss) < mss)
					break;
		}
		SOCKBUF_UNLOCK(&so->so_rcv);
	}
	tp->snd_cwnd = mss;

#ifdef RTV_SSTHRESH
	if (rt->rt_rmx.rmx_ssthresh) {
		/*
		 * There's some sort of gateway or interface
		 * buffer limit on the path.  Use this to set
		 * the slow start threshhold, but set the
		 * threshold to no less than 2*mss.
		 */
		tp->snd_ssthresh = max(2 * mss, rt->rt_rmx.rmx_ssthresh);
	}
#endif /* RTV_MTU */
	return (mss);
}
