# Filename: devinfo.awk
#
# Input file format
# 	All entries for each type of device must be present
# 	and should appear in the positions shown below.
# 
# <DeviceName> <CBS> <Device_specific_info>
#
# Sample entries for various devices:
#
# 	DISK 00D03 ID 5 MAXTOR
#       Minimum RAID entry example:
#
# 	RAID 00A00 ID 2 SW 02 LV 3 DC 5 SID 7 RAID3
# 	RAID 00D03 ID 0 SW 04 LV 5 DC 5 SID 7 RAID5
#       Example with NO_PAGER, PAGE_TO:
#
# 	RAID 00B00 ID 3 SW 02 LV 3 DC 5 SID 7 RAID3 NO_PAGER
# 	RAID 00C00 ID 4 SW 02 LV 3 DC 5 SID 7 RAID3 PAGE_TO rz0b
#
#       Example with Block count (device capacity in Gbytes)
#
# 	RAID 00C00 ID 4 SW 02 LV 3 DC 5 SID 7 BC 2 RAID3 PAGE_TO rz0b
#
# 	RAID 00D03 ID 0 SW 04 LV 5 DC 5 SID 7 RAID5
# 	TAPE 00D03 ID 2 DAT
# 	MIO  01D02 I04
#	ENET 02B15
#
#       Sample entry for MDC
#
#       MDC 02A03 M03 16
#
#       Sample entries for SCSI-16:
#       There could be a maximum of 2 controllers 
#       SIO  01A00 S04
# 	RAID 01A00 ID 2 SW 02 LV 3 DC 5 SID 7 RAID3 CTLR0
# 	TAPE 01A00 ID 2 DAT CTLR1
#
#       2-row system is expected to be defined as follows:
#       
#       CAB_ROWS 2
#
# Output file format
#   CABINET <cabNum> BP <BpNum> S <SlotNum> [<Device/Controller names>]
#
# Modification History:
#
# $Log: devinfo.awk,v $
# Revision 1.11  1995/03/28  00:07:19  chowell
# chowell - added a version string.
#
# Revision 1.10  1994/08/19  20:50:52  mouli
# Added support for MDC, SCSI-16 and 2-row. Error handling
# is improved.
#
# Revision 1.7  1994/02/03  00:22:55  mouli
# Updated the documentation for NO_PAGER and PAGE_TO keywords.
#
# Revision 1.6  1993/12/20  19:41:49  mouli
# Added support for NOPAGER and PAGE_TO keywords.
#
# Revision 1.5  1993/12/16  19:01:02  mouli
# Added NOPAGER and PAGE_TO keywords for OS debugging.
#
# Revision 1.4  1993/05/13  20:52:11  mouli
# Added HIPPI device support.
#
# Revision 1.3  1993/03/22  01:38:09  mouli
# 1. took off the requirement to have block count in user-created
#    device config file. Standard devices (disk,raid) will have
#    an entry in a file with two entries (device_name, block_count)
#    The awk script uses this file for outputing block count info.
#
# Revision 1.2  1993/03/18  21:25:08  mouli
# 1. Added ENET & MIO processing
# 2. printdev() now handles generic additional info such as brand
#    name, device name, fab-rev.
#
# Revision 1.1  1993/03/12  01:32:30  mouli
# Initial revision
#
# 	03/09/93	mouli	Initial version
#
##############################################################################
#
# What string:
#
# @(#) devinfo.awk DIAG_REL_2.0 Thu Jan  5 12:00:18 PST 1995
#
##############################################################################

# Set up Table for Cable to Backplane mapping
# Initialize keyword and entry count table
BEGIN {
	self = "devinfo.awk"
	if (errfile == "")
	{
		printf "%s:no errfile\n", self
		exit 2
	}
# Mapping of Cable label to an index
	bp["A"] = 0	# Cable A
	bp["B"] = 1	# Cable B
	bp["C"] = 2	# Cable C
	bp["D"] = 3	# Cable D
	bp["E"] = 4	# Cable E (Power control)

        TRUE = 1
        FALSE = 0
# keywords
	TAPE_KWD  = "TAPE"
	DISK_KWD = "DISK"
	RAID_KWD = "RAID"
	ENET_KWD = "ENET"
	MIO_KWD  = "MIO"
        SIO_KWD  = "SIO"
        MDC_KWD  = "MDC"
	HIPPI_KWD  = "HIPPI"
        CTLR_KWD  = "CTLR"
        DEF_CTLR  = "CTLR0"
        MAXCTLRS  = 2
        NOPAGER_KWD = "NOPAGER"
        PAGETO_KWD = "PAGE_TO"
	CABROW_KWD = "CAB_ROWS"
#       Set up the variables for 2-row system
# The choice of SPECIAL_CAB, BP, SLOT is such that this
# will be very first record this scripts output is sorted.
# so that we can output it as the first record in SYSCONFIG.TXT

        MAXROWS    = 2
	SPECIAL_CAB = -1
	SPECIAL_BP  = "-A"
	SPECIAL_SLOT  = -1

# Initialize expected number of fields required for various entries
# The minimum DISK Record has 5 fields
#    DISK 00D03 ID 5 MAXTOR   
# The maximum DISK Record has 10 fields
#    DISK 00D03 ID 5 BC 1 MAXTOR PAGE_TO rz0b CTLR0

	DISK_REC_MIN = 5
	DISK_REC_MAX = 10

# The minimum RAID Record has 13 fields
# 	RAID 00A00 ID 2 SW 02 LV 3 DC 5 SID 7 RAID3
# The maximum RAID Record has 18 fields
# 	RAID 00A00 ID 2 SW 02 LV 3 DC 5 SID 7 BC 1 PAGE_TO rz0b RAID3 CTLR0

	RAID_REC_MIN = 13
	RAID_REC_MAX = 18

# The minimum TAPE record has 5 fields
# 	TAPE 00D03 ID 2 DAT
# The maximum TAPE record has 5 fields
# 	TAPE 00D03 ID 2 DAT

        TAPE_REC_MIN = 5
        TAPE_REC_MAX = 6

# Set up expected field counts

	ecount[TAPE_KWD] = TAPE_REC_MIN
	ecount[DISK_KWD] = DISK_REC_MIN
	ecount[RAID_KWD] = RAID_REC_MIN
	ecount[ENET_KWD] = 2
	ecount[MIO_KWD] =  3
	ecount[SIO_KWD] =  3
	ecount[MDC_KWD] =  4
	ecount[HIPPI_KWD] =  3
	ecount[NOPAGER_KWD] = 2
	ecount[PAGETO_KWD] = 3
        ecount[CABROW_KWD] = 2

	
}

# printdev(name, cbs,addinfo)
#   Separate out the Cabinet, Backplane, Slot info 
#   use the manufacturer's name while outputing
#   Note that the backplanes are output as Alpha (A,B,C,D)

function printdev(name, cbs, addinfo)
{
	printf "CABINET %s ", substr(cbs, 1, 2)
	printf "BP %s ", substr(cbs, 3, 1)
	printf "S %s", substr(cbs, 4, 2)
        if (name != "") printf " %s", name
        if (addinfo != "") printf " %s", addinfo
        printf "\n"
}
# valid_cbs(cbs)
#  Given a cbs string of form "CCBSS", it validates the
#  string. Currently, it checks if the string length is
#  as expected. It should do some more checking the range
#  of numbers for CC (cabinet) and SS (slot) and B (backplane)
#
#  Returns: TRUE if CBS spec is valie
#           FALSE otherwise, outputs error message
#
function valid_cbs(cbs)
{
  if (length(cbs) != 5)
    {
      printf "%s Line %d: Invalid CBS specification\n", FILENAME,NR >> errfile
      return (FALSE)
    }
  return (TRUE)
  
}

# valid_ctlr(cname)
#  Given a controller name, it validates the
#  string. 
#
#  Returns: TRUE if valid controller name
#           FALSE otherwise, outputs error message
#
function valid_ctlr(cname)
{
  for (i = 0; i < MAXCTLRS; i++)
   {
	vname = sprintf("%s%d", CTLR_KWD, i)
	if (vname == cname)
	{
	   return (TRUE)
	}
   }
  printf "%s Line %d: Invalid controller specification %s\n", FILENAME,NR,cname >> errfile
  return(FALSE)
}
# raid_rec_init()
#   Performs initialization for RAID record processing
#   
function raid_rec_init()
{
# ID .. NAME are the minimum required fields
# BC and CTL are optional. Note that the indices are
# so chosen that we can check with a simple loop 
# whether or not mandatory fields are missing in the input record.

 ID = 0
 SW = 1
 LV = 2
 DC = 3
 SID= 4
 NAME = 5

# indices of optional fields here

 BC = 6
 CTL = 7
 NOPAG = 8
 PAG   = 9
 NSUB = 10

# NSUB counts the total number of subfields other than RAID and CBS
# and the arguments to the subfields. NREQ_RAID is the number of
# mandatory fields for RAID record
#
 NREQ_RAID = 6

# Register no fields have been encountered

 for (i = 0; i < 10; i++)
   raid_f[i] = FALSE
}
# tape_rec_init()
#   Performs initialization for TAPE record processing
# 
function tape_rec_init()
{

# NAME is the minimum required field
# CTL are optional. Note that the indices are
# so chosen that we can check with a simple loop 
# whether or not mandatory fields are missing in the input record.

 ID = 0
 NAME = 1

# indices of optional fields here

 CTL = 2
 NSUB = 3

# NSUB is the number of subfields in TAPE record excluding TAPE and CBS
# and the arguments to the sub fields.
# NREQ_TAPE is the number of mandatory fields for TAPE record

 NREQ_TAPE = 2

 for (i = 0; i < NSUB; i++)
   tape_f[i] = FALSE

}
# process_tape_record()
#   If the tape record is valid, it outputs the record
#   calling printdev().
#
# Returns: Name of tape device - if valid record
#          Null otherwise
#
function process_tape_record()
{
  if (!valid_cbs($2))
    return ("")

  tape_rec_init()

# We processed the TAPE keyword (field 1) and the CBS entry (field 2)
# So start from 3rd field in the input record

  f = 3
  err = 0
  ctl_name = ""
  while (f <= NF)
    {
      if ($f == "ID") f_idx = ID
      else if (index($f, CTLR_KWD) == 1) f_idx = CTL
      else f_idx = NAME
     if (tape_f[f_idx])
	{
	  printf "%s Line %d: Duplicate entry for %s field in TAPE record\n", FILENAME, NR, $f >> errfile
	  return ""

	}
      else
	{
	  if (f_idx == CTL)
	    {
	      ctl_name = $f
	      if (valid_ctlr(ctl_name))
	      {
		      tape_f[f_idx] = TRUE
	      }
	      else err = 1
	      f++
	      continue
	    }
	  else if (f_idx == NAME)
	    {
	      tape_name = $f
	      tape_f[f_idx] = TRUE
	      f++
	      continue
	    }
	  else if (f_idx == ID)
	    {
	      if ($(f+1) == "")
		{
		  printf "%s Line %d: %s field for TAPE requires an argument\n",FILENAME, NR,$f >> errfile
		  err = 1
		}
	      else 
		{
# we assume that the following field is its arg
		  tape_f[f_idx] = TRUE
		}
# Increment field count by the ID+its arg any way.
	      f += 2
	      continue
	    }
        }
    }
# end while

   for (i = 0; i < NREQ_TAPE; i++)
     {
       if (!tape_f[i])
	 {
	   printf "%s Line %d: Mandatory field(s) %d in TAPE record missing\n", FILENAME, NR,i >> errfile
	   err = 1
	 }
     }
     if (err) return ""

# we have a valid tape record. If no controller is
# specified append a default controller.
     info = ""
     if (ctl_name == "") info = info " " DEF_CTLR
     else info = info " " ctl_name
#     printf "%d: tape_name = %s info = %s\n", NR, tape_name, info >> errfile
     printdev(tape_name, $2, info)
     return tape_name

}
# disk_rec_init()
#   Performs initialization for DISK record processing
#
function disk_rec_init()
{

# ID .. NAME are the minimum required fields
# BC and CTL are optional. Note that the indices are
# so chosen that we can check with a simple loop 
# whether or not mandatory fields are missing in the input record.

 ID = 0
 NAME = 1

# indices of optional fields here

 BC = 2
 CTL = 3
 NOPAG = 4
 PAG   = 5
 NSUB = 6

# NSUB is the number of subfields in DISK record excluding DISK and CBS
# and the arguments to the sub fields. NREQ is the total number of
# required mandatory fields for DIS record

 NREQ_DISK = 2

# Register no fields have been encountered
 for (i = 0; i < NSUB; i++)
   disk_f[i] = FALSE

}
# process_disk_record()
#   If the disk record is valid, it outputs the record
#   calling printdev().
#
# Returns: Name of disk device - if valid record
#          Null otherwise

function process_disk_record()
{
  if (!valid_cbs($2))
    return ("")

  disk_rec_init()

# We processed the DISK keyword (field 1) and the CBS entry (field 2)
# So start from 3rd field in the input record

  f = 3
  err = 0
  pagedev = ""
  ctl_name = ""
  while (f <= NF)
    {
      if ($f == "ID") f_idx = ID
      else if ($f == "BC") f_idx = BC
      else if ($f == NOPAGER_KWD) f_idx = NOPAG
      else if ($f == PAGETO_KWD) f_idx = PAG
      else if (index($f, CTLR_KWD) == 1) f_idx = CTL
      else f_idx = NAME

      if (disk_f[f_idx])
	{
	  printf "%s Line %d: Duplicate entry for %s field in DISK record\n", FILENAME, NR,$f >> errfile
	  return ""

	}
      else
	{
# The following processing counts on the fact that
# all fields except CTLR and DISK name require one argument
	  if (f_idx == CTL)
	    {
	      ctl_name = $f
	      if (valid_ctlr(ctl_name))
	      {
		      disk_f[f_idx] = TRUE
	      }
	      else err = 1
	      f++
	      continue
	    }
	  else if (f_idx == NAME)
	    {
	      disk_name = $f
	      disk_f[f_idx] = TRUE
	      f++
	      continue
	    }
	  else if (f_idx == NOPAG)
	    {
	      disk_f[f_idx] = TRUE
	      f++
	      continue
	    }
	  else
	    {
	      if ($(f+1) == "")
		{
		  printf "%s Line %d: %s field for DISK requires an argument\n",FILENAME, NR,$f >> errfile
		  err = 1
		}
	      else 
		{
# we assume that the following field is its arg
		  disk_f[f_idx] = TRUE
		  if (f_idx == PAG) pagedev = $(f+1)
		}

	    }
# Increment field count by the ID+its arg any way.
	  f += 2
	  continue
	}
    }
# end of while

   for (i = 0; i < NREQ_DISK; i++)
     {
       if (!disk_f[i])
	 {
	   printf "%s Line %d: Mandatory field(s) %d in DISK record missing\n", FILENAME, NR,i >> errfile
	   err = 1
	 }
     }
   if (disk_f[NOPAG] && disk_f[PAG])
     {
       printf "%s Line %d: %s and %s are mutually exclusive\n", FILENAME, NR, PAGETO_KWD, NOPAGER_KWD >> errfile
       err = 1
     }
     if (err) return ""

# We have a valid record
     info = ""
     if (disk_f[NOPAG]) info = NOPAGER_KWD
     else if (disk_f[PAG]) info = PAGETO_KWD " " pagedev

     if (ctl_name == "") info = info " " DEF_CTLR
     else info = info " " ctl_name
#     printf "%d: disk_name = %s info = %s\n", NR, disk_name, info >> errfile
     printdev(disk_name, $2, info)
     return disk_name

}
# process_raid_record()
#   If the raid record is valid, it outputs the record
#   calling printdev().
#
# Returns: Name of raid device - if valid record
#          Null otherwise

function process_raid_record()
{
#  printf "%d: validate_raid\n", NR >> errfile
  if (!valid_cbs($2))
    return ("")

  raid_rec_init()

# We processed the RAID keyword (field 1) and the CBS entry (field 2)
# So start from 3rd field in the input record

  f = 3
  err = 0
  ctl_name = ""
  page_dev = ""
  while (f <= NF)
    {
      if ($f == "ID") f_idx = ID
      else if ($f == "SW") f_idx = SW
      else if ($f == "LV") f_idx = LV
      else if ($f == "DC") f_idx = DC
      else if ($f == "SID") f_idx = SID
      else if ($f == "BC") f_idx = BC
      else if ($f == NOPAGER_KWD) f_idx = NOPAG
      else if ($f == PAGETO_KWD) f_idx = PAG
      else if (index($f, "RAID") == 1) f_idx = NAME
      else if (index($f, CTLR_KWD) == 1) f_idx = CTL
      else 
	{
	  printf "%s Line %d: Unknown field %s in RAID record\n", FILENAME, NR, $f >> errfile
	  return ""
	}
      if (raid_f[f_idx])
	{
	  printf "%s Line %d: Duplicate entry for %s field in RAID record\n", FILENAME, NR,$f >> errfile
	  return ""
	}
      else
	{

# The following processing counts on the fact that
# all fields except CTLR and RAID name require one argument
	  if (f_idx == CTL)
	    {
	      ctl_name = $f
	      if (valid_ctlr(ctl_name))
	      {
		      raid_f[f_idx] = TRUE
	      }
	      else err = 1
	      f++
	      continue
	    }
	  else if (f_idx == NAME)
	    {
	      raid_name = $f
	      raid_f[f_idx] = TRUE
	      f++
	      continue
	    }
	  else if (f_idx == NOPAG)
	    {
	      raid_f[f_idx] = TRUE
	      f++
	      continue
	    }
	  else
	    {
	      if ($(f+1) == "")
		{
		  printf "%s Line %d: %s field for RAID requires an argument\n",FILENAME, NR,$f >> errfile
		  err = 1
		}
	      else 
		{
# we assume that the following field is its arg
		  raid_f[f_idx] = TRUE
		  if (f_idx == PAG) pagedev = $(f+1)
		}

	    }
# Increment field count by the ID+its arg any way.
	  f += 2
	  continue
	}

    }
   for (i = 0; i < NREQ_RAID; i++)
     {
       if (!raid_f[i])
	 {
	   printf "%s Line %d: Mandatory field(s) %d in RAID record missing\n", FILENAME, NR,i >> errfile
	   err = 1
	 }
     }
   if (raid_f[NOPAG] && raid_f[PAG])
     {
       printf "%s Line %d: %s and %s are mutually exclusive\n", FILENAME, NR, PAGETO_KWD, NOPAGER_KWD >> errfile
       err = 1
     }
     if (err) return ""
# we have a valid RAID record
     info = ""
     if (raid_f[NOPAG]) info = NOPAGER_KWD
     else if (raid_f[PAG]) info = PAGETO_KWD " " pagedev

     if (ctl_name == "") info = info " " DEF_CTLR
     else info = info " " ctl_name
#     printf "%d: raid_name = %s info = %s\n", NR, raid_name, info >> errfile
     printdev(raid_name, $2, info)
     return raid_name
  
}
#
# Process the device configuration (Main Loop)
{
	if ($1 == TAPE_KWD || $1 == DISK_KWD || 
	    $1 == RAID_KWD || $1 == ENET_KWD || 
	    $1 == MIO_KWD || $1 == SIO_KWD || 
	    $1 == MDC_KWD || $1 == HIPPI_KWD || 
	    $1 == CABROW_KWD )
	{
#		Check sanity of record 
		if ($1 == RAID_KWD || $1 == DISK_KWD || $1 == TAPE_KWD)
		{
			if ($1 == RAID_KWD)
			{
				minval = RAID_REC_MIN;
				maxval = RAID_REC_MAX;
			}
			else if ($1 == DISK_KWD) 
			{
				minval =  DISK_REC_MIN;
				maxval =  DISK_REC_MAX;
			}
			else
			{
				minval =  TAPE_REC_MIN;
				maxval =  TAPE_REC_MAX;
                        }
			if ((NF < minval) || (NF > maxval))
			{
				printf "%s Line %d: Incorrect %s entry\n",FILENAME, NR,$1 >errfile
				next
			}
		}
		else
		{
			if (NF != ecount[$1]) 
			{
				printf "%s Line %d: Incorrect %s entry\n",FILENAME,NR,$1 >errfile
				next
			}
		}
# Process each device info and append a default controller if needed

		if ($1 == TAPE_KWD)
		{
		  process_tape_record()
		}
		else if ($1 == DISK_KWD)
		{
		  process_disk_record()
		}
		else if ($1 == RAID_KWD)
		{
		  process_raid_record()

		}
		else if ($1 == ENET_KWD)
		{
		  if (valid_cbs($2))
		      printdev($1, $2, "")
		}
		else if ($1 == MIO_KWD || $1 == SIO_KWD || $1 == HIPPI_KWD)
		{
		  if (valid_cbs($2))
			printdev($1, $2, $3)
		}
	 	else if($1 == MDC_KWD)
		{
		  if (valid_cbs($2))
		      {

			mdc_info = ""
			for (f = 3; f <= NF; f++)
			  mdc_info = mdc_info " " $f
			printdev($1, $2, mdc_info)
		      }
		}
		else if ($1 == CABROW_KWD)
		  {
# Note that we don't want to validate the CAB_ROWS keyword for 
# validity of CBS, since CAB_ROWS is specified for the entire
# system and not for a slot.
		    nrows = sprintf("%d", $2);
		    if (nrows <= 0 || nrows > MAXROWS)
		      {
			printf "%s Line %d: CAB_ROWS can't be more than %d\n", FILENAME, NR, MAXROWS >> errfile
			next
		      }
		    printf "CABINET %d BP %s S %s %s\n", SPECIAL_CAB, SPECIAL_BP,SPECIAL_SLOT, $0			
		  }
	}
	next
}
/^$/
{
# No action for blank lines
}
