# Filename: cat.awk
#
# Input file 
# 	sorted and generated by devinfo.awk
#
#   CABINET <cabNum> BP <A|B|C|D> S <slotNum> <DeviceDescription>
#   Sample entries:
#
#  			 Fields
#	   1      2    3    4   5   6     7        8
# 
#	CABINET   00   BP   A   S   00   RAID5 
#	CABINET   00   BP   A   S   02   MIO      I04
#	CABINET   00   BP   A   S   07   ENET 
#	CABINET   00   BP   D   S   03   DISK      MAXTOR1240
#	CABINET   00   BP   D   S   03   RAID5 
#	CABINET   00   BP   D   S   03   TAPE      DAT
#       CABINET   01   BP   D   S   03   RAID3     CTLR1
#       CABINET   01   BP   D   S   03   TAPE      DAT         CTLR1
#	CABINET   01   BP   D   S   03   SIO      S04
#
# Output file format
#	Concatenates multiple device names for a given slot on a bp
#	and outputs them one per cabinet
#
# Modification History:
#
# $Log: cat.awk,v $
# Revision 1.7  1995/03/28  00:07:01  chowell
# chowell - added a version string
#
# Revision 1.6  1994/09/17  01:04:32  mouli
# Fixed the bug in outputing the controller info for the
# last record. The processing in END takes care of it.
#
# Revision 1.5  1994/08/19  20:49:21  mouli
# Added support for MDC, SCSI-16 and 2-row. Added more error
# checking for generating a consistent file.
#
# Revision 1.3  1993/12/20  19:41:20  mouli
# Added support for NOPAGER and PAGE_TO keywords.
#
# Revision 1.2  1993/03/18  21:23:36  mouli
# 1. Modified to output cable label numbers instead of backplane numbers.
# 2. Added outputting the 8th field of the input file which could be one of
#    {device brand name, fab-rev, symbolic name}
#
# Revision 1.1  1993/03/12  01:31:18  mouli
# Initial revision
#
# 	03/09/93	mouli	Initial version
#
##############################################################################
#
# What string:
#
# @(#) cat.awk DIAG_REL_2.0 Thu Jan  5 12:00:18 PST 1995
#
##############################################################################

BEGIN {
# Initialize variables and keywords
	lastcab = -1
	lastbp = ""
	lasts = -1
# MAXCTLR is the max number of controllers (defined in config support)
# in the system. The SIO (scsi-16) can have this maximum
# whereas MIO can have only one.
        MAXCTLR = 2
        MAXCTLR_ON_MIO = 1
        CTLR_KWD = "CTLR"
        ENET_KWD = "ENET"
        MIO_KWD  = "MIO"
        MDC_KWD  = "MDC"
        HIPPI_KWD = "HIPPI"
        SIO_KWD   = "SIO"
        TRUE = 1
        FALSE = 0
# first to indicate that the first record is processed
	first = TRUE
	initvars()
}
# initvars()
#   Initialize the scsi device string for each controller
#   initialize the booleans that keep track of expansion boards
#   
# Return values: None
#
function initvars()
{
        initctlstr()
        has_hippi = FALSE
        has_mio   = FALSE
        has_sio   = FALSE
	has_mdc   = FALSE
        has_enet  = FALSE
}
#
# initctlstr()
#   For each controller, initialize the string of devices attached to it
#   Controller name/device string is an associative array
# Return values: None
#
function initctlstr()
{
  for (i = 0; i < MAXCTLR; i++)
    {
      name = sprintf("%s%d", CTLR_KWD, i);
      ctlstr[name] = ""
    }
}
# checkio(tc, tb, ts)
#   For the given cabinet, backplane, slot, this function checks
#   if it has conflicting io devices specified. 
#   If there is a conflicting device
#   combination it outputs an error message in to the error message
#   file specified at the time of invocation. The caller of the awk
#   script should check the error file for appropriate action.
#   The following are conflicting devices on the same slot: 
#            MIO and HIPPI, MIO and MDC, MIO and SIO, 
#            SIO and HIPPI, SIO and MDC, MDC and HIPPI
#            ENET without an MIO/SIO 
#            MIO slot with more than MAXCTLR_ON_MIO
#            SIO slot with more than MAXCTLR
#            
#  Return values: None
#               
#  Variables used:
#      has_mio, has_hippi, has_sio, has_mdc, has_enet
#
# 


function checkio(tc, tb, ts)
{	
#   printf "checkio: NR %d has_mio %d has_hippi %d has_sio %d has_mdc %d has_enet %d\n",    NR, has_mio, has_hippi, has_sio, has_mdc, has_enet >> errfile
   if (has_mio && has_hippi)
      printf "Incorrect entry: %02d%s%02d has both MIO and HIPPI\n", tc,tb,ts >> errfile
   if (has_mio && has_mdc)
      printf "Incorrect entry: %02d%s%02d has both MIO and MDC\n", tc,tb,ts >> errfile
   if (has_mio && has_sio)
     printf "Incorrect entry: %02d%s%02d has both MIO and SIO\n", tc,tb,ts >> errfile
   checkctlr(tc, tb, ts)
   if (has_hippi && has_mdc)
      printf "Incorrect entry: %02d%s%02d has both HIPPI and MDC\n", tc,tb,ts >> errfile
   if (has_hippi && has_sio)
      printf "Incorrect entry: %02d%s%02d has both HIPPI and SIO\n", tc,tb,ts >> Errfile
   if (has_mdc && has_sio)
      printf "Incorrect entry: %02d%s%02d has both MDC and SIO\n", tc,tb,ts >> errfile
   if (has_enet && (!has_mio) && (!has_sio))
      printf "Incorrect entry: %02d%s%02d has ENET but no MIO/SIO\n", tc,tb,ts >> errfile


}
# checkctlr(tc, tb, ts)
#   For a given cab/bp/slot, it checks the conflicts with
#   number of controllers with certain expansion boards. If there
#   are conflicts. it outputs error messages to the file specified
#   with parameter 'errfile' when this awk script is invoked.
#   If controllers/scsi devices are specified without an MIO/SIO, it
#   flags it as an error
#
#   Return values: None
#
#   Variables used:
#     ctlstr[] -  associative array of device strings for controllers
#     has_mio, has_sio - (boolean flags)
#
function checkctlr(tc, tb, ts)
{
#  printf "checkctlr() NR: %d has_mio = %d has_sio = %d, has_mdc = %d, has_hippi = %d, has_enet = %d\n", NR, has_mio, has_sio, has_mdc, has_hippi, has_enet >> errfile
  ctlr_cnt = 0
  for (i in ctlstr)
    {
      if (ctlstr[i] != "") 
	{
#	  printf "NR: %d %s: %s\n",NR, i, ctlstr[i] >> errfile
	  ++ctlr_cnt;
        }
    }
  if (has_mio)
    {
      if (ctlr_cnt > MAXCTLR_ON_MIO)
	printf "Incorrect entry: %02d%s%02d MIO with more than %d controller(s)\n", tc,tb,ts, MAXCTLR_ON_MIO >> errfile
    }
  else if (has_sio)
    {
      if (ctlr_cnt > MAXCTLR)
	printf "Incorrect entry: %02d%s%02d SIO with more than %d controller(s)\n", tc,tb,ts, MAXCTLR_ON_MIO >> errfile
    }
  else if (ctlr_cnt != 0)
    {
	printf "Incorrect entry: %02d%s%02d has devices without an MIO/SIO\n", tc,tb,ts >> errfile      
#        printf "has_mio %d has_sio %d ctlr_cnt = %d\n", has_mio, has_sio, ctlr_cnt >> errfile
    }
}
# process_record(isnew)
#  isnew - boolean
#          TRUE (nonzero) indicates that the input record
#          is for a different cab/bp/slot than the previous one
#          FALSE (zero) indicates that the input record that we process
#          has the same cab/bp/slot as for the last record.
#
#  The function processes the input record. It registers the
#  expansion ports the slot has and builds a string of devices specified 
#  on different controllers.
#
#  Return values: None
#  Variables used:
#     has_mio, has_mdc, has_sio, has_hippi, has_enet - Boolean flags
#     ctlstr[] - Device string for each controller
#     isctl
function process_record(isnew)
{	
  if (isnew)
    {
      initvars()

# different from the last cab,bp,slot. so output the
# CAB,BP,SLOT info

      for (f = 1; f < 7; f++)
        printf "%s ", $f
    }
  if ($7 == "MIO") has_mio = TRUE
  else if ($7 == "SIO") has_sio = TRUE
  else if ($7 == "HIPPI") has_hippi = TRUE
  else if ($7 == "MDC") has_mdc = TRUE
  else if ($7 == "ENET") has_enet = TRUE
  for (f = 8; f <= NF; f++)
  {

      isctl = index($f, CTLR_KWD)
      if (isctl == 1)
      {
# Gather device strings for this controller.
# 

 	 for (p = 7; p < f; p++)
	  {
	       ctlstr[$f] = ctlstr[$f] " " $p
	  }
       }
    }
   if (!isctl) 
   {
# Entries that do not have controller keyword.
#			    printf "NR: %d not ctlr keyword\n", NR >> errfile
       for (f = 7; f <= NF; f++)
	 printf " %s", $f
    }
     isctl = 0

}

# Main action routine for all input records
# Action for all input records 
{
	if ($1 == "CABINET")
	{
		curcab = int($2)
		curbp = $4
		curs = int($6)
		#printf "Cab = %d Bp = %s Slot = %d\n", curcab, curbp, curs >> errfile
		if ((curcab == lastcab) && (curbp  == lastbp)  && (curs   == lasts))
		{
# Current input record has the cab,bp,slot spec as the one
# we processed last.
		  process_record(FALSE)
		}
		else 
		{
# We've encountered a different cab/backplane/slot
#  		    printf "\nNR: %d new rec has_mio %d has_sio %d has_mdc %d has_hippi %d has_enet %d\n", NR, has_mio, has_sio,has_mdc,has_hippi,has_enet >> errfile

			if (!first)
			{
# Not the 1st input record and end of a unique cab/bp/slot entry
# Check any conflicting i/o devices for the last cab,bp,slot

			    checkio(lastcab, lastbp,lasts)

# output the device string attached to each controller
# controller/device string is an associative array

                             for (i in ctlstr)
                             {
                                if (ctlstr[i] != "")
                                   printf " %s %s", i, ctlstr[i]
                             }
 			     printf "\n"
			}
			else {
# first input record.
                               first = FALSE
                        }
# We just processed one unique cab/bp/slot/entry
# process the new record.

                        process_record(TRUE)

		}
# Save current cabinet,backplane,slot

		lastcab = curcab
		lastbp = curbp
		lasts = curs
	}
}
END {
# We may still have ctlr string for the last
# record. If there is any, output it
	for (i in ctlstr)
		if (ctlstr[i] != "") printf " %s %s", i, ctlstr[i]

# output a newline at end of input processing
	printf "\n"
}
